<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Intl\Tests\Data\Provider;

use Symfony\Component\Intl\Data\Provider\LanguageDataProvider;
use Symfony\Component\Intl\Intl;
use Symfony\Component\Intl\Locale;

/**
 * @author Bernhard Schussek <bschussek@gmail.com>
 */
abstract class AbstractLanguageDataProviderTest extends AbstractDataProviderTest
{
    // The below arrays document the state of the ICU data bundled with this package.

    protected static $languages = array(
        'aa',
        'ab',
        'ace',
        'ach',
        'ada',
        'ady',
        'ae',
        'aeb',
        'af',
        'afh',
        'agq',
        'ain',
        'ak',
        'akk',
        'akz',
        'ale',
        'aln',
        'alt',
        'am',
        'an',
        'ang',
        'anp',
        'ar',
        'ar_001',
        'arc',
        'arn',
        'aro',
        'arp',
        'arq',
        'arw',
        'ary',
        'arz',
        'as',
        'asa',
        'ase',
        'ast',
        'av',
        'avk',
        'awa',
        'ay',
        'az',
        'azb',
        'ba',
        'bal',
        'ban',
        'bar',
        'bas',
        'bax',
        'bbc',
        'bbj',
        'be',
        'bej',
        'bem',
        'bew',
        'bez',
        'bfd',
        'bfq',
        'bg',
        'bho',
        'bi',
        'bik',
        'bin',
        'bjn',
        'bkm',
        'bla',
        'bm',
        'bn',
        'bo',
        'bpy',
        'bqi',
        'br',
        'bra',
        'brh',
        'brx',
        'bs',
        'bss',
        'bua',
        'bug',
        'bum',
        'byn',
        'byv',
        'ca',
        'cad',
        'car',
        'cay',
        'cch',
        'ce',
        'ceb',
        'cgg',
        'ch',
        'chb',
        'chg',
        'chk',
        'chm',
        'chn',
        'cho',
        'chp',
        'chr',
        'chy',
        'ckb',
        'co',
        'cop',
        'cps',
        'cr',
        'crh',
        'cs',
        'csb',
        'cu',
        'cv',
        'cy',
        'da',
        'dak',
        'dar',
        'dav',
        'de',
        'de_AT',
        'de_CH',
        'del',
        'den',
        'dgr',
        'din',
        'dje',
        'doi',
        'dsb',
        'dtp',
        'dua',
        'dum',
        'dv',
        'dyo',
        'dyu',
        'dz',
        'dzg',
        'ebu',
        'ee',
        'efi',
        'egl',
        'egy',
        'eka',
        'el',
        'elx',
        'en',
        'en_AU',
        'en_CA',
        'en_GB',
        'en_US',
        'enm',
        'eo',
        'es',
        'es_419',
        'es_ES',
        'es_MX',
        'esu',
        'et',
        'eu',
        'ewo',
        'ext',
        'fa',
        'fan',
        'fat',
        'ff',
        'fi',
        'fil',
        'fit',
        'fj',
        'fo',
        'fon',
        'fr',
        'fr_CA',
        'fr_CH',
        'frc',
        'frm',
        'fro',
        'frp',
        'frr',
        'frs',
        'fur',
        'fy',
        'ga',
        'gaa',
        'gag',
        'gan',
        'gay',
        'gba',
        'gbz',
        'gd',
        'gez',
        'gil',
        'gl',
        'glk',
        'gmh',
        'gn',
        'goh',
        'gom',
        'gon',
        'gor',
        'got',
        'grb',
        'grc',
        'gsw',
        'gu',
        'guc',
        'gur',
        'guz',
        'gv',
        'gwi',
        'ha',
        'hai',
        'hak',
        'haw',
        'he',
        'hi',
        'hif',
        'hil',
        'hit',
        'hmn',
        'ho',
        'hr',
        'hsb',
        'hsn',
        'ht',
        'hu',
        'hup',
        'hy',
        'hz',
        'ia',
        'iba',
        'ibb',
        'id',
        'ie',
        'ig',
        'ii',
        'ik',
        'ilo',
        'inh',
        'io',
        'is',
        'it',
        'iu',
        'izh',
        'ja',
        'jam',
        'jbo',
        'jgo',
        'jmc',
        'jpr',
        'jrb',
        'jut',
        'jv',
        'ka',
        'kaa',
        'kab',
        'kac',
        'kaj',
        'kam',
        'kaw',
        'kbd',
        'kbl',
        'kcg',
        'kde',
        'kea',
        'ken',
        'kfo',
        'kg',
        'kgp',
        'kha',
        'kho',
        'khq',
        'khw',
        'ki',
        'kiu',
        'kj',
        'kk',
        'kkj',
        'kl',
        'kln',
        'km',
        'kmb',
        'kn',
        'ko',
        'koi',
        'kok',
        'kos',
        'kpe',
        'kr',
        'krc',
        'kri',
        'krj',
        'krl',
        'kru',
        'ks',
        'ksb',
        'ksf',
        'ksh',
        'ku',
        'kum',
        'kut',
        'kv',
        'kw',
        'ky',
        'la',
        'lad',
        'lag',
        'lah',
        'lam',
        'lb',
        'lez',
        'lfn',
        'lg',
        'li',
        'lij',
        'liv',
        'lkt',
        'lmo',
        'ln',
        'lo',
        'lol',
        'loz',
        'lt',
        'ltg',
        'lu',
        'lua',
        'lui',
        'lun',
        'luo',
        'lus',
        'luy',
        'lv',
        'lzh',
        'lzz',
        'mad',
        'maf',
        'mag',
        'mai',
        'mak',
        'man',
        'mas',
        'mde',
        'mdf',
        'mdr',
        'men',
        'mer',
        'mfe',
        'mg',
        'mga',
        'mgh',
        'mgo',
        'mh',
        'mi',
        'mic',
        'min',
        'mk',
        'ml',
        'mn',
        'mnc',
        'mni',
        'moh',
        'mos',
        'mr',
        'mrj',
        'ms',
        'mt',
        'mua',
        'mul',
        'mus',
        'mwl',
        'mwr',
        'mwv',
        'my',
        'mye',
        'myv',
        'mzn',
        'na',
        'nan',
        'nap',
        'naq',
        'nb',
        'nd',
        'nds',
        'ne',
        'new',
        'ng',
        'nia',
        'niu',
        'njo',
        'nl',
        'nl_BE',
        'nmg',
        'nn',
        'nnh',
        'no',
        'nog',
        'non',
        'nov',
        'nqo',
        'nr',
        'nso',
        'nus',
        'nv',
        'nwc',
        'ny',
        'nym',
        'nyn',
        'nyo',
        'nzi',
        'oc',
        'oj',
        'om',
        'or',
        'os',
        'osa',
        'ota',
        'pa',
        'pag',
        'pal',
        'pam',
        'pap',
        'pau',
        'pcd',
        'pdc',
        'pdt',
        'peo',
        'pfl',
        'phn',
        'pi',
        'pl',
        'pms',
        'pnt',
        'pon',
        'prg',
        'pro',
        'ps',
        'pt',
        'pt_BR',
        'pt_PT',
        'qu',
        'quc',
        'qug',
        'raj',
        'rap',
        'rar',
        'rgn',
        'rif',
        'rm',
        'rn',
        'ro',
        'ro_MD',
        'rof',
        'rom',
        'root',
        'rtm',
        'ru',
        'rue',
        'rug',
        'rup',
        'rw',
        'rwk',
        'sa',
        'sad',
        'sah',
        'sam',
        'saq',
        'sas',
        'sat',
        'saz',
        'sba',
        'sbp',
        'sc',
        'scn',
        'sco',
        'sd',
        'sdc',
        'se',
        'see',
        'seh',
        'sei',
        'sel',
        'ses',
        'sg',
        'sga',
        'sgs',
        'sh',
        'shi',
        'shn',
        'shu',
        'si',
        'sid',
        'sk',
        'sl',
        'sli',
        'sly',
        'sm',
        'sma',
        'smj',
        'smn',
        'sms',
        'sn',
        'snk',
        'so',
        'sog',
        'sq',
        'sr',
        'srn',
        'srr',
        'ss',
        'ssy',
        'st',
        'stq',
        'su',
        'suk',
        'sus',
        'sux',
        'sv',
        'sw',
        'swb',
        'swc',
        'syc',
        'syr',
        'szl',
        'ta',
        'tcy',
        'te',
        'tem',
        'teo',
        'ter',
        'tet',
        'tg',
        'th',
        'ti',
        'tig',
        'tiv',
        'tk',
        'tkl',
        'tkr',
        'tl',
        'tlh',
        'tli',
        'tly',
        'tmh',
        'tn',
        'to',
        'tog',
        'tpi',
        'tr',
        'tru',
        'trv',
        'ts',
        'tsd',
        'tsi',
        'tt',
        'ttt',
        'tum',
        'tvl',
        'tw',
        'twq',
        'ty',
        'tyv',
        'tzm',
        'udm',
        'ug',
        'uga',
        'uk',
        'umb',
        'und',
        'ur',
        'uz',
        'vai',
        've',
        'vec',
        'vep',
        'vi',
        'vls',
        'vmf',
        'vo',
        'vot',
        'vro',
        'vun',
        'wa',
        'wae',
        'wal',
        'war',
        'was',
        'wo',
        'wuu',
        'xal',
        'xh',
        'xmf',
        'xog',
        'yao',
        'yap',
        'yav',
        'ybb',
        'yi',
        'yo',
        'yrl',
        'yue',
        'za',
        'zap',
        'zbl',
        'zea',
        'zen',
        'zgh',
        'zh',
        'zh_Hans',
        'zh_Hant',
        'zu',
        'zun',
        'zxx',
        'zza',
    );

    protected static $alpha2ToAlpha3 = array(
        'aa' => 'aar',
        'ab' => 'abk',
        'ae' => 'ave',
        'af' => 'afr',
        'ak' => 'aka',
        'am' => 'amh',
        'an' => 'arg',
        'ar' => 'ara',
        'as' => 'asm',
        'av' => 'ava',
        'ay' => 'aym',
        'az' => 'aze',
        'ba' => 'bak',
        'be' => 'bel',
        'bg' => 'bul',
        'bh' => 'bih',
        'bi' => 'bis',
        'bm' => 'bam',
        'bn' => 'ben',
        'bo' => 'bod',
        'br' => 'bre',
        'bs' => 'bos',
        'ca' => 'cat',
        'ce' => 'che',
        'ch' => 'cha',
        'co' => 'cos',
        'cr' => 'cre',
        'cs' => 'ces',
        'cu' => 'chu',
        'cv' => 'chv',
        'cy' => 'cym',
        'da' => 'dan',
        'de' => 'deu',
        'dv' => 'div',
        'dz' => 'dzo',
        'ee' => 'ewe',
        'el' => 'ell',
        'en' => 'eng',
        'eo' => 'epo',
        'es' => 'spa',
        'et' => 'est',
        'eu' => 'eus',
        'fa' => 'fas',
        'ff' => 'ful',
        'fi' => 'fin',
        'fj' => 'fij',
        'fo' => 'fao',
        'fr' => 'fra',
        'fy' => 'fry',
        'ga' => 'gle',
        'gd' => 'gla',
        'gl' => 'glg',
        'gn' => 'grn',
        'gu' => 'guj',
        'gv' => 'glv',
        'ha' => 'hau',
        'he' => 'heb',
        'hi' => 'hin',
        'ho' => 'hmo',
        'hr' => 'hrv',
        'ht' => 'hat',
        'hu' => 'hun',
        'hy' => 'hye',
        'hz' => 'her',
        'ia' => 'ina',
        'id' => 'ind',
        'ie' => 'ile',
        'ig' => 'ibo',
        'ii' => 'iii',
        'ik' => 'ipk',
        'io' => 'ido',
        'is' => 'isl',
        'it' => 'ita',
        'iu' => 'iku',
        'ja' => 'jpn',
        'jv' => 'jav',
        'ka' => 'kat',
        'kg' => 'kon',
        'ki' => 'kik',
        'kj' => 'kua',
        'kk' => 'kaz',
        'kl' => 'kal',
        'km' => 'khm',
        'kn' => 'kan',
        'ko' => 'kor',
        'kr' => 'kau',
        'ks' => 'kas',
        'ku' => 'kur',
        'kv' => 'kom',
        'kw' => 'cor',
        'ky' => 'kir',
        'la' => 'lat',
        'lb' => 'ltz',
        'lg' => 'lug',
        'li' => 'lim',
        'ln' => 'lin',
        'lo' => 'lao',
        'lt' => 'lit',
        'lu' => 'lub',
        'lv' => 'lav',
        'mg' => 'mlg',
        'mh' => 'mah',
        'mi' => 'mri',
        'mk' => 'mkd',
        'ml' => 'mal',
        'mn' => 'mon',
        'mr' => 'mar',
        'ms' => 'msa',
        'mt' => 'mlt',
        'my' => 'mya',
        'na' => 'nau',
        'nb' => 'nob',
        'nd' => 'nde',
        'ne' => 'nep',
        'ng' => 'ndo',
        'nl' => 'nld',
        'nn' => 'nno',
        'nr' => 'nbl',
        'nv' => 'nav',
        'ny' => 'nya',
        'oc' => 'oci',
        'oj' => 'oji',
        'om' => 'orm',
        'or' => 'ori',
        'os' => 'oss',
        'pa' => 'pan',
        'pi' => 'pli',
        'pl' => 'pol',
        'ps' => 'pus',
        'pt' => 'por',
        'qu' => 'que',
        'rm' => 'roh',
        'rn' => 'run',
        'ro' => 'ron',
        'ru' => 'rus',
        'rw' => 'kin',
        'sa' => 'san',
        'sc' => 'srd',
        'sd' => 'snd',
        'se' => 'sme',
        'sg' => 'sag',
        'si' => 'sin',
        'sk' => 'slk',
        'sl' => 'slv',
        'sm' => 'smo',
        'sn' => 'sna',
        'so' => 'som',
        'sq' => 'sqi',
        'sr' => 'srp',
        'ss' => 'ssw',
        'st' => 'sot',
        'su' => 'sun',
        'sv' => 'swe',
        'sw' => 'swa',
        'ta' => 'tam',
        'te' => 'tel',
        'tg' => 'tgk',
        'th' => 'tha',
        'ti' => 'tir',
        'tk' => 'tuk',
        'tn' => 'tsn',
        'to' => 'ton',
        'tr' => 'tur',
        'ts' => 'tso',
        'tt' => 'tat',
        'ty' => 'tah',
        'ug' => 'uig',
        'uk' => 'ukr',
        'ur' => 'urd',
        'uz' => 'uzb',
        've' => 'ven',
        'vi' => 'vie',
        'vo' => 'vol',
        'wa' => 'wln',
        'wo' => 'wol',
        'xh' => 'xho',
        'yi' => 'yid',
        'yo' => 'yor',
        'za' => 'zha',
        'zh' => 'zho',
        'zu' => 'zul',
    );

    /**
     * @var LanguageDataProvider
     */
    protected $dataProvider;

    protected function setUp()
    {
        parent::setUp();

        $this->dataProvider = new LanguageDataProvider(
            $this->getDataDirectory().'/'.Intl::LANGUAGE_DIR,
            $this->createEntryReader()
        );
    }

    abstract protected function getDataDirectory();

    public function testGetLanguages()
    {
        $this->assertEquals(static::$languages, $this->dataProvider->getLanguages());
    }

    /**
     * @dataProvider provideLocales
     */
    public function testGetNames($displayLocale)
    {
        $languages = array_keys($this->dataProvider->getNames($displayLocale));

        sort($languages);

        $this->assertEquals(static::$languages, $languages);
    }

    public function testGetNamesDefaultLocale()
    {
        Locale::setDefault('de_AT');

        $this->assertSame(
            $this->dataProvider->getNames('de_AT'),
            $this->dataProvider->getNames()
        );
    }

    /**
     * @dataProvider provideLocaleAliases
     */
    public function testGetNamesSupportsAliases($alias, $ofLocale)
    {
        // Can't use assertSame(), because some aliases contain scripts with
        // different collation (=order of output) than their aliased locale
        // e.g. sr_Latn_ME => sr_ME
        $this->assertEquals(
            $this->dataProvider->getNames($ofLocale),
            $this->dataProvider->getNames($alias)
        );
    }

    /**
     * @dataProvider provideLocales
     */
    public function testGetName($displayLocale)
    {
        $names = $this->dataProvider->getNames($displayLocale);

        foreach ($names as $language => $name) {
            $this->assertSame($name, $this->dataProvider->getName($language, $displayLocale));
        }
    }

    public function testGetNameDefaultLocale()
    {
        Locale::setDefault('de_AT');

        $names = $this->dataProvider->getNames('de_AT');

        foreach ($names as $language => $name) {
            $this->assertSame($name, $this->dataProvider->getName($language));
        }
    }

    public function provideLanguagesWithAlpha3Equivalent()
    {
        return array_map(
            function ($value) { return array($value); },
            array_keys(static::$alpha2ToAlpha3)
        );
    }

    /**
     * @dataProvider provideLanguagesWithAlpha3Equivalent
     */
    public function testGetAlpha3Code($language)
    {
        $this->assertSame(static::$alpha2ToAlpha3[$language], $this->dataProvider->getAlpha3Code($language));
    }

    public function provideLanguagesWithoutAlpha3Equivalent()
    {
        return array_map(
            function ($value) { return array($value); },
            array_diff(static::$languages, array_keys(static::$alpha2ToAlpha3))
        );
    }

    /**
     * @dataProvider provideLanguagesWithoutAlpha3Equivalent
     * @expectedException \Symfony\Component\Intl\Exception\MissingResourceException
     */
    public function testGetAlpha3CodeFailsIfNoAlpha3Equivalent($currency)
    {
        $this->dataProvider->getAlpha3Code($currency);
    }
}
