blueprint = {}

function blueprint.Init(boundsSize)
	-- Blueprint functions for base scanning and printing
	-- All materials used are stored and given a numerical id
	blueprint.blocksTable = {}
	blueprint.nextBlockId = 1

	-- The actual layout of the blocks
	blueprint.layoutTableBackground = {}
	blueprint.layoutTableForeground = {}

	-- Any tile mods applied to the blocks
	blueprint.layoutTableBackgroundMods = {}
	blueprint.layoutTableForegroundMods = {}

	-- The placement of objects
	blueprint.objectTable = {}

	-- A copy of the config table, so it doesn't go out of scope
	blueprint.configTable = {}

	-- A copy of the bounding box size, for print previewing
	blueprint.boundingBoxSize = copyTable(boundsSize)
end

------------------------------------------------------------------------------------
-- blocksTable
------------------------------------------------------------------------------------
-- Get the id for a given material name
function blueprint.blockId(matName)
	if (matName == nil) then
		return nil
	end

	local id = blueprint.blocksTable[matName]

	if (id == nil) then
		blueprint.blocksTable[matName] = blueprint.nextBlockId
		id = blueprint.nextBlockId
		blueprint.nextBlockId = blueprint.nextBlockId + 1
	end

	return id
end

-- Get the material name for a given id
function blueprint.materialFromId(id)
	for _name, _id in pairs(blueprint.blocksTable) do
		if (_id == id) then
			return _name
		end
	end

	return nil
end
------------------------------------------------------------------------------------
-- layoutTables
------------------------------------------------------------------------------------
-- Set the block id for a given material and position
function blueprint.setBlock(x, y, matName, layer)
	if (layer == "background") then
		if (blueprint.layoutTableBackground[y] == nil) then
			blueprint.layoutTableBackground[y] = {}
		end
		blueprint.layoutTableBackground[y][x] = blueprint.blockId(matName)
	elseif (layer == "foreground") then
		if (blueprint.layoutTableForeground[y] == nil) then
			blueprint.layoutTableForeground[y] = {}
		end
		blueprint.layoutTableForeground[y][x] = blueprint.blockId(matName)
	end
end

-- Get the material name of the block at a given position
function blueprint.getBlock(x, y, layer)
	local id = nil
	if (layer == "background") then
		if (blueprint.layoutTableBackground[y] ~= nil) then
			id = blueprint.layoutTableBackground[y][x]
		end
	elseif (layer == "foreground") then
		if (blueprint.layoutTableForeground[y] ~= nil) then
			id = blueprint.layoutTableForeground[y][x]
		end
	else
		return nil
	end

	if (id == nil) then
		return nil
	end

	local matName = blueprint.materialFromId(id)
	
	return matName
end
------------------------------------------------------------------------------------
-- modTables
------------------------------------------------------------------------------------
-- Set the mod id for a given material and position
function blueprint.setMod(x, y, modName, layer)
	if (layer == "background") then
		if (blueprint.layoutTableBackgroundMods[y] == nil) then
			blueprint.layoutTableBackgroundMods[y] = {}
		end
		blueprint.layoutTableBackgroundMods[y][x] = blueprint.blockId(modName)
	elseif (layer == "foreground") then
		if (blueprint.layoutTableForegroundMods[y] == nil) then
			blueprint.layoutTableForegroundMods[y] = {}
		end
		blueprint.layoutTableForegroundMods[y][x] = blueprint.blockId(modName)
	end
end

-- Get the material name of the block at a given position
function blueprint.getMod(x, y, layer)
	local id = nil
	if (layer == "background") then
		if (blueprint.layoutTableBackgroundMods[y] ~= nil) then
			id = blueprint.layoutTableBackgroundMods[y][x]
		end
	elseif (layer == "foreground") then
		if (blueprint.layoutTableForegroundMods[y] ~= nil) then
			id = blueprint.layoutTableForegroundMods[y][x]
		end
	else
		return nil
	end

	if (id == nil) then
		return nil
	end

	local modName = blueprint.materialFromId(id)
	
	return modName
end
------------------------------------------------------------------------------------
-- objectTables
------------------------------------------------------------------------------------
-- Store an obect at a given position
function blueprint.setObject(x, y, id)
	-- init
	objectParameterTable = {}
	
	if (id == nil) then
		blueprint.objectTable[y][x] = objectParameterTable
		return
	end

	-- read values from world
	objectParameterTable.name = world.entityName(id)
	objectParameterTable.facing = world.callScriptedEntity(id, "entity.direction")
	if (objectParameterTable.facing == nil) then
		objectParameterTable.facing = 1
	end
	if (world.containerSize(id)) then
		objectParameterTable.contents = world.containerTakeAll(id)
	end
	if (world.callScriptedEntity(id, "object.outputNodeCount") > 0) then
		-- wiring would go here but is currently impossible
	end
	-- create entry
	if (blueprint.objectTable[y] == nil) then
		blueprint.objectTable[y] = {}
	end
	blueprint.objectTable[y][x] = objectParameterTable
end
------------------------------------------------------------------------------------
-- Serialisation
------------------------------------------------------------------------------------
-- returns a table for immediate config use
function blueprint.toConfigTable()
	local tbl = {}
	
	tbl.boundingBoxSize = blueprint.boundingBoxSize
	tbl.nextBlockId = blueprint.nextBlockId
	tbl.blocksTable = blueprint.blocksTable
	tbl.layoutTableBackground = blueprint.layoutTableBackground
	tbl.layoutTableForeground = blueprint.layoutTableForeground
	tbl.layoutTableBackgroundMods = blueprint.layoutTableBackgroundMods
	tbl.layoutTableForegroundMods = blueprint.layoutTableForegroundMods
	
	tbl.objectTable = blueprint.objectTable
	
	return { miab_basestore_blueprint = tbl }
end

-- populates the config table and points the relevant stuff at it
function blueprint.fromEntityConfig()
	blueprint.configTable = config.getParameter("miab_basestore_blueprint", nil)

	if (blueprint.configTable ~= nil) then
		blueprint.boundingBoxSize = blueprint.configTable.boundingBoxSize
		blueprint.nextBlockId = blueprint.configTable.nextBlockId
		blueprint.blocksTable = blueprint.configTable.blocksTable
		blueprint.layoutTableBackground = blueprint.configTable.layoutTableBackground
		blueprint.layoutTableForeground = blueprint.configTable.layoutTableForeground
		blueprint.layoutTableBackgroundMods = blueprint.configTable.layoutTableBackgroundMods
		blueprint.layoutTableForegroundMods = blueprint.configTable.layoutTableForegroundMods
		
		blueprint.objectTable = blueprint.configTable.objectTable
		
		for _mat, _id in pairs(blueprint.blocksTable) do
			blueprint.blocksTable[_mat] = tonumber(_id)
		end
		for _y, _tbl in pairs(blueprint.layoutTableBackground) do
			for _x, _id in pairs(_tbl) do
				blueprint.layoutTableBackground[_y][_x] = tonumber(_id)
			end
		end
		for _y, _tbl in pairs(blueprint.layoutTableForeground) do
			for _x, _id in pairs(_tbl) do
				blueprint.layoutTableForeground[_y][_x] = tonumber(_id)
			end
		end
		for _y, _tbl in pairs(blueprint.layoutTableBackgroundMods) do
			for _x, _id in pairs(_tbl) do
				blueprint.layoutTableBackgroundMods[_y][_x] = tonumber(_id)
			end
		end
		for _y, _tbl in pairs(blueprint.layoutTableForegroundMods) do
			for _x, _id in pairs(_tbl) do
				blueprint.layoutTableForegroundMods[_y][_x] = tonumber(_id)
			end
		end
	end
end
		
function blueprint.DumpJSON()
	-- see: http://jsonlint.com/ for json validation tool
	local serialised = "\n"
	serialised = serialised .. "-------------------------------------------\nRecipe serialisation begins\nReplace \"!itemname\" and save in \"pilch_sciencestation/custom\" folder as \"itemname.recipe\"\n-------------------------------------------\n"
	serialised = serialised .. "{\n\t\"input\" : [\n\t\t{ \"item\" : \"titaniumbar\", \"count\" : 10 },\n\t\t{ \"item\" : \"money\", \"count\" : 400 }\n\t],\n"
	serialised = serialised .. "\t\"output\" : {\n\t\t\"item\" : \"!itemname\", \"count\" : 1\n\t},\n"
	serialised = serialised .. "\t\"groups\" : [ \"pilch_sciencestation\", \"furniture\", \"objects\", \"all\" ]\n}\n"
	serialised = serialised .. "-------------------------------------------\nRecipe serialisation ends\n-------------------------------------------\n"
	serialised = serialised .. "Config patch serialisation begins\nReplace \"!itemname\" and insert after the first line in \"pilch_sciencestation/player.config.patch\"\n-------------------------------------------\n"
	serialised = serialised .. "\t{\n\t\t\"op\" : \"add\",\n\t\t\"path\" : \"/defaultBlueprints/tier1/-\",\n\t\t\"value\" : { \"item\" : \"!itemname\" }\n\t},\n"
	serialised = serialised .. "-------------------------------------------\nConfig patch serialisation ends\n"
	serialised = serialised .. "-------------------------------------------\nObject serialisation begins\nReplace \"!itemname\", \"!longdescription\" and \"!shortdescription\" and save in \"pilch_sciencestation/custom\" folder as \"itemname.object\"\n-------------------------------------------\n"
	serialised = serialised .. "{\n\t\"objectName\" : \"!itemname\",\n\t\"rarity\" : \"Common\",\n\t\"description\" : \"!longdescription\",\n\t\"shortdescription\" : \"!shortdescription\",\n"
	serialised = serialised .. "\t\"race\" : \"generic\",\n\t\"category\" : \"tool\",\n\t\"price\" : 1,\n\t\"printable\" : false,\n\n"
	serialised = serialised .. "\t\"inventoryIcon\" : \"/objects/basestorage/miab_basestore_printer/miab_basestore_printer_icon.png\",\n"
	serialised = serialised .. "\t\"orientations\" : [\n\t\t{\n\t\t\t\"dualImage\" : \"/objects/basestorage/miab_basestore_printer/miab_basestore_printer.png:<color>.<frame>\",\n"
	serialised = serialised .. "\t\t\t\"imagePosition\" : [-8, 0],\n\t\t\t\"frames\" : 5,\n\t\t\t\"animationCycle\" : 1,\n\t\t\t\"spaceScan\" : 0.1\n\t\t}\n\t],\n\n"
	serialised = serialised .. "\t\"animation\" : \"/objects/basestorage/miab_basestore_printer/miab_basestore_printer.animation\",\n"
	serialised = serialised .. "\t\t\"animationParts\" : {\n\t\t\"normal_operation_image\" : \"/objects/basestorage/miab_basestore_printer/miab_basestore_printer.png\"\n\t},\n"
	serialised = serialised .. "\t\"animationPosition\" : [-8, 0],\n\n"
	serialised = serialised .. "\t\"scripts\" : [\n\t\t\"/objects/basestorage/miab_basestore_printer/miab_basestore_print_activator.lua\",\n"
	serialised = serialised .. "\t\t\"/scripts/basestorage/miab_basestore_printer.lua\",\n\t\t\"/scripts/basestorage/miab_basestore_blueprint.lua\",\n"
	serialised = serialised .. "\t\t\"/scripts/basestorage/miab_basestore_util.lua\"\n\t],\n\t\"scriptDelta\" : 5,\n\n\t\"miab_printer_offset\" : [1, 0],\n\n"

	serialised = serialised .. "\t\"miab_basestore_blueprint\" : {\n"
	serialised = serialised .. "\t\t\"boundingBoxSize\" : [\n"
	serialised = serialised .. "\t\t\t" .. tostring(blueprint.boundingBoxSize[1]) .. ", " .. tostring(blueprint.boundingBoxSize[2]) .. "\n"
	serialised = serialised .. "\t\t],\n"
	serialised = serialised .. tableToJSON("\t\t", "blocksTable", blueprint.blocksTable, ",")
	serialised = serialised .. "\t\t\"nextBlockId\" : " .. blueprint.nextBlockId .. ",\n"
	serialised = serialised .. tableToJSON("\t\t", "layoutTableBackground", blueprint.layoutTableBackground, ",")
	serialised = serialised .. tableToJSON("\t\t", "layoutTableForeground", blueprint.layoutTableForeground, ",")
	serialised = serialised .. tableToJSON("\t\t", "layoutTableBackgroundMods", blueprint.layoutTableBackgroundMods, ",")
	serialised = serialised .. tableToJSON("\t\t", "layoutTableForegroundMods", blueprint.layoutTableForegroundMods, ",")
	serialised = serialised .. tableToJSON("\t\t", "objectTable", blueprint.objectTable, "")
	serialised = serialised .. "\t}\n}\n"
	serialised = serialised .. "-------------------------------------------\nObject serialisation ends\n-------------------------------------------"
	sb.logInfo(serialised)
end

function tableToJSON(prefix, name, val, suffix)
	local serialised = ""
	if (type(val) == "boolean") then
		if (val == true) then
			serialised = serialised .. prefix .. "\"" .. name .. "\" : true" .. suffix .. "\n"
		else
			serialised = serialised .. prefix .. "\"" .. name .. "\" : false" .. suffix .. "\n"
		end
	elseif (type(val) == "number") then
		serialised = serialised .. prefix .. "\"" .. name .. "\" : " .. tostring(val) .. suffix .. "\n"
	elseif (type(val) == "string") then
		serialised = serialised .. prefix .. "\"" .. name .. "\" : \"" .. val .. "\"" .. suffix .. "\n"
	elseif (type(val) == "table") then
		local _k, _v
		local itemCount, itemCurrent
		itemCount = blueprint.tablelength(val)
		itemCurrent = 0
		serialised = serialised .. prefix .. "\"" .. name .. "\" : {" .. "\n"
		for _k, _v in pairs(val) do
			itemCurrent = itemCurrent + 1
			if (itemCount == itemCurrent) then
				serialised = serialised .. tableToJSON(prefix .. "\t", _k, _v, "")
			else
				serialised = serialised .. tableToJSON(prefix .. "\t", _k, _v, ",")
			end
		end
		serialised = serialised .. prefix .. "}" .. suffix .. "\n"
	else
		serialised = serialised .. prefix .. "\"" .. name .. "\" : \"Serialisation error: not basic type or table\"" .. suffix .. "\n"
	end
	return serialised
end

------------------------------------------------------------------------------------
-- UTIL
------------------------------------------------------------------------------------
function blueprint.isInsideBoundingBox(pos, bBox)
	-- checks if pos x,y coordinates are inside the boundary box bBox defined by x1,y1,x2,y2
	local distBL = world.distance(pos, { bBox[1], bBox[2] })
	local distTR = world.distance(pos, { bBox[3], bBox[4] })
	
	if (distBL[1] < 0) then return false end
	if (distBL[2] < 0) then return false end
	if (distTR[1] > 0) then return false end
	if (distTR[2] > 0) then return false end

	return true
end

function blueprint.clearBlock(pos, layer)
	world.damageTiles({pos}, layer, pos, "blockish", 10000, 0)
end

function blueprint.tablelength(T)
  local count = 0
  for _ in pairs(T) do count = count + 1 end
  return count
end

