function init(args)
	object.setInteractive(true)

	-- wiring
	object.setAllOutputNodes(false)
	onNodeConnectionChange()
	self.corners = {}
	
	-- Animation
	self.animationColour = nil
	self.defaultAnimColour = config.getParameter("miab_defaultAnimColour", "green")
	setDefaultAnimation()
end

function onInteraction(args)
	if (object.isInputNodeConnected(0)) then
		return
	end
	
	onActivation()
end

function onActivation(args)
	if (not readerBusy()) then
		-- Options for read process
		local readerOptions = {}
		readerOptions.readerPosition = object.toAbsolutePosition({ 0.0, 0.0 })
		readerOptions.spawnPrinterPosition = object.toAbsolutePosition({ 0, 4 })
		readerOptions.breakStuff = config.getParameter("miab_breakStuff", true)
		readerOptions.clearOnly = config.getParameter("miab_clearOnly", false)
		readerOptions.plotJSON = config.getParameter("miab_dumpJSON", false)
		readerOptions.printerCount = config.getParameter("miab_printerCount", 1)
		readerOptions.areaToIgnore = config.getParameter("miab_fixed_area_to_ignore_during_scan_bounding_box", nil) -- [left, bottom, right, top]
		readerOptions.animationDuration = 3
		
		readerOptions.areaToScan = defineAreaToScan()
		
		if (readerOptions.areaToScan ~= nil) then
			-- start reading process
			readStart(readerOptions)
		end
	end
end

function update(dt)
	-- this needs to be polled until done
	readModule()
end

function defineAreaToScan()
	local scannerSize = config.getParameter("miab_scannerSize", {3.0, 2.0})
	local scannerOrigin = config.getParameter("miab_scannerOrigin", {3.0, 1.0})
	local fixedAreaToScan = config.getParameter("miab_fixed_area_to_scan_bounding_box", nil) -- [left, bottom, right, top]
	
	local bounds = {0, 0, 0, 0}
	local pos1 = entity.position() -- default corners are the corners of the scanner itself
	local pos2 = entity.position()
	local ents = object.getOutputNodeIds(0)
	local useScannerAsCorner = true

	-- use a predifined area to scan
	if (fixedAreaToScan) then
		return fixedAreaToScan
	end
	
	-- handle different ammounts of receivers and wired or not
	local corners = countConnectedCorners(entity.id())
	if (corners == 0) then
		return nil
	elseif (corners == 1) then
		-- use one corner and the scanner itself
		pos2 = world.entityPosition(self.corners[1])
	else
		-- use two corners
		pos1 = world.entityPosition(self.corners[1])
		pos2 = world.entityPosition(self.corners[2])
		useScannerAsCorner = false
	end
	
	-- find smallest X Corner coordinate of the two corners
	-- and use that for the left of the bounding box
	-- use the bigger one for right side
	-- using world.distance()
	local dist = world.distance(pos2, pos1)
	local xMax = dist[1]
	local yMax = dist[2]
	if (xMax > 1) then
		bounds[1] = pos1[1] -- left
		bounds[3] = pos2[1] -- right
		scannerCornerX = "left" -- if one of the corners is the scanner it is the left one
	elseif (xMax < 1) then
		bounds[1] = pos2[1] -- left
		bounds[3] = pos1[1] -- right
		scannerCornerX = "right" -- if one of the corners is the scanner it is the right one
	else
		return nil
	end

	-- find smallest Y Corner coordinate of the two corners
	-- and use that for the bottom of the bounding box
	-- use the bigger one for top side
	if (pos1[2] < pos2[2]) then
		bounds[2] = pos1[2] -- down
		bounds[4] = pos2[2] -- up
	elseif (pos1[2] > pos2[2]) then
		bounds[2] = pos2[2] -- down
		bounds[4] = pos1[2] -- up
	else
		return nil
	end
	
	if (useScannerAsCorner) then
		-- we have to shift a bit accoring to the size of the scanner
		-- X coordinate shift due to scanner size
		if (scannerCornerX == "left") then
			-- if the scanner is left corner
			bounds[1] = bounds[1] + (scannerSize[1] - scannerOrigin[1])
		elseif (scannerCornerX == "right") then
			-- if the scanner is right corner
			bounds[3] = bounds[3] + (scannerSize[1] - scannerOrigin[1]) - scannerSize[1] +1
		else
			return nil
		end
		-- Y coordinate is not of interest. we allways use the origin of the scanner.
		-- not adapting Y will never leed to the scanner beeing eaten by itself
	end

	-- so far we are directly ON the corners.
	-- we sorted out which one is bottom left and which one is top right
	-- we also shifted one of the corners if it was the scanner itself accoring to the size of the scanner in order not to eat the scanner itself

	-- we now apply an offset for the box.
	-- we dont want to scan AT the corners, but inside of the corners (in X direction "inside")
	-- Offsets for scanning objects:
	-- 1 block right, 0 blocks up for bottom left
	-- 1 block left, 0 blocks down for top right
	bounds[1] = bounds[1] + 1.0 --( <- offset from bottom left
	bounds[2] = bounds[2] + 0.0 --(
	bounds[3] = bounds[3] - 1.0 --( <- offset from top right corner_2
	bounds[4] = bounds[4] - 0.0 --(

	-- although this is already checked we check again if the box has a zero or negativ area
	-- "negative" meaing that what is left and right (or up and down) might be labled wrong
	dist = world.distance({bounds[3], bounds[4]}, {bounds[1], bounds[2]})
	xMax = dist[1]
	yMax = dist[2]
	if (xMax < 1) or (yMax < 1) then
		return nil
	end
	return bounds
end

----- Wiring -----
-- returns the number of scanning corners connected to our output node
function countConnectedCorners(entID)
	local count = 0
	for nodeID, j in pairs(world.callScriptedEntity(entID, "object.getOutputNodeIds", 0)) do
		if (world.entityName(nodeID) == "miab_basestore_receiver") then
			count = count + 1
			self.corners[count] = nodeID
		else
			count = count + countConnectedCorners(nodeID)
		end
	end
	return count
end

function onNodeConnectionChange(args)
	object.setInteractive(not object.isInputNodeConnected(0))
	if (object.isInputNodeConnected(0)) then
		onInputNodeChange({ level = object.getInputNodeLevel(0) })
	end
end

function onInputNodeChange(args)
	if (args.level) and (not self.level) then
		onActivation()
	end
	self.level = args.level
end

----- Animation -----
function setDefaultAnimation()
	if (self.defaultAnimColour == "green") then
		setGreenAnimation()
	elseif (self.defaultAnimColour == "yellow") then
		setYellowAnimation()
	elseif (self.defaultAnimColour == "red") then
		setRedAnimation()
	end
end

function setGreenAnimation ()
	if (self.animationColour ~= "green") then
		animator.setAnimationState("DisplayState", "Green_State")
		self.animationColour = "green"
	end
end

function setYellowAnimation ()
	if (self.animationColour ~= "yellow") then
		animator.setAnimationState("DisplayState", "Yellow_State")
		self.animationColour = "yellow"
	end
end

function setRedAnimation ()
	if (self.animationColour ~= "red") then
		animator.setAnimationState("DisplayState", "Red_State")
		self.animationColour = "red"
	end
end

function setBlueprintAnimation ()
	if (self.animationColour ~= "blueprint") then	
		animator.setAnimationState("DisplayState", "Blueprint_State")
		self.animationColour = "blueprint"
	end
end