// 全局配置文件
var AudioConfig = {
    opacity: 0.90,            // 透明度
    color: '255,255,255',   // 颜色
    shadowBlur: 15,          // 发光程度
    rotation: 0,             // 旋转角度
    isRing: true,            // 显示环
    isInnerRing: true,      // 显示内环
    isOuterRing: true,      // 显示外环
    offsetX: 0.5,            // X坐标偏移
    offsetY: 0.5,            // Y坐标偏移
    isClickOffset: false,  // 鼠标坐标偏移
    radius: 0.5,             // 半径
    amplitude: 5,            // 振幅
    distance: 0,             // 内外环距离
    isLineTo: false,        // 是否连线
    pointNum: 120,           // 点的数量
    lineWidth: 5,            // 线宽
    isBall: true,            // 显示小球
    ballSpacer: 3,           // 小球间隔
    ballSize: 3,             // 小球大小
    isDate: true,            // 是否显示日期
    dateStyle: 1,            // 时间显示风格
    fontSize: 60,            // 字体大小
    language: 'zh_cn',      // 语言
    audioSamples: []         // 音频数组
};

!function ($) {

    // 优化动画
    // =================================================================================================================

    window.requestAnimFrame = (function () {
        return window.requestAnimationFrame
            || window.webkitRequestAnimationFrame
            || window.mozRequestAnimationFrame
            || function (callback) {
                window.setTimeout(callback, 1000 / 60);
            };
    })();

    // 创建设置canvas并添加到body
    // =================================================================================================================

    /* 设置canvas的高度和宽度 */
    function setCanvasSize() {
        canvasWidth = canvas.width = window.innerWidth || document.documentElement.clientWidth || document.body.clientWidth;
        canvasHeight = canvas.height = window.innerHeight || document.documentElement.clientHeight || document.body.clientHeight;
        minLength = Math.min(canvasWidth, canvasHeight);
        originX = canvasWidth * AudioConfig.offsetX;
        originY = canvasHeight * AudioConfig.offsetY;
    }

    /* 设置context属性 */
    function setContext() {
        context.fillStyle = 'rgb(' + AudioConfig.color + ')';
        // 线条属性
        context.lineWidth = AudioConfig.lineWidth;
        context.strokeStyle = 'rgb(' + AudioConfig.color + ')';
        // 阴影属性
        context.shadowColor = 'rgb(' + AudioConfig.color + ')';
        context.shadowBlur = AudioConfig.shadowBlur;
        // 文字属性
        context.font = AudioConfig.fontSize + 'px 微软雅黑';
        context.textAlign = 'center';
        context.textBaseline = 'middle';
    }

    // 设置canvas基本属性
    var canvas = document.createElement('canvas');
    var canvasWidth, canvasHeight;  // canvas宽度和高度
    var originX, originY;           // 原点位置
    var minLength = 300;            // 最小长度
    setCanvasSize();
    canvas.id = 'canvas-audio'; // canvas ID
    $(canvas).css({
        'position':'fixed',
        'top':0,
        'left':0,
        'z-index': -1,
        'opacity': AudioConfig.opacity
    });
    $('body').append(canvas);

    // 设置context基本属性
    var context = canvas.getContext('2d');
    setContext();
    moment.lang(AudioConfig.language);

    // 根据audioSamples生成音频圆环
    // =================================================================================================================

    // 圆环点的集合
    var ringArray = [];  // 圆环音频数组
    var ballArray = [];  // 小球音频数组
    var pointArray1, pointArray2, pointArray3 = [];  // 坐标数组

    // 上一个点的集合
    var lastAudioSamples = [];
    for(var i = 0; i < 120; i++) {
        lastAudioSamples[i] = 0;
    }

    var rotationAngle = 0;  // 旋转角度


    /* 根据数量提取音频取样值 */
    function getRingArray(num) {
        var AudioArray = [].concat(AudioConfig.audioSamples || 0);
        var max = AudioArray.length - num;
        var isfirst = true;
        for(var i = 0; i < max; i++) {
            if(isfirst) {
                AudioArray.shift();
                isfirst = false;
            } else {
                AudioArray.pop();
                isfirst = true;
            }
        }
        return AudioArray;
    }

    /* 根据间隔提取音频取样值 */
    function getBallArray(num) {
        var AudioArray = [];
        for(var i = 0; i < 120; i += num) {
            AudioArray.push(AudioConfig.audioSamples[i] || 0);
        }
        return AudioArray;
    }

    /* 获取点的角度 */
    function getDeg(index, point, angle) {
        return Math.PI / 180 * ( 360 / point ) * ( index + angle / 3 )
    }

    /* 获取点的坐标 */
    function getXY(radius, deg) {
        return {
            'x': Math.cos(deg) * radius + originX,
            'y': Math.sin(deg) * radius + originY
        };
    }

    /* 生成点的集合 */
    function setPoint() {
        // 清空之前点的集合
        pointArray1 = [];
        pointArray2 = [];
        pointArray3 = [];
        // 重新生成点的集合
        for(var i = 0; i < ringArray.length; i++) {
            var deg = getDeg(i, ringArray.length, rotationAngle);  // 该点的度数
            // 获取当前audioSamples[i]的值
            var audioValue = AudioConfig.audioSamples[i] ? AudioConfig.audioSamples[i] : 0;
            /** 若小于上一个点的音频取样值，则取上一个值，
             * decline保证音频取样值减弱时，audioValue平缓下降而不是保持原状
             * 当然，decline越小过渡越缓慢，越大过渡越迅速（甚至失效）
             */
            var decline = 0.25;  // 衰弱值，建议0 - 0.3之内
            audioValue = Math.max(audioValue, lastAudioSamples[i] - decline);
            audioValue = Math.min(audioValue, 1.5);  // 溢出部分按值1.25处理
            lastAudioSamples[i] = audioValue;  // 更新lastAudioSamples
            // 设置当前点对应的半径
            var radius1 = AudioConfig.radius * ( minLength / 2 ) - AudioConfig.distance - audioValue * ( AudioConfig.amplitude * 15 );  // 第一层圆环半径
            var radius2 = AudioConfig.radius * ( minLength / 2 ) + AudioConfig.distance + audioValue * ( AudioConfig.amplitude * 15 );  // 第二层圆环半径
            // 获取当前点对应的坐标
            var point1 = getXY(radius1, deg);
            var point2 = getXY(radius2, deg);
            // 将点的坐标储存至坐标数组
            pointArray1.push({'x': point1.x, 'y': point1.y});
            pointArray2.push({'x': point2.x, 'y': point2.y});
        }

        // 第三层点的集合
        for(var i = 0; i < ballArray.length; i++) {
            var deg = getDeg(i, ballArray.length, rotationAngle);  // 该点的度数
            // 获取当前ballArray[i]的值
            var audioValue = AudioConfig.audioSamples[i] ? AudioConfig.audioSamples[i] : 0;
            audioValue = Math.min(audioValue, 1);  // 溢出部分按值1处理
            var radius = AudioConfig.radius * ( minLength / 2 ) + ( AudioConfig.distance + 50 ) + audioValue * 75;  // 第三层圆环半径
            // 获取点的坐标并储存至坐标数组
            var point = getXY(radius, deg);
            pointArray3.push({'x': point.x, 'y': point.y});
        }

        // 角度偏移
        if(AudioConfig.rotation) {
            rotationAngle += AudioConfig.rotation;
            if(rotationAngle >= 360 && rotationAngle <= 360) {
                rotationAngle = 0;
            }
        }
    }

    /* 绘制音频圆环 */
    function drawRing() {
        if(AudioConfig.isRing) {
            if(AudioConfig.isInnerRing) {
                // 第一层圆环连线
                context.beginPath();
                context.moveTo(pointArray1[0].x, pointArray1[0].y);
                for (var i = 0; i < ringArray.length; i++) {
                    context.lineTo(pointArray1[i].x, pointArray1[i].y);
                }
                context.closePath();
                context.stroke();
            }
            if(AudioConfig.isOuterRing) {
                // 第二层圆环连线
                context.beginPath();
                context.moveTo(pointArray2[0].x, pointArray2[0].y);
                for (var i = 0; i < ringArray.length; i++) {
                    context.lineTo(pointArray2[i].x, pointArray2[i].y);
                }
                context.closePath();
                context.stroke();
            }
        }
        if(AudioConfig.isLineTo) {
            // 内外环连线
            context.beginPath();
            for (var i = 0; i < ringArray.length; i++) {
                context.moveTo(pointArray1[i].x, pointArray1[i].y);
                context.lineTo(pointArray2[i].x, pointArray2[i].y);
            }
            context.closePath();
            context.stroke();
        }
        if(AudioConfig.isBall) {
            //第三层小球圆环
            for (var i = 0; i < ballArray.length; i++) {
                context.beginPath();
                context.arc(pointArray3[i].x - 0.5, pointArray3[i].y - 0.5, AudioConfig.ballSize, 0, 360, false);
                context.fill();
                context.closePath();
            }
        }
    }

    // 绘制时间
    // =================================================================================================================

    /**
     * YYYY：年 MM；月  DD：日
     * HH：小时(二十四小时制) hh；小时(十二小时制) mm；分钟 ss：秒
     * a：时间段 dddd；星期
     */

    /* 获取时间 */
    function getDate() {
        switch (AudioConfig.dateStyle) {
            case 1:
                return moment().format('hh:mm:ss a');
            case 2:
                return moment().format('HH:mm:ss');
        }
    }

    /* 绘制时间 */
    function drawDate () {
        // moment.lang('zh-cn');
        context.fillText( getDate(), originX, originY - AudioConfig.fontSize / 2);
        context.font = AudioConfig.fontSize / 2 + 'px 微软雅黑';
        context.fillText( moment().format('MMM Do YY dddd'), originX, originY + AudioConfig.fontSize / 4);
    }

    // 交互事件
    // =================================================================================================================

    // 尺寸改变
    $(window).resize(function () {
        setCanvasSize()
    });

    // 点击事件
    $('body').click(function (e) {
        if(AudioConfig.isClickOffset) {
            x = e.clientX || canvasWidth * AudioConfig.offsetX;
            y = e.clientY || canvasHeight * AudioConfig.offsetY;
            AudioConfig.offsetX = x / canvasWidth;
            AudioConfig.offsetY = y / canvasHeight;
        }
    });

    // 音频监听
    // audioArray.length = 128; audioArray值的范围一般为0 - 1,燃一点曲子< 5，很难超过10
    // =================================================================================================================
    function wallpaperAudioListener(audioArray) {
        AudioConfig.audioSamples = audioArray;
        $(canvas).css('opacity', AudioConfig.opacity);
        setContext();  // 重新设置context属性
        // 获取偏移坐标
        originX = canvasWidth * AudioConfig.offsetX;
        originY = canvasHeight * AudioConfig.offsetY;
        // 绘制音频圆环
        context.clearRect(0, 0, canvasWidth, canvasHeight);
        ringArray = getRingArray(AudioConfig.pointNum);
        ballArray = getBallArray(AudioConfig.ballSpacer);
        setPoint(); drawRing();
        moment.lang(AudioConfig.language);
        AudioConfig.isDate && drawDate();  // 绘制时间
    }
    window.wallpaperRegisterAudioListener && window.wallpaperRegisterAudioListener(wallpaperAudioListener);

/*
    // 测试test
    // =================================================================================================================


    for(var i; i < 128; i++) {
        AudioConfig.audioSamples[i] = 0;
    }
    ringArray = getRingArray(AudioConfig.pointNum);
    ballArray = getBallArray(AudioConfig.ballSpacer);
    setPoint();
    drawRing();
*/

}(window.jQuery);


