﻿using UnityEngine;
using UnityEditor;
using System.IO;
using System.Reflection;

[InitializeOnLoad]
public static class FileExtensions {

	public static string prevGuid;

	private static GUIStyle labelStyle;

	const int ONE_COLUMN = 0;
	const int TWO_COLUMNS = 1;

	static FileExtensions() {
		EditorApplication.projectWindowItemOnGUI += ListItemOnGUI;
	}

	public static void DrawOutline(Rect rect, string text, GUIStyle style, Color outColor, Color inColor, float size) {
		float halfSize = size * 0.5F;

		style.normal.textColor = outColor;

		rect.x -= halfSize;
		GUI.Label(rect, text, style);

		rect.x += size;
		GUI.Label(rect, text, style);

		rect.x -= halfSize;
		rect.y -= halfSize;
		GUI.Label(rect, text, style);

		rect.y += size;
		GUI.Label(rect, text, style);

		rect.y -= halfSize;
		style.normal.textColor = inColor;
		GUI.Label(rect, text, style);
	}

	private static void ListItemOnGUI(string guid, Rect rect) {
		if (prevGuid != guid) {
			prevGuid = guid;

			if ( labelStyle == null) {
				labelStyle = new GUIStyle(EditorStyles.boldLabel);
			}

			string assetPath = AssetDatabase.GUIDToAssetPath(guid);
			Object obj = AssetDatabase.LoadAssetAtPath<UnityEngine.Object>(assetPath);

			if (obj != null && AssetDatabase.IsMainAsset(obj) && !IsDirectory(obj)) {
				string extension = Path.GetExtension(assetPath);
				if (BigThumbnails) {
					DrawOutline(rect, extension, labelStyle, Color.black, Color.white, 2.0f);
				}
				else {
					string fileName = Path.GetFileNameWithoutExtension(assetPath);

					GUIContent content = new GUIContent(fileName);
					if (EditorGUIUtility.isProSkin) {
						labelStyle.normal.textColor = Color.grey;
					}
					else {
						labelStyle.normal.textColor = Color.white;
					}

					#if UNITY_5_5
						GUI.Label(rect.Translate(16f + GUI.skin.label.CalcSize(content).x, 1f), extension, labelStyle);
					#else
						GUI.Label(rect.Translate(16f + GUI.skin.label.CalcSize(content).x - GUI.skin.label.margin.left * (TWO_COLUMNS - ColumnsCount), 1f), extension, labelStyle);
					#endif
				}
			}
		}
	}

	private static bool BigThumbnails {
		get {
			return ColumnsCount == TWO_COLUMNS && ListAreaGridSize > 16f;
		}
	}

	private static int ColumnsCount {
		get {
			var projectWindow = GetProjectWindow();
			var viewModeFieldInfo = projectWindow.GetType().GetField("m_ViewMode", BindingFlags.Instance | BindingFlags.NonPublic);
			return (int)viewModeFieldInfo.GetValue(projectWindow);
		}
	}

	private static float ListAreaGridSize {
		get {
			var projectWindow = GetProjectWindow();
			var propertyInfo = projectWindow.GetType().GetProperty("listAreaGridSize", BindingFlags.Instance | BindingFlags.Public);
			return (float)propertyInfo.GetValue(projectWindow, null);
		}
	}

	private static EditorWindow GetProjectWindow() {
		if (EditorWindow.focusedWindow != null && EditorWindow.focusedWindow.titleContent.text == "Project") {
			return EditorWindow.focusedWindow;
		}

		return GetExistingWindowByName("Project");
	}

	private static EditorWindow GetExistingWindowByName(string name) {
		EditorWindow[] windows = Resources.FindObjectsOfTypeAll<EditorWindow>();
		foreach (var item in windows) {
			if (item.titleContent.text == name) {
				return item;
			}
		}

		return default(EditorWindow);
	}

	private static Rect Translate(this Rect rect, float x, float y) {
		rect.x += x;
		rect.y += y;

		return rect;
	}

	private static bool IsDirectory(Object obj) {
		return obj != null && obj is DefaultAsset && !AssetDatabase.IsForeignAsset(obj);
	}
}
