/*
 * Decompiled with CFR 0.152.
 */
package mobac.program.download;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import mobac.exceptions.DownloadFailedException;
import mobac.exceptions.UnrecoverableDownloadException;
import mobac.program.interfaces.HttpMapSource;
import mobac.program.interfaces.MapSourceListener;
import mobac.program.interfaces.MapSpace;
import mobac.program.model.Settings;
import mobac.program.model.TileImageType;
import mobac.program.tilestore.TileStore;
import mobac.program.tilestore.TileStoreEntry;
import mobac.utilities.Utilities;
import mobac.utilities.stream.ThrottledInputStream;
import org.apache.log4j.Logger;

public class TileDownLoader {
    public static String ACCEPT = "text/html, image/png, image/jpeg, image/gif, */*;q=0.1";
    private static Logger log;
    private static Settings settings;

    public static byte[] getImage(int x, int y, int zoom, HttpMapSource mapSource) throws IOException, InterruptedException, UnrecoverableDownloadException {
        MapSpace mapSpace = mapSource.getMapSpace();
        int maxTileIndex = mapSpace.getMaxPixels(zoom) / mapSpace.getTileSize();
        if (x > maxTileIndex) {
            throw new RuntimeException("Invalid tile index x=" + x + " for zoom " + zoom);
        }
        if (y > maxTileIndex) {
            throw new RuntimeException("Invalid tile index y=" + y + " for zoom " + zoom);
        }
        TileStore ts = TileStore.getInstance();
        Settings s = Settings.getInstance();
        TileStoreEntry tile = null;
        if (s.tileStoreEnabled) {
            tile = ts.getTile(x, y, zoom, mapSource);
            boolean expired = TileDownLoader.isTileExpired(tile);
            if (tile != null) {
                if (expired) {
                    log.trace("Expired: " + mapSource.getName() + " " + tile);
                } else {
                    log.trace("Tile of map source " + mapSource.getName() + " used from tilestore");
                    byte[] data = tile.getData();
                    TileDownLoader.notifyCachedTileUsed(data.length);
                    return data;
                }
            }
        }
        byte[] data = null;
        if (tile == null) {
            data = TileDownLoader.downloadTileAndUpdateStore(x, y, zoom, mapSource);
            TileDownLoader.notifyTileDownloaded(data.length);
        } else {
            byte[] updatedData = TileDownLoader.updateStoredTile(tile, mapSource);
            if (updatedData != null) {
                data = updatedData;
                TileDownLoader.notifyTileDownloaded(data.length);
            } else {
                data = tile.getData();
                TileDownLoader.notifyCachedTileUsed(data.length);
            }
        }
        return data;
    }

    private static void notifyTileDownloaded(int size) {
        if (Thread.currentThread() instanceof MapSourceListener) {
            ((MapSourceListener)((Object)Thread.currentThread())).tileDownloaded(size);
        }
    }

    private static void notifyCachedTileUsed(int size) {
        if (Thread.currentThread() instanceof MapSourceListener) {
            ((MapSourceListener)((Object)Thread.currentThread())).tileLoadedFromCache(size);
        }
    }

    public static byte[] downloadTileAndUpdateStore(int x, int y, int zoom, HttpMapSource mapSource) throws UnrecoverableDownloadException, IOException, InterruptedException {
        return TileDownLoader.downloadTileAndUpdateStore(x, y, zoom, mapSource, Settings.getInstance().tileStoreEnabled);
    }

    public static byte[] downloadTileAndUpdateStore(int x, int y, int zoom, HttpMapSource mapSource, boolean useTileStore) throws UnrecoverableDownloadException, IOException, InterruptedException {
        if (zoom < 0) {
            throw new UnrecoverableDownloadException("Negative zoom!");
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(zoom, x, y);
        if (conn == null) {
            throw new UnrecoverableDownloadException("Tile x=" + x + " y=" + y + " zoom=" + zoom + " is not a valid tile in map source " + mapSource);
        }
        log.trace("Downloading " + conn.getURL());
        TileDownLoader.prepareConnection(conn);
        conn.connect();
        int code = conn.getResponseCode();
        byte[] data = TileDownLoader.loadBodyDataInBuffer(conn);
        if (code != 200) {
            throw new DownloadFailedException(conn, code);
        }
        TileDownLoader.checkContentType(conn, data);
        TileDownLoader.checkContentLength(conn, data);
        String eTag = conn.getHeaderField("ETag");
        long timeLastModified = conn.getLastModified();
        long timeExpires = conn.getExpiration();
        Utilities.checkForInterruption();
        TileImageType imageType = Utilities.getImageType(data);
        if (imageType == null) {
            throw new UnrecoverableDownloadException("The returned image is of unknown format");
        }
        if (useTileStore) {
            TileStore.getInstance().putTileData(data, x, y, zoom, mapSource, timeLastModified, timeExpires, eTag);
        }
        Utilities.checkForInterruption();
        return data;
    }

    public static byte[] updateStoredTile(TileStoreEntry tile, HttpMapSource mapSource) throws UnrecoverableDownloadException, IOException, InterruptedException {
        int x = tile.getX();
        int y = tile.getY();
        int zoom = tile.getZoom();
        HttpMapSource.TileUpdate tileUpdate = mapSource.getTileUpdate();
        switch (tileUpdate) {
            case ETag: {
                boolean unchanged = TileDownLoader.hasTileETag(tile, mapSource);
                if (!unchanged) break;
                if (log.isTraceEnabled()) {
                    log.trace("Data unchanged on server (eTag): " + mapSource + " " + tile);
                }
                return null;
            }
            case LastModified: {
                boolean isNewer = TileDownLoader.isTileNewer(tile, mapSource);
                if (isNewer) break;
                if (log.isTraceEnabled()) {
                    log.trace("Data unchanged on server (LastModified): " + mapSource + " " + tile);
                }
                return null;
            }
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(zoom, x, y);
        if (conn == null) {
            throw new UnrecoverableDownloadException("Tile x=" + x + " y=" + y + " zoom=" + zoom + " is not a valid tile in map source " + mapSource);
        }
        if (log.isTraceEnabled()) {
            log.trace(String.format("Checking %s %s", mapSource.getName(), tile));
        }
        TileDownLoader.prepareConnection(conn);
        boolean conditionalRequest = false;
        switch (tileUpdate) {
            case IfNoneMatch: {
                if (tile.geteTag() == null) break;
                conn.setRequestProperty("If-None-Match", tile.geteTag());
                conditionalRequest = true;
                break;
            }
            case IfModifiedSince: {
                if (tile.getTimeLastModified() <= 0L) break;
                conn.setIfModifiedSince(tile.getTimeLastModified());
                conditionalRequest = true;
            }
        }
        conn.connect();
        Settings s = Settings.getInstance();
        int code = conn.getResponseCode();
        if (conditionalRequest && code == 304) {
            if (s.tileStoreEnabled) {
                tile.update(conn.getExpiration());
                TileStore.getInstance().putTile(tile, mapSource);
            }
            if (log.isTraceEnabled()) {
                log.trace("Data unchanged on server: " + mapSource + " " + tile);
            }
            return null;
        }
        byte[] data = TileDownLoader.loadBodyDataInBuffer(conn);
        if (code != 200) {
            throw new DownloadFailedException(conn, code);
        }
        TileDownLoader.checkContentType(conn, data);
        TileDownLoader.checkContentLength(conn, data);
        String eTag = conn.getHeaderField("ETag");
        long timeLastModified = conn.getLastModified();
        long timeExpires = conn.getExpiration();
        Utilities.checkForInterruption();
        TileImageType imageType = Utilities.getImageType(data);
        if (imageType == null) {
            throw new UnrecoverableDownloadException("The returned image is of unknown format");
        }
        if (s.tileStoreEnabled) {
            TileStore.getInstance().putTileData(data, x, y, zoom, mapSource, timeLastModified, timeExpires, eTag);
        }
        Utilities.checkForInterruption();
        return data;
    }

    public static boolean isTileExpired(TileStoreEntry tileStoreEntry) {
        if (tileStoreEntry == null) {
            return true;
        }
        long expiredTime = tileStoreEntry.getTimeExpires();
        if (expiredTime >= 0L) {
            long maxExpirationTime = TileDownLoader.settings.tileMaxExpirationTime + tileStoreEntry.getTimeDownloaded();
            long minExpirationTime = TileDownLoader.settings.tileMinExpirationTime + tileStoreEntry.getTimeDownloaded();
            expiredTime = Math.max(minExpirationTime, Math.min(maxExpirationTime, expiredTime));
        } else {
            expiredTime = tileStoreEntry.getTimeDownloaded() + TileDownLoader.settings.tileDefaultExpirationTime;
        }
        return expiredTime < System.currentTimeMillis();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static byte[] loadBodyDataInBuffer(HttpURLConnection conn) throws IOException {
        InputStream input = conn.getInputStream();
        byte[] data = null;
        try {
            long bandwidthLimit;
            if (Thread.currentThread() instanceof MapSourceListener && (bandwidthLimit = Settings.getInstance().getBandwidthLimit()) > 0L) {
                input = new ThrottledInputStream(input);
            }
            data = Utilities.getInputBytes(input);
        }
        catch (IOException e) {
            InputStream errorIn = conn.getErrorStream();
            try {
                byte[] errData = Utilities.getInputBytes(errorIn);
                log.trace("Retrieved " + errData.length + " error bytes for a HTTP " + conn.getResponseCode());
            }
            catch (Exception ee) {
                log.debug("Error retrieving error stream content: " + e);
            }
            finally {
                Utilities.closeStream(errorIn);
            }
            throw e;
        }
        finally {
            Utilities.closeStream(input);
        }
        log.trace("Retrieved " + data.length + " bytes for a HTTP " + conn.getResponseCode());
        if (data.length == 0) {
            return null;
        }
        return data;
    }

    protected static boolean isTileNewer(TileStoreEntry tile, HttpMapSource mapSource) throws IOException {
        long oldLastModified = tile.getTimeLastModified();
        if (oldLastModified <= 0L) {
            log.warn("Tile age comparison not possible: tile in tilestore does not contain lastModified attribute");
            return true;
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(tile.getZoom(), tile.getX(), tile.getY());
        conn.setRequestMethod("HEAD");
        conn.setRequestProperty("Accept", ACCEPT);
        long newLastModified = conn.getLastModified();
        if (newLastModified == 0L) {
            return true;
        }
        return newLastModified > oldLastModified;
    }

    protected static boolean hasTileETag(TileStoreEntry tile, HttpMapSource mapSource) throws IOException {
        String eTag = tile.geteTag();
        if (eTag == null || eTag.length() == 0) {
            log.warn("ETag check not possible: tile in tilestore does not contain ETag attribute");
            return true;
        }
        HttpURLConnection conn = mapSource.getTileUrlConnection(tile.getZoom(), tile.getX(), tile.getY());
        conn.setRequestMethod("HEAD");
        conn.setRequestProperty("Accept", ACCEPT);
        String onlineETag = conn.getHeaderField("ETag");
        if (onlineETag == null || onlineETag.length() == 0) {
            return true;
        }
        return onlineETag.equals(eTag);
    }

    protected static void prepareConnection(HttpURLConnection conn) throws ProtocolException {
        conn.setRequestMethod("GET");
        Settings s = Settings.getInstance();
        conn.setConnectTimeout(1000 * s.httpConnectionTimeout);
        conn.setReadTimeout(1000 * s.httpReadTimeout);
        if (conn.getRequestProperty("User-agent") == null) {
            conn.setRequestProperty("User-agent", s.getUserAgent());
        }
        conn.setRequestProperty("Accept", ACCEPT);
    }

    protected static void checkContentType(HttpURLConnection conn, byte[] data) throws UnrecoverableDownloadException {
        String contentType = conn.getContentType();
        if (contentType != null && !(contentType = contentType.toLowerCase()).startsWith("image/")) {
            if (log.isTraceEnabled() && contentType.startsWith("text/")) {
                log.trace("Content (" + contentType + "): " + new String(data));
            }
            throw new UnrecoverableDownloadException("Content type of the loaded image is unknown: " + contentType, UnrecoverableDownloadException.ERROR_CODE_CONTENT_TYPE);
        }
    }

    protected static void checkContentLength(HttpURLConnection conn, byte[] data) throws UnrecoverableDownloadException {
        int len = conn.getContentLength();
        if (len < 0) {
            return;
        }
        if (data.length != len) {
            throw new UnrecoverableDownloadException("Content length is not as declared by the server: retrived=" + data.length + " bytes  expected-content-length=" + len + " bytes");
        }
    }

    static {
        String defaultReadTimeout = System.getProperty("sun.net.client.defaultReadTimeout");
        if (defaultReadTimeout == null) {
            System.setProperty("sun.net.client.defaultReadTimeout", "15000");
        }
        System.setProperty("http.maxConnections", "20");
        log = Logger.getLogger(TileDownLoader.class);
        settings = Settings.getInstance();
    }
}

