require "Reloading/ISReloadableWeapon"

QGWeaponManualIMNC = ISReloadableWeapon:derive("QGWeaponManualIMNC");

--************************************************************************--
--** QGWeaponManualIMNC:initialise
--**
--************************************************************************--
function QGWeaponManualIMNC:initialise()

end

--************************************************************************--
--** QGWeaponManualIMNC:new
--**
--** Constructor for the QGWeaponManualIMNC reloadable
--************************************************************************--
function QGWeaponManualIMNC:new()
	local o = {}
	--o.data = {}
	o = ISReloadableWeapon:new();
    setmetatable(o, self)
    self.__index = self
	return o;
end

--************************************************************************--
--** QGWeaponManualIMNC:isLoaded
--**
--** Returns whether or not the gun will fire when the mouse is next
--** clicked
--**
--************************************************************************--
function QGWeaponManualIMNC:isLoaded(difficulty)
	return self.currentCapacity > 0;
end

--************************************************************************--
--** QGWeaponManualIMNC:fireShot
--**
--** Action performed when a shot is fired. Should typically decrease
--** the current amount of ammo in the weapon
--**
--************************************************************************--
function QGWeaponManualIMNC:fireShot(weapon, difficulty)
	if self.currentCapacity > 0 then
		self.currentCapacity = self.currentCapacity - 1;
	end
	self.racked = 0
	self:syncReloadableToItem(weapon)
end

--************************************************************************--
--** QGWeaponManualIMNC:canReload
--**
--** Whether the character attempting to reload has the necessary
--** prerequisites to perform the reload action. Called prior to
--** the timed action and not to be confused with isReloadValid
--**
--************************************************************************--
function QGWeaponManualIMNC:canReload(chr)
	if(self.currentCapacity < self.maxCapacity
	and chr:getInventory():FindAndReturn(self.ammoType) ~= nil) then
		return true;
	end
	return false;
end

function QGWeaponManualIMNC:canUnload(chr)
	if(self.currentCapacity > 0) then
		return true;
	end
	return false;
end

--************************************************************************--
--** QGWeaponManualIMNC:isReloadValid
--**
--** Function for the TimedAction that determines whether the reload
--** action is still valid. If the player does something that should
--** interrupt the action, this should return false
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--** @return true if the action may continue to be performed
--**
--************************************************************************--
function QGWeaponManualIMNC:isReloadValid(char, square, difficulty)
	if(self.currentCapacity < self.maxCapacity
		and char:getInventory():FindAndReturn(self.ammoType) ~= nil) then
			return true;
	end
	self.reloadInProgress = false;
	return false;
end

function QGWeaponManualIM:isUnloadValid(char, square, difficulty)
	if(self.currentCapacity > 0) then
		return true;
	end
	self.unloadInProgress = false;
	return false;
end

--************************************************************************--
--** QGWeaponManualIMNC:reloadStart
--**
--** Function that should be performed upon the start of the timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--************************************************************************--
function QGWeaponManualIMNC:reloadStart(char, square, difficulty)
	self.reloadInProgress = true;
end

function QGWeaponManualIM:unloadStart(char, square, difficulty)
	self.unloadInProgress = true;
end

--************************************************************************--
--** QGWeaponManualIMNC:reloadPerform
--**
--** Function that should be performed upon successful completion of the
--** timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--** @param difficulty - the current reload difficulty
--** @param weapon - the item being reloaded
--**
--************************************************************************--
function QGWeaponManualIMNC:reloadPerform(char, square, difficulty, weapon)
	getSoundManager():PlayWorldSound(self.insertSound, char:getSquare(), 0, 10, 1.0, false);
	self.currentCapacity = self.currentCapacity + 1;
	-- remove the necessary ammo
	char:getInventory():RemoveOneOf(self.ammoType);
	self.reloadInProgress = false;
	self:syncReloadableToItem(weapon);
	char:getXp():AddXP(Perks.Reloading, 1);
	if(self.currentCapacity == self.maxCapacity) then
		return false;
	end
	return true;
end

function QGWeaponManualIMNC:unloadPerform(char, square, difficulty, weapon)
	getSoundManager():PlayWorldSound(self.rackSound, char:getSquare(), 0, 10, 1.0, false);
	self.currentCapacity = self.currentCapacity - 1;
	char:getInventory():AddItem('QG.'..self.ammoType);
	self.unloadInProgress = false;
	self:syncReloadableToItem(weapon);
	char:getXp():AddXP(Perks.Reloading, 1);
	if(self.currentCapacity == 0) then
		return false;
	end
	return true;
end

--************************************************************************--
--** QGWeaponManualIMNC:rackingStart
--**
--** Function that should be performed upon the start of the timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--************************************************************************--
function QGWeaponManualIMNC:rackingStart(char, square, weapon)
    getSoundManager():PlayWorldSound(self.rackSound, char:getSquare(), 0, 10, 1.0, false);
end

--************************************************************************--
--** QGWeaponManualIMNC:rackingPerform
--**
--** Function that should be performed upon successful completion of the
--** timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--** @param weapon - the item being reloaded
--**
--************************************************************************--
function QGWeaponManualIMNC:rackingPerform(char, square, weapon)
    self.racked = 1;
	self:syncReloadableToItem(weapon);
end

function QGWeaponManualIM:isChainUnloading()
	return true;
end

--************************************************************************--
--** QGWeaponManualIMNC:getReloadTime
--**
--** Returns the time take to perform the reload action
--**
--************************************************************************--
function QGWeaponManualIMNC:getReloadTime()
	return self.reloadTime;
end

--************************************************************************--
--** QGWeaponManualIMNC:getRackTime
--**
--** Returns the time take to perform the reload action
--**
--************************************************************************--
function QGWeaponManualIMNC:getRackTime()
	return self.rackTime;
end

--************************************************************************--
--** QGWeaponManualIMNC:syncItemToReloadable
--**
--** Function that copies details from an Item's modData to the instance
--** of this QGWeaponManualIMNC
--**
--** @param weapon - the weapon from which the reloadable information
--** should be retrieved
--**
--************************************************************************--
function QGWeaponManualIMNC:syncItemToReloadable(weapon)
	ISReloadableWeapon.syncItemToReloadable(self, weapon);
	local modData = weapon:getModData();
	self.racked = modData.racked;
end

--************************************************************************--
--** QGWeaponManualIMNC:syncReloadableToItem
--**
--** Function that copies details from the instance of this
--** QGWeaponManualIMNC to an Item's modData
--**
--** @param weapon - the weapon to which the reloadable information
--** should be copied
--**
--************************************************************************--
function QGWeaponManualIMNC:syncReloadableToItem(weapon)
	ISReloadableWeapon.syncReloadableToItem(self, weapon);
	local modData = weapon:getModData();
	modData.racked = self.racked;
end

--************************************************************************--
--** QGWeaponManualIMNC:setupReloadable
--**
--** Function that initialises all the required modData on an item.
--**
--** @param weapon - the weapon to setup
--** @param v - the lua table containing the key value pairs to attach
--** to the modData
--************************************************************************--
function QGWeaponManualIMNC:setupReloadable(weapon, v)
	ISReloadableWeapon.setupReloadable(self, weapon, v);
	local modData = weapon:getModData();
	modData.racked = 0;
end

function QGWeaponManualIMNC:printWeaponDetails(item)
    self:printItemDetails(item);
    local modData = item:getModData();
    local outString  = '';
    if(modData.shootSoundPartial ~= nil) then
        outString = outString..', shootSoundPartial: '..modData.shootSoundPartial;
    else
        outString = outString..', shootSoundPartial == nil';
    end
    if(modData.defaultSwingSound ~= nil) then
        outString = outString..', defaultSwingSound: '..modData.defaultSwingSound;
    else
        outString = outString..', defaultSwingSound == nil';
    end
    print(outString);
    print('***************************************************************');
    print();
    print();
end


function QGWeaponManualIMNC:printReloadableWeaponDetails()
    self:printReloadableDetails();
    local outString  = '';
    if(self.containsClip ~= nil) then
        outString = outString..', shootSoundPartial: '..self.shootSoundPartial;
    else
        outString = outString..', shootSoundPartial == nil';
    end
    if(self.clipName ~= nil) then
        outString = outString..', defaultSwingSound: '..self.defaultSwingSound;
    else
        outString = outString..', defaultSwingSound == nil';
    end
    print(outString);
    print('***************************************************************');
    print();
    print();
end
