require "Reloading/ISReloadableWeapon"

QSWeaponManualIMNC = ISReloadableWeapon:derive("QSWeaponManualIMNC");

--************************************************************************--
--** QSWeaponManualIMNC:initialise
--**
--************************************************************************--
function QSWeaponManualIMNC:initialise()

end

--************************************************************************--
--** QSWeaponManualIMNC:new
--**
--** Constructor for the QSWeaponManualIMNC reloadable
--************************************************************************--
function QSWeaponManualIMNC:new()
	local o = {}
	--o.data = {}
	o = ISReloadableWeapon:new();
    setmetatable(o, self)
    self.__index = self
	return o;
end

--************************************************************************--
--** QSWeaponManualIMNC:isLoaded
--**
--** Returns whether or not the gun will fire when the mouse is next
--** clicked
--**
--************************************************************************--
function QSWeaponManualIMNC:isLoaded(difficulty)
	return self.currentCapacity > 0;
end

--************************************************************************--
--** QSWeaponManualIMNC:fireShot
--**
--** Action performed when a shot is fired. Should typically decrease
--** the current amount of ammo in the weapon
--**
--************************************************************************--
function QSWeaponManualIMNC:fireShot(weapon, difficulty)
	if self.currentCapacity > 0 then
		self.currentCapacity = self.currentCapacity - 1;
	end
	self.racked = 0
	self:syncReloadableToItem(weapon)
end

--************************************************************************--
--** QSWeaponManualIMNC:canReload
--**
--** Whether the character attempting to reload has the necessary
--** prerequisites to perform the reload action. Called prior to
--** the timed action and not to be confused with isReloadValid
--**
--************************************************************************--
function QSWeaponManualIMNC:canReload(chr)
	if(self.currentCapacity < self.maxCapacity
	and chr:getInventory():FindAndReturn(self.ammoType) ~= nil) then
		return true;
	end
	return false;
end

function QSWeaponManualIMNC:canUnload(chr)
	if(self.currentCapacity > 0) then
		return true;
	end
	return false;
end

--************************************************************************--
--** QSWeaponManualIMNC:isReloadValid
--**
--** Function for the TimedAction that determines whether the reload
--** action is still valid. If the player does something that should
--** interrupt the action, this should return false
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--** @return true if the action may continue to be performed
--**
--************************************************************************--
function QSWeaponManualIMNC:isReloadValid(char, square, difficulty)
	if(self.currentCapacity < self.maxCapacity
		and char:getInventory():FindAndReturn(self.ammoType) ~= nil) then
			return true;
	end
	self.reloadInProgress = false;
	return false;
end

function QSWeaponManualIM:isUnloadValid(char, square, difficulty)
	if(self.currentCapacity > 0) then
		return true;
	end
	self.unloadInProgress = false;
	return false;
end

--************************************************************************--
--** QSWeaponManualIMNC:reloadStart
--**
--** Function that should be performed upon the start of the timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--************************************************************************--
function QSWeaponManualIMNC:reloadStart(char, square, difficulty)
	self.reloadInProgress = true;
end

function QSWeaponManualIM:unloadStart(char, square, difficulty)
	self.unloadInProgress = true;
end

--************************************************************************--
--** QSWeaponManualIMNC:reloadPerform
--**
--** Function that should be performed upon successful completion of the
--** timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--** @param difficulty - the current reload difficulty
--** @param weapon - the item being reloaded
--**
--************************************************************************--
function QSWeaponManualIMNC:reloadPerform(char, square, difficulty, weapon)
	getSoundManager():PlayWorldSound(self.insertSound, char:getSquare(), 0, 10, 1.0, false);
	self.currentCapacity = self.currentCapacity + 1;
	-- remove the necessary ammo
	char:getInventory():RemoveOneOf(self.ammoType);
	self.reloadInProgress = false;
	self:syncReloadableToItem(weapon);
	char:getXp():AddXP(Perks.Reloading, 1);
	if(self.currentCapacity == self.maxCapacity) then
		return false;
	end
	return true;
end

function QSWeaponManualIMNC:unloadPerform(char, square, difficulty, weapon)
	getSoundManager():PlayWorldSound(self.rackSound, char:getSquare(), 0, 10, 1.0, false);
	self.currentCapacity = self.currentCapacity - 1;
	char:getInventory():AddItem('QS.'..self.ammoType);
	self.unloadInProgress = false;
	self:syncReloadableToItem(weapon);
	char:getXp():AddXP(Perks.Reloading, 1);
	if(self.currentCapacity == 0) then
		return false;
	end
	return true;
end

--************************************************************************--
--** QSWeaponManualIMNC:rackingStart
--**
--** Function that should be performed upon the start of the timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--**
--************************************************************************--
function QSWeaponManualIMNC:rackingStart(char, square, weapon)
    getSoundManager():PlayWorldSound(self.rackSound, char:getSquare(), 0, 10, 1.0, false);
end

--************************************************************************--
--** QSWeaponManualIMNC:rackingPerform
--**
--** Function that should be performed upon successful completion of the
--** timed action
--**
--** @param char - the character performing the action. Must not be nil
--** @param square - not used
--** @param weapon - the item being reloaded
--**
--************************************************************************--
function QSWeaponManualIMNC:rackingPerform(char, square, weapon)
    self.racked = 1;
	self:syncReloadableToItem(weapon);
end

function QSWeaponManualIM:isChainUnloading()
	return true;
end

--************************************************************************--
--** QSWeaponManualIMNC:getReloadTime
--**
--** Returns the time take to perform the reload action
--**
--************************************************************************--
function QSWeaponManualIMNC:getReloadTime()
	return self.reloadTime;
end

--************************************************************************--
--** QSWeaponManualIMNC:getRackTime
--**
--** Returns the time take to perform the reload action
--**
--************************************************************************--
function QSWeaponManualIMNC:getRackTime()
	return self.rackTime;
end

--************************************************************************--
--** QSWeaponManualIMNC:syncItemToReloadable
--**
--** Function that copies details from an Item's modData to the instance
--** of this QSWeaponManualIMNC
--**
--** @param weapon - the weapon from which the reloadable information
--** should be retrieved
--**
--************************************************************************--
function QSWeaponManualIMNC:syncItemToReloadable(weapon)
	ISReloadableWeapon.syncItemToReloadable(self, weapon);
	local modData = weapon:getModData();
	self.racked = modData.racked;
end

--************************************************************************--
--** QSWeaponManualIMNC:syncReloadableToItem
--**
--** Function that copies details from the instance of this
--** QSWeaponManualIMNC to an Item's modData
--**
--** @param weapon - the weapon to which the reloadable information
--** should be copied
--**
--************************************************************************--
function QSWeaponManualIMNC:syncReloadableToItem(weapon)
	ISReloadableWeapon.syncReloadableToItem(self, weapon);
	local modData = weapon:getModData();
	modData.racked = self.racked;
end

--************************************************************************--
--** QSWeaponManualIMNC:setupReloadable
--**
--** Function that initialises all the required modData on an item.
--**
--** @param weapon - the weapon to setup
--** @param v - the lua table containing the key value pairs to attach
--** to the modData
--************************************************************************--
function QSWeaponManualIMNC:setupReloadable(weapon, v)
	ISReloadableWeapon.setupReloadable(self, weapon, v);
	local modData = weapon:getModData();
	modData.racked = 0;
end

function QSWeaponManualIMNC:printWeaponDetails(item)
    self:printItemDetails(item);
    local modData = item:getModData();
    local outString  = '';
    if(modData.shootSoundPartial ~= nil) then
        outString = outString..', shootSoundPartial: '..modData.shootSoundPartial;
    else
        outString = outString..', shootSoundPartial == nil';
    end
    if(modData.defaultSwingSound ~= nil) then
        outString = outString..', defaultSwingSound: '..modData.defaultSwingSound;
    else
        outString = outString..', defaultSwingSound == nil';
    end
    print(outString);
    print('***************************************************************');
    print();
    print();
end


function QSWeaponManualIMNC:printReloadableWeaponDetails()
    self:printReloadableDetails();
    local outString  = '';
    if(self.containsClip ~= nil) then
        outString = outString..', shootSoundPartial: '..self.shootSoundPartial;
    else
        outString = outString..', shootSoundPartial == nil';
    end
    if(self.clipName ~= nil) then
        outString = outString..', defaultSwingSound: '..self.defaultSwingSound;
    else
        outString = outString..', defaultSwingSound == nil';
    end
    print(outString);
    print('***************************************************************');
    print();
    print();
end
