function Buffin:LoadData()
	local defaults = {
		char = {
			buffs = {
				Self = {},
				Weapon = {},
				Group = {},
				Single = {},
				Combat = {},
				Flask = {},
			},
			last = {
				Self = {},
				Weapon = {},
				Group = {},
				Single = {},
				Combat = {},
				Flask = {},
			},
			options = {
				locked = false,
				scale = 1,
				attendance = 0,
				pvp = false,
				learningIncludesSpec = true,
			},
			combatOptions = {
				locked = false,
				scale = 1,
				alpha = 1,
				framePositions = {},
				showDuration = 60,
			},
		},
		profile = {buffs = {}, options = {}},
	}
	Buffin:tcopy(defaults.profile.buffs, defaults.char.buffs)
	self.db = LibStub("AceDB-3.0"):New("BuffinDB", defaults, true)
	self.db.RegisterCallback(self, "OnProfileChanged", "DoProfileSwitch")

	if not self.db.char.version or self.db.char.version < 93 then
		for id,position in pairs(self.db.char.combatOptions.framePositions) do -- This is a bug that shouldn't creep in anymore, sometimes it was saving in the whole UIParent table in the SV.
			if position[2] ~= nil then position[2] = nil end
		end
		if self.db.char.options.position and self.db.char.options.position[2] ~= nil then self.db.char.options.position[2] = nil end
		self.db.char.version = 93
	end

	--apply options
	if self.db.char.options.locked then self.btn.normal:SetAlpha(0) end
	self.btn:SetScale(self.db.char.options.scale)
	
	self.updaterFunc = function()
		self:SetupGroupRange()
	end
	self.buffs = Buffin.buffsByClassByType[self.class];
	self.lookupByID = {}; self.lookupByName = {}; self.lookupByCombatID = {}
	self.group = {classes={},ranges={},unitGUIDLookup={},guidUnitLookup={}}
	self.playerGUID = UnitGUID('player')
	self.primarySpecialisation = GetPrimaryTalentTree()
	
	self:MakeOptions()

	LibStub("AceConfig-3.0"):RegisterOptionsTable("Buffin", self.options)
	self.blizOptions = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Buffin")
end

function Buffin:MakeOptions()
	self.options = {
		type = "group",
		name = "Buffin Options",
		childGroups = 'tab',
		handler = self,
		args = {
			force = {
				type = 'execute',
				name = 'Brute Force!',
				hidden = true,
				func = "Force",
			},
			buffs = {
				type = 'group',
				name = "Buffs",
				order = 1,
				args = {
					explanation = {
						type = 'description',
						name = "This tab customises what buffs you would like Buffin to work with.\n\nChoosing 'Dynamic' will cause Buffin to guess at the buff you want and on whom you want it cast, if it doesn't guess correctly you should set a different option here.\nChoosing 'Learn' will cause Buffin to remember the last spell you cast in it's category and suggest that. Selecting individual buffs manually forces them to be active and will, in the case of Single buffs, remember on whom you want to cast them.\n",
						order = 0,
					},
				},
			},
			opts = {
				type = 'group',
				name = "Buff Options",
				order = 2, 
				get = 'GetOpt',
				set = 'SetOpt',
				args = {
					button = {
						type = 'group',
						inline = true,
						name = "Buffs",
						order = 1,
						args = {
							locked = {
								type = 'toggle',
								name = "Locked",
								order = 1,
								width = 'half',
								set = function(i,v) 
									self:SetOpt(i, v)
									if v == true then self.btn.normal:SetAlpha(0)
									else self.btn.normal:SetAlpha(1) end
								end,
							},
							scale = {
								type = 'range',
								name = "Size",
								order = 2,
								min = 0.5,
								max = 10,
								step = 0.1,
								set = function(i,v)
									self:SetOpt(i, v)
									self.btn:SetScale(v)
								end,
							},
							key = {
								type = 'input', --'keybinding'
								name = "Keybind",
								dialogControl = 'KeybindingWithWheel',
								order = 3,
								get = function() 
									LoadBindings(GetCurrentBindingSet())
									return GetBindingKey('CLICK BuffinButton:LeftButton')
								end,
								set = function(i,v) 
									local currentBinding = GetBindingKey('CLICK BuffinButton:LeftButton')
									if currentBinding then 
										if currentBinding == 'MOUSEWHEELUP' then SetBinding('MOUSEWHEELUP', 'CAMERAZOOMIN')
										elseif currentBinding == 'MOUSEWHEELDOWN' then SetBinding('MOUSEWHEELDOWN', 'CAMERAZOOMOUT')
										else SetBinding(currentBinding) end
									end
									SetBindingClick(v, 'BuffinButton')
									SaveBindings(GetCurrentBindingSet())
								end, 
							},
						},
					},
					conditions = {
						type = 'group',
						inline = true,
						name = 'Buffing Conditions',
						order = 2,
						args = {
							attendance = {
								type = 'range',
								name = "Group Attendance",
								desc = "The percent of group members who must be prescent for you to buff group buffs.",
								order = 1,
								min = 0,
								max = 1,
								step = 0.1,
								bigStep = 0.2,
								isPercent = true,
								disabled = function() return not Buffin.buffs.Group end,
							},
							pvp = {
								type = 'toggle',
								name = "Any PvP",
								desc = "Will buff anyone in range if you're flagged for PvP, disregarding your setting for Group Attendance.",
								order = 2,
								width = 'half',
								disabled = function() return not Buffin.buffs.Group end,
							},
							learningIncludesSpec = {
								type = 'toggle',
								name = "Learning via Spec",
								desc = [[Enabling this option causes Buffin's learning to take into account your current spec.
								Disabling this option means that Buffin will recommend to you your last cast spell in that category, regardless of what spec you were in when you cast it.]],
								order = 3,
							},
						}
					},
					expiration = {
						type = 'group',
						inline = true,
						name = "Expiration",
						order = 3,
						args = {
							help = {
								type = 'description',
								name = "These are the number of seconds a buff will need to have remaining before being considered for rebuffing. For example, a value of 0 means that the buff will only be replaced if it doesn't exist whereas a value of 1800 will replace a buff if it has less than 30 minutes remaining.\nNote: This may not have any meaning for some skills that buffin tracks, such as pet summoning, etc.",
								order = 0,
							},
						},
					},
				},
			},
			profiles = {
				type = 'group',
				name = "Buff Profiles",
				order = 3,
				args = {
					help = {
						type = 'description',
						name = "You can choose to save your current setup into a stored profile here, or load previously stored profiles into your current setup.\nNote: You won't see changes to profiles activate on this screen, the changes made here are reflected in the Buffs tab.",
						order = 1,
					},
					load = {
						type = 'group',
						name = "Load",
						inline = true,
						order = 2,
						args = {
							existing = {
								type = 'select',
								name = "Load a profile",
								values = "GetProfiles",
								set = "LoadProfile",
							},
							default = {
								type = 'execute',
								name = "Load Defaults",
								desc = "Resets settings to their defaults for your current setup.",
								func = function() self:LoadProfile({},'Default') end,
							},
						},
					},
					save = {
						type = 'group',
						name = "Save",
						inline = true,
						order = 3,
						set = "SaveProfile",
						args = {
							new = {
								type = 'input',
								name = "New",
								order = 1,
							},
							existing = {
								type = 'select',
								name = "Overwrite Existing",
								order = 2,
								values = "GetProfiles",
							},
						},
					},
					delete = {
						type = 'group',
						name = "Delete",
						inline = true,
						order = 4,
						args = {
							existing = {
								type = 'select',
								name = "Delete Existing",
								order = 1,
								values = "GetProfiles",
								set = "DeleteProfile",
							},
							reset = {
								type = 'execute',
								name = "Reset Database",
								desc = "Hit this if you run into trouble. Might be especially handy after updating the addon to a new version.\nIMPORTANT: This will remove all your settings for all characters and reset the database to its default state.",
								confirm = true,
								order = 2,
								func = function() self.db:ResetDB('Default') end, --TODO: Remove bindings.
							},
						},
					},
					
				},
			},
			combatOpts = {
				type = 'group',
				name = "Combat",
				order = 4, 
				get = 'GetOpt',
				set = 'SetOpt',
				args = {
					button = {
						type = 'group',
						inline = true,
						name = "Indicators",
						order = 1,
						args = {
							locked = {
								type = 'toggle',
								name = "Locked",
								order = 1,
								width = 'half',
								disabled = InCombatLockdown,
								set = function(i,v) 
									self:SetOpt(i, v)
									for spell,frame in pairs(self.combatFrames) do
										frame:SetAlpha(v and 0 or 1);
										if v then frame:Hide() else frame:Show() end
									end
								end,
							},
							scale = {
								type = 'range',
								name = "Scale",
								order = 2,
								min = 0.5,
								max = 10,
								step = 0.1,
								set = function(i,v)
									self:SetOpt(i, v)
									for _,frame in pairs(Buffin.combatFrames) do frame:SetScale(v) end
									for i=1,#Buffin.spareCombatFrames do Buffin.spareCombatFrames[i]:SetScale(v) end
								end,
							},
							alpha = {
								type = 'range',
								name = "Transparency",
								order = 3,
								min = 0,
								max = 1,
								step = 0.1,
								set = function(i,v)
									self:SetOpt(i,v)
									for _,frame in pairs(Buffin.combatFrames) do frame:SetAlpha(v) end
									for i=1,#Buffin.spareCombatFrames do Buffin.spareCombatFrames[i]:SetAlpha(v) end
								end,
								isPercent = true,
							},
							showDuration = {
								type = 'range',
								name = "Show Duration",
								desc = "The time (in seconds) at which duration remaining is shown on combat icons. Zero indicates that the duration should be always shown.",
								order = 4,
								min = 0,
								max = 600,
								step = 30,
							},
							--[[test = {
								type = 'toggle',
								name = "Show test frames",
								order = 4,
								set = function(i,v)
									self:SetOpt(i,v)
									for spell,frame in pairs(Buffin.combatFrames) do 
										if v then Buffin:CombatFrameStart(spell, 60)
										else Buffin:CombatFrameStop(spell) end
									end
									Buffin:ScheduleTimer(function() 
										for spell,frame in pairs(Buffin.combatFrames) do
											Buffin:CombatFrameStop()
										end
										Buffin.db.char.combatOptions.test = false
										LibStub('AceConfigRegistry-3.0'):NotifyChange('Buffin')
									end, 60)
								end,
							}]]
						},
					},
					keybinds = {
						type = 'group',
						inline = true,
						name = "Keybindings",
						hidden = true,
						order = 2,
						set = function(i,v)
							local current = self:GetOpt(i)
							if current then SetBinding(current, nil) end --Clear current
							self:SetOpt(i,v)
							for spell,frame in pairs(self.combatFrames) do
								if spell.id == i.arg then
									SetBindingClick(v, frame:GetName())
									break
								end
							end
						end,
						args = {
							help = {
								type = 'description',
								name = "You can bind any keys to these buttons, even the same keys to different buttons, however, if you do this for buttons that're active at the same time then it's undefined which one will actually get bound.",
								order = 0,
							},
						},
					},
				},
			},
		}
	}

	local mIndex = 10
	for mode, groups in pairs(self.buffs) do
		if mode ~= 'Combat' then
			self.options.args.buffs.args[mode] = {
				type = 'group',
				name = mode,
				inline = true,
				order = mIndex,
				args = {},
			}
		else
			self.options.args.buffs.args[mode] = {
				type = 'multiselect',
				name = mode,
				get = function(i,id)
					i[#i+1] = 'Unique'
					return Buffin:GetBuff(i,id)
				end,
				set = function(i,id,v)
					i[#i+1] = 'Unique'
					Buffin:SetBuff(i,id,v)
					Buffin:SetupCombatFrames()
				end,
				arg = true,
			}
		end

		local gIndex = 1
		for group, spells in pairs(groups) do
			local hasSpells = 0
			local values = {}
			local aggregateName = ""
			for id, options in pairs(spells) do
				hasSpells = hasSpells + 1

				local name, texture, buffName, buffTexture, firstBuffName
				if not options.type then options.type = 'spell' end
				if options.type == 'spell' then 
					name,_,texture = GetSpellInfo(id)
				elseif options.type == 'item' then name,_,_,_,_,_,_,_,_,texture = GetItemInfo(id) end
				if type(options.buff) == 'number' then 
					buffName,_,buffTexture = GetSpellInfo(type(options.buff) == 'table' and options.buff[1] or options.buff)
				elseif type(options.buff) == 'table' then
					--Uniqueify by name
					local tmpTbl = {}
					for i=1,#options.buff do
						local n,_,t = GetSpellInfo(options.buff[i])
						if firstBuffName == nil then firstBuffName = n end
						tmpTbl[n] = t
					end

					buffName,buffTexture = {},{}
					for n,t in pairs(tmpTbl) do
						table.insert(buffName, n)
						table.insert(buffTexture, t)
					end
				end

				if name then
					aggregateName = aggregateName .. (firstBuffName or name) .. "\n"
			
					values[id] = firstBuffName or name

					options.id = id;
					options.name = name;
					options.buffName = buffName 
					options.texture = texture;
					options.buffTexture = buffTexture
					options.mode = mode;
					options.group = group
					if mode == 'Weapon' then 
						if group == 'Main' then options.slot = 16
						elseif group == 'Off' then options.slot = 17
						elseif group == 'Ranged' then options.slot = 18 end
					end
					if type(options.buff) == 'table' then
						for i=1,#options.buff do options.buff[options.buff[i]] = true end
					end

					local usable = GetSpellInfo(options.name) --Use named lookup (only returns for spells you know when passed a spellname)
					if options.type == 'item' or usable then options.available = true
					elseif not usable then 
						-- Sometimes GetSpellInfo just plain doesn't work (Freezing Trap!) so give IsUsableSpell another go..
						--[[local usable, nomana = IsUsableSpell(id) --IsUsableSpell is returning true for everything it seems.
						if usable or nomana then options.available = true end]]
						options.available = false
					else options.available = false end
 
					local lookup
					if mode == 'Combat' then lookup = self.lookupByCombatID
					else lookup = self.lookupByID end

					lookup[id] = options
					
					if id == 58149 then lookup[79637] = options end --FIXME: hack to be able to learn Flask of Enhancements.

					if type(options.buff) == 'number' then lookup[options.buff] = options
					elseif type(options.buff) == 'table' then for i=1,#options.buff do lookup[options.buff[i]] = options end end

					if mode ~= 'Combat' then
						 self.lookupByName[name] = options
						if buffName and type(buffName) == 'string' then self.lookupByName[buffName] = options
						elseif buffName and type(buffName) == 'table' then
							for i=1,#buffName do
								self.lookupByName[buffName[i]] = options
							end
						end
					end

					if self.lookupByClash[id] then options.priority = self.lookupByClash[id] end

					if options.secure then 
						self.options.args.combatOpts.args.keybinds.hidden = false
						self.options.args.combatOpts.args.keybinds.args['bind'..id] = {
							type = 'keybinding',
							name = firstBuffName or name,
							arg = id,
						}
					end
				--else self:dump(id, GetItemInfo(id))
				end
			end
			
			if hasSpells > 0 then 
				values[-2] = "Dynamic"
				if mode ~= 'Combat' then
					if group ~= 'Unique' then 
						values[-1] = "Learn"
						values[-3] = "None"
					end

					self.options.args.buffs.args[mode].args[group] = {
						type = group == 'Unique' and 'multiselect' or 'select',
						name = group,
						values = values,
						order = gIndex+1,
						get = "GetBuff",
						set = "SetBuff",
						arg = group == 'Unique' or false,
						--disabled = function() local i={mode,'', arg=group}; return self:Dynamic(i) end
					}
				else
					self.options.args.buffs.args[mode].values = values
				end
				
				
				local gName = self:GetCoallescedModeGroupName(mode, group)
				if not self.options.args.opts.args.expiration.args[gName] and mode ~= 'Combat' then
					self.options.args.opts.args.expiration.args[gName] = {
						type = 'range',
						name = gName,
						desc = function(i) return aggregateName .. "\n" .. ((Buffin:GetOpt(i) ~= nil and Buffin:GetOpt(i) > 0) and SecondsToTime(Buffin:GetOpt(i)) or 'Only when expired') end,
						min = 0,
						max = 1800,
						bigStep = 60,
						order = gIndex+1,
						get = 'GetOpt',
						set = 'SetOpt',
					}
				end
				gIndex = gIndex+10
			end
		end
		mIndex = mIndex+10
	end
	LibStub('AceConfigRegistry-3.0'):NotifyChange('Buffin')
end

function Buffin:GetBuff(i, id)
	local mode, group = i[#i-1], i[#i]

	if self.db.char.buffs[mode][group] == nil then -- Dynamic
		if not id then return -2 -- select
		else return id==-2 end	-- multiselect
	elseif self.db.char.buffs[mode][group] == false then return -3 end

	if not id then -- select
		for id,_ in pairs(self.db.char.buffs[mode][group]) do
			return id -- return the first id in the list
		end
	else return self.db.char.buffs[mode][group][id] or false end -- multiselect
end

function Buffin:SetBuff(i, v, state)
	local mode,group = i[#i-1], i[#i]
	if not i.arg or not self.db.char.buffs[mode][group] then self.db.char.buffs[mode][group] = {} end --force unique

	if state == false then self.db.char.buffs[mode][group][v] = nil -- Unique
	elseif v == -2 then self:Dynamic(mode, group, true) -- Dynamic
	elseif v == -3 then self.db.char.buffs[mode][group] = false -- None
	else self.db.char.buffs[mode][group][v] = true end

	self:ChooseNextSpell()
end

function Buffin:GetOpt(i)
	if i[#i-2] == 'opts' then return self.db.char.options[i[#i]]
	elseif i[#i-2] == 'combatOpts' then return self.db.char.combatOptions[i[#i]] end
end

function Buffin:GetExpirationTime(mode, group)
	return self.db.char.options[self:GetCoallescedModeGroupName(mode, group)] or 0
end

function Buffin:SetOpt(i, v)
	if i[#i-2] == 'opts' then
		self.db.char.options[i[#i]] = v
		self:ChooseNextSpell()
	elseif i[#i-2] == 'combatOpts' then
		self.db.char.combatOptions[i[#i]] = v
	end
end

function Buffin:GetCoallescedModeGroupName(mode, group)
	return (group == 'Unique' or mode == 'Weapon') and mode --[[.. ' (generic)']] or group
end

function Buffin:Dynamic(mode,group,v)
	if v == nil then return (not self.db.char.buffs[mode][group]) 
	else 
		if not v then self.db.char.buffs[mode][group] = {}
		else self.db.char.buffs[mode][group] = nil end
	end
end

function Buffin:GetProfiles(i)
	local profiles = self.db:GetProfiles();
	local ret = {}
	for key,val in pairs(profiles) do
		if val ~= 'Default' then 
			if(string.sub(val, 1, string.len(self.class)) == self.class) then
				val = string.sub(val, string.len(self.class)+2, -1)
				ret[val] = val
			end
		end
	end
	return ret
end

function Buffin:LoadProfile(i,v)
	if v ~= 'Default' then v = self.class .. " " .. v end
	if v ~= self.db:GetCurrentProfile() then self.db:SetProfile(v)
	else self:DoProfileSwitch(nil, nil, v) end
end

function Buffin:SaveProfile(i,v)
	v = self.class .. " " .. v
	self.savingProfile = true
	if v ~= self.db:GetCurrentProfile() then self.db:SetProfile(v)
	else self:DoProfileSwitch(nil, nil, v) end
	self.savingProfile = nil
end

function Buffin:DeleteProfile(i,v)
	v = self.class .. " " .. v
	self.db.UnregisterCallback(self, "OnProfileChanged")
	self.db:SetProfile('Default')
	self.db:DeleteProfile(v)
	self.db.RegisterCallback(self, "OnProfileChanged", "DoProfileSwitch")
end

function Buffin:DoProfileSwitch(_,_, profile)  
	if profile ~= 'Default' and string.sub(profile, 1, string.len(self.class)) ~= self.class then
		self.db.UnregisterCallback(self, "OnProfileChanged")
		self.db:SetProfile('Default')
		self.db.RegisterCallback(self, "OnProfileChanged", "DoProfileSwitch")
		self:Print('Tried to load a profile from a different class. Maybe from an external addon or command? Reverted to the default configuration.');
	end

	if self.savingProfile then self:tcopy(self.db.profile.buffs, self.db.char.buffs)
	else self:tcopy(self.db.char.buffs, self.db.profile.buffs) end

	self:Force()
	self:SetupCombatFrames()
end

function Buffin:tcopy(to, from) 
	for k,v in pairs(from) do
		if(type(v)=="table") then
			to[k] = {}
			Buffin:tcopy(to[k], v);
		else
			to[k] = v;
		end
	end
end


