local ADDON_NAME = ...;

CreateFrame('Frame', 'WorldQuester', UIParent, 'ButtonFrameTemplate');

local QUEST_MAPS = {1014,1015,1017,1018,1021,1024,1033};
local ILVL_REGEX = "装备等级 (%d+)";
local REWARD_TYPES = {
	NONE = 0,
	GOLD = 1,
	CURRENCY = 2,
	ARTIFACT = 3,
	XP = 4,
	ITEM = 5
}
local SORT_TYPES = {
	REWARD = 0,
	TIME = 1,
	NEW = 2,
	RARITY = 3,
	FACTION = 4,
}
local SORT_FUNCTIONS = {
	[SORT_TYPES.REWARD] = function(a,b)
		if a.reward.type ~= b.reward.type then
			return a.reward.type > b.reward.type;
		end
		if a.reward.type == REWARD_TYPES.ITEM then
			return a.reward.ilvl > b.reward.ilvl;
		end
		return a.reward.ammount > b.reward.ammount;
	end,
	[SORT_TYPES.TIME] = function(a,b) return a.time < b.time; end,
	[SORT_TYPES.NEW] = function(a,b) return a.added > b.added; end,
	[SORT_TYPES.RARITY] = function(a,b) return a.rarity > b.rarity; end,
	[SORT_TYPES.FACTION] = function(a,b) return a.faction > b.faction; end,
}
local SORT_ICONS = {
	[SORT_TYPES.REWARD] = 'TRADE_ARCHAEOLOGY_CHESTOFTINYGLASSANIMALS',
	[SORT_TYPES.TIME] = 'SPELL_HOLY_BORROWEDTIME',
	[SORT_TYPES.NEW] = 'inv_misc_metagem_b',
	[SORT_TYPES.RARITY] = 'INV_Misc_GroupNeedMore',
	[SORT_TYPES.FACTION] = 'Achievement_BG_getXflags_no_die'
}
local RARITY_COLORS = {
	{0.18*3,0.16*3,0.15*3}, -- normal
	{0,0.43,0.86}, -- rare
	{0.63,0.20,0.93}, -- epic
}
local FACTION_ICONS = {
	[1900] = 1394952, -- court of farondis
	[1883] = 1394953, -- dreamweavers
	[1828] = 1394954, -- highmountain tribe
	--[] = 1394955, -- kirin tor
	[1859] = 1394956, -- nightfallen
	[1948] = 1394957, -- valarjar
	[1894] = 1394958, -- wardens
}

local function FormatCopper(c)
	c = floor(tonumber(c) or 0);
	local cAbs = abs(c);
	local g = floor(c/1e4);
	if cAbs == 0 then
		return '-';
	elseif cAbs < 1e2 then -- less than 1 silver
		return format("|cFFE07938%d|r",c);
	elseif cAbs < 1e4 then -- less than 1 gold
		return format("|cFFDDDDDD%d |cFFE07938%02d|r",c/1e2,c%1e2);
	elseif c >= 2147483648e4 then -- (2^31)*10000
		return '|cFFFF0000Overflow';
	elseif g >= 1e6 then
		g = format('%d,%03d,%03d',g/1e6,(g%1e6)/1e3,g%1e3);
	elseif g >= 1e3 then
		g = format('%d,%03d',g/1e3,g%1e3);
	end
	return format("|cFFFFEE00%s |cFFDDDDDD%02d |cFFE07938%02d|r",g,(c%1e4)/1e2,c%1e2);
end

local TT = CreateFrame('GameTooltip'); -- Scanning tooltip, used in UpdateTask to get real item info.
TT:SetOwner(UIParent);
TT:AddFontStrings(TT:CreateFontString(),TT:CreateFontString());

local Tooltip_ItemFrame = CreateFrame('Frame', nil, GameTooltip, 'EmbeddedItemTooltip');
Tooltip_ItemFrame:SetPoint('BOTTOMLEFT', GameTooltip, 'BOTTOMLEFT', 10, 16);
Tooltip_ItemFrame.Tooltip.shoppingTooltips = GameTooltip.shoppingTooltips;

WorldQuester.Quests = {};
WorldQuester.Sort = SORT_TYPES.NEW;

function WorldQuester:UpdateTask(task, map)
	local saved = WorldQuesterSaved_Quests;
	if not saved then return end
	
	local title, faction = C_TaskQuest.GetQuestInfoByQuestID(task.questId)
	local expires = C_TaskQuest.GetQuestTimeLeftMinutes(task.questId);
	local qtype, rarity, isElite = select(3, GetQuestTagInfo(task.questId));
	
	if expires == 0 or expires > 20160 then
		return;
	end
	
	local reward = {type = REWARD_TYPES.NONE};
	if GetQuestLogRewardMoney(task.questId) > 0 then
		reward.type = REWARD_TYPES.GOLD;
		reward.ammount = GetQuestLogRewardMoney(task.questId) or 0;
	elseif GetNumQuestLogRewardCurrencies(task.questId) > 0 then
		reward.type = REWARD_TYPES.CURRENCY;
		local name, icon, ammount = GetQuestLogRewardCurrencyInfo(1, task.questId);
		reward.name = name;
		reward.icon = icon;
		reward.ammount = ammount;
	elseif GetQuestLogRewardArtifactXP(task.questId) > 0 then
		reward.type = REWARD_TYPES.ARTIFACT;
		reward.ammount = GetQuestLogRewardArtifactXP(task.questId);
	elseif GetQuestLogRewardXP(task.questId) > 0 then
		reward.type = REWARD_TYPES.XP;
		reward.ammount = GetQuestLogRewardXP(task.questId);
	elseif GetNumQuestLogRewards(task.questId) > 0 then
		reward.type = REWARD_TYPES.ITEM;
		local itemName, itemTexture, quantity, quality, isUsable, itemID = GetQuestLogRewardInfo(1, task.questId);
		reward.item = itemID;
		reward.ammount = quantity;
		TT:ClearLines();
		TT:SetQuestLogItem("reward",1,task.questId);
		reward.name, reward.link = TT:GetItem();
		if not reward.link then
			self.MissingData = true;
		end
		reward.ilvl = 0;
		for i = 3,TT:GetNumRegions() do
			local ilvl = (select(i, TT:GetRegions()):GetText() or ''):match(ILVL_REGEX);
			if ilvl then reward.ilvl = tonumber(ilvl); end
		end
	end
	
	local isNew = false;
	local added = time();
	if saved[task.questId] then
		added = saved[task.questId][1];
		isNew = time()-added <= 3600; -- first seen less than an hour ago.
	else
		saved[task.questId] = {added, expires};
		self.MinimapButton.Flash:Show();
	end
	
	if reward.type == REWARD_TYPES.NONE or not title then
		self.MissingData = true;
	end
	
	self.Quests[task.questId] = {
		id = task.questId,
		map = map,
		new = isNew,
		faction = faction or 0,
		added = added,
		task = task,
		reward = reward,
		title = title,
		time = expires,
		type = qtype,
		rarity = rarity or 1,
		elite = isElite,
		expired = false,
	};
end

function WorldQuester:Scan()
	if WorldMapFrame:IsShown() then
		return;
	end
	self.MissingData = false;
	local active = {};
	for i,map in pairs(QUEST_MAPS) do
		SetMapByID(map);
		local tasks = C_TaskQuest.GetQuestsForPlayerByMapID(map);
		for j,task in pairs(tasks or {}) do
			active[task.questId] = true;
			self:UpdateTask(task,map);
		end
	end
	
	for i,q in pairs(self.Quests) do
		if not active[q.id] then
			q.expired = true;
		end
	end
	
	WorldQuester:UpdateUI();
	if self.MissingData then
		C_Timer.After(1, function()
			self:Scan()
		end)
	end
end

function WorldQuester:SelectQuest(q)
	WorldMapFrame:Show();
	SetMapByID(q.map);
	WorldMap_UpdateQuestBonusObjectives();
	local tasks = C_TaskQuest.GetQuestsForPlayerByMapID(q.map);
	for j,task in pairs(tasks or {}) do
		local poi = _G["WorldMapFrameTaskPOI"..j];
		if poi.questID == q.id then
			poi:Click();
			return;
		end
	end
	WorldMapFrame:Hide();
end

function WorldQuester:ShowQuestTooltip(f, q)
	local Tooltip = GameTooltip;
	Tooltip:SetOwner(f, "ANCHOR_RIGHT", 18, 0);
	
	local color = WORLD_QUEST_QUALITY_COLORS[q.rarity];
	Tooltip:SetText(q.title, color.r, color.g, color.b);
	
	if q.faction > 0 then
		local factionName = GetFactionInfoByID(q.faction);
		if factionName then
			Tooltip:AddLine(factionName);
		end
	end
	
	local color = NORMAL_FONT_COLOR;
	Tooltip:AddLine(BONUS_OBJECTIVE_TIME_LEFT:format(SecondsToTime(q.time*60)), color.r, color.g, color.b);
	
	for i = 1, q.task.numObjectives do
		local objectiveText, objectiveType, finished = GetQuestObjectiveInfo(q.id, i, false);
		if ( objectiveText and #objectiveText > 0 ) then
			local color = finished and GRAY_FONT_COLOR or HIGHLIGHT_FONT_COLOR;
			Tooltip:AddLine(QUEST_DASH .. objectiveText, color.r, color.g, color.b, true);
		end
	end
	
	Tooltip:AddLine(" ");
	Tooltip:AddLine(QUEST_REWARDS, NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b, true);
	
	local HC = HIGHLIGHT_FONT_COLOR;
	if q.reward.type == REWARD_TYPES.GOLD then
		SetTooltipMoney(Tooltip, q.reward.ammount);
	elseif q.reward.type == REWARD_TYPES.CURRENCY then
		Tooltip:AddLine(BONUS_OBJECTIVE_REWARD_WITH_COUNT_FORMAT:format(q.reward.icon, q.reward.ammount, q.reward.name), HC.r, HC.g, HC.b);
	elseif q.reward.type == REWARD_TYPES.ARTIFACT then
		Tooltip:AddLine(BONUS_OBJECTIVE_ARTIFACT_XP_FORMAT:format(q.reward.ammount), HC.r, HC.g, HC.b);
	elseif q.reward.type == REWARD_TYPES.XP then
		Tooltip:AddLine(BONUS_OBJECTIVE_EXPERIENCE_FORMAT:format(q.reward.ammount), HC.r, HC.g, HC.b);
	end
	
	Tooltip:Show();
	
	if q.reward.type == REWARD_TYPES.ITEM then
		EmbeddedItemTooltip_SetItemByQuestReward(Tooltip_ItemFrame, 1, q.id);
		local w,h = GameTooltip:GetSize();
		local ew,eh = Tooltip_ItemFrame:GetSize();
		GameTooltip:SetSize(max(w,ew+54), h+eh+12);
		Tooltip_ItemFrame.Tooltip:SetWidth(GameTooltip:GetWidth()-34);
		if IsShiftKeyDown() then
			GameTooltip_ShowCompareItem(Tooltip_ItemFrame.Tooltip);
		end
	else
		Tooltip_ItemFrame:Hide();
	end
end

function WorldQuester:InitUI()
	self:SetAttribute("UIPanelLayout-area", "left");
	self:SetAttribute("UIPanelLayout-pushable", 1);
	self:SetAttribute("UIPanelLayout-defined", true);
	self:SetSize(350,500);
	self:SetShown(false);
	
	local portrait = self:CreateTexture('BACKGROUND');
	portrait:SetMask("Interface/CharacterFrame/TempPortraitAlphaMask");
	portrait:SetTexture('Interface/ICONS/Achievement_Quests_Completed_08');
	portrait:SetPoint('TOPLEFT',-8,8);
	
	local Container = LibElci.Create(self,'Frame');
	Container:SetPoint('TOPLEFT',4,-4);
	Container:SetPoint('BOTTOMRIGHT',-4,3)
	self.Container = Container;
	
	local Title = Container:Create('FontString');
	Title:SetPoint('TOP',10,-1.5);
	Title:SetText('世界任务助手');
	
	local NGA = Container:Create('FontString');
	NGA:SetPoint('TOP',10,-35);
	NGA:SetText('NeverGankAsuka Windlord汉化');
	local SortButton = Container:Create('Button',function(c)
		self.Sort = (self.Sort+1)%3;
		c.Icon:SetTexture('Interface/ICONS/'..SORT_ICONS[self.Sort]);
		self:UpdateUI();
	end);

	SortButton:SetSize(22,22);
	SortButton:SetPoint('TOPRIGHT',-14,-28);
	SortButton:SetBackdrop({});
	SortButton.Icon = SortButton:Create('Texture');
	SortButton.Icon:SetPoint('CENTER');
	SortButton.Icon:SetDrawLayer('BACKGROUND');
	SortButton.Icon:SetMask("Interface\\CharacterFrame\\TempPortraitAlphaMask");
	SortButton.Icon:SetTexture('Interface/ICONS/'..SORT_ICONS[self.Sort]);
	SortButton.Border = SortButton:Create('Texture');
	SortButton.Border:SetTexture('Interface/Garrison/GarrLanding-TradeskillTimerFill');
	SortButton.Border:SetTexture('Interface/COMMON/BlueMenuRing');
	SortButton.Border:SetPoint('TOPLEFT',SortButton,'TOPLEFT',-8,8);
	SortButton.Border:SetPoint('BOTTOMRIGHT',SortButton,'BOTTOMRIGHT',16,-16);
	SortButton.OnUpdate = function(c,v)
		v = v*6;
		c.Icon:SetSize(22+v, 22+v);
	end
	SortButton:OnUpdate(0)
	
	local ScrollingList = Container:Create('ScrollingList');
	ScrollingList.ItemHeight = 24;
	ScrollingList.ScrollFactor = 30;
	ScrollingList:SetPoint('TOPLEFT', self.Inset, 'TOPLEFT', 3, -3);
	ScrollingList:SetPoint('BOTTOMRIGHT', self.Inset, 'BOTTOMRIGHT', -3, 3);
	local handler = function(c)
		WorldQuester:SelectQuest(c.quest);
		print(format("|cffffff00|Hquest:%d:110|h[%s]|h|r",c.quest.id,c.quest.title));
		if c.quest.reward.type == REWARD_TYPES.ITEM then
			print(c.quest.reward.link);
		end
	end
	for i = 1,20 do
		local f = ScrollingList:AddButton(-2, handler);
		f:SetPushedColor(0.4,0.4,0.4);
		-- TODO: create a mimic of WorldMapTooltip
		f.OnEnter = function(c)
			self:ShowQuestTooltip(c, c.quest);
		end
		f.OnLeave = function(c)
			GameTooltip:Hide();
			Tooltip_ItemFrame:Hide();
		end
		f:SetBackdrop({
			bgFile = "Interface/AddOns/WorldQuester/Noise.blp",
			edgeFile = "Interface/AddOns/WorldQuester/Noise.blp",
			edgeSize = 1,
		});
		f.BorderFactor = 1.5;
		f:SetPushedColor(1,1,1,1);
		f.Reward = f:Create('Texture');
		f.Reward:SetTexCoord(0.075,0.925,0.075,0.925);
		f.Reward:SetSize(22,22);
		f.Reward:SetPoint('LEFT',2,0);
		--f.RewardCount = f:Create('FontString',12,'OUTLINE');
		--f.RewardCount:SetPoint('BOTTOMRIGHT', f.Reward, 'BOTTOMRIGHT', 0, 3);
		f.Title = f:Create('FontString');
		f.Title:SetPoint('TOPLEFT',f.Reward,'TOPRIGHT',4,-1);
		f.Title:SetShadowColor(0,0,0,1);
		f.Title:SetShadowOffset(1,-1);
		f.RewardText = f:Create('FontString',13);
		f.RewardText:SetPoint('LEFT',f.Reward,'RIGHT',4,1);
		f.RewardText:SetShadowColor(0,0,0,1);
		f.RewardText:SetShadowOffset(1,-1);
		f.Time = f:Create('FontString', 13);
		f.Time:SetPoint('RIGHT', -4, 1);
		f.Time:SetShadowColor(0,0,0,1);
		f.Time:SetShadowOffset(1,-1);
		f.Faction = f:Create('Texture');
		f.Faction:SetPoint('RIGHT');
		f.Faction:SetSize(32,22);
		f.Faction:SetTexCoord( 4/64, (4+56)/64, 12/64, (12+40)/64 );
		f.Faction:SetGradientAlpha("HORIZONTAL",1,1,1,0,1,1,1,0.5);
		f.New = f:Create('FontString',10);--(f,"ARTWORK","GameFontHighlight");
		f.New:SetPoint('RIGHT', -56, 1);
		f.New:SetText(NEW_CAPS);
		f.NewGlow = f:Create('Texture');
		f.NewGlow:SetAtlas('collections-newglow');
		f.NewGlow:SetPoint('TOPLEFT', f.New, 'TOPLEFT', -20, 10);
		f.NewGlow:SetPoint('BOTTOMRIGHT', f.New, 'BOTTOMRIGHT', 20, -10);
	end
	ScrollingList.PopulateFrame = function(c,f,q)
		f.quest = q;
		f:SetColor( unpack(RARITY_COLORS[q.rarity or 1]) );
		f.questID = q.id;
		f.numObjectives = q.task.numObjectives;
		f.worldQuest = true;
		--f.Title:SetText(q.title);
		local t = q.time or 0;
		f.Time:SetText( (t>1440 and format('%d天',t/1440)) or (t>60 and format('%d小时',t/60) or format('%d分钟',t)) );
		if q.faction and FACTION_ICONS[q.faction] then
			f.Faction:SetTexture(FACTION_ICONS[q.faction]);
			f.Faction:SetShown(true);
		else
			f.Faction:SetShown(false);
		end
		
		f.New:SetShown(q.new);
		f.NewGlow:SetShown(q.new);
		--f.RewardCount:SetText('');
		if q.reward.type == REWARD_TYPES.GOLD then
			f.Reward:SetTexture('Interface/ICONS/INV_Misc_Coin_02');
			f.Reward:SetShown(true);
			f.RewardText:SetText(FormatCopper(q.reward.ammount));
			f.RewardText:SetShown(true);
		elseif q.reward.type == REWARD_TYPES.CURRENCY then
			if not q.reward.icon then
				self.MissingData = true;
				f.Reward:SetTexture('Interface/ICONS/INV_Misc_QuestionMark');
			else
				f.Reward:SetTexture(q.reward.icon);
			end
			f.Reward:SetShown(true);
			f.RewardText:SetText((q.reward.ammount or '')..' '..(q.reward.name or ''));
			f.RewardText:SetShown(true);
		elseif q.reward.type == REWARD_TYPES.ITEM then
			local name, link = GetItemInfo(q.reward.item or 0);
			local icon = GetItemIcon(q.reward.item or 0);
			if icon == '' then icon = nil; end
			f.Reward:SetTexture(icon or 'Interface/ICONS/INV_Misc_QuestionMark');
			f.Reward:SetShown(true);
			f.RewardText:SetText(
				(q.reward.link or link or '') ..
				((q.reward.ammount>1) and (' x'..q.reward.ammount) or '') ..
				((q.reward.ilvl>0) and (' ('..q.reward.ilvl..')') or ''));
			f.RewardText:SetShown(true);
			--if q.reward.ilvl > 0 then
			--	f.RewardCount:SetText(q.reward.ilvl);
			--end
			if not (q.reward.item and name and link and icon) then
				self.MissingData = true;
			end
		else
			f.Reward:SetShown(false);
			f.RewardText:SetShown(false);
		end
		
	end
	self.ScrollingList = ScrollingList;
	
	self.FactionButtons = {};
	
	for i = 1,4 do
		local btn = self.Container:Create('Button', function(c)
			self.FactionFilter = (c.Faction ~= self.FactionFilter) and c.Faction;
			self.ScrollingList.Pos = 0;
			self:UpdateUI()
		end);
		btn:SetPoint('BOTTOMLEFT', 10+70*(i-1), 3);
		btn:SetColor(0,0,0,0);
		btn:SetTextColor(1,1,1,1);
		btn:SetPushedColor(0,0,0,0);
		btn:SetSize(60,20);
		
		btn.Text:SetPoint('TOPLEFT',32,0);
		btn.Text:SetJustifyH('LEFT');
		btn.Text:SetFont('Fonts/ARIALN.TooltipF', 14, 'OUTLINE');
		btn.Icon = btn:Create('Texture');
		btn.Icon:SetPoint('LEFT');
		btn.Icon:SetSize(22,22);
		btn.Icon:SetDrawLayer('BACKGROUND');
		btn.Icon:SetMask("Interface\\CharacterFrame\\TempPortraitAlphaMask");
		
		btn.Border = btn:CreateTexture();
		btn.Border:SetTexture('Interface/AddOns/WorldQuester/FactionFrame.blp');
		btn.Border:SetPoint('TOPLEFT',-4,6);
		
		btn.Background = btn:CreateTexture();
		btn.Background:SetDrawLayer('BACKGROUND');
		btn.Background:SetPoint('RIGHT',-3,0);
		btn.Background:SetSize(32,15);
		
		self.FactionButtons[i] = btn;
	end
	
end

function WorldQuester:InitMinimapButton()
	
	local Button = CreateFrame('Button', 'WorldQuesterMinimapButton', Minimap);
	Button:SetFrameStrata("MEDIUM")
	Button:SetSize(31, 31)
	Button:SetFrameLevel(8)
	Button:RegisterForDrag("LeftButton")
	Button:EnableMouse(true);
	Button:SetMovable(true);
	Button:IsUserPlaced(true);
	Button:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight")
	Button:SetScript('OnClick', function()
		ToggleFrame(self);
	end)
	
	self.MinimapButton = Button;
	
	local Border = Button:CreateTexture(nil, "OVERLAY")
	Border:SetSize(53, 53)
	Border:SetTexture("Interface\\Minimap\\MiniMap-TrackingBorder")
	Border:SetPoint("TOPLEFT")
	
	local Icon = Button:CreateTexture(nil, "ARTWORK")
	Icon:SetSize(20, 20)
	Icon:SetTexture('Interface/ICONS/Achievement_Quests_Completed_08')
	Icon:SetPoint("CENTER")
	
	local Flash = Button:CreateTexture(nil, "OVERLAY");
	Flash:SetTexture('Interface/UNITPOWERBARALT/PandarenTraining_Circular_Flash');
	Flash:SetPoint('CENTER',2,0);
	Flash:SetBlendMode('ADD');
	Flash:SetSize(64,64);
	Flash:Hide();
	Button.Flash = Flash;
	local FlashV = 0;
	C_Timer.NewTicker(0.05, function()
		FlashV = (FlashV+0.025)%1;
		Flash:SetVertexColor(1,1,1, 0.25+abs((FlashV*1.5)-0.75));
	end)
	
	Button:SetScript('OnMouseDown', function() Icon:SetSize(16,16); end);
	Button:SetScript('OnMouseUp', function() Icon:SetSize(20,20); end);
	
	local minimapShapes = {
		["ROUND"] = {true, true, true, true},
		["SQUARE"] = {false, false, false, false},
		["CORNER-TOPLEFT"] = {false, false, false, true},
		["CORNER-TOPRIGHT"] = {false, false, true, false},
		["CORNER-BOTTOMLEFT"] = {false, true, false, false},
		["CORNER-BOTTOMRIGHT"] = {true, false, false, false},
		["SIDE-LEFT"] = {false, true, false, true},
		["SIDE-RIGHT"] = {true, false, true, false},
		["SIDE-TOP"] = {false, false, true, true},
		["SIDE-BOTTOM"] = {true, true, false, false},
		["TRICORNER-TOPLEFT"] = {false, true, true, true},
		["TRICORNER-TOPRIGHT"] = {true, false, true, true},
		["TRICORNER-BOTTOMLEFT"] = {true, true, false, true},
		["TRICORNER-BOTTOMRIGHT"] = {true, true, true, false},
	}

	local function RepositionIcon(self)
		local mx, my = Minimap:GetCenter();
		local s = Minimap:GetEffectiveScale();
		local cx, cy = GetCursorPosition();
		local angle = math.atan2((cy/s) - my, (cx/s) - mx)
		local x, y, q = math.cos(angle), math.sin(angle), 1
		if x < 0 then q = q + 1 end
		if y > 0 then q = q + 2 end
		local minimapShape = GetMinimapShape and GetMinimapShape() or "ROUND"
		local quadTable = minimapShapes[minimapShape]
		if quadTable[q] then
			x, y = x*80, y*80
		else
			local diagRadius = 103.13708498985; --math.sqrt(2*(80)^2)-10
			x = math.max(-80, math.min(x*diagRadius, 80))
			y = math.max(-80, math.min(y*diagRadius, 80))
		end
		Button:ClearAllPoints();
		Button:SetPoint("CENTER", Minimap, "CENTER", x, y);
	end

	Button:SetScript('OnDragStart', function()
		Button:SetScript("OnUpdate", RepositionIcon);
		Icon:SetSize(16,16);
		Button:StartMoving();
	end);

	Button:SetScript('OnDragStop', function()
		Button:SetScript("OnUpdate", nil);
		Icon:SetSize(20,20);
		Button:StopMovingOrSizing();
	end);
	
	RepositionIcon(Button);
	
end

function WorldQuester:UpdateUI()
	local frame = WorldQuester;
	if not frame:IsShown() then return end
	
	local active_factions = {};
	for i, meta in pairs(GetQuestBountyInfoForMapID(1014)) do
		active_factions[meta.factionID] = {select(4,GetQuestObjectiveInfo(meta.questID,1,false))};
	end
	
	if self.FactionFilter and not active_factions[self.FactionFilter] then
		self.FactionFilter = nil;
	end
	
	local data = {};
	local i = 1;
	for id,q in pairs(self.Quests) do
		if (q.faction == self.FactionFilter or not self.FactionFilter) and not q.expired then
			data[i] = q;
			i = i + 1;
		end
	end
	if self.Sort then
		sort(data, SORT_FUNCTIONS[self.Sort]);
	end
	frame.ScrollingList.Data = data;
	frame.ScrollingList:Update();
	
	
	local i = 1;
	for id, status in pairs(active_factions) do
		local btn = self.FactionButtons[i];
		btn:SetText(format('%d/%d', unpack(status)));
		btn.Icon:SetTexture(FACTION_ICONS[id]);
		btn:Show();
		if self.FactionFilter == id then
			btn.Background:SetColorTexture(0,0,0,0.8);
		else
			btn.Background:SetColorTexture(0,0,0,0.3);
		end
		btn.Faction = id;
		i = i+1;
	end
	for i = i, #self.FactionButtons do
		self.FactionButtons[i]:Hide();
	end
	
end

WorldQuester:SetScript('OnShow',function(self)
	self.MinimapButton.Flash:Hide();
	self:Scan();
end)

SlashCmdList.WORLDQUESTER = function()ToggleFrame(WorldQuester)end;
SLASH_WORLDQUESTER1 = "/wq";

WorldQuester:SetScript('OnEvent',function(self,e,...)
	
	if e == 'ADDON_LOADED' then
		self:UnregisterEvent(e);
		if type(WorldQuesterSaved) ~= 'table' then
			WorldQuesterSaved = {};
		end
		if type(WorldQuesterSaved_Quests) ~= 'table' then
			WorldQuesterSaved_Quests = {};
		end
		for id,d in pairs(WorldQuesterSaved_Quests) do
			if d[1]+d[2]*60 <= time() then -- remove expired quests.
				WorldQuesterSaved_Quests[id] = nil;
			end
		end
		self:InitUI();
		self:InitMinimapButton();
		
	elseif e == 'MODIFIER_STATE_CHANGED' and (...):match('SHIFT') then
		if IsShiftKeyDown() then
			GameTooltip_ShowCompareItem(Tooltip_ItemFrame.Tooltip);
		else
			GameTooltip.shoppingTooltips[1]:Hide();
			GameTooltip.shoppingTooltips[2]:Hide();
		end
		
	end
end);
WorldQuester:RegisterEvent('ADDON_LOADED');
WorldQuester:RegisterEvent('MODIFIER_STATE_CHANGED');

C_Timer.NewTicker(30,function() -- update ticker
	WorldQuester:Scan();
end)
