﻿local _, addonNamespace = ...

local SlotIconManager = {
    STYLE = {
        SMALL = 1,
        UPGRADES = 2,
    }
}
SlotIconManager.__index = SlotIconManager

addonNamespace.SlotIconManager = SlotIconManager

local SLOT_FRAME_NAME = 1
local SLOT_ALIGNMENT = 2
local SLOT_ID = 3
local REQUIRED_FLAG = 4

function SlotIconManager:new(mode)
    return setmetatable({
        mode = mode,
        slotIcons = {},
        slotText = {},
        visible = false,
        hostVisible = false,
        style = nil,
        slotsInfo = nil,
        slotNames = nil,
    }, SlotIconManager):init()
end

function SlotIconManager:init()
    if self.mode == "player" then
        self.slotsInfo = {
            HeadSlot = { "CharacterHeadSlot", "RIGHT", nil, false },
            NeckSlot = { "CharacterNeckSlot", "RIGHT", nil, false },
            ShoulderSlot = { "CharacterShoulderSlot", "RIGHT", nil, false },
            BackSlot = { "CharacterBackSlot", "RIGHT", nil, false },
            ChestSlot = { "CharacterChestSlot", "RIGHT", nil, false },
            --            ShirtSlot         = {"CharacterShirtSlot",         "RIGHT", nil, false},
            --            TabardSlot        = {"CharacterTabardSlot",        "RIGHT", nil, false},
            WristSlot = { "CharacterWristSlot", "RIGHT", nil, false },
            HandsSlot = { "CharacterHandsSlot", "LEFT", nil, false },
            WaistSlot = { "CharacterWaistSlot", "LEFT", nil, false },
            LegsSlot = { "CharacterLegsSlot", "LEFT", nil, false },
            FeetSlot = { "CharacterFeetSlot", "LEFT", nil, false },
            Finger0Slot = { "CharacterFinger0Slot", "LEFT", nil, false },
            Finger1Slot = { "CharacterFinger1Slot", "LEFT", nil, false },
            Trinket0Slot = { "CharacterTrinket0Slot", "LEFT", nil, false },
            Trinket1Slot = { "CharacterTrinket1Slot", "LEFT", nil, false },
            MainHandSlot = { "CharacterMainHandSlot", "LEFT", nil, false },
            SecondaryHandSlot = { "CharacterSecondaryHandSlot", "RIGHT", nil, nil },
        }

        self.slotsInfo.NeckSlot[REQUIRED_FLAG] = true
        self.slotsInfo.BackSlot[REQUIRED_FLAG] = true
        self.slotsInfo.Finger0Slot[REQUIRED_FLAG] = true
        self.slotsInfo.Finger1Slot[REQUIRED_FLAG] = true
        self.slotsInfo.MainHandSlot[REQUIRED_FLAG] = true

        local fontStyle = "NewSubSpellFont";
        self.averageItemLevelFontString = CharacterModelFrame:CreateFontString("CharacterModelAverageItemLevel", "OVERLAY", fontStyle)
        self.averageItemLevelFontString:SetPoint("CENTER", CharacterModelFrame, "BOTTOM", 0, 51)
    elseif self.mode == "target" then
        self.slotsInfo = {
            HeadSlot = { "InspectHeadSlot", "RIGHT", nil, false },
            NeckSlot = { "InspectNeckSlot", "RIGHT", nil, false },
            ShoulderSlot = { "InspectShoulderSlot", "RIGHT", nil, false },
            BackSlot = { "InspectBackSlot", "RIGHT", nil, false },
            ChestSlot = { "InspectChestSlot", "RIGHT", nil, false },
            --            ShirtSlot         = {"InspectShirtSlot",         "RIGHT", nil, false},
            --            TabardSlot        = {"InspectTabardSlot",        "RIGHT", nil, false},
            WristSlot = { "InspectWristSlot", "RIGHT", nil, false },
            HandsSlot = { "InspectHandsSlot", "LEFT", nil, false },
            WaistSlot = { "InspectWaistSlot", "LEFT", nil, false },
            LegsSlot = { "InspectLegsSlot", "LEFT", nil, false },
            FeetSlot = { "InspectFeetSlot", "LEFT", nil, false },
            Finger0Slot = { "InspectFinger0Slot", "LEFT", nil, false },
            Finger1Slot = { "InspectFinger1Slot", "LEFT", nil, false },
            Trinket0Slot = { "InspectTrinket0Slot", "LEFT", nil, false },
            Trinket1Slot = { "InspectTrinket1Slot", "LEFT", nil, false },
            MainHandSlot = { "InspectMainHandSlot", "LEFT", nil, false },
            SecondaryHandSlot = { "InspectSecondaryHandSlot", "RIGHT", nil, nil },
        }

        self.slotsInfo.NeckSlot[REQUIRED_FLAG] = true
        self.slotsInfo.BackSlot[REQUIRED_FLAG] = true
        self.slotsInfo.Finger0Slot[REQUIRED_FLAG] = true
        self.slotsInfo.Finger1Slot[REQUIRED_FLAG] = true
        self.slotsInfo.MainHandSlot[REQUIRED_FLAG] = true

        local fontStyle = "NewSubSpellFont";
        self.averageItemLevelFontString = InspectModelFrame:CreateFontString("InspectModelAverageItemLevel", "OVERLAY", fontStyle)
        self.averageItemLevelFontString:SetPoint("CENTER", InspectModelFrame, "BOTTOM", 0, 51)
    else
        self.slotsInfo = nil
    end

    self.slotNames = {}

    for slotName, slotInfo in pairs(self.slotsInfo) do
        slotInfo[SLOT_ID] = GetInventorySlotInfo(slotName)
        table.insert(self.slotNames, slotName)
    end

    return self
end

function SlotIconManager:SetHostVisibility(flag)
    if self.hostVisible ~= flag then
        self.hostVisible = flag
        if self.hostVisible then
            self:_ShowUI()
        else
            self:_HideUI()
        end
    end
end

function SlotIconManager:Refresh()
    if self.visible and self.hostVisible then
        if self.mode == "player" then
            self:_FindItemInfo("player")
        elseif self.mode == "target" then
            if InspectFrame.unit then
                self.visible = true
                self:_FindItemInfo(InspectFrame.unit)
            else
                self:_HideUI()
            end
        end
    end
end

function SlotIconManager:SetStyle(value)
    self.style = value
    if self.visible then
        self:_HideUI()
        self:_ShowUI()
    end
end

function SlotIconManager:_ShowUI()
    if not self.visible then
        if self.mode == "player" then
            self.visible = true
            self:_FindItemInfo("player")
        elseif self.mode == "target" and InspectFrame.unit then
            self.visible = true
            self:_FindItemInfo(InspectFrame.unit)
        end
    end
end

function SlotIconManager:_HideUI()
    if self.visible then
        self.visible = false
        self:_HideAllSlotIcons()
        self:_HideAllSlotText()
        self.averageItemLevelFontString:Hide()
    end
end

function SlotIconManager:_GetSlotNames()
    return self.slotNames
end

function SlotIconManager:_GetSlotInfo(slotName, slotInfoFieldId)
    return self.slotsInfo[slotName][slotInfoFieldId]
end

function SlotIconManager:_HideAllSlotIcons()
    for _, slotIcons in pairs(self.slotIcons) do
        for _, slotIcon in pairs(slotIcons) do
            addonNamespace.ReleaseSlotIcon(slotIcon)
        end
    end

    self.slotIcons = {}
end

function SlotIconManager:_HideAllSlotText()
    for i, slotText in pairs(self.slotText) do
        slotText:Hide()
    end
end

function SlotIconManager:_ShowSlotIcon(slotName, textureName, tooltip)
    local previousSlotIcon, slotIcon
    local slotIconIndex = 1

    if self.slotIcons[slotName] then
        for i, item in pairs(self.slotIcons[slotName]) do
            if item:isHidden() then
                slotIcon = item
                break
            end

            slotIconIndex = slotIconIndex + 1
            previousSlotIcon = item
        end
    else
        self.slotIcons[slotName] = {}
    end

    if not slotIcon then
        local parent = previousSlotIcon and previousSlotIcon.frame or _G[self:_GetSlotInfo(slotName, SLOT_FRAME_NAME)]
        local alignment = self:_GetSlotInfo(slotName, SLOT_ALIGNMENT)
        local iconScale = 1
        local iconSize = 16 / iconScale
        local iconSpacing = 1 / iconScale
        local dx = previousSlotIcon and iconSpacing or 10 / iconScale
        local dy = not previousSlotIcon and 4 or 0

        slotIcon = addonNamespace.AllocateSlotIcon(parent)

        if alignment == "LEFT" then
            slotIcon.frame:SetPoint("BOTTOMRIGHT", parent, "BOTTOMLEFT", -dx, dy)
        elseif alignment == "RIGHT" then
            slotIcon.frame:SetPoint("BOTTOMLEFT", parent, "BOTTOMRIGHT", dx, dy)
        end

        slotIcon.frame:SetSize(iconSize, iconSize)
        slotIcon.frame:SetScale(iconScale)

        table.insert(self.slotIcons[slotName], slotIcon)
    end

    slotIcon:Render(textureName, tooltip)
end

function SlotIconManager:_ShowSlotText(slotName, value)
    local key = slotName .. "-" .. self.style

    if not self.slotText[key] then
        local parent = _G[self:_GetSlotInfo(slotName, SLOT_FRAME_NAME)]
        local alignment = self:_GetSlotInfo(slotName, SLOT_ALIGNMENT)
        local fontStyle = nil
        if bit.band(self.style, SlotIconManager.STYLE.SMALL) == SlotIconManager.STYLE.SMALL then
            fontStyle = "SystemFont_Small"
        else
            fontStyle = "SystemFont_Med1"
        end
        self.slotText[key] = parent:CreateFontString("KILFrameNew_" .. slotName, "OVERLAY", fontStyle)
        local dx = 10
        local dy = -4
        if alignment == "LEFT" then
            self.slotText[key]:SetPoint("TOPRIGHT", parent, "TOPLEFT", -dx, dy)
        elseif alignment == "RIGHT" then
            self.slotText[key]:SetPoint("TOPLEFT", parent, "TOPRIGHT", dx, dy)
        end
    end

    self.slotText[key]:Show()
    self.slotText[key]:SetText(value)
end

function SlotIconManager:_FindItemInfo(who)
    self:_HideUI()

    if not who then
        return
    end

    self.visible = true

    local ilvlSum = 0
    local totalCurrentUpgrades = 0
    local totalMaxUpgrades = 0
    local mainHandItem = nil
    local mainHandItemLevel = 0
    local secondaryHandItem = nil
    local secondaryHandItemLevel = 0

    for i, slotName in pairs(self:_GetSlotNames()) do
        local slotId = self:_GetSlotInfo(slotName, SLOT_ID)
        local itemString = GetInventoryItemLink(who, slotId)
        local ilvl = 0

        if itemString then
            local itemInfo = addonNamespace.ItemStringInfo:new(itemString)
            local color = itemInfo:getQualityColor()

            ilvl = itemInfo:getItemLevel() or 0

            if slotName == "MainHandSlot" then
                mainHandItem = itemInfo
                mainHandItemLevel = ilvl
            elseif slotName == "SecondaryHandSlot" then
                secondaryHandItem = itemInfo
                secondaryHandItemLevel = ilvl
            end

            local ilvlString = "" .. ilvl

            local current, max = itemInfo:GetUpgrades()
            if current and max then
                if bit.band(self.style, SlotIconManager.STYLE.UPGRADES) == SlotIconManager.STYLE.UPGRADES then
                    ilvlString = ilvlString .. " (" .. current .. "/" .. max .. ")"
                end
                totalCurrentUpgrades = totalCurrentUpgrades + current
                totalMaxUpgrades = totalMaxUpgrades + max
            end

            if KibsItemLevel.db.profile.Color and color then
                ilvlString = "|c" .. color .. ilvlString .. "|r"
            end

            self:_ShowSlotText(slotName, ilvlString)

            local tooltip = addonNamespace.Tooltip:new()
            local enchantInfo = itemInfo:getEnchantInfo()

            if enchantInfo then
                local consumable = enchantInfo:getConsumableItem()
                local formula = enchantInfo:getFormulaItem()
                local receipe = enchantInfo:getReceipeSpell()
                local texture =
                consumable and consumable:getTextureName()
                        or formula and formula:getTextureName()
                        or receipe and receipe:getTextureName()
                        or "INTERFACE/ICONS/INV_Misc_QuestionMark"

                if consumable then
                    tooltip:AddHyperlink(consumable:getLink())
                elseif formula then
                    tooltip:AddHyperlink(formula:getLink())
                elseif receipe then
                    tooltip:AddHyperlink(receipe:getLink())
                elseif KIBC_EnchantToSpellID[enchantInfo:getId()] then
                    local spellInfo = addonNamespace.SpellInfo:new("enchant:" .. KIBC_EnchantToSpellID[enchantInfo:getId()])
                    texture = spellInfo:getTextureName()
                    tooltip:AddHyperlink(spellInfo:getLink())
                else
                    tooltip:AddText(string.format(KibsItemLevel.L["Unknown enchant #%d"], enchantInfo:getId()))
                end

                self:_ShowSlotIcon(slotName, texture, tooltip)
            elseif UnitLevel(who) == 100 and self:_GetSlotInfo(slotName, REQUIRED_FLAG) then
                tooltip:AddText(KibsItemLevel.L["Missing enchant"])
                self:_ShowSlotIcon(slotName, "INTERFACE/BUTTONS/UI-GROUPLOOT-PASS-UP", tooltip)
            end

            for _, socketInfo in pairs(itemInfo:getSockets()) do
                local texture
                local tooltip = addonNamespace.Tooltip:new()

                if not socketInfo:isEmpty() then
                    texture = socketInfo:getGem():getTextureName()
                    tooltip:AddHyperlink(socketInfo:getGem():getLink())
                elseif UnitLevel(who) == 100 then
                    texture = socketInfo:getTextureName()
                    tooltip:AddText(KibsItemLevel.L["Missing gem"])
                end

                if texture then
                    self:_ShowSlotIcon(slotName, texture, tooltip)
                end
            end
        end

        ilvlSum = ilvlSum + ilvl
    end

    local itemCount = table.getn(self:_GetSlotNames())

    if mainHandItem and mainHandItem:isTwoHandedWeapon() and not secondaryHandItem then
        ilvlSum = ilvlSum + mainHandItemLevel
    end

    local averageItemLevel = ilvlSum / itemCount

    self:_ShowAverageItemLevelFontString(averageItemLevel, totalCurrentUpgrades, totalMaxUpgrades)
end

function SlotIconManager:_ShowAverageItemLevelFontString(averageItemLevel, totalCurrentUpgrades, totalMaxUpgrades)
    local text

    if KibsItemLevel.db.profile.UpgradesSummary then
        local format = KibsItemLevel.L["平均装等: %.1f (%d/%d)"]
        text = string.format(format, averageItemLevel, totalCurrentUpgrades, totalMaxUpgrades)
    else
        local format = KibsItemLevel.L["平均装等: %.1f"]
        text = string.format(format, averageItemLevel)
    end

    self.averageItemLevelFontString:Show()
    self.averageItemLevelFontString:SetText(NORMAL_FONT_COLOR_CODE
            .. text
            .. FONT_COLOR_CODE_CLOSE)
end
