/* Copyright 2011 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the sample code usage restrictions document for further information.
 *
 */

package com.esri.arcgis.android.samples.georssfeeds;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import android.app.Activity;
import android.app.DatePickerDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.graphics.Color;
import android.os.AsyncTask;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.DatePicker;
import android.widget.TextView;
import android.widget.Toast;

import com.esri.android.map.Callout;
import com.esri.android.map.GraphicsLayer;
import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISDynamicMapServiceLayer;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.core.geometry.Point;
import com.esri.core.map.Graphic;
import com.esri.core.symbol.SimpleMarkerSymbol;


public class GeoRSSFeeds extends Activity {
	
	MapView map = null;
	GraphicsLayer graphicsLayer = null;
	
	static final int DATE_DIALOG_ID = 0;
	private int mYear = 2011;
	private int mMonth = 11;
	private int mDay = 1;
	private ProgressDialog progress = null;

    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);

		// Retrieve the map and initial extent from XML layout
		map = (MapView)findViewById(R.id.map);
		// Add dynamic layer to MapView
		map.addLayer(new ArcGISDynamicMapServiceLayer("" +
				"http://server.arcgisonline.com/ArcGIS/rest/services/ESRI_StreetMap_World_2D/MapServer"));
		
		graphicsLayer = new GraphicsLayer();
		
		map.addLayer(graphicsLayer);
		
		MyOnSingleTapListener listener = new MyOnSingleTapListener(this);
		map.setOnSingleTapListener(listener);		
		
		Button datePicker = (Button) findViewById(R.id.datePicker1);
		
		datePicker.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {
				showDialog(DATE_DIALOG_ID);
			}
		});		
		
    }  
    
	protected Dialog onCreateDialog(int id) {
		switch (id) {
		case DATE_DIALOG_ID:
			return new DatePickerDialog(this, mDateSetListener, mYear, mMonth, mDay);
		}
		return null;
	}    
    
	private DatePickerDialog.OnDateSetListener mDateSetListener = new DatePickerDialog.OnDateSetListener() {
		public void onDateSet(DatePicker view, int year, int monthOfYear, int dayOfMonth) {
			mYear = year;
			mMonth = monthOfYear;
			mDay = dayOfMonth;

			String sday;
			String smonth;
			String syear;

			if (mDay < 10) {
				sday = "0" + mDay;
			} else {
				sday = "" + mDay;
			}

			mMonth++;
			if (mMonth < 10) {
				smonth = "0" + mMonth;
			} else {
				smonth = "" + mMonth;
			}

			syear = mYear - 2000 + "";

			String date = syear + smonth + sday;

			AsyncEventFetching aef = new AsyncEventFetching();
			aef.execute(date);

		}
	};    
    
	private void getGeoRSS(String date) {
		/*
		 * This is where the feeds are brought in for tornado reports, hail
		 * reports and wind reports.
		 * The reports come in with X,Ys and comments. The reported locations
		 * are added as features to a GraphicsLayer.
		 */
		graphicsLayer.removeAll();
		getWindEvents(date);
		getHailEvents(date);
		getTornadoEvents(date);
	}	
	
	private void getWindEvents(String date) {

		// Find all of the WIND events. Use URLConnection and read in each line
		// of the CSV file.
		try {
			URL wind = new URL("http://www.spc.noaa.gov/climo/reports/" + date
					+ "_rpts_wind.csv");
			URLConnection windc = wind.openConnection();
			BufferedReader in = new BufferedReader(new InputStreamReader(
					windc.getInputStream()));
			String inputLine;
			String[] columnNames = {};
			Point pt;
			Graphic graphic;
			
			Map<String, Object> attr = new HashMap<String, Object>();
			
			ArrayList<Graphic> windEvents = new ArrayList<Graphic>();
			SimpleMarkerSymbol symbol = new SimpleMarkerSymbol(Color.BLUE, 20,
					SimpleMarkerSymbol.STYLE.DIAMOND);

			while ((inputLine = in.readLine()) != null) {
				String[] values = inputLine.split(",");
				if (values[0].equals("Time")) {
					columnNames = values;
					continue;
				}

				double lat = Double.parseDouble(values[5]);
				double lon = Double.parseDouble(values[6]);

				pt = new Point(lon, lat);

				// Create attributes to populate graphic.
				int index = 0;
				for (String key : columnNames) {
					attr.put(key, values[index]);
					index++;
				}
				
				graphic = new Graphic(pt, symbol, attr, null);
				windEvents.add(graphic);
			}
			
			// Call this to add reports to the GraphicsLayer...
			addReports(windEvents);

			// Close the input stream...
			in.close();

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}	

	private void getHailEvents(String date) {

		// Find all of the WIND events. Use URLConnection and read in each line
		// of the CSV file.
		try {
			URL wind = new URL("http://www.spc.noaa.gov/climo/reports/" + date
					+ "_rpts_wind.csv");
			URLConnection windc = wind.openConnection();
			BufferedReader in = new BufferedReader(new InputStreamReader(
					windc.getInputStream()));
			String inputLine;
			String[] columnNames = {};
			Point pt;
			Graphic graphic;
			
			Map<String, Object> attr = new HashMap<String, Object>();
			
			ArrayList<Graphic> hailEvents = new ArrayList<Graphic>();
			SimpleMarkerSymbol symbol = new SimpleMarkerSymbol(Color.GREEN, 20,
					SimpleMarkerSymbol.STYLE.DIAMOND);

			while ((inputLine = in.readLine()) != null) {
				String[] values = inputLine.split(",");
				if (values[0].equals("Time")) {
					columnNames = values;
					continue;
				}

				double lat = Double.parseDouble(values[5]);
				double lon = Double.parseDouble(values[6]);

				pt = new Point(lon, lat);

				// Create attributes to populate graphic.
				int index = 0;
				for (String key : columnNames) {
					attr.put(key, values[index]);
					index++;
				}
				
				graphic = new Graphic(pt, symbol, attr, null);
				hailEvents.add(graphic);
			}
			
			// Call this to add reports to the GraphicsLayer...
			addReports(hailEvents);

			// Close the input stream...
			in.close();

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}		
	
	private void getTornadoEvents(String date) {

		// Find all of the TORNADO events. Use URLConnection and read in each
		// line of the CSV file.
		try {
			URL wind = new URL("http://www.spc.noaa.gov/climo/reports/" + date
					+ "_rpts_torn.csv");
			URLConnection windc = wind.openConnection();
			BufferedReader in = new BufferedReader(new InputStreamReader(
					windc.getInputStream()));
			String inputLine;
			String[] columnNames = {};
			Point pt;
			Graphic graphic;
			
			Map<String, Object> attr = new HashMap<String, Object>();
			
			ArrayList<Graphic> tornadoEvents = new ArrayList<Graphic>();
			
			SimpleMarkerSymbol symbol = new SimpleMarkerSymbol(Color.RED, 20,
					SimpleMarkerSymbol.STYLE.DIAMOND);

			while ((inputLine = in.readLine()) != null) {
				String[] values = inputLine.split(",");
				if (values[0].equals("Time")) {
					columnNames = values;
					continue;
				}

				double lat = Double.parseDouble(values[5]);
				double lon = Double.parseDouble(values[6]);

				pt = new Point(lon, lat);

				// Create attributes to populate graphic.
				int index = 0;
				for (String key : columnNames) {
					attr.put(key, values[index]);
					index++;
				}
				
				graphic = new Graphic(pt, symbol, attr, null);
				tornadoEvents.add(graphic);

			}

			addReports(tornadoEvents);
			in.close();

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	private void addReports(ArrayList<Graphic> graphics) {
		Graphic graphic;
		// if there is something to add, do it...
		if (graphics.size() > 0) {

			for(int i=0; i < graphics.size(); i++){
				graphic = graphics.get(i);
				graphicsLayer.addGraphic(graphic);
				
			}
		}

	}	
	
	@Override
	protected void onPause() {
		super.onPause();
		map.pause();
	}
	@Override 	protected void onResume() {
		super.onResume(); 
		map.unpause();
	}

	
	private class MyOnSingleTapListener implements OnSingleTapListener {

		// Here, we use a single tap to popup the attributes for a report...
		Context _ctx;
		private static final long serialVersionUID = 1L;
		public MyOnSingleTapListener(Context ctx) {
			_ctx = ctx;
		}

		public void onSingleTap(float x, float y) {
			Callout mapCallout = map.getCallout();
			mapCallout.hide();
			Point pnt = map.toMapPoint(x, y);
			
			int[] grs = graphicsLayer.getGraphicIDs(x, y, 20);
			Log.d("Test", "Graphics number is " + grs.length);
			
			if(grs.length > 0){
				Graphic g = graphicsLayer.getGraphic(grs[0]);
				Map<String, Object> atts = g.getAttributes();
				String text = "";
				for (int i = 0; i < atts.size(); i++) {
					text = text + atts.keySet().toArray()[i] + ": "
							+ atts.values().toArray()[i] + "\n";
				}

				TextView tv = new TextView(_ctx);
				tv.setText(text);	
				
				// Here, we populate the Callout with the attribute information
				// from the report.
				mapCallout.setOffset(0, -3);
				mapCallout.setCoordinates(pnt);
				mapCallout.setMaxHeight(200);
				mapCallout.setMaxWidth(400);
				mapCallout.setStyle(R.xml.mycalloutprefs);
				mapCallout.setContent(tv);

				mapCallout.show();				
			}

		}
	}
	
	private class AsyncEventFetching extends AsyncTask<String, Void, Void> {

		protected void onPreExecute() {
			if (map.getCallout().isShowing()) {
				map.getCallout().hide();
			}
			progress = ProgressDialog.show(GeoRSSFeeds.this, "",
					"Fetching Geo RSS, Please wait....", true);

		}

		protected Void doInBackground(String... dateStrs) {
			if (dateStrs.length > 0) {
				getGeoRSS(dateStrs[0]);
			}
			return null;
		}

		protected void onPostExecute(Void result) {
			progress.dismiss();
			if (graphicsLayer != null && graphicsLayer.getNumberOfGraphics() == 0) {
				Toast.makeText(GeoRSSFeeds.this,
						"There were no storm reports on this date.",
						Toast.LENGTH_SHORT).show();
			} else {
				Toast.makeText(GeoRSSFeeds.this,
						"tap on an event to see it's information.",
						Toast.LENGTH_SHORT).show();
			}
			
		}

	}	
	
}