/* Copyright 2012 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the sample code usage restrictions document for further information.
 *
 */

package com.esri.arcgis.android.samples.viewshed;

import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;

import android.app.Activity;
import android.app.ProgressDialog;
import android.graphics.Color;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Handler.Callback;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.ImageButton;
import android.widget.Toast;

import com.esri.android.map.GraphicsLayer;
import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISTiledMapServiceLayer;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.Point;
import com.esri.core.map.Graphic;
import com.esri.core.symbol.SimpleFillSymbol;
import com.esri.core.symbol.SimpleMarkerSymbol;
import com.esri.core.symbol.SimpleMarkerSymbol.STYLE;
import com.esri.core.tasks.ags.geoprocessing.GPFeatureRecordSetLayer;
import com.esri.core.tasks.ags.geoprocessing.GPLinearUnit;
import com.esri.core.tasks.ags.geoprocessing.GPParameter;
import com.esri.core.tasks.ags.geoprocessing.GPResultResource;
import com.esri.core.tasks.ags.geoprocessing.Geoprocessor;

/**
 * This sample application illustrates the usage of Geoprocessing. It allows the
 * user to select a point on the map and see all the areas that are visible
 * within the chosen distance.
 * @version $Revision: 1.0 $
 */

public class Viewshed extends Activity {


	protected static final int CLOSE_LOADING_WINDOW = 0;
	protected static final int CANCEL_LOADING_WINDOW = 1;
	MapView map = null;
	private ArrayList<GPParameter> params;
	Geoprocessor gp;
	GraphicsLayer gLayer;
	ProgressDialog dialog = null;
	Timer cancelViewShed = new Timer();
	private ImageButton delete, go;
	Point mappoint;


	Handler uiHandler = new Handler(new Callback() {
		@Override
		public boolean handleMessage(final Message msg) {
			switch (msg.what) {
			case CLOSE_LOADING_WINDOW:
				if (dialog != null) {
					dialog.dismiss();
				}
				cancelViewShed.cancel();
				break;
			case CANCEL_LOADING_WINDOW:
				if (dialog != null) {
					dialog.dismiss();
				}
				Toast toast = Toast.makeText(Viewshed.this,
						"ViewShed too long, canceled", Toast.LENGTH_SHORT);
				toast.show();
				break;
			}
			return false;
		}

	});

	/** Called when the activity is first created. 
	 * @param savedInstanceState Bundle
	 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);
		
		map = (MapView)findViewById(R.id.map);
		// Add Tiled layer to MapView
		ArcGISTiledMapServiceLayer baselayer = new ArcGISTiledMapServiceLayer("http://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer");
		map.addLayer(baselayer);
		// Create a graphics layer for Viewshed drawing
		gLayer = new GraphicsLayer();
		map.addLayer(gLayer);
		// Map Buttons
		delete = (ImageButton)findViewById(R.id.deletebutton);
		go = (ImageButton)findViewById(R.id.gobutton);
		
		Toast.makeText(this, "Single tap on the map", Toast.LENGTH_SHORT).show();

		/**
		 * Clear all graphics from the graphics layer. The method is called when
		 * the trash can button is clicked by the user.
		 */
		delete.setOnClickListener(new OnClickListener() {
			
			@Override
			public void onClick(View v) {
				if(v.isEnabled()){
					gLayer.removeAll();
				}
			}
		});
		
		/**
		 * In response to the clicking of the arrow button
		 */
		go.setOnClickListener(new OnClickListener() {
			
			@Override
			public void onClick(View v) {
				if(v.isEnabled()){
					start(mappoint);
				}else{
					Log.d("TAG", "View not enabled");
				}
			}
		});

		map.setOnSingleTapListener(new OnSingleTapListener() {
			private static final long serialVersionUID = 1L;

			@Override
			public void onSingleTap(final float x, final float y) {
				mappoint = map.toMapPoint(x, y);
				Graphic g = new Graphic(mappoint,new SimpleMarkerSymbol(Color.RED, 10, STYLE.CIRCLE));
				gLayer.addGraphic(g);
			}
		});
		
		//Retrieve the non-configuration instance data that was previously returned. 
		Object init = getLastNonConfigurationInstance();
		if (init != null) {
			map.restoreState((String) init);
		}		
		
	}

	class ViewShedQuery extends AsyncTask<ArrayList<GPParameter>, Void, GPParameter[]> {

		GPParameter[] outParams = null;

		/**
		 * Method onPostExecute.
		 * @param result GPParameter[]
		 */
		@Override
		protected void onPostExecute(GPParameter[] result) {
			if (result == null)
				return;
			for (int i = 0; i < result.length; i++) {
				if (result[i] instanceof GPFeatureRecordSetLayer) {

					GPFeatureRecordSetLayer fsl = (GPFeatureRecordSetLayer) result[i];
					for (Graphic feature : fsl.getGraphics()) {
						Graphic g = new Graphic(feature.getGeometry(),new SimpleFillSymbol(Color.CYAN));
						gLayer.addGraphic(g);
					}
				}
			}
			uiHandler.sendEmptyMessage(CLOSE_LOADING_WINDOW);
		}

		/**
		 * Method doInBackground.
		 * @param params1 ArrayList<GPParameter>[]
		 * @return GPParameter[]
		 */
		@Override
		protected GPParameter[] doInBackground(ArrayList<GPParameter>... params1) {

			gp = new Geoprocessor("http://sampleserver1.arcgisonline.com/ArcGIS/rest/services/Elevation/ESRI_Elevation_World/GPServer/Viewshed");
			/*
			 * API v1.1 requires SpatialReference parameter
			 */
			gp.setOutSR(map.getSpatialReference());
			try {
				GPResultResource rr = gp.execute(params1[0]);
				outParams = rr.getOutputParameters();
			} catch (Exception e) {
				e.printStackTrace();
			}
			return outParams;
		}
	}

	/**
	 * 1) Create the GP object by passing it the url. 2) Create the input
	 * parameters and add to the GPParameter object. 3) Pass these params to the
	 * execute method.
	 * 
	 * @param mPoint
	 *            - point selected on the map by the user
	 */
	@SuppressWarnings("unchecked")
	public void start(Point mPoint) {
		// First input parameter
		GPFeatureRecordSetLayer gpf = new GPFeatureRecordSetLayer("Input_Observation_Point");
		gpf.setSpatialReference(map.getSpatialReference());
		gpf.setGeometryType(Geometry.Type.POINT);
		// Add the point selected by the user
		Graphic f = new Graphic(mPoint,new SimpleMarkerSymbol(Color.RED,25,STYLE.DIAMOND));
		gpf.addGraphic(f);

		// Second input parameter
		GPLinearUnit gpl = new GPLinearUnit("Viewshed_Distance");
		gpl.setUnits("esriMeters");
		gpl.setDistance(8046.72);

		// Add params
		params = new ArrayList<GPParameter>();
		params.add(gpf);
		params.add(gpl);

		try {
			dialog = ProgressDialog.show(Viewshed.this, "", "Loading. Please wait...", true, true);
			new ViewShedQuery().execute(params);
			cancelViewShed = new Timer();
			cancelViewShed.schedule(new TimerTask() {

				@Override
				public void run() {
					uiHandler.sendEmptyMessage(CANCEL_LOADING_WINDOW);
				}
			}, 60000);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	@Override
	protected void onPause() {
		super.onPause();
		map.pause();
	}
	
	@Override 	
	protected void onResume() {
		super.onResume(); 
		map.unpause();
	}	

}
