package com.esri.arcgis.android.samples.identifytask;
/* Copyright 2010 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the Sample code usage restrictions document for further information.
 *
 */

import java.util.ArrayList;
import java.util.List;

import android.app.Activity;
import android.content.Context;
import android.graphics.Color;
import android.os.AsyncTask;
import android.os.Bundle;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.TextView;

import com.esri.android.action.IdentifyResultSpinner;
import com.esri.android.action.IdentifyResultSpinnerAdapter;
import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISTiledMapServiceLayer;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Point;
import com.esri.core.tasks.ags.identify.IdentifyParameters;
import com.esri.core.tasks.ags.identify.IdentifyResult;
import com.esri.core.tasks.ags.identify.IdentifyTask;



/**
 * This sample allows the user to identify data based on single tap
 * and view the results in a callout window which has a spinner
 * in its layout. Also the user can select any of the results displayed
 * and view its details. The details are the attribute values.
 * 
 * The output value shown in the spinner is the display field.
 *
 */



public class Identify extends Activity {
	
	
	MapView map = null;
	
	IdentifyParameters params;
	
	/** Called when the activity is first created. */
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		map = new MapView(this);
		
		ArcGISTiledMapServiceLayer basemap = new ArcGISTiledMapServiceLayer("http://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer");
		
		map.addLayer(basemap);


		ArcGISTiledMapServiceLayer layer = new ArcGISTiledMapServiceLayer("http://services.arcgisonline.com/ArcGIS/rest/services/Demographics/USA_Average_Household_Size/MapServer");
		map.addLayer(layer);		
		
		// Create an extent for initial extent
		Envelope env = new Envelope(-19332033.11, -3516.27, -1720941.80, 11737211.28);
		// Set the MapView initial extent
		map.setExtent(env);
		setContentView(map);		
		params = new IdentifyParameters();
		params.setTolerance(20);
		params.setDPI(98);
		params.setLayers(new int[]{4});
		params.setLayerMode(IdentifyParameters.ALL_LAYERS);
		
		
		map.setOnSingleTapListener(new OnSingleTapListener() {
	
			private static final long serialVersionUID = 1L;			
			
			public void onSingleTap(final float x, final float y) {
				
				if(!map.isLoaded()){
						return;
				}
					//establish the identify parameters	
					Point identifyPoint = map.toMapPoint(x, y);				
					params.setGeometry(identifyPoint);
					params.setSpatialReference(map.getSpatialReference());									
					params.setMapHeight(map.getHeight());
					params.setMapWidth(map.getWidth());
					Envelope env = new Envelope();
					map.getExtent().queryEnvelope(env);
					params.setMapExtent(env);
					
					MyIdentifyTask mTask = new MyIdentifyTask(identifyPoint);
					mTask.execute(params);					
				}

		});	
		
	}
	 
	


	private ViewGroup createIdentifyContent(final List<IdentifyResult> results){

          LinearLayout layout = new LinearLayout(this);
          layout.setLayoutParams(new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));
          layout.setOrientation(LinearLayout.HORIZONTAL);
          
          IdentifyResultSpinner spinner = new IdentifyResultSpinner(this, (List<IdentifyResult>) results);
          spinner.setClickable(true);  
          MyIdentifyAdapter adapter = new MyIdentifyAdapter(this, results);
          spinner.setAdapter(adapter);
          spinner.setLayoutParams(new LayoutParams(LayoutParams.FILL_PARENT,LayoutParams.WRAP_CONTENT));          
          layout.addView(spinner);
          
          
          
         return layout;
    }
    
	/**
	 * This class allows the user to customize the string shown in the callout.
	 * By default its the display field name.
	 *
	 */
	public class MyIdentifyAdapter extends IdentifyResultSpinnerAdapter{
		String m_show = null;
		List<IdentifyResult> resultList;
		int currentDataViewed = -1;
		Context m_context;


		public MyIdentifyAdapter(Context context, List<IdentifyResult> results) {
			super(context,results);
			this.resultList = results;
			this.m_context = context;
			
		}

		//This is the view that will get added to the callout
		//Create a text view and assign the text that should be visible in the callout		
		public View getView(int position, View convertView, ViewGroup parent) {
			String outputVal = null;
			TextView txtView;
			IdentifyResult curResult = this.resultList.get(position);
	
			
			if(curResult.getAttributes().containsKey("Name")){			
				outputVal = curResult.getAttributes().get("Name").toString();			
			}
			
			txtView = new TextView(this.m_context);
			txtView.setText(outputVal);
			txtView.setTextColor(Color.BLACK);
			txtView.setLayoutParams(new ListView.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.WRAP_CONTENT));
			txtView.setGravity(Gravity.CENTER_VERTICAL);
		
			return txtView;
		}
		
	}
	
	
	@Override
	protected void onPause() {
		super.onPause();
		map.pause();
    }
	@Override 	
	protected void onResume() {
		super.onResume(); 
		map.unpause();
	}
	
	private class MyIdentifyTask extends AsyncTask<IdentifyParameters, Void, IdentifyResult[]> {

		IdentifyTask mIdentifyTask;
		Point mAnchor;
		MyIdentifyTask(Point anchorPoint) {
			mAnchor = anchorPoint;
		}
		@Override
		protected IdentifyResult[] doInBackground(IdentifyParameters... params) {
			IdentifyResult[] mResult = null;
			if (params != null && params.length > 0) {
				IdentifyParameters mParams = params[0];
				try {
					mResult = mIdentifyTask.execute(mParams);
				} catch (Exception e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				
			}
			return mResult;
		}
		
		
		
		

		@Override
		protected void onPostExecute(IdentifyResult[] results) {
			// TODO Auto-generated method stub
			
			ArrayList<IdentifyResult> resultList = new ArrayList<IdentifyResult>();
			for (int index=0; index < results.length; index++){
				
				if(results[index].getAttributes().get(results[index].getDisplayFieldName())!=null)
					resultList.add(results[index]);
			}
			
			//map.getCallout().show(map.toMapPoint(x,y), createIdentifyContent(resultList));
			map.getCallout().show(mAnchor, createIdentifyContent(resultList));
		}

		@Override
		protected void onPreExecute() {
			//mIdentifyTask = new IdentifyTask("http://sampleserver3.arcgisonline.com/ArcGIS/rest/services/Earthquakes/EarthquakesFromLastSevenDays/MapServer");
			mIdentifyTask = new IdentifyTask("http://services.arcgisonline.com/ArcGIS/rest/services/Demographics/USA_Average_Household_Size/MapServer");
		}
		
		
		
	}
	
}