/* Copyright 2010 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the Sample code usage restrictions document for further information.
 *
 */

package com.esri.arcgis.android.samples.agslocator;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;

import android.app.Activity;
import android.app.ProgressDialog;
import android.graphics.Color;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import android.os.Handler.Callback;
import android.os.Message;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.Toast;

import com.esri.android.map.GraphicsLayer;
import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISDynamicMapServiceLayer;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.SpatialReference;
import com.esri.core.map.Graphic;
import com.esri.core.symbol.SimpleMarkerSymbol;
import com.esri.core.symbol.TextSymbol;
import com.esri.core.tasks.ags.geocode.Locator;
import com.esri.core.tasks.ags.geocode.LocatorGeocodeResult;
import com.esri.core.tasks.ags.geocode.LocatorReverseGeocodeResult;

public class AGS_Locator extends Activity {
	
		
	/*
	 * ArcGIS Android elements
	 */
	MapView map = null;
	GraphicsLayer graphicsLayer = null;
	Locator al;

	/*
	 * Android UI elements
	 */
	Graphic markerGraphic;
	EditText edtText;
	Button btnExecuteGeocode;

	/*
	 * UI Handler
	 */
	ProgressDialog dialog = null;
	static final int CLOSE_LOADING_WINDOW = 0;
	static final int CANCEL_LOADING_WINDOW = 1;
	Timer cancelLocate = new Timer();

	private Handler uiHandler = new Handler(new Callback() {

		public boolean handleMessage(Message msg) {
			switch (msg.what) {
			case CLOSE_LOADING_WINDOW:
				if (dialog != null) {
					dialog.dismiss();
				}
				cancelLocate.cancel();
				break;
			case CANCEL_LOADING_WINDOW:
				if (dialog != null) {
					dialog.dismiss();
				}
				Toast toast = Toast.makeText(AGS_Locator.this,
						"Locate canceled", Toast.LENGTH_SHORT);
				toast.show();
				break;
			}
			return false;
		}

	});

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {

		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		/*
		 * initialize ArcGIS Android MapView, Graphics Layer, Android UI
		 * elements
		 */
		map = (MapView) findViewById(R.id.map);
		map.addLayer(new ArcGISDynamicMapServiceLayer("http://server.arcgisonline.com/ArcGIS/rest/services/ESRI_StreetMap_World_2D/MapServer"));
        graphicsLayer = new GraphicsLayer();
        map.addLayer(graphicsLayer);
        
		edtText = (EditText) findViewById(R.id.TextEditView);
		btnExecuteGeocode = (Button) findViewById(R.id.executeGeocode);

		// perform reverse geocode on single tap.
		map.setOnSingleTapListener(new OnSingleTapListener() {

			private static final long serialVersionUID = 1L;

			public void onSingleTap(final float x, final float y) {


				graphicsLayer.removeAll();

				// retrieve the user clicked location
				final Point loc = map.toMapPoint(x, y);

				// initialize arcgis locator
				al = new Locator(
						"http://sampleserver1.arcgisonline.com/ArcGIS/rest/services/Locators/ESRI_Geocode_USA/GeocodeServer");
				try {

					/*
					 * API v1.1 method signature changed to take SpatialReference
					 * parameter instead of int wkid.
					 */
					SpatialReference sr = map.getSpatialReference();
					// retrieve LocatorReverseGeocodeResult
					LocatorReverseGeocodeResult result = al.reverseGeocode(loc,
							1000.00, sr, sr);

					// checks if State and Zip is present in the result
					if (result.getAddressFields().get("State").length() != 2
							&& result.getAddressFields().get("Zip").length() != 5) {

    					map.getCallout().show(loc, message("No Address Found."));

					} else {

						// display the result in map callout
						String msg = "Address:"
								+ result.getAddressFields().get("Address")
								+ "\n" + "City:"
								+ result.getAddressFields().get("City") + "\n"
								+ " State:"
								+ result.getAddressFields().get("State") + "\n"
								+ " Zip:"
								+ result.getAddressFields().get("Zip");
						map.getCallout().show(loc, message(msg));

					}
				} catch (Exception e) {

					e.printStackTrace();
//					map.getCallout().setAnchor(Callout.ANCHOR_POSITION_FLOATING);
					map.getCallout().show(loc, message("No Address Found."));

				}

			}
		});
		// execute geocoding task on button click.
		btnExecuteGeocode.setOnClickListener(new View.OnClickListener() {

			public void onClick(View v) {

				graphicsLayer.removeAll();
				map.getCallout().hide();
				findMapPoint();

			}
		});
	}

	/*
	 * Cutomize the map Callout text
	 */
	private TextView message(String text) {

		final TextView msg = new TextView(this);
		msg.setText(text);
		msg.setTextSize(12);
		msg.setTextColor(Color.BLACK);
		return msg;

	}

	/*
	 * Execute geocode task asynchronously.
	 */
	class AgsLocator
			extends
			AsyncTask<java.util.Map<java.lang.String, java.lang.String>, Void, java.util.List<LocatorGeocodeResult>> {

		@Override
		protected void onPostExecute(java.util.List<LocatorGeocodeResult> result) {

			if (result.size() == 0) {
				Toast toast = Toast.makeText(AGS_Locator.this,
						"No result found.", Toast.LENGTH_SHORT);
				toast.show();

			} else {
				dialog = ProgressDialog.show(AGS_Locator.this, "",
						"Loading. Please wait...", true, true);
				cancelLocate = new Timer();
				cancelLocate.schedule(new TimerTask() {

					@Override
					public void run() {
						uiHandler.sendEmptyMessage(CANCEL_LOADING_WINDOW);
					}

				}, 60000);

				// Create graphic to add locator result to map
				
				Geometry geomS = result.get(0).getLocation();
				SimpleMarkerSymbol smsS = new SimpleMarkerSymbol(Color.RED, 10, SimpleMarkerSymbol.STYLE.DIAMOND);
				Graphic graphicS = new Graphic( geomS, smsS );
				
				Geometry geomT = result.get(0).getLocation();
				TextSymbol tsT = new TextSymbol(16, result.get(0).getAddress(), Color.BLACK);
				tsT.setOffsetX(15);
				tsT.setOffsetY(15);
				Graphic graphicT = new Graphic( geomT, tsT );

				/*
				 * add the updated graphic to graphics layer and display the
				 * result on the map
				 */
				graphicsLayer.addGraphic(graphicS);
				graphicsLayer.addGraphic(graphicT);

				// zoom to the locator result
				map.zoomTo(result.get(0).getLocation(), 2);

				uiHandler.sendEmptyMessage(CLOSE_LOADING_WINDOW);


			}
		}

		@Override
		protected List<LocatorGeocodeResult> doInBackground(
				Map<String, String>... params) {

			SpatialReference sr = map.getSpatialReference();
			List<LocatorGeocodeResult> results = null;

			// initialize arcgis locator
			al = new Locator(
					"http://sampleserver1.arcgisonline.com/ArcGIS/rest/services/Locators/ESRI_Geocode_USA/GeocodeServer");
			try {

				
				/*
				 * API v1.1 method signature changed to take SpatialReference
				 * parameter instead of int wkid.
				 */
				// perform geocode operation
				results = al.geocode(params[0], null, sr);
			} catch (Exception e) {
				e.getMessage();
			}
			return results;
		}

	}

	@SuppressWarnings("unchecked")
	public void findMapPoint() {

		try {
			HashMap<String, String> addressFields = new HashMap<String, String>();
			// get the user entered address and create arcgis
			// locator address fields
			String line = edtText.getText().toString();

			StringTokenizer st = new StringTokenizer(line, ",");

			if (st.countTokens() == 4) {

				addressFields.put("Address", st.nextToken());
				addressFields.put("City", st.nextToken());
				addressFields.put("State", st.nextToken());
				addressFields.put("Zip", st.nextToken());

			} else if (st.countTokens() == 3) {

				addressFields.put("Address", st.nextToken());
				addressFields.put("City", st.nextToken());
				addressFields.put("State", st.nextToken());

			} else {

				Toast toast = Toast.makeText(AGS_Locator.this,
						"Please enter address in correct format.",
						Toast.LENGTH_SHORT);
				toast.show();

			}

			if (addressFields.get("Address") != null
					&& addressFields.get("State") != null) {

				new AgsLocator().execute(addressFields);

			}

		} catch (Exception e) {
			e.printStackTrace();
		}

	}	
	
	@Override
	protected void onPause() {
		super.onPause();
		map.pause();
 }
	@Override 	
	protected void onResume() {
		super.onResume(); 
		map.unpause();
	}		
	
}