/* Copyright 2012 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the �Sample code usage restrictions� document for further information.
 *
 */

package com.esri.arcgis.android.samples.popupinwebmapforviewing;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;

import android.app.Activity;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.graphics.Color;
import android.os.AsyncTask;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup.LayoutParams;
import android.widget.LinearLayout;

import com.esri.android.map.Layer;
import com.esri.android.map.MapView;
import com.esri.android.map.PopupContainer;
import com.esri.android.map.PopupView;
import com.esri.android.map.ags.ArcGISDynamicMapServiceLayer;
import com.esri.android.map.ags.ArcGISFeatureLayer;
import com.esri.android.map.ags.ArcGISLayerInfo;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.SpatialReference;
import com.esri.core.map.FeatureSet;
import com.esri.core.map.Graphic;
import com.esri.core.map.popup.PopupInfo;
import com.esri.core.tasks.ags.query.Query;
import com.esri.core.tasks.ags.query.QueryTask;

public class PopupInWebmapForViewing extends Activity {
	private MapView map;
  private PopupContainer popupContainer;
  private PopupDialog popupDialog;
  private ProgressDialog progressDialog;
  private AtomicInteger count;
	
  /** Called when the activity is first created. */    
  public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    
    // Load a webmap.
    map = new MapView(this, "http://www.arcgis.com/home/item.html?id=81d2dcf906cf4df4889ec36c8dc0c1f9", "", "");
    setContentView(map);
    
    // Tap on the map and show popups for selected features.
    map.setOnSingleTapListener(new OnSingleTapListener() {
      private static final long serialVersionUID = 1L;

      public void onSingleTap(float x, float y) {    	
        if (map.isLoaded()) {
        	// Instantiate a PopupContainer
        	popupContainer = new PopupContainer(map.getContext());
        	int id = popupContainer.hashCode();
        	popupDialog = null;
        	// Display spinner.
        	if (progressDialog == null || !progressDialog.isShowing())
        		progressDialog = ProgressDialog.show(map.getContext(), "", "Querying...");

        	// Loop through each layer in the webmap
        	int tolerance = 20;
					Envelope env = new Envelope(map.toMapPoint(x, y), 20 * map.getResolution(), 20 * map.getResolution());
        	Layer[] layers = map.getLayers();
        	count = new AtomicInteger();
        	for (Layer layer : layers) {
        		// If the layer has not been initialized or is invisible, do nothing.
      			if (!layer.isInitialized() || !layer.isVisible())
      				continue;
      			
      			if (layer instanceof ArcGISFeatureLayer) { 
      				// Query feature layer and display popups
      				ArcGISFeatureLayer featureLayer = (ArcGISFeatureLayer) layer;          				
      				if (featureLayer.getPopupInfo() != null && featureLayer.getPopupInfo().isInitialized()) {
      					// Query feature layer which is associated with a popup definition.
      					count.incrementAndGet();
      					new RunQueryFeatureLayerTask(x, y, tolerance, id).execute(featureLayer);
      				}
      			}
      			else if (layer instanceof ArcGISDynamicMapServiceLayer) { 
      				// Query dynamic map service layer and display popups.
      				ArcGISDynamicMapServiceLayer dynamicLayer = (ArcGISDynamicMapServiceLayer) layer;
      				// Retrieve layer info for each sub-layer of the dynamic map service layer.
      				ArcGISLayerInfo[] layerinfos = dynamicLayer.getAllLayers();
      				if ( layerinfos == null )
      					continue;
      				
      				// Loop through each sub-layer
      				for (ArcGISLayerInfo layerInfo : layerinfos) {
      					// Obtain PopupInfo for sub-layer.
      					PopupInfo popupInfo = dynamicLayer.getPopupInfo(layerInfo.getId());
      					// Skip sub-layer which is without a popup definition.
      					if ( popupInfo == null || ! popupInfo.isInitialized() ) {
      						continue;
      					}
      					// Check if a sub-layer is visible.
      					ArcGISLayerInfo info = layerInfo;
      					while ( info != null && info.isVisible() ) {
      						info = info.getParentLayer();
      					}
      					// Skip invisible sub-layer
      					if ( info != null && ! info.isVisible() ) {
      						continue;
      					};

      					// Check if the sub-layer is within the scale range
      					double maxScale = (layerInfo.getMaxScale() != 0) ? layerInfo.getMaxScale():popupInfo.getMaxScale();
      					double minScale = (layerInfo.getMinScale() != 0) ? layerInfo.getMinScale():popupInfo.getMinScale();

      					if ((maxScale == 0 || map.getScale() > maxScale) && (minScale == 0 || map.getScale() < minScale)) {
      						// Query sub-layer which is associated with a popup definition and is visible and in scale range.
      						count.incrementAndGet();
      						new RunQueryDynamicLayerTask(env, popupInfo, dynamicLayer.getSpatialReference(), id).execute(dynamicLayer.getUrl() + "/" + layerInfo.getId());
      					}
      				}
      			}      			
        	}
        }
      }
    });
  }
  
  private void createPopupViews(Graphic[] graphics, PopupInfo popupInfo, final int id) {
  	// Create a list of PopupView from the array of graphics.
		final ArrayList<PopupView> popupViewList = new ArrayList<PopupView>();
		for (int i = 0; i < graphics.length; i++) {
			PopupView popupView = new PopupView(map.getContext(), popupInfo, graphics[i]);
			popupViewList.add(popupView);
		}			
		// Add the PopupViews to PopupContainer and display the popups.
		if ( id != popupContainer.hashCode() ) {
			// Dismiss spinner
			if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
				progressDialog.dismiss();

			return;
		}
		
		// Add PopupViews and set colors for the popup
		popupContainer.addPopupViews(popupViewList.toArray(new PopupView[0]));
		popupContainer.setPopupBackgroundColor(Color.WHITE);
		popupContainer.setPopupTextColor(Color.BLACK);
    popupContainer.setPopupSelectorColors(Color.BLACK, Color.WHITE, Color.RED, Color.GRAY);

		if (popupDialog == null) {
			// Dismiss spinner.
			if (progressDialog != null && progressDialog.isShowing()) 
				progressDialog.dismiss();
			
			// Create a dialog for the popups and display it.
			popupDialog = new PopupDialog(map.getContext(), popupContainer);
			popupDialog.show();
		}
  }
  
  // Query feature layer by hit test
  private class RunQueryFeatureLayerTask extends AsyncTask<ArcGISFeatureLayer, Void, Graphic[]> {

		private int tolerance;
		private float x;
		private float y;
		private ArcGISFeatureLayer featureLayer;
		private int id;

		public RunQueryFeatureLayerTask(float x, float y, int tolerance, int id) {
			super();
			this.x = x;
			this.y = y;
			this.tolerance = tolerance;
			this.id = id;
		}

		@Override
		protected Graphic[] doInBackground(ArcGISFeatureLayer... params) {
			for (ArcGISFeatureLayer featureLayer : params) {
				this.featureLayer = featureLayer;
				// Retrieve graphic ids near the point.
				int[] ids = featureLayer.getGraphicIDs(x, y, tolerance);
				if (ids != null && ids.length > 0) {
					ArrayList<Graphic> graphics = new ArrayList<Graphic>();
					for (int id : ids) {
						// Obtain graphic based on the id.
						Graphic g = featureLayer.getGraphic(id);
						if (g == null)
							continue;
						graphics.add(g);
					}
					// Return an array of graphics near the point.
					return graphics.toArray(new Graphic[0]);
				}
			}
			return null;
		}

		@Override
		protected void onPostExecute(Graphic[] graphics) {
			count.decrementAndGet();
			// Validate parameter.
			if (graphics == null || graphics.length == 0) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
					progressDialog.dismiss();
				
				return;
			}
			// Check if the requested PopupContainer id is the same as the current PopupContainer.
			// Otherwise, abandon the obsoleted query result.
			if ( id != popupContainer.hashCode() ) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
					progressDialog.dismiss();
				
				return;
			}
			
			PopupInfo popupInfo = featureLayer.getPopupInfo();
			if ((popupInfo == null || !popupInfo.isInitialized()) ) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0)  
					progressDialog.dismiss();

				return;
			}
			count.incrementAndGet();
			createPopupViews(graphics, popupInfo, id);
		}

	}
  
  // Query dynamic map service layer by QueryTask
  private class RunQueryDynamicLayerTask extends AsyncTask<String, Void, FeatureSet> {
		Envelope env;
		PopupInfo popupInfo;
		private SpatialReference sr;
		private int id;

		public RunQueryDynamicLayerTask(Envelope env, PopupInfo popupInfo, SpatialReference sr, int id) {
			super();
			this.env = env;
			this.popupInfo = popupInfo;
			this.sr = sr;
			this.id = id;
		}

		@Override
		protected FeatureSet doInBackground(String... urls) {
			for (String url : urls) {
				// Retrieve graphics within the envelope.
				Query query = new Query();
				query.setInSpatialReference(sr);
				query.setOutSpatialReference(sr);
				query.setGeometry(env);
				query.setMaxFeatures(10);
				query.setOutFields(new String[] { "*" });

				QueryTask queryTask = new QueryTask(url);
				try {
					FeatureSet results = queryTask.execute(query);
					return results;
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			return null;
		}

		@Override
		protected void onPostExecute(final FeatureSet result) {
			// Validate parameter.
			count.decrementAndGet();
			if (result == null) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
					progressDialog.dismiss();

				return;
			}
			Graphic[] graphics = result.getGraphics();
			if (graphics == null || graphics.length == 0) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
					progressDialog.dismiss();

				return;
			}
			// Check if the requested PopupContainer id is the same as the current PopupContainer.
			// Otherwise, abandon the obsoleted query result.
			if (id != popupContainer.hashCode()) {
				// Dismiss spinner
				if (progressDialog != null && progressDialog.isShowing() && count.intValue() == 0) 
					progressDialog.dismiss();

				return;
			}
			count.incrementAndGet();
			createPopupViews(graphics, popupInfo, id);
		}
	}
  
  // A customize full screen dialog.
  private class PopupDialog extends Dialog {
	  private View popupContainer;
	  
	  public PopupDialog(Context context, PopupContainer popupContainer) {
		  super(context, android.R.style.Theme);
		  this.popupContainer = popupContainer;
	  }
	  
	  @Override
	  protected void onCreate(Bundle savedInstanceState) {
		  super.onCreate(savedInstanceState);
		  LayoutParams params = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
			LinearLayout layout = new LinearLayout(getContext());
			layout.addView(popupContainer, android.widget.LinearLayout.LayoutParams.FILL_PARENT, android.widget.LinearLayout.LayoutParams.FILL_PARENT);
			setContentView(layout, params);
		}
	  
  }
  
}