/* Copyright 2012 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the �Sample code usage restrictions� document for further information.
 *
 */

package com.esri.arcgis.android.samples.popupinwebmapforediting;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.provider.MediaStore;
import android.view.View;
import android.view.ViewGroup.LayoutParams;
import android.widget.LinearLayout;
import android.widget.Toast;

import com.esri.android.map.Layer;
import com.esri.android.map.MapView;
import com.esri.android.map.PopupContainer;
import com.esri.android.map.PopupContainer.PopupEditButton;
import com.esri.android.map.PopupView;
import com.esri.android.map.ags.ArcGISFeatureLayer;
import com.esri.android.map.event.OnLongPressListener;
import com.esri.android.map.event.OnSingleTapListener;
import com.esri.android.map.event.PopupEditingListener;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.Point;
import com.esri.core.map.CallbackListener;
import com.esri.core.map.FeatureEditResult;
import com.esri.core.map.Graphic;
import com.esri.core.map.popup.PopupInfo;

public class PopupInWebmapForEditing extends Activity {
	MapView map;
  PopupContainer popupContainer;
  PopupDialog popupDialog;
  Point newPoint;
  boolean geometryEditing = false;
	
  /** Called when the activity is first created. */    
  @Override
public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    
    // Load a webmap.
    map = new MapView(this, "http://www.arcgis.com/home/item.html?id=07602ed54cc24784aeac769c7ba88536", "", "");
    setContentView(map);
        
    map.setOnSingleTapListener(new OnSingleTapListener() {
      private static final long serialVersionUID = 1L;

      public void onSingleTap(float x, float y) {    	
        if (map.isLoaded()) {
        	if (geometryEditing) {
        		// Tap on the map to get new location and show popup again after tap on Edit Geometry button.
        		newPoint = map.toMapPoint(x, y);
        		geometryEditing = false;
            popupDialog.show();            
        	}
        	else {
        		// Tap on the map and show popups for selected editable features.
	        	// Instantiate a PopupContainer
	        	popupContainer = new PopupContainer(map.getContext());
	        	int id = popupContainer.hashCode();
	        	popupDialog = null;
	        	
	        	// Loop through each layer in the webmap
	        	int tolerance = 20;
	        	Layer[] layers = map.getLayers();
	        	for (Layer layer : layers) {
	        		// If the layer has not been initialized or is invisible, do nothing.
	      			if (!layer.isInitialized() || !layer.isVisible())
	      				continue;
	      			if (layer instanceof ArcGISFeatureLayer) { 
	      				// Query feature layer and display popups
	      				ArcGISFeatureLayer featureLayer = (ArcGISFeatureLayer) layer;          				
	      				if (featureLayer.getPopupInfo() != null && featureLayer.getPopupInfo().isInitialized()) {
	      					// Query feature layer which is associated with a popup definition.
	      					new RunQueryFeatureLayerTask(x, y, tolerance, id).execute(featureLayer);
	      				}
	      			}      		
	        	}
	        	
	      	  popupContainer.setPopupEditingListener(new MyPopupEditingListener(true));
	        }
        }
      }
    });
    
    // Long press on the map and add a new feature to the point feature layer.
    map.setOnLongPressListener(new OnLongPressListener() {
      private static final long serialVersionUID = 1L;
      private ArcGISFeatureLayer featureLayer = null;

      public void onLongPress(float x, float y) {    	
        if (map.isLoaded()) {
      	  // Get the point featurelayer
        	Layer[] layers = map.getLayers();
      	  for (Layer layer : layers) {
      		  if (layer instanceof ArcGISFeatureLayer) {
      			  ArcGISFeatureLayer fl = (ArcGISFeatureLayer) layer;
      			  if (fl.getGeometryType() == Geometry.Type.POINT) {
      				  featureLayer = fl;
      				  break;
      			  }
      		  }
      	  }

      	  if (featureLayer == null) 
      	  	return;
      	  PopupInfo popupInfo = featureLayer.getPopupInfo();
      	  if (popupInfo == null || !popupInfo.isInitialized())
      	  	return;
      	  
      	  // Create a new graphic
      	  Point point = map.toMapPoint(x, y);
      	  Graphic graphic = featureLayer.createFeatureWithType(featureLayer.getTypes()[0], point);

        	// Instantiate a PopupContainer
        	popupContainer = new PopupContainer(map.getContext());
        	// Add PopupView
        	PopupView popupView = new PopupView(map.getContext(), popupInfo, graphic);
        	popupView.setTag(featureLayer);
        	ArrayList<PopupView> popupViewList = new ArrayList<PopupView>();
        	popupViewList.add(popupView);
					popupContainer.addPopupViews(popupViewList.toArray(new PopupView[0]));

        	// Set to editing mode
        	popupContainer.setEditingMode(true);
        	// Set visibility for the Edit buttons
      	  popupContainer.setPopupEditButtonVisibility(PopupEditButton.DELETE, View.INVISIBLE);
      	  popupContainer.setPopupEditButtonVisibility(PopupEditButton.GEOMETRY, View.INVISIBLE);
      	  popupContainer.setPopupEditButtonVisibility(PopupEditButton.CANCEL, View.INVISIBLE);

        	// Set listener to handle editing events
      	  popupContainer.setPopupEditingListener(new MyPopupEditingListener(false));
      	  
					// Create a dialog for the popups and display it.
      	  popupDialog = new PopupDialog(map.getContext(), popupContainer);
      	  popupDialog.show();
        }
      }
    });
  }
  
  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
  	
    if (resultCode == Activity.RESULT_OK && data != null && popupContainer != null) {
    	// Add the selected media as attachment.
      Uri selectedImage = data.getData();
      popupContainer.getSelectedPopup().addAttachment(selectedImage);
    }
  }

  // Implement PopupEditingListener.
  private class MyPopupEditingListener implements PopupEditingListener {
  	private boolean existingFeature;
  	ArcGISFeatureLayer featureLayer;

  	public MyPopupEditingListener(boolean existingFeature) {
  		this.existingFeature = existingFeature;
  	}
  	
  	// Add an attachment to the feature.
	  public void onAddAttachment() {
	  	// Start media gallery to allow users to pick a image or video as attachment.
  		startActivityForResult(new Intent(Intent.ACTION_PICK, MediaStore.Images.Media.INTERNAL_CONTENT_URI), 1);
	  }     		  

	  // Post edits to the server.
  	public void onSave(Map<String, Object> attributes) {  		
  		featureLayer  = getFeatureLayer();
  		if (featureLayer == null)
  			return;
  		
  		// Get graphic
  		// Get geometry
  		Geometry geometry;
  		if (newPoint != null) {
  			// New geometry
  			geometry = newPoint;
  			newPoint = null;
  		}
  		else {
  			// Existing geometry
  			geometry = popupContainer.getSelectedPopup().getGraphic().getGeometry();
  		}
  		// Create graphic from the geometry and the updated attributes.
  		Graphic graphic = new Graphic(geometry,
				popupContainer.getSelectedPopup().getGraphic().getSymbol(), attributes,
				popupContainer.getSelectedPopup().getGraphic().getInfoTemplate());
  		if (existingFeature) {
  			// Update existing graphic
        featureLayer.applyEdits(null, null, new Graphic[] {graphic}, getEditCallbackListener(existingFeature));
  		}
  		else {
  			// Add new graphic
        featureLayer.applyEdits(new Graphic[] {graphic}, null, null, getEditCallbackListener(existingFeature));
  		}      
  		
      // Dismiss the dialog
      popupDialog.dismiss();
	  }
	  
  	// Delete a feature
	  public boolean onDelete() {
  		featureLayer  = getFeatureLayer();
  		if (featureLayer == null)
  			return false;

			((Activity)map.getContext()).runOnUiThread(new Runnable() {

				public void run() {
					AlertDialog.Builder builder = new AlertDialog.Builder(map.getContext());
					builder.setTitle("Delete Feature");
					builder.setMessage("Are you sure you want to delete this feature?");		
					builder.setCancelable(false);
					builder.setPositiveButton("Ok", new DialogInterface.OnClickListener() {
						
						public void onClick(DialogInterface dialog, int id) {
						// Get the graphic
				      Graphic graphic = popupContainer.getSelectedPopup().getGraphic();     
				      popupDialog.dismiss();
				      // Delete the feature
				      featureLayer.applyEdits(null, new Graphic[] {graphic}, null, new CallbackListener<FeatureEditResult[][]>() {                    
				        public void onError(Throwable e) {
				        	// Fail to delete the feature.
				          runOnUiThread(new Runnable() {                        
				            public void run() {
				              Toast.makeText(PopupInWebmapForEditing.this, "delete failed.", Toast.LENGTH_SHORT).show();
				            }
				          });    
				        }
				        
				        public void onCallback(FeatureEditResult[][] objs) {
				        	// Feature has been deleted.
				          runOnUiThread(new Runnable() {
				            public void run() {
				              Toast.makeText(PopupInWebmapForEditing.this, "delete succeeded.", Toast.LENGTH_SHORT).show();
				            }
				          });				          
				        }
				      });      
				      
						}
					});
					builder.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
						
						public void onClick(DialogInterface dialog, int id) {
							dialog.cancel();
						}
					});
					
					AlertDialog alert = builder.create();
					alert.show();
				}
			});

			return (popupDialog != null && popupDialog.isShowing())?false:true;

  	}
	  
	  // Cancel editing
	  public void onCancel() {
		  popupContainer.cancelEditing();
	  }
	  
	  // Edit geometry
	  public void onGeometryEditing(Geometry geometry) {
  		if ( geometry.getType() == Geometry.Type.POINT ) {
        geometryEditing = true;
        popupDialog.hide();
        Toast.makeText(PopupInWebmapForEditing.this, "tap on a new location.", Toast.LENGTH_SHORT).show();
      }

  	}

	  // Start editing session
		public void onStartEditing() {
			popupContainer.setPopupEditButtonVisibility(PopupEditButton.CANCEL, View.INVISIBLE);
		}
  
		private ArcGISFeatureLayer getFeatureLayer() {
			Object tag = popupContainer.getSelectedPopup().getTag();
			if (tag == null)
				return null;
			return (ArcGISFeatureLayer) tag;
		}
		
		// Callback 
		private CallbackListener<FeatureEditResult[][]> getEditCallbackListener(final boolean existingFeature) {
			return new CallbackListener<FeatureEditResult[][]>() {
        
        public void onError(Throwable e) {
        	// Fail to save graphic
          runOnUiThread(new Runnable() {                        
            public void run() {
              Toast.makeText(PopupInWebmapForEditing.this, "save failed.", Toast.LENGTH_SHORT).show();
            }
          });
        }
        
        public void onCallback(FeatureEditResult[][] objs) {
        	// Graphic has been saved.
          runOnUiThread(new Runnable() {
            public void run() {
              Toast.makeText(PopupInWebmapForEditing.this, "save succeeded.", Toast.LENGTH_SHORT).show();
            }
          });
          
          // Save attachments to the server if newly added attachments exist.
          // Retrieve object id of the feature
          int oid; 
          if (existingFeature) {
          	oid = objs[2][0].getObjectId();
          }
          else {
          	oid = objs[0][0].getObjectId();
          }
          // Get newly added attachments
          List<File> attachments = popupContainer.getSelectedPopup().getAddedAttachments();
          if (attachments != null && attachments.size() > 0) {
          	for (File attachment : attachments) {
          		// Save newly added attachment based on the object id of the feature.
          		featureLayer.addAttachment(oid, attachment, new CallbackListener<FeatureEditResult>() {
          			public void onError(Throwable e) {
          				// Failed to save new attachments.
                  runOnUiThread(new Runnable() {
                    public void run() {
                      Toast.makeText(PopupInWebmapForEditing.this, "save failed for new attachment.", Toast.LENGTH_SHORT).show();
                    }
                  });
          			}
          			
          			public void onCallback(FeatureEditResult arg0) {
          				// New attachments have been saved.
                  runOnUiThread(new Runnable() {
                    public void run() {
                      Toast.makeText(PopupInWebmapForEditing.this, "save succeeded for new attachment.", Toast.LENGTH_SHORT).show();
                    }
                  });
          			}
          		});
          	}
          }
          
          // Delete attachments if some attachments have been mark as delete.
          // Get ids of attachments which are marked as delete.
          List<Integer> attachmentIDs = popupContainer.getSelectedPopup().getDeletedAttachmentIDs();
          if (attachmentIDs != null && attachmentIDs.size() > 0) {
          	int[] ids = new int[attachmentIDs.size()];
          	for (int i = 0; i < attachmentIDs.size(); i++) {
          		ids[i] = attachmentIDs.get(i);
          	}
          	// Delete attachments
        		featureLayer.deleteAttachments(oid, ids, new CallbackListener<FeatureEditResult[]>() {
        			public void onError(Throwable e) {
        				// Failed to delete attachments
                runOnUiThread(new Runnable() {
                  public void run() {
                    Toast.makeText(PopupInWebmapForEditing.this, "save failed for deleted attachment.", Toast.LENGTH_SHORT).show();
                  }
                });
        			}
        			
        			public void onCallback(FeatureEditResult[] objs) {
        				// Attachments have been removed.
                runOnUiThread(new Runnable() {
                  public void run() {
                    Toast.makeText(PopupInWebmapForEditing.this, "save succeeded for deleted attachment.", Toast.LENGTH_SHORT).show();
                  }
                });
        			}
        		});
          }
        }
      
			};
		}
  }
  
  // Query feature layer by hit test
  private class RunQueryFeatureLayerTask extends AsyncTask<ArcGISFeatureLayer, Void, Graphic[]> {

		private int tolerance;
		private float x;
		private float y;
		private ArcGISFeatureLayer featureLayer;
		private int id;

		public RunQueryFeatureLayerTask(float x, float y, int tolerance, int id) {
			super();
			this.x = x;
			this.y = y;
			this.tolerance = tolerance;
			this.id = id;
		}

		@Override
		protected Graphic[] doInBackground(ArcGISFeatureLayer... params) {
			for (ArcGISFeatureLayer featureLayer1 : params) {
				this.featureLayer = featureLayer1;
				// Retrieve graphic ids near the point.
				int[] ids = featureLayer1.getGraphicIDs(x, y, tolerance);
				if (ids != null && ids.length > 0) {
					ArrayList<Graphic> graphics = new ArrayList<Graphic>();
					for (int id1 : ids) {
						// Obtain graphic based on the id.
						Graphic g = featureLayer1.getGraphic(id1);
						if (g == null)
							continue;
						graphics.add(g);
					}
					// Return an array of graphics near the point.
					return graphics.toArray(new Graphic[0]);
				}
			}
			return null;
		}

		@Override
		protected void onPostExecute(Graphic[] graphics) {
			// Validate parameter.
			if (graphics == null || graphics.length == 0)
				return;
			// Check if the requested PopupContainer id is the same as the current PopupContainer.
			// Otherwise, abandon the obsoleted query result.
			if ( id != popupContainer.hashCode() ) 
				return;
			
			PopupInfo popupInfo = featureLayer.getPopupInfo();
			if (popupInfo == null || !popupInfo.isInitialized())
				return;
			
			// Create a list of PopupView from the array of graphics.
			final ArrayList<PopupView> popupViewList = new ArrayList<PopupView>();
			for (int i = 0; i < graphics.length; i++) {
				PopupView popupView = new PopupView(map.getContext(), popupInfo, graphics[i]);
				popupView.setTag(featureLayer);
				popupViewList.add(popupView);
			}
			
			// Add the PopupViews to PopupContainer and display the popups.
			if ( id != popupContainer.hashCode() ) 
				return;
			
			// Add PopupViews and set colors for the popup
			popupContainer.addPopupViews(popupViewList.toArray(new PopupView[0]));

			if (popupDialog == null) {
				// Create a dialog for the popups and display it.
				popupDialog = new PopupDialog(map.getContext(), popupContainer);
				popupDialog.show();
			}
		}

	}
  
  // A customized full screen dialog.
  private class PopupDialog extends Dialog {
	  private View popupContainer1;
	  
	  public PopupDialog(Context context, PopupContainer popupContainer) {
		  super(context, android.R.style.Theme);
		  this.popupContainer1 = popupContainer;
	  }
	  
	  @Override
	  protected void onCreate(Bundle savedInstanceState) {
		  super.onCreate(savedInstanceState);
		  LayoutParams params = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
			LinearLayout layout = new LinearLayout(getContext());
			layout.addView(popupContainer1, LayoutParams.FILL_PARENT, LayoutParams.FILL_PARENT);
			setContentView(layout, params);
		}
	  
  }
  
}