/* Copyright 2012 ESRI
 *
 * All rights reserved under the copyright laws of the United States
 * and applicable international laws, treaties, and conventions.
 *
 * You may freely redistribute and use this sample code, with or
 * without modification, provided you include the original copyright
 * notice and use restrictions.
 *
 * See the �Sample code usage restrictions� document for further information.
 *
 */

package com.esri.android.sample;

import java.io.FileNotFoundException;
import java.util.UUID;

import android.app.Activity;
import android.os.Bundle;
import android.util.Log;

import com.esri.android.map.GraphicsLayer;
import com.esri.android.map.GroupLayer;
import com.esri.android.map.Layer;
import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISTiledMapServiceLayer;
import com.esri.android.map.event.OnLongPressListener;
import com.esri.android.map.event.OnStatusChangedListener;
import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.Polygon;
import com.esri.core.map.Graphic;
import com.esri.core.symbol.advanced.Message;
import com.esri.core.symbol.advanced.MessageHelper;
import com.esri.core.symbol.advanced.MessageProcessor;
import com.esri.core.symbol.advanced.SymbolDictionary.DictionaryType;

/**
 * This sample illustrates the use of MessageProcessor to process Advanced Symbology features 
 * whose definitions are stored locally on the device. The MessageProcessor class provides the 
 * capability to take a message received from an external source and convert this into a symbol which 
 * is displayed on a map in a graphics layer. The message processor class requires a group layer 
 * which has been added to an initialized map. 
 * The MessageProcessor constructor also requires a DictionaryType which indicates the type of symbol 
 * dictionary being used.
 * The Message class encapsulates information from an incoming or outgoing message. Apart from a 
 * message ID all other properties in the message are held as name-value pairs.
 *  
 **/

public class AdvancedSymbologySampleActivity extends Activity {

	MapView map;
	ArcGISTiledMapServiceLayer tiledMapService;
	MessageProcessor processor;

	Point center;
	Point centerOffSet;

	String singlepointcode = "SFGPEVC--------";
	String multipointcode = "GFMPNC----****X";	
	private int symbolCount = 10;

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		map = (MapView)findViewById(R.id.map);
		final GroupLayer groupLayer = new GroupLayer();
		tiledMapService = new ArcGISTiledMapServiceLayer("http://services.arcgisonline.com/ArcGIS/rest/services/World_Topo_Map/MapServer");
		map.addLayer(tiledMapService);
		map.addLayer(groupLayer);


		try {
			processor = new MessageProcessor(DictionaryType.MIL2525C, groupLayer);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}

		center = new Point(-567329.8, 4878732);
		centerOffSet = new Point(-567339.8, 4878742);

		map.setOnStatusChangedListener(new OnStatusChangedListener() {
			private static final long serialVersionUID = 1L;

			public void onStatusChanged(Object source, STATUS status) {
				if (source == map && status == STATUS.INITIALIZED) {
					map.centerAt(center, false);
					map.setScale(720000.0);
					addPoints();
				}
			}
		});	


		map.setOnLongPressListener(new OnLongPressListener() {

			private static final long serialVersionUID = 1L;

			public void onLongPress(float x, float y) {
				Layer[] layers = groupLayer.getLayers();
				for(Layer layer: layers) {
					if (layer instanceof GraphicsLayer) {
						GraphicsLayer glayer = (GraphicsLayer)layer;
						int[] graphics = glayer.getGraphicIDs(x, y, 10);
						if (graphics != null && graphics.length > 0) {
							Log.d("Test", "Graphic is found");
							Graphic gr = glayer.getGraphic(graphics[0]);
							Geometry geom = gr.getGeometry();
							Point targetcontrolpt = null;
							if (geom instanceof Point) {
								Point pt = (Point)geom;
								Point scrnpt = map.toScreenPoint(pt);
								scrnpt = new Point(scrnpt.getX() + 50, scrnpt.getY() -50);
								targetcontrolpt = map.toMapPoint(scrnpt);
								Log.d("Test", "x: " + pt.getX() + "; y: " + pt.getY());
							}
							Message message = processor.createMessageFrom(gr);
							String controlPoints = (String)message.getProperty(MessageHelper.MESSAGE_2525C_CONTROL_POINTS_PROPERTY_NAME);
							Log.i("Test", "control point:" + controlPoints);
							if (targetcontrolpt !=null) {
								message.setProperty(MessageHelper.MESSAGE_2525C_CONTROL_POINTS_PROPERTY_NAME, targetcontrolpt.getX() + "," + targetcontrolpt.getY());
							}
							message.setProperty("_Action", "update");
							processor.processMessage(message);
						}
					}
				}
			}
		});
	}


	/*
	 * addPoints method to create and process message to add
	 * points to display in MapView Graphics Layer. 
	 */
	private void addPoints() {
		Polygon extent = map.getExtent();
		Envelope env = new Envelope();
		extent.queryEnvelope(env);
		center = map.getCenter();

		double resolution = map.getResolution();
		double middley = env.getCenterY();
		double leftx = env.getLowerLeft().getX();
		double strideX = resolution * (map.getWidth() / symbolCount);
		double strideXOffset = resolution * (map.getWidth() /(symbolCount * 2));
		double strideYOffset = resolution * (map.getHeight() / 16);
		double strideY8 = resolution * (map.getHeight() / 8);

		double cornerXOffset = resolution * (map.getWidth() /(symbolCount * 4));
		double cornerYoffset = resolution * (map.getHeight() / 32);

		double y1 = middley - strideYOffset;
		double y5 = middley + strideYOffset + strideY8 * 3;

		// Iterate through the symbols to process points on map
		for(int i = 0; i < symbolCount; i++) {
			double x1 = leftx + strideXOffset + strideX * i;

			// Message creation
			Message lmessage = new Message();
			UUID luuid = UUID.randomUUID();
			lmessage.setID(luuid.toString());
			lmessage.setProperty("_Type", "position_report");
			lmessage.setProperty("_Action", "update");
			String controlpoints = x1 + ","  + y1;
			lmessage.setProperty("_Control_Points", controlpoints);
			lmessage.setProperty("sic", "SFGPUCRRL--E---");
			lmessage.setProperty("_WKID", "3857");
			lmessage.setProperty("UniqueDesignation", "2020");
			// process message and add point
			processor.processMessage(lmessage);

			// Different symbol type definition
			Message msg2 = new Message();
			msg2.setID(UUID.randomUUID().toString());
			msg2.setProperty("_Type", "position_report");
			msg2.setProperty("_Action", "update");
			controlpoints = getMultiPoint(x1, y5, cornerXOffset, cornerYoffset);            		
			msg2.setProperty("_Control_Points", controlpoints);
			msg2.setProperty("sic", multipointcode);
			msg2.setProperty("_WKID", "3857");
			msg2.setProperty("UniqueDesignation", "Mad dog");
			processor.processMessage(msg2);	        

		}
	}

	/*
	 * getMultiPoint method 
	 */
	private String getMultiPoint(double centerPointX, double centerPointY, double cornerOffsetX, double cornerOffsetY) {
		double leftx = centerPointX - cornerOffsetX;
		double rightx = centerPointX + cornerOffsetX;
		double topy = centerPointY + cornerOffsetY;
		double bottomy = centerPointY - cornerOffsetY;
		StringBuilder builder = new StringBuilder();
		builder.append(leftx).append(",").append(topy).append(";");
		builder.append(rightx).append(",").append(topy).append(";");
		builder.append(rightx).append(",").append(bottomy).append(";");
		builder.append(leftx).append(",").append(bottomy);

		return builder.toString();
	}		

	@Override 
	protected void onDestroy() { 
		super.onDestroy();
	}
	@Override
	protected void onPause() {
		super.onPause();
		map.pause();
	}
	@Override 	protected void onResume() {
		super.onResume(); 
		map.unpause();
	}

}