/* Copyright 2012 ESRI
*
* All rights reserved under the copyright laws of the United States
* and applicable international laws, treaties, and conventions.
*
* You may freely redistribute and use this sample code, with or
* without modification, provided you include the original copyright
* notice and use restrictions.
*
* See the sample code usage restrictions document for further information.
*
*/

package com.esri.arcgis.android.samples.featurecollection;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonParseException;
import org.codehaus.jackson.JsonParser;
import org.codehaus.jackson.JsonToken;

import android.app.Activity;
import android.os.Bundle;
import android.util.Log;
import android.widget.Toast;

import com.esri.android.map.MapView;
import com.esri.android.map.ags.ArcGISFeatureLayer;
import com.esri.android.map.ags.ArcGISTiledMapServiceLayer;
import com.esri.android.map.ags.ArcGISFeatureLayer.MODE;
import com.esri.android.map.ags.ArcGISFeatureLayer.Options;
import com.esri.android.map.event.OnStatusChangedListener;
import com.esri.core.geometry.GeometryEngine;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.SpatialReference;
import com.esri.core.map.FeatureSet;
import com.esri.core.map.Graphic;

/**
 * The sample shows how to use a feature collection with layer definition.
 * Essentially we read the contents of a flicker feed and get the graphics from
 * the feed.
 * Create a feature set using the graphics, add the feature set, layer definition to the feature layer/
 * 
 *
 *
 */
public class FeatureCollection extends Activity {

	MapView map;
	
	//Essentially create your layer definition from the rest url, append /?f=json to your url
	//Replace " with \"
	String layerDefinition = "{\"geometryType\": \"esriGeometryPoint\",\"objectIdField\":\"ObjectID\",\"drawingInfo\": {\"renderer\": {\"type\": \"simple\",\"symbol\": {\"type\": \"esriPMS\",\"url\": \"http://help.arcgis.com/en/webapi/javascript/arcgis/demos/images/flickr.png\",\"contentType\": \"image/png\",\"width\": 20,\"height\": 20}}},\"fields\": [{\"name\": \"ObjectID\",\"alias\": \"ObjectID\",\"type\": \"esriFieldTypeOID\"},{\"name\": \"description\",\"alias\": \"Description\",\"type\": \"esriFieldTypeString\"},{\"name\": \"title\",\"alias\": \"Title\",\"type\": \"esriFieldTypeString\"}]}";
	
	ArcGISFeatureLayer fLayer;

	@Override
	protected void onCreate(Bundle savedInstanceState) {		
		super.onCreate(savedInstanceState);
		map = new MapView(this);
		ArcGISTiledMapServiceLayer tmsl = new ArcGISTiledMapServiceLayer(
				"http://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer");
		map.addLayer(tmsl);

		setContentView(map);
		

		
		/**
		 * Once the map is initialized read the json response
		 * Get the graphics from the flicker fee
		 * Add these graphics to the feature set
		 * Pass the layer definition, feature set, and mode to the feature layer
		 * 
		 */
		map.setOnStatusChangedListener(new OnStatusChangedListener() {
			private static final long serialVersionUID = 1L;

			@Override
			public void onStatusChanged(Object source, STATUS status) {
				if (source == map && status == STATUS.INITIALIZED) {
					String response = getJsonResponse();
					ArrayList<Graphic> grList = generateGraphics(response);
					Graphic[] grs = new Graphic[grList.size()];
					grs = grList.toArray(grs);
					FeatureSet fs = new FeatureSet();
					fs.setGraphics(grs);
					Options options = new Options();
					options.mode = MODE.SNAPSHOT;

					try {
						fLayer = new ArcGISFeatureLayer(layerDefinition, fs,
								options);
						map.addLayer(fLayer);
						Toast.makeText(FeatureCollection.this, "Flicker feed contents added to feature layer", Toast.LENGTH_SHORT).show();
					} catch (Exception e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}

				}

			}
		});

	}

	/**
	 * Class used to store the content retrieved 
	 * from the flicker feed
	 */
	static class Item {
		private String title;
		private String description;
		private double latitude;
		private double longitude;

		public String getTitle() {
			return title;
		}

		public void setTitle(String title) {
			this.title = title;
		}

		public String getDescription() {
			return description;
		}

		public void setDescription(String description) {
			this.description = description;
		}

		public double getLatitude() {
			return latitude;
		}

		public void setLatitude(double latitude) {
			this.latitude = latitude;
		}

		public double getLongitude() {
			return longitude;
		}

		public void setLongitude(double longitude) {
			this.longitude = longitude;
		}

	}
	
	/**
	 * The method is used to get graphics from the flicker feed.
	 * @param jsonString
	 * @return list of graphics
	 */

	private ArrayList<Graphic> generateGraphics(String jsonString) {
		ArrayList<Item> items = new ArrayList<Item>();
		ArrayList<Graphic> grList = new ArrayList<Graphic>();
		JsonFactory f = new JsonFactory();
		try {
			JsonParser parser = f.createJsonParser(jsonString);
			if (!isObjectStart(parser)) {
				return null;
			}

			while (parser.nextToken() != JsonToken.END_OBJECT) {
				String fieldName = parser.getCurrentName();
				Log.d("Test", "fiedName is " + fieldName);
				parser.nextToken();
				if ("items".equals(fieldName)) {
					if (parser.getCurrentToken() == JsonToken.START_ARRAY) {

						while (parser.nextToken() != JsonToken.END_ARRAY) {
							Item item = new Item();
							if (parser.getCurrentToken() == JsonToken.START_OBJECT) {
								while (parser.nextToken() != JsonToken.END_OBJECT) {
									String secondField = parser
											.getCurrentName();
									parser.nextToken();
									if ("title".equals(secondField)) {
										item.setTitle("Test title");
									} else if ("description"
											.equals(secondField)) {
										item.setDescription("Test description");
									} else if ("latitude".equals(secondField)) {
										String latStr = parser.getText();
										item.setLatitude(Double.valueOf(latStr));
									} else if ("longitude".equals(secondField)) {
										String lonStr = parser.getText();
										item.setLongitude(Double
												.valueOf(lonStr));
									} else {
										parser.skipChildren();
									}
								}
							}
							items.add(item);
						}
					}
				} else {
					parser.skipChildren();
				}

			}

		} catch (JsonParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			// return grList;
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			// return grList;
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			// return grList;
		} finally {
			SpatialReference webMercator = SpatialReference.create(102100);
			for (Item item : items) {
				if (item.getLatitude() == 0.000000) {
					continue;
				}
				Point pointwm = GeometryEngine.project(item.getLongitude(),
						item.getLatitude(), webMercator);
				HashMap<String, Object> attrs = new HashMap<String, Object>();
				attrs.put("description", item.getDescription());
				attrs.put("title", item.getTitle());
				Graphic gr = new Graphic(pointwm, null, attrs, null);
				grList.add(gr);

			}

		}

		return grList;
	}

	private String getJsonResponse() {
		String strResponse = null;
		HttpClient client = new DefaultHttpClient();
		try {
			List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
			nameValuePairs.add(new BasicNameValuePair("format", "json"));
			URI uri = URIUtils.createURI("http", "api.flickr.com", -1,
					"/services/feeds/geo",
					URLEncodedUtils.format(nameValuePairs, "UTF-8"), null);
			HttpGet httpGet = new HttpGet(uri);
			HttpResponse response = client.execute(httpGet);
			HttpEntity entity = response.getEntity();
			strResponse = EntityUtils.toString(entity, "UTF-8");
			strResponse = strResponse.substring(15, strResponse.length() - 1);
		} catch (ClientProtocolException e) {

		} catch (IOException e) {

		} catch (URISyntaxException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return strResponse;
	}
	
	final public static boolean isObjectStart(JsonParser parser) throws Exception {
	    return parser.getCurrentToken() == null ? parser.nextToken() == JsonToken.START_OBJECT
	        : parser.getCurrentToken() == JsonToken.START_OBJECT;

	  }

	

	@Override
	protected void onPause() {
		// TODO Auto-generated method stub
		super.onPause();
		map.pause();
	}

	@Override
	protected void onResume() {
		// TODO Auto-generated method stub
		super.onResume();
		map.unpause();
	}

}
