=begin rdoc

= simple_building_generator.rb
Copyright 2015 by Mohammad Rizky Kurniawan - All Rights Reserved

== Information
Author:: Mohammad Rizky Kurniawan
Name:: simple_building_generator.rb
Version:: 1.06
SU Version:: 2013
Date:: 12/08/2015
Description:: 
  Convert single face (marked with user-defined material) into building (box-shape) and apply user-defined
material to them. This script contains code from offset.rb by Rick Wilson from http://www.smustard.com

History::
* 1.06:: 12/08/2015
  * Added context menu for quick building generation
* 1.05:: 12/08/2015
  * Added option "Save configuration"
  * Added option "Use previous saved configuration"
  * Changing process message to MB_MULTILINE
    because when someone insert a lot of materials
    the confirmation button get pushed to out of
    screen
  * The floorplan now check back material of the face
* 1.04:: 11/08/2015
  * Added option "Different Rooftop Edges Material"
* 1.03:: 09/08/2015
  * Added option "Texture Random Mode"
* 1.02:: 08/08/2015
  * Added option to "try fit material"
* 1.01:: 06/08/2015
  * Fixing bug that creates rooftop edges facing upside down
* 1.00:: 05/08/2015
  * Initial release

Contact::
Simple Building Generator : http://sketchucation.com/forums/viewtopic.php?f=323&t=62725
Thread on sketchUcation
E-Mail                    : mohrizkyk@gmail.com

Notes::
  By reading at this source, I would like to say thank you for using Simple Building Generator or to learn
how this plugin work, I hope that my comments in this source could help you in your research :D

Thanks::
- Rick Wilson for the offset.rb script. Without this the rooftop would be edgeless :(
- CityGen team (Thomas Thomassen, Remus Knowles, Chris Fullmer) as predecessor
- For whoever create the video http://www.youtube.com/watch?v=XCEN0qQOsIA (How to Create a City in Blender)
  you really showed me how a building should be :D :D

=end
require 'sketchup.rb'

UI.menu("Plugins").add_item("Simple Building Generator") {
  sbg_main
}

# ===== UPDATE [12/08/2015] V1.06 [START] =====
UI.add_context_menu_handler do |sbg_context_menu|
  sbg_context_menu.add_separator
  sbg_submenu = sbg_context_menu.add_submenu("Simple Building Generator")
  if (sbg_submenu)
    sbg_submenu.add_item("Fixed Level Building") {
      sbg_quick_generate_fixed_level_building
    }
    sbg_submenu.add_item("Random Level Building") {
      sbg_quick_generate_random_level_building
    }
  end
end
$sbg_urgent_message = "Please see the plugin page in \"Release\" section how to download the plugin."
def sbg_quick_generate_fixed_level_building
  status = UI.messagebox($sbg_urgent_message)
  message_end =     "Thank you for using building generator v1.07" +
                "\n\nPlease report any bug or share any works you've made" +
                  "\nwith this plugin." +
                  "\nIt will motivate me a lot :)" +
                "\n\nAlso don't hesitate to ask me if you find any difficulty" +
                  "\nto use this plugin." +
                "\n\n==============================" +
                  "\nsketchucation.com/forums/viewtopic.php?f=323&t=62725" +
                  "\n=============================="
  UI.messagebox(message_end)
  return
end

def sbg_quick_generate_random_level_building
  UI.messagebox($sbg_urgent_message)
  message_end =     "Thank you for using building generator v1.07" +
                "\n\nPlease report any bug or share any works you've made" +
                  "\nwith this plugin." +
                  "\nIt will motivate me a lot :)" +
                "\n\nAlso don't hesitate to ask me if you find any difficulty" +
                  "\nto use this plugin." +
                "\n\n==============================" +
                  "\nsketchucation.com/forums/viewtopic.php?f=323&t=62725" +
                  "\n=============================="
  UI.messagebox(message_end)
end
# ===== UPDATE [12/08/2015] V1.06 [END] =====

# ===== UPDATE [12/08/2015] V1.05 [START] =====
$sbg_saved_conf_general        = []
$sbg_saved_conf_general_status = false
$sbg_saved_conf_gf             = []
$sbg_saved_conf_gf_status      = false
$sbg_saved_conf_f              = []
$sbg_saved_conf_f_status       = false
$sbg_saved_conf_re             = []
$sbg_saved_conf_re_status      = false
# ===== UPDATE [12/08/2015] V1.05 [END] =====

def sbg_main
  # ===== General Input [START] =====
  # Variable for store process report message
  message = "Process check :"

  prompts  = ["Floorplan material name",
              "Erase floorplan face when done?",
              "Ground floor height (in m)",
              "Floor height (in m)",
              "Randomize rooftop floor material?",
              "Rooftop floor material",
              "Rooftop edge height (in m)",
              "Fit texture?",          # ===== UPDATE [08/08/2015] V1.02 =====
              "Random texture mode?",  # ===== UPDATE [09/08/2015] V1.03 =====
              "Save current setting?", # ===== UPDATE [12/08/2015] V1.05 =====
              "Use previous setting?"] # ===== UPDATE [12/08/2015] V1.05 =====
  defaults = ["",
              "No",
              0.m,
              0.m,
              "No",
              "",
              0.m,
              "No",         # ===== UPDATE [08/08/2015] V1.02 =====
              "Individual", # ===== UPDATE [09/08/2015] V1.03 =====
              "No",         # ===== UPDATE [12/08/2015] V1.05 =====
              "No"]         # ===== UPDATE [12/08/2015] V1.05 =====
  list     = ["",
              "Yes|No",
              "",
              "",
              "Yes|No",
              "",
              "",
              "Yes|No",         # ===== UPDATE [08/08/2015] V1.02 =====
              "Individual|Set", # ===== UPDATE [09/08/2015] V1.03 =====
              "Yes|No",         # ===== UPDATE [12/08/2015] V1.05 =====
              "Yes|No"]         # ===== UPDATE [12/08/2015] V1.05 =====
  arr_main_input = UI.inputbox(prompts, defaults, list, "Simple Building Generator")
  # ===== UPDATE [12/08/2015] V1.05 [START] =====
  # Using previous saved setting
  if (arr_main_input[10] == "Yes")
    # If previous saved setting exist then load it, otherwise fail
    if ($sbg_saved_conf_general_status)
      arr_main_input = $sbg_saved_conf_general
    else
      UI.messagebox("There are no previous saved configuration.\n\nNotes : SBG doesn't save any configuration\nafter you quit SketchUp.")
      return
    end
  end
  # ===== UPDATE [12/08/2015] V1.05 [END] =====

  # ===== General Input validation [START] =====
  # Check if floorplan material exist in the library.
  # If not don't bother to search through selection
  status = check_material_existence arr_main_input[0]
  if (!status)
    UI.messagebox("Floorplan material not exist in the materials library.")
    return
  elsif (status)
    message = message + "\n- Target material : [" + arr_main_input[0] + "]"
  else
    UI.messagebox("Unexpected error : ERR001")
    return
  end

  # Check if there is face using floorplan material in selection
  # If not, don't continue
  floorplan_count = sbg_count_floorplan arr_main_input[0]
  if (floorplan_count == 0)
    UI.messagebox("There is no face using material [" + arr_main_input[0] + "] in selection.")
    return
  elsif (floorplan_count > 0)
    message = message + "\n- Floorplan found : [" + floorplan_count.to_s + "]"
  end

  # Insert erase face answer into process report
  if (arr_main_input[1] == "Yes")
    message = message + "\n- Delete face : [Yes]"
  elsif (arr_main_input[1] == "No")
    message = message + "\n- Delete face : [No]"
  else
    UI.messagebox("Unexpected error : ERR002")
    return
  end

  # Check ground floor height
  if (arr_main_input[2] > 0)
    message = message + "\n- Ground floor : [" + arr_main_input[2].to_s + "]"
  elsif (arr_main_input[2] == 0)
    message = message + "\n- Ground floor : [None]"
  elsif (arr_main_input[2] < 0)
    UI.messagebox("Ground floor height must be 0 or more!")
    return
  else
    UI.messagebox("Unexpected error : ERR003")
    return
  end

  # Check individual floor height
  if (arr_main_input[3] > 0)
    message = message + "\n- Floor : [" + arr_main_input[3].to_s + "]"
  elsif (arr_main_input[3] == 0)
    message = message + "\n- Floor : [None]"
  elsif (arr_main_input[3] < 0)
    UI.messagebox("Floor height must be 0 or more!")
    return
  else
    UI.messagebox("Unexpected error : ERR004")
    return
  end

  # Insert randomize rooftop floor material answer into process report
  if (arr_main_input[4] == "Yes")
    message = message + "\n- Rooftop floor material : [Random]"
  elsif (arr_main_input[4] == "No")
    message = message + "\n- Rooftop floor material : [Fixed]"
  else
    UI.messagebox("Unexpected error : ERR005")
    return
  end

  # Rooftop floor material check
  if (arr_main_input[4] == "Yes")
    message = message + "\n- Rooftop floor material : \n=========="
    # If rooftop floor material is randomized,
    # then split the material name by comma and then check if all material exist or not
    mtl_count = 0
    rooftop_floor_mtl = arr_main_input[5].split(",")
    rooftop_floor_mtl.each { |mtl_name|
      # The material name should not same as the target material name
      if (mtl_name == arr_main_input[0])
        UI.messagebox("Material that will be used must not same as target material")
        return
      end

      # Check if material exist or not
      status = check_material_existence mtl_name
      if (!status)
        UI.messagebox("Material not found : [" + mtl_name + "]")
        return
      elsif (status)
        message = message + "\n:= " + mtl_name
      else
        UI.messagebox("Unexpected error : ERR006")
        return
      end

      mtl_count = mtl_count + 1
    }
    message = message + "\n=========="

    if (mtl_count <= 1)
      UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
      return
    end
  else
    # The material name should not same as the target material name
    if (arr_main_input[5] == arr_main_input[0])
      UI.messagebox("Material that will be used must not same as target material")
      return
    end

    # Check if material exist or not
    status = check_material_existence arr_main_input[5]
    if (!status)
      UI.messagebox("Material not found : [" + arr_main_input[5] + "]")
      return
    elsif (status)
      message = message + "\n- Rooftop floor material : [" + arr_main_input[5] + "]"
    else
      UI.messagebox("Unexpected error : ERR007")
      return
    end
  end

  # Check rooftop edges height
  if (arr_main_input[6] > 0)
    message = message + "\n- Rooftop edges : [" + arr_main_input[6].to_s + "]"
  elsif (arr_main_input[6] == 0)
    message = message + "\n- Rooftop edges : [None]"
  elsif (arr_main_input[6] < 0)
    UI.messagebox("Rooftop edges height must be 0 or more!")
    return
  else
    UI.messagebox("Unexpected error : ERR008")
    return
  end

  # ===== UPDATE [08/08/2015] V1.02 [START] =====
  # Insert fit to face answer into process report
  if (arr_main_input[7] == "Yes")
    message = message + "\n- Fit texture : [Yes]"
  elsif (arr_main_input[7] == "No")
    message = message + "\n- Fit texture : [No]"
  else
    UI.messagebox("Unexpected error : ERR009")
    return
  end
  # ===== UPDATE [08/08/2015] V1.02 [END] =====

  # ===== UPDATE [11/08/2015] V1.04 [START] =====
  # Insert fit to face answer into process report
  if (arr_main_input[8] == "Individual")
    message = message + "\n- Random mode : [Individual]"
  elsif (arr_main_input[8] == "Set")
    message = message + "\n- Random mode : [Set]"
  else
    UI.messagebox("Unexpected error : ERR010")
    return
  end
  # ===== UPDATE [11/08/2015] V1.02 [END] =====
  # ===== General Input validation [END] =====

  # ===== UPDATE [12/08/2015] V1.05 [START] =====
  # Display report
  status = UI.messagebox(message, MB_MULTILINE)
  status = UI.messagebox("Do you want to proceed?", MB_YESNO)
  # ===== UPDATE [12/08/2015] V1.05 [END] =====
  if (status == IDNO)
    UI.messagebox("Aborted!")
    return
  else
    # Check : if no ground floor or floor, the building will not be generated
    if (arr_main_input[2] <= 0 && arr_main_input[3] <= 0)
      UI.messagebox("Could not generate building, no ground floor or floor specified!\n\nAborting!")
      return
    end
  end

  # ===== UPDATE [12/08/2015] V1.05 [START] =====
  # Save current setting
  if (arr_main_input[9] == "Yes")
    $sbg_saved_conf_general        = arr_main_input
    $sbg_saved_conf_general_status = true
  end
  # ===== UPDATE [12/08/2015] V1.05 [END] =====
  # ===== General Input [END] =====

  # ===== Ground Floor Input [START] =====
  arr_gf_input = []
  # Use ground floor?
  if (arr_main_input[2] > 0)
    # Variable for store process report message
    message = "Process check :"

    prompts  = ["Randomize ground floor material?",
                "Ground floor material name",
                "Save current setting?", # ===== UPDATE [12/08/2015] V1.05 =====
                "Use previous setting?"] # ===== UPDATE [12/08/2015] V1.05 =====
    defaults = ["No",
                "",
                "No", # ===== UPDATE [12/08/2015] V1.05 =====
                "No"] # ===== UPDATE [12/08/2015] V1.05 =====
    list     = ["Yes|No",
                "",
                "Yes|No", # ===== UPDATE [12/08/2015] V1.05 =====
                "Yes|No"] # ===== UPDATE [12/08/2015] V1.05 =====
    arr_gf_input = UI.inputbox(prompts, defaults, list, "Ground Floor Setting")
    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Using previous saved setting
    if (arr_gf_input[3] == "Yes")
      # If previous saved setting exist then load it, otherwise fail
      if ($sbg_saved_conf_gf_status)
        arr_gf_input = $sbg_saved_conf_gf
      else
        UI.messagebox("There are no previous saved configuration.\n\nNotes : SBG doesn't save any configuration\nafter you quit SketchUp.")
        return
      end
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====

    # ===== Ground Floor Input validation [START] =====
    # Insert randomize ground floor material answer into process report
    if (arr_gf_input[0] == "Yes")
      message = message + "\n- Ground floor material : [Random]"
    elsif (arr_gf_input[0] == "No")
      message = message + "\n- Ground floor material : [Fixed]"
    else
      UI.messagebox("Unexpected error : ERR011")
      return
    end

    # Ground floor material check
    if (arr_gf_input[0] == "Yes")
      message = message + "\n- Ground floor material : \n=========="
      # If ground floor material is randomized,
      # then split the material name by comma and then check if all material exist or not
      mtl_count = 0
      ground_floor_mtl = arr_gf_input[1].split(",")
      # ===== UPDATE [09/08/2015] V1.03 [START] =====
      # If randomize mode is set and the rooftop floor material is randomized then check
      # if ground floor material array have the same size as rooftop floor material array
      if (arr_main_input[8] == "Set" && arr_main_input[4] == "Yes")
        cnt_rooftop_floor_mtl = arr_main_input[5].split(",").size
        cnt_ground_floor_mtl = arr_gf_input[1].split(",").size
        if (cnt_rooftop_floor_mtl != cnt_ground_floor_mtl)
          sbg_err_message = "Rooftop Floor has " + cnt_rooftop_floor_mtl.to_s + " materials." +
                          "\nGround Floor has " + cnt_ground_floor_mtl.to_s  + " materials." +
                        "\n\nIf you choose \"Set\" random mode, all randomized material" +
                          "\nmust have same value." +
                        "\n\nYou can input repeated value, example :" +
                          "\n- rooftop_1,rooftop_1,rooftop_2" +
                          "\n- ground_1,ground_2,ground_3" +
                          "\nBuilding 1 and 2 will have same rooftop material but different" +
                          "\nground floor material. Building 3 will always have same rooftop."
          UI.messagebox(sbg_err_message)
          return
        end
      end
      # ===== UPDATE [09/08/2015] V1.03 [END] =====
      ground_floor_mtl.each { |mtl_name|
        # The material name should not same as the target material name
        if (mtl_name == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence mtl_name
        if (!status)
          UI.messagebox("Material not found : [" + mtl_name + "]")
          return
        elsif (status)
          message = message + "\n:= " + mtl_name
        else
          UI.messagebox("Unexpected error : ERR012")
          return
        end

        mtl_count = mtl_count + 1
      }
      message = message + "\n=========="

      if (mtl_count <= 1)
        UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
        return
      end
    else
      # The material name should not same as the target material name
      if (arr_gf_input[1] == arr_main_input[0])
        UI.messagebox("Material that will be used must not same as target material")
        return
      end

      # Check if material exist or not
      status = check_material_existence arr_gf_input[1]
      if (!status)
        UI.messagebox("Material not found : [" + arr_gf_input[1] + "]")
        return
      elsif (status)
        message = message + "\n- Ground floor material : [" + arr_gf_input[1] + "]"
      else
        UI.messagebox("Unexpected error : ERR013")
        return
      end
    end
    # ===== Ground Floor Input validation [END] =====

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Display report
    status = UI.messagebox(message, MB_MULTILINE)
    status = UI.messagebox("Do you want to proceed?", MB_YESNO)
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
    if (status == IDNO)
      UI.messagebox("Aborted!")
      return
    end

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Save current setting
    if (arr_gf_input[2] == "Yes")
      $sbg_saved_conf_gf        = arr_gf_input
      $sbg_saved_conf_gf_status = true
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
  end
  # ===== Ground Floor Input [END] =====

  # ===== Floor Input [START] =====
  arr_f_input = []
  # Multi level building?
  if (arr_main_input[3] > 0)
    # Variable for store process report message
    message = "Process check :"

    prompts  = ["Randomize floor material?",
                "Floor material name",
                "Floor level",
                "Randomize floor level?",
                "Floor level (min)",
                "Floor level (max)",
                "Save current setting?", # ===== UPDATE [12/08/2015] V1.05 =====
                "Use previous setting?"] # ===== UPDATE [12/08/2015] V1.05 =====
    defaults = ["No",
                "",
                1,
                "No",
                0,
                0,
                "No", # ===== UPDATE [12/08/2015] V1.05 =====
                "No"] # ===== UPDATE [12/08/2015] V1.05 =====
    list     = ["Yes|No",
                "",
                "",
                "Yes|No",
                "",
                "",
                "Yes|No", # ===== UPDATE [12/08/2015] V1.05 =====
                "Yes|No"] # ===== UPDATE [12/08/2015] V1.05 =====
    arr_f_input = UI.inputbox(prompts, defaults, list, "Floor Setting")
    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Using previous saved setting
    if (arr_f_input[7] == "Yes")
      # If previous saved setting exist then load it, otherwise fail
      if ($sbg_saved_conf_f_status)
        arr_f_input = $sbg_saved_conf_f
      else
        UI.messagebox("There are no previous saved configuration.\n\nNotes : SBG doesn't save any configuration\nafter you quit SketchUp.")
        return
      end
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====

    # ===== Floor Input validation [START] =====
    # Insert randomize floor material answer into process report
    if (arr_f_input[0] == "Yes")
      message = message + "\n- Floor material : [Random]"
    elsif (arr_f_input[0] == "No")
      message = message + "\n- Floor material : [Fixed]"
    else
      UI.messagebox("Unexpected error : ERR014")
      return
    end

    # Floor material check
    if (arr_f_input[0] == "Yes")
      message = message + "\n- Floor material : \n=========="
      # If ground floor material is randomized,
      # then split the material name by comma and then check if all material exist or not
      mtl_count = 0
      floor_mtl = arr_f_input[1].split(",")
      # ===== UPDATE [09/08/2015] V1.03 [START] =====
      # If randomize mode is set and the rooftop floor material or ground floor material
      # is randomized then check if floor material array have the same size as rooftop
      # floor material array or ground floor material array
      if (arr_main_input[8] == "Set" && (arr_main_input[4] == "Yes" || arr_gf_input[0] == "Yes"))
        # Only need to take one, because they should've been checked before
        cnt_randomized_mtl = 0
        if (arr_main_input[4] == "Yes")
          cnt_randomized_mtl = arr_main_input[5].split(",").size
        elsif (arr_gf_input[0] == "Yes")
          cnt_randomized_mtl = arr_gf_input[1].split(",").size
        else
          UI.messagebox("Unexpected error : ERR015")
          return
        end
        
        cnt_floor_mtl = arr_f_input[1].split(",").size
        if (cnt_randomized_mtl != cnt_floor_mtl)
          sbg_err_message = "Rooftop/Ground Floor has " + cnt_randomized_mtl.to_s + " materials." +
                          "\nFloor has " + cnt_floor_mtl.to_s  + " materials." +
                        "\n\nIf you choose \"Set\" random mode, all randomized material" +
                          "\nmust have same value." +
                        "\n\nYou can input repeated value, example :" +
                          "\n- rooftop_1,rooftop_1,rooftop_2" +
                          "\n- ground_1,ground_2,ground_3" +
                          "\nBuilding 1 and 2 will have same rooftop material but different" +
                          "\nground floor material. Building 3 will always have same rooftop."
          UI.messagebox(sbg_err_message)
          return
        end
      end
      # ===== UPDATE [09/08/2015] V1.03 [END] =====
      floor_mtl.each { |mtl_name|
        # The material name should not same as the target material name
        if (mtl_name == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence mtl_name
        if (!status)
          UI.messagebox("Material not found : [" + mtl_name + "]")
          return
        elsif (status)
          message = message + "\n:= " + mtl_name
        else
          UI.messagebox("Unexpected error : ERR016")
          return
        end

        mtl_count = mtl_count + 1
      }
      message = message + "\n=========="

      if (mtl_count <= 1)
        UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
        return
      end
    else
      # The material name should not same as the target material name
      if (arr_f_input[1] == arr_main_input[0])
        UI.messagebox("Material that will be used must not same as target material")
        return
      end

      # Check if material exist or not
      status = check_material_existence arr_f_input[1]
      if (!status)
        UI.messagebox("Material not found : [" + arr_f_input[1] + "]")
        return
      elsif (status)
        message = message + "\n- Floor material : [" + arr_f_input[1] + "]"
      else
        UI.messagebox("Unexpected error : ERR017")
        return
      end
    end

    # Floor level check
    # Floor level is fixed
    if (arr_f_input[3] == "No")
      if (arr_f_input[2] < 1)
        UI.messagebox("If floor level is fixed, value must be 1 or more.")
        return
      elsif (arr_f_input[2] >= 1)
        message = message + "\n- Floor level : [" + arr_f_input[2].to_s + "]"
      else
        UI.messagebox("Unexpected error : ERR018")
        return
      end
    # Floor level is randomized
    elsif (arr_f_input[3] == "Yes")
      # Minimum floor level validation
      if (arr_f_input[4] < 0)
        UI.messagebox("Minimum floor level must be 0 or more.")
        return
      elsif (arr_f_input[4] >= 0)
        message = message + "\n- Minimum floor level : [" + arr_f_input[4].to_s + "]"
      else
        UI.messagebox("Unexpected error : ERR019")
        return
      end

      # Maximum floor level validation
      if (arr_f_input[4] >= arr_f_input[5])
        UI.messagebox("Maximum floor level must be greater than minimum floor level.")
        return
      elsif (arr_f_input[4] < arr_f_input[5])
        message = message + "\n- Maximum floor level : [" + arr_f_input[5].to_s + "]"
      else
        UI.messagebox("Unexpected error : ERR020")
        return
      end
    else
      UI.messagebox("Unexpected error : ERR021")
      return
    end
    # ===== Floor Input validation [END] =====

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Display report
    status = UI.messagebox(message, MB_MULTILINE)
    status = UI.messagebox("Do you want to proceed?", MB_YESNO)
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
    if (status == IDNO)
      UI.messagebox("Aborted!")
      return
    else
      # Check : If there is no ground floor and
      #         the floor level or floor minimum level
      #         is 0, some target may not produce building.
      #         Therefore abort.
      if (arr_main_input[2] <= 0 && ((arr_f_input[3] == "No" && arr_f_input[2] <= 0) || (arr_f_input[3] == "Yes" && arr_f_input[4] <= 0)))
        UI.messagebox("If there is no ground level, floor level or minimum floor level must 1 or more!\n\nAborting!")
        return
      end
    end

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Save current setting
    if (arr_f_input[6] == "Yes")
      $sbg_saved_conf_f        = arr_f_input
      $sbg_saved_conf_f_status = true
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
  end
  # ===== Floor Input [END] =====

  # ===== Rooftop Edges Input [START] =====
  arr_re_input = []
  # Use rooftop edges?
  if (arr_main_input[6] > 0)
    # Variable for store process report message
    message = "Process check :"

    prompts  = ["Randomize rooftop edges default material?", # ===== UPDATE [11/08/2015] V1.04 [START] =====
                "Rooftop edges default material name",       # |
                "Different outer faces material?",           # |
                "Randomize outer faces material?",           # |
                "Outer faces material name",                 # |
                "Different top face material?",              # |
                "Randomize top face material?",              # |
                "Top face material name",                    # |
                "Different inner faces material?",           # |
                "Randomize inner faces material?",           # |
                "Inner face material name",                  # ===== UPDATE [11/08/2015] V1.04 [END] =====
                "Save current setting?",                     # ===== UPDATE [12/08/2015] V1.05 =====
                "Use previous setting?"]                     # ===== UPDATE [12/08/2015] V1.05 =====
    defaults = ["No",
                "",
                "No", # ===== UPDATE [11/08/2015] V1.04 [START] =====
                "No", # |
                "",   # |
                "No", # |
                "No", # |
                "",   # |
                "No", # |
                "No", # |
                "",   # ===== UPDATE [11/08/2015] V1.04 [END] =====
                "No", # ===== UPDATE [12/08/2015] V1.05 =====
                "No"] # ===== UPDATE [12/08/2015] V1.05 =====
    list     = ["Yes|No",
                "",
                "Yes|No", # ===== UPDATE [11/08/2015] V1.04 [START] =====
                "Yes|No", # |
                "",       # |
                "Yes|No", # |
                "Yes|No", # |
                "",       # |
                "Yes|No", # |
                "Yes|No", # |
                "",       # ===== UPDATE [11/08/2015] V1.04 [END] =====
                "Yes|No", # ===== UPDATE [12/08/2015] V1.05 =====
                "Yes|No"] # ===== UPDATE [12/08/2015] V1.05 =====
    arr_re_input = UI.inputbox(prompts, defaults, list, "Rooftop Edges Setting")
    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Using previous saved setting
    if (arr_re_input[12] == "Yes")
      # If previous saved setting exist then load it, otherwise fail
      if ($sbg_saved_conf_re_status)
        arr_re_input = $sbg_saved_conf_re
      else
        UI.messagebox("There are no previous saved configuration.\n\nNotes : SBG doesn't save any configuration\nafter you quit SketchUp.")
        return
      end
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====

    # ===== Rooftop Edges Input validation [START] =====
    # ===== UPDATE [11/08/2015] V1.04 [START] =====
    # If each sides uses different material then no need to check rooftop edges
    # default material, otherwise check it
    if (arr_re_input[2] == "No" || arr_re_input[5] == "No" || arr_re_input[8] == "No")
      # Insert randomize rooftop edges default material answer into process report
      if (arr_re_input[0] == "Yes")
        message = message + "\n- Rooftop edges default material : [Random]"
      elsif (arr_re_input[0] == "No")
        message = message + "\n- Rooftop edges default material : [Fixed]"
      else
        UI.messagebox("Unexpected error : ERR022")
        return
      end

      # Rooftop edges default material check
      if (arr_re_input[0] == "Yes")
        message = message + "\n- Rooftop edges default material : \n=========="
        # If rooftop edges default material is randomized, then split the material name
        # by comma and then check if all material exist or not
        re_mtl = arr_re_input[1].split(",")

        # If randomize mode is set and the rooftop floor material, ground floor material,
        # floor material, or any different rooftop edges side is randomized then check if
        # rooftop edges default material array have the same size as  material array before
        if (arr_main_input[8] == "Set" && ((arr_main_input[4] == "Yes" || arr_gf_input[0] == "Yes" || arr_f_input[0] == "Yes") || ((arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes") || (arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes") || (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes"))))
          chk_random_count = true
          msg_randomized_mtl = ""
          msg_randomized_mtl2 = "Rooftop Edges Default"
          cnt_randomized_mtl = 0

          # Count the rooftop edges default material list
          cnt_curr_mtl = arr_re_input[1].split(",").size

          # Count the rooftop floor material list if randomized, and compare the list
          # with rooftop edges default material list
          if (arr_main_input[4] == "Yes")
            cnt_randomized_mtl = arr_main_input[5].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Floor"
              chk_random_count = false
            end
          end
          
          # Count the ground floor material list if randomized, and compare the list
          # with rooftop edges default material list
          if (arr_gf_input[0] == "Yes")
            cnt_randomized_mtl = arr_gf_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Ground Floor"
              chk_random_count = false
            end
          end
          
          # Count the floor material list if randomized, and compare the list
          # with rooftop edges default material list
          if (arr_f_input[0] == "Yes")
            cnt_randomized_mtl = arr_f_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Floor"
              chk_random_count = false
            end
          end
          
          # Count the outer rooftop edges material list if randomized, and compare
          # the list with rooftop edges default material list
          if (arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes")
            cnt_randomized_mtl = arr_re_input[4].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Outer Rooftop Edges"
              chk_random_count = false
            end
          end
          
          # Count the top rooftop edge material list if randomized, and compare
          # the list with rooftop edges default material list
          if (arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes")
            cnt_randomized_mtl = arr_re_input[7].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Top Rooftop Edge"
              chk_random_count = false
            end
          end
          
          # Count the inner rooftop edge material list if randomized, and compare
          # the list with rooftop edges default material list
          if (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes")
            cnt_randomized_mtl = arr_re_input[10].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Inner Rooftop Edges"
              chk_random_count = false
            end
          end
          
          if (chk_random_count == false)
            sbg_err_message =  msg_randomized_mtl + " has " + cnt_randomized_mtl.to_s + " materials." +
                            "\n" + msg_randomized_mtl2 + " has " + cnt_curr_mtl.to_s  + " materials." +
                          "\n\nIf you choose \"Set\" random mode, all randomized material" +
                            "\nmust have same value." +
                          "\n\nYou can input repeated value, example :" +
                            "\n- rooftop_1,rooftop_1,rooftop_2" +
                            "\n- ground_1,ground_2,ground_3" +
                            "\nBuilding 1 and 2 will have same rooftop material but different" +
                            "\nground floor material. Building 3 will always have same rooftop."
            UI.messagebox(sbg_err_message)
            return
          end
        end

        # Check each existence and insert the name to process check message
        mtl_count = 0
        re_mtl.each { |mtl_name|
          # The material name should not same as the target material name
          if (mtl_name == arr_main_input[0])
            UI.messagebox("Material that will be used must not same as target material")
            return
          end

          # Check if material exist or not
          status = check_material_existence mtl_name
          if (!status)
            UI.messagebox("Material not found : [" + mtl_name + "]")
            return
          elsif (status)
            message = message + "\n:= " + mtl_name
          else
            UI.messagebox("Unexpected error : ERR023")
            return
          end

          mtl_count = mtl_count + 1
        }
        message = message + "\n=========="

        if (mtl_count <= 1)
          UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
          return
        end
      else
        # The material name should not same as the target material name
        if (arr_re_input[1] == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence arr_re_input[1]
        if (!status)
          UI.messagebox("Material not found : [" + arr_re_input[1] + "]")
          return
        elsif (status)
          message = message + "\n- Rooftop edges default material : [" + arr_re_input[1] + "]"
        else
          UI.messagebox("Unexpected error : ERR024")
          return
        end
      end
    end

    # The outer sides uses different material
    if (arr_re_input[2] == "Yes")
      # Insert randomize outer rooftop edges material answer into process report
      if (arr_re_input[3] == "Yes")
        message = message + "\n- Outer rooftop edges material : [Random]"
      elsif (arr_re_input[3] == "No")
        message = message + "\n- Outer rooftop edges material : [Fixed]"
      else
        UI.messagebox("Unexpected error : ERR025")
        return
      end

      # Outer rooftop edges material check
      # Outer rooftop edges material is randomized
      if (arr_re_input[3] == "Yes")
        message = message + "\n- Outer rooftop edges material : \n=========="
        # If outer rooftop edges material is randomized, then split the material name
        # by comma and then check if all material exist or not
        re_mtl = arr_re_input[4].split(",")

        # If randomize mode is set and the rooftop floor material, ground floor material,
        # floor material, rooftop edges default material or any different rooftop edges
        # side is randomized then check if outer rooftop edges material array have the
        # same size as material array before
        if (arr_main_input[8] == "Set" && ((arr_main_input[4] == "Yes" || arr_gf_input[0] == "Yes" || arr_f_input[0] == "Yes" || arr_re_input[0] == "Yes") || ((arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes") || (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes"))))
          chk_random_count = true
          msg_randomized_mtl = ""
          msg_randomized_mtl2 = "Outer Rooftop Edges"
          cnt_randomized_mtl = 0

          # Count the outer rooftop edges material list
          cnt_curr_mtl = arr_re_input[4].split(",").size

          # Count the rooftop floor material list if randomized, and compare the list
          # with outer rooftop edges material list
          if (arr_main_input[4] == "Yes")
            cnt_randomized_mtl = arr_main_input[5].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Floor"
              chk_random_count = false
            end
          end
          
          # Count the ground floor material list if randomized, and compare the list
          # with outer rooftop edges material list
          if (arr_gf_input[0] == "Yes")
            cnt_randomized_mtl = arr_gf_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Ground Floor"
              chk_random_count = false
            end
          end
          
          # Count the floor material list if randomized, and compare the list
          # with outer rooftop edges material list
          if (arr_f_input[0] == "Yes")
            cnt_randomized_mtl = arr_f_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Floor"
              chk_random_count = false
            end
          end
          
          # Count the rooftop edges default material list if randomized and
          # if some rooftop edges is using default, and compare the list with
          # outer rooftop edges material list
          if (arr_re_input[0] == "Yes" && (arr_re_input[5] == "No" || arr_re_input[8] == "No"))
            cnt_randomized_mtl = arr_re_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Edges Default"
              chk_random_count = false
            end
          end
          
          # Count the top rooftop edge material list if randomized, and compare
          # the list with outer rooftop edges material list
          if (arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes")
            cnt_randomized_mtl = arr_re_input[7].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Top Rooftop Edge"
              chk_random_count = false
            end
          end
          
          # Count the inner rooftop edge material list if randomized, and compare
          # the list with outer rooftop edges material list
          if (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes")
            cnt_randomized_mtl = arr_re_input[10].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Inner Rooftop Edges"
              chk_random_count = false
            end
          end
          
          if (chk_random_count == false)
            sbg_err_message =  msg_randomized_mtl + " has " + cnt_randomized_mtl.to_s + " materials." +
                            "\n" + msg_randomized_mtl2 + " has " + cnt_curr_mtl.to_s  + " materials." +
                          "\n\nIf you choose \"Set\" random mode, all randomized material" +
                            "\nmust have same value." +
                          "\n\nYou can input repeated value, example :" +
                            "\n- rooftop_1,rooftop_1,rooftop_2" +
                            "\n- ground_1,ground_2,ground_3" +
                            "\nBuilding 1 and 2 will have same rooftop material but different" +
                            "\nground floor material. Building 3 will always have same rooftop."
            UI.messagebox(sbg_err_message)
            return
          end
        end

        # Check each existence and insert the name to process check message
        mtl_count = 0
        re_mtl.each { |mtl_name|
          # The material name should not same as the target material name
          if (mtl_name == arr_main_input[0])
            UI.messagebox("Material that will be used must not same as target material")
            return
          end

          # Check if material exist or not
          status = check_material_existence mtl_name
          if (!status)
            UI.messagebox("Material not found : [" + mtl_name + "]")
            return
          elsif (status)
            message = message + "\n:= " + mtl_name
          else
            UI.messagebox("Unexpected error : ERR026")
            return
          end

          mtl_count = mtl_count + 1
        }
        message = message + "\n=========="

        if (mtl_count <= 1)
          UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
          return
        end
      # Outer rooftop edges material is fixed
      else
        # The material name should not same as the target material name
        if (arr_re_input[4] == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence arr_re_input[4]
        if (!status)
          UI.messagebox("Material not found : [" + arr_re_input[4] + "]")
          return
        elsif (status)
          message = message + "\n- Outer rooftop edges material : [" + arr_re_input[4] + "]"
        else
          UI.messagebox("Unexpected error : ERR027")
          return
        end
      end
    end

    # The top side uses different material
    if (arr_re_input[5] == "Yes")
      # Insert randomize top rooftop edge material answer into process report
      if (arr_re_input[6] == "Yes")
        message = message + "\n- Top rooftop edge material : [Random]"
      elsif (arr_re_input[6] == "No")
        message = message + "\n- Top rooftop edge material : [Fixed]"
      else
        UI.messagebox("Unexpected error : ERR028")
        return
      end

      # Top rooftop edge material check
      # Top rooftop edge material is randomized
      if (arr_re_input[6] == "Yes")
        message = message + "\n- Top rooftop edge material : \n=========="
        # If top rooftop edges material is randomized, then split the material name
        # by comma and then check if all material exist or not
        re_mtl = arr_re_input[7].split(",")

        # If randomize mode is set and the rooftop floor material, ground floor material,
        # floor material, rooftop edges default material or any different rooftop edges
        # side is randomized then check if top rooftop edge material array have the
        # same size as material array before
        if (arr_main_input[8] == "Set" && ((arr_main_input[4] == "Yes" || arr_gf_input[0] == "Yes" || arr_f_input[0] == "Yes" || arr_re_input[0] == "Yes") || ((arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes") || (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes"))))
          chk_random_count = true
          msg_randomized_mtl = ""
          msg_randomized_mtl2 = "Top Rooftop Edge"
          cnt_randomized_mtl = 0

          # Count the top rooftop edge material list
          cnt_curr_mtl = arr_re_input[7].split(",").size

          # Count the rooftop floor material list if randomized, and compare the list
          # with top rooftop edge material list
          if (arr_main_input[4] == "Yes")
            cnt_randomized_mtl = arr_main_input[5].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Floor"
              chk_random_count = false
            end
          end
          
          # Count the ground floor material list if randomized, and compare the list
          # with top rooftop edge material list
          if (arr_gf_input[0] == "Yes")
            cnt_randomized_mtl = arr_gf_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Ground Floor"
              chk_random_count = false
            end
          end
          
          # Count the floor material list if randomized, and compare the list
          # with top rooftop edge material list
          if (arr_f_input[0] == "Yes")
            cnt_randomized_mtl = arr_f_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Floor"
              chk_random_count = false
            end
          end
          
          # Count the rooftop edges default material list if randomized and
          # if some rooftop edges is using default, and compare the list with
          # top rooftop edge material list
          if (arr_re_input[0] == "Yes" && (arr_re_input[2] == "No" || arr_re_input[8] == "No"))
            cnt_randomized_mtl = arr_re_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Edges Default"
              chk_random_count = false
            end
          end
          
          # Count the top rooftop edge material list if randomized, and compare
          # the list with top rooftop edge material list
          if (arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes")
            cnt_randomized_mtl = arr_re_input[4].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Outer Rooftop Edge"
              chk_random_count = false
            end
          end
          
          # Count the inner rooftop edge material list if randomized, and compare
          # the list with top rooftop edge material list
          if (arr_re_input[8] == "Yes" && arr_re_input[9] == "Yes")
            cnt_randomized_mtl = arr_re_input[10].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Inner Rooftop Edges"
              chk_random_count = false
            end
          end
          
          if (chk_random_count == false)
            sbg_err_message =  msg_randomized_mtl + " has " + cnt_randomized_mtl.to_s + " materials." +
                            "\n" + msg_randomized_mtl2 + " has " + cnt_curr_mtl.to_s  + " materials." +
                          "\n\nIf you choose \"Set\" random mode, all randomized material" +
                            "\nmust have same value." +
                          "\n\nYou can input repeated value, example :" +
                            "\n- rooftop_1,rooftop_1,rooftop_2" +
                            "\n- ground_1,ground_2,ground_3" +
                            "\nBuilding 1 and 2 will have same rooftop material but different" +
                            "\nground floor material. Building 3 will always have same rooftop."
            UI.messagebox(sbg_err_message)
            return
          end
        end

        # Check each existence and insert the name to process check message
        mtl_count = 0
        re_mtl.each { |mtl_name|
          # The material name should not same as the target material name
          if (mtl_name == arr_main_input[0])
            UI.messagebox("Material that will be used must not same as target material")
            return
          end

          # Check if material exist or not
          status = check_material_existence mtl_name
          if (!status)
            UI.messagebox("Material not found : [" + mtl_name + "]")
            return
          elsif (status)
            message = message + "\n:= " + mtl_name
          else
            UI.messagebox("Unexpected error : ERR029")
            return
          end

          mtl_count = mtl_count + 1
        }
        message = message + "\n=========="

        if (mtl_count <= 1)
          UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
          return
        end
      # Top rooftop edge material is fixed
      else
        # The material name should not same as the target material name
        if (arr_re_input[7] == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence arr_re_input[7]
        if (!status)
          UI.messagebox("Material not found : [" + arr_re_input[7] + "]")
          return
        elsif (status)
          message = message + "\n- Outer rooftop edges material : [" + arr_re_input[7] + "]"
        else
          UI.messagebox("Unexpected error : ERR030")
          return
        end
      end
    end

    # The inner sides uses different material
    if (arr_re_input[8] == "Yes")
      # Insert randomize inner rooftop edges material answer into process report
      if (arr_re_input[9] == "Yes")
        message = message + "\n- Inner rooftop edges material : [Random]"
      elsif (arr_re_input[9] == "No")
        message = message + "\n- Inner rooftop edges material : [Fixed]"
      else
        UI.messagebox("Unexpected error : ERR031")
        return
      end

      # Inner rooftop edges material check
      # Inner rooftop edges material is randomized
      if (arr_re_input[9] == "Yes")
        message = message + "\n- Inner rooftop edges material : \n=========="
        # If inner rooftop edges material is randomized, then split the material name
        # by comma and then check if all material exist or not
        re_mtl = arr_re_input[10].split(",")

        # If randomize mode is set and the rooftop floor material, ground floor material,
        # floor material, rooftop edges default material or any different rooftop edges
        # side is randomized then check if inner rooftop edges material array have the
        # same size as material array before
        if (arr_main_input[8] == "Set" && ((arr_main_input[4] == "Yes" || arr_gf_input[0] == "Yes" || arr_f_input[0] == "Yes" || arr_re_input[0] == "Yes") || ((arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes") || (arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes"))))
          chk_random_count = true
          msg_randomized_mtl = ""
          msg_randomized_mtl2 = "Inner Rooftop Edges"
          cnt_randomized_mtl = 0

          # Count the inner rooftop edges material list
          cnt_curr_mtl = arr_re_input[10].split(",").size

          # Count the rooftop floor material list if randomized, and compare the list
          # with inner rooftop edges material list
          if (arr_main_input[4] == "Yes")
            cnt_randomized_mtl = arr_main_input[5].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Floor"
              chk_random_count = false
            end
          end
          
          # Count the ground floor material list if randomized, and compare the list
          # with inner rooftop edges material list
          if (arr_gf_input[0] == "Yes")
            cnt_randomized_mtl = arr_gf_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Ground Floor"
              chk_random_count = false
            end
          end
          
          # Count the floor material list if randomized, and compare the list
          # with inner rooftop edges material list
          if (arr_f_input[0] == "Yes")
            cnt_randomized_mtl = arr_f_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Floor"
              chk_random_count = false
            end
          end
          
          # Count the rooftop edges default material list if randomized and
          # if some rooftop edges is using default, and compare the list with
          # inner rooftop edges material list
          if (arr_re_input[0] == "Yes" && (arr_re_input[2] == "No" || arr_re_input[5] == "No"))
            cnt_randomized_mtl = arr_re_input[1].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Rooftop Edges Default"
              chk_random_count = false
            end
          end
          
          # Count the outer rooftop edges material list if randomized, and compare
          # the list with inner rooftop edges material list
          if (arr_re_input[2] == "Yes" && arr_re_input[3] == "Yes")
            cnt_randomized_mtl = arr_re_input[4].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Outer Rooftop Edges"
              chk_random_count = false
            end
          end
          
          # Count the top rooftop edge material list if randomized, and compare
          # the list with inner rooftop edges material list
          if (arr_re_input[5] == "Yes" && arr_re_input[6] == "Yes")
            cnt_randomized_mtl = arr_re_input[7].split(",").size
            # If not same then the check will fail
            if (cnt_curr_mtl != cnt_randomized_mtl)
              msg_randomized_mtl = "Top Rooftop Edge"
              chk_random_count = false
            end
          end
          
          if (chk_random_count == false)
            sbg_err_message =  msg_randomized_mtl + " has " + cnt_randomized_mtl.to_s + " materials." +
                            "\n" + msg_randomized_mtl2 + " has " + cnt_curr_mtl.to_s  + " materials." +
                          "\n\nIf you choose \"Set\" random mode, all randomized material" +
                            "\nmust have same value." +
                          "\n\nYou can input repeated value, example :" +
                            "\n- rooftop_1,rooftop_1,rooftop_2" +
                            "\n- ground_1,ground_2,ground_3" +
                            "\nBuilding 1 and 2 will have same rooftop material but different" +
                            "\nground floor material. Building 3 will always have same rooftop."
            UI.messagebox(sbg_err_message)
            return
          end
        end

        # Check each existence and insert the name to process check message
        mtl_count = 0
        re_mtl.each { |mtl_name|
          # The material name should not same as the target material name
          if (mtl_name == arr_main_input[0])
            UI.messagebox("Material that will be used must not same as target material")
            return
          end

          # Check if material exist or not
          status = check_material_existence mtl_name
          if (!status)
            UI.messagebox("Material not found : [" + mtl_name + "]")
            return
          elsif (status)
            message = message + "\n:= " + mtl_name
          else
            UI.messagebox("Unexpected error : ERR032")
            return
          end

          mtl_count = mtl_count + 1
        }
        message = message + "\n=========="

        if (mtl_count <= 1)
          UI.messagebox("Need more than 1 material if using randomized material, separated by comma.")
          return
        end
      # Inner rooftop edges material is fixed
      else
        # The material name should not same as the target material name
        if (arr_re_input[10] == arr_main_input[0])
          UI.messagebox("Material that will be used must not same as target material")
          return
        end

        # Check if material exist or not
        status = check_material_existence arr_re_input[10]
        if (!status)
          UI.messagebox("Material not found : [" + arr_re_input[10] + "]")
          return
        elsif (status)
          message = message + "\n- Outer rooftop edges material : [" + arr_re_input[10] + "]"
        else
          UI.messagebox("Unexpected error : ERR033")
          return
        end
      end
    end
    # ===== UPDATE [11/08/2015] V1.04 [END] =====
    # ===== Rooftop Edges Input validation [END] =====

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Display report
    status = UI.messagebox(message, MB_MULTILINE)
    status = UI.messagebox("Do you want to proceed?", MB_YESNO)
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
    if (status == IDNO)
      UI.messagebox("Aborted!")
      return
    end

    # ===== UPDATE [12/08/2015] V1.05 [START] =====
    # Save current setting
    if (arr_re_input[11] == "Yes")
      $sbg_saved_conf_re        = arr_re_input
      $sbg_saved_conf_re_status = true
    end
    # ===== UPDATE [12/08/2015] V1.05 [END] =====
  end
  # ===== Rooftop Edges Input [END] =====


  # Final Confirmation
  status = UI.messagebox("This process may take some time.\n\nHave you save your work?", MB_YESNO)
  if (status == IDNO)
    status = UI.messagebox("You really want to continue without saving your work first?", MB_YESNO)
    if (status == IDNO)
      UI.messagebox("Aborted!\n\nSBG doesn't want anything bad happened to your work during process.\nThank you.")
      return
    end
  end

  # Explore and generate selection
  dbg_search_floorplan (arr_main_input, arr_gf_input, arr_f_input, arr_re_input)
end

def dbg_search_floorplan (conf_general, conf_gf, conf_f, conf_re)
  # arr_main_input -> conf_general
   target_mtl_name = conf_general[0]
        face_erase = conf_general[1]
   ground_floor_ht = conf_general[2]
          floor_ht = conf_general[3]
  rand_rooftop_mtl = conf_general[4]
       rooftop_mtl = conf_general[5]
   cnt_rooftop_mtl = 1
  if (rand_rooftop_mtl == "Yes")
        rooftop_mtl = conf_general[5].split(",")
    cnt_rooftop_mtl = rooftop_mtl.size
  end
  rooftop_edges_ht = conf_general[6]
           mtl_fit = conf_general[7] # ===== UPDATE [08/08/2015] V1.02 =====
       random_mode = conf_general[8] # ===== UPDATE [09/08/2015] V1.03 =====

  #   arr_gf_input -> conf_gf
  rand_ground_floor_mtl = conf_gf[0]
       ground_floor_mtl = conf_gf[1]
   cnt_ground_floor_mtl = 1
  if (rand_ground_floor_mtl == "Yes")
        ground_floor_mtl = conf_gf[1].split(",")
    cnt_ground_floor_mtl = ground_floor_mtl.size
  end

  #    arr_f_input -> conf_f
    rand_floor_mtl = conf_f[0]
         floor_mtl = conf_f[1]
     cnt_floor_mtl = 1
  if (rand_floor_mtl == "Yes")
        floor_mtl = conf_f[1].split(",")
    cnt_floor_mtl = floor_mtl.size
  end
       floor_level = conf_f[2]
  rand_floor_level = conf_f[3]
   min_floor_level = conf_f[4]
   max_floor_level = conf_f[5]

  #   arr_re_input -> conf_re
  rand_rooftop_edges_mtl = conf_re[0]
       rooftop_edges_mtl = conf_re[1]
   cnt_rooftop_edges_mtl = 1
  if (rand_rooftop_edges_mtl == "Yes")
        rooftop_edges_mtl = conf_re[1].split(",")
    cnt_rooftop_edges_mtl = rooftop_edges_mtl.size
  end
  # ===== UPDATE [11/08/2015] V1.04 [START] =====
  diff_rooftop_edges_outer_mtl = conf_re[2]
  rand_rooftop_edges_outer_mtl = conf_re[3]
       rooftop_edges_outer_mtl = conf_re[4]
   cnt_rooftop_edges_outer_mtl = 1
  if (diff_rooftop_edges_outer_mtl == "Yes" && rand_rooftop_edges_outer_mtl == "Yes")
        rooftop_edges_outer_mtl = conf_re[4].split(",")
    cnt_rooftop_edges_outer_mtl = rooftop_edges_outer_mtl.size
  end
  diff_rooftop_edges_top_mtl = conf_re[5]
  rand_rooftop_edges_top_mtl = conf_re[6]
       rooftop_edges_top_mtl = conf_re[7]
   cnt_rooftop_edges_top_mtl = 1
  if (diff_rooftop_edges_top_mtl == "Yes" && rand_rooftop_edges_top_mtl == "Yes")
        rooftop_edges_top_mtl = conf_re[7].split(",")
    cnt_rooftop_edges_top_mtl = rooftop_edges_top_mtl.size
  end
  diff_rooftop_edges_inner_mtl = conf_re[8]
  rand_rooftop_edges_inner_mtl = conf_re[9]
       rooftop_edges_inner_mtl = conf_re[10]
   cnt_rooftop_edges_inner_mtl = 1
  if (diff_rooftop_edges_inner_mtl == "Yes" && rand_rooftop_edges_inner_mtl == "Yes")
        rooftop_edges_inner_mtl = conf_re[10].split(",")
    cnt_rooftop_edges_inner_mtl = rooftop_edges_inner_mtl.size
  end
  # ===== UPDATE [11/08/2015] V1.04 [END] =====

  # START OPERATION
  mod = Sketchup.active_model
  ret = mod.start_operation('Simple Building Generator', true)
  if (!ret)
    UI.messagebox("Unexpected error : ERR034")
    return
  end

  # Get selection
  sel = Sketchup.active_model.selection
  sel.each { |obj|
    # Process each face in selection
    if obj.is_a? Sketchup::Face
      # ===== UPDATE [12/08/2015] V1.05 [START] =====
      face_is_floorplan = sbg_face_is_floorplan obj, target_mtl_name
      if (face_is_floorplan)
        # Process information for method

        # ===== UPDATE [09/08/2015] V1.03 [START] =====
        # For using "Set" random mode
        tex_set_index = 0
        if (random_mode == "Set")
          if (rand_ground_floor_mtl == "Yes")
            tex_set_index = rand(cnt_ground_floor_mtl)
          elsif (rand_floor_mtl == "Yes")
            tex_set_index = rand(cnt_floor_mtl)
          elsif (rand_rooftop_mtl == "Yes")
            tex_set_index = rand(cnt_rooftop_mtl)
          elsif (rand_rooftop_edges_mtl == "Yes")
            tex_set_index = rand(cnt_rooftop_edges_mtl)
          end
        end
        # ===== UPDATE [09/08/2015] V1.03 [END] =====

        # floorplan
        param1 = obj

        # ground_floor_ht
        param2 = ground_floor_ht

        # ground_floor_mtl
        param3 = ""
        if (rand_ground_floor_mtl == "Yes")
          tex_index = rand(cnt_ground_floor_mtl)
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          # If using the "Set" random mode, override the texture selection
          # with global selection
          if (random_mode == "Set")
            tex_index = tex_set_index
          end
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          param3 = ground_floor_mtl[tex_index]
        elsif (rand_ground_floor_mtl == "No")
          param3 = ground_floor_mtl
        end

        # floor_ht
        param4 = floor_ht

        # floor_mtl
        param5 = ""
        if (rand_floor_mtl == "Yes")
          tex_index = rand(cnt_floor_mtl)
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          # If using the "Set" random mode, override the texture selection
          # with global selection
          if (random_mode == "Set")
            tex_index = tex_set_index
          end
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          param5 = floor_mtl[tex_index]
        elsif (rand_floor_mtl == "No")
          param5 = floor_mtl
        end

        # floor_level
        param6 = 0
        if (rand_floor_level == "Yes")
          param6 = rand(max_floor_level - min_floor_level + 1) + min_floor_level
        elsif (rand_floor_level == "No")
          param6 = floor_level
        end

        # rooftop_mtl
        param7 = ""
        if (rand_rooftop_mtl == "Yes")
          tex_index = rand(cnt_rooftop_mtl)
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          # If using the "Set" random mode, override the texture selection
          # with global selection
          if (random_mode == "Set")
            tex_index = tex_set_index
          end
          # ===== UPDATE [09/08/2015] V1.03 [START] =====
          param7 = rooftop_mtl[tex_index]
        elsif (rand_rooftop_mtl == "No")
          param7 = rooftop_mtl
        end

        # rooftop_edges_ht
        param8 = rooftop_edges_ht

        # ===== UPDATE [11/08/2015] V1.04 [START] =====
        # rooftop_edges_outer_mtl
        param9 = ""
        # Different outer rooftop edges material
        if (diff_rooftop_edges_outer_mtl == "Yes")
          # Outer rooftop edges material is randomized
          if (rand_rooftop_edges_outer_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_outer_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param9 = rooftop_edges_outer_mtl[tex_index]
          # Outer rooftop edges material is fixed
          else
            param9 = rooftop_edges_outer_mtl
          end
        # Use default rooftop edges material
        else
          # Rooftop edges default material is randomized
          if (rand_rooftop_edges_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param9 = rooftop_edges_mtl[tex_index]
          # Rooftop edges default material is fixed
          else
            param9 = rooftop_edges_mtl
          end
        end

        # rooftop_edges_top_mtl
        param10 = ""
        # Different top rooftop edge material
        if (diff_rooftop_edges_top_mtl == "Yes")
          # Top rooftop edges material is randomized
          if (rand_rooftop_edges_top_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_top_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param10 = rooftop_edges_top_mtl[tex_index]
          # Top rooftop edges material is fixed
          else
            param10 = rooftop_edges_top_mtl
          end
        # Use default rooftop edges material
        else
          # Rooftop edges default material is randomized
          if (rand_rooftop_edges_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param10 = rooftop_edges_mtl[tex_index]
          # Rooftop edges default material is fixed
          else
            param10 = rooftop_edges_mtl
          end
        end

        # rooftop_edges_inner_mtl
        param11 = ""
        # Different inner rooftop edge material
        if (diff_rooftop_edges_inner_mtl == "Yes")
          # Inner rooftop edges material is randomized
          if (rand_rooftop_edges_inner_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_inner_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param11 = rooftop_edges_inner_mtl[tex_index]
          # Inner rooftop edges material is fixed
          else
            param11 = rooftop_edges_inner_mtl
          end
        # Use default rooftop edges material
        else
          # Rooftop edges default material is randomized
          if (rand_rooftop_edges_mtl == "Yes")
            tex_index = rand(cnt_rooftop_edges_mtl)
            # If using the "Set" random mode, override the texture selection
            # with global selection
            if (random_mode == "Set")
              tex_index = tex_set_index
            end
            param11 = rooftop_edges_mtl[tex_index]
          # Rooftop edges default material is fixed
          else
            param11 = rooftop_edges_mtl
          end
        end

        # Toggle texture fit to geometry
        param12 = false
        if (mtl_fit == "Yes")
          param12 = true
        end
        # ===== UPDATE [11/08/2015] V1.04 [END] =====

        # Process the face to building
        sbg_generate_building param1, param2, param3, param4, param5, param6, param7, param8, param9, param10, param11, param12 # ===== UPDATE [11/08/2015] V1.04 =====

        # Erase when done
        if (face_erase == "Yes")
          obj.erase!
        end
      end
      # ===== UPDATE [12/08/2015] V1.05 [END] =====
    end
  }

  # COMMIT OPERATION
  ret = mod.commit_operation
  if (!ret)
    UI.messagebox("Unexpected error : ERR035")
    return
  end

  # ===== UPDATE [12/08/2015] V1.05 [START] =====
  message_end =     "Thank you for using building generator v1.07" +
                "\n\nPlease report any bug or share any works you've made" +
                  "\nwith this plugin." +
                  "\nIt will motivate me a lot :)" +
                "\n\nAlso don't hesitate to ask me if you find any difficulty" +
                  "\nto use this plugin." +
                "\n\n==============================" +
                  "\nsketchucation.com/forums/viewtopic.php?f=323&t=62725" +
                  "\n=============================="
  UI.messagebox(message_end)
  # ===== UPDATE [12/08/2015] V1.05 [END] =====
end

def sbg_generate_building (floorplan,
                           ground_floor_ht,
                           ground_floor_mtl,
                           floor_ht,
                           floor_mtl,
                           floor_level,
                           rooftop_mtl,
                           rooftop_edges_ht,
                           rooftop_edges_outer_mtl,  # ===== UPDATE [11/08/2015] V1.04 =====
                           rooftop_edges_top_mtl,    # |
                           rooftop_edges_inner_mtl,  # ===== UPDATE [11/08/2015] V1.04 =====
                           mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
  nor = floorplan.normal
  if (nor)
    if (nor.z < 0)
      floorplan.reverse!
    end
  else
    UI.messagebox("Unexpected error : ERR036")
    return
  end

  arr_ground_floor_pts_down = []
   arr_ground_floor_pts_top = []

  # arr_floor_pts_top also act as rooftop floor points
  arr_floor_pts_down = []
   arr_floor_pts_top = []

  arr_rooftop_edges_pts_down = []
   arr_rooftop_edges_pts_top = []

  # ===== Get Bounding Vertex Location [START] =====
  vertex_count = 0
  arr_vertex = floorplan.vertices
  arr_vertex.each { |vertex|
    x = vertex.position[0]
    y = vertex.position[1]
    z = vertex.position[2]

    # Using ground floor
    if (ground_floor_ht > 0)
      arr_ground_floor_pts_down[vertex_count] = [x,y,z]

      z = z + ground_floor_ht
      arr_ground_floor_pts_top[vertex_count] = [x,y,z]
    end

    # Using floor
    if (floor_ht > 0 && floor_level > 0)
      arr_floor_pts_down[vertex_count] = [x,y,z]

      z = z + (floor_ht * floor_level)
      arr_floor_pts_top[vertex_count] = [x,y,z]
    else
      # If not using floor the arr_floor_pts_top still need to filled for rooftop points.
      arr_floor_pts_top[vertex_count] = [x,y,z]
    end

    # Using rooftop edges
    if (rooftop_edges_ht > 0)
      arr_rooftop_edges_pts_down[vertex_count] = [x,y,z]

      z = z + rooftop_edges_ht
      arr_rooftop_edges_pts_top[vertex_count] = [x,y,z]
    end

    vertex_count = vertex_count + 1
  }
  # ===== Get Bounding Vertex Location [END] =====

  # ===== Generate Building [START] =====
  world = Sketchup.active_model.entities
  group = world.add_group
  group_ent = group.entities
  # Create the face for each side of the building
  for index in 1..vertex_count
    # The 3rd vertex location should above the 2nd vertex location
    # The 4th vertex location should above the 1st vertex location
    if (index < vertex_count)
      # Using ground floor
      if (ground_floor_ht > 0)
        ground_floor_face = group_ent.add_face arr_ground_floor_pts_down[index - 1], arr_ground_floor_pts_down[index], arr_ground_floor_pts_top[index], arr_ground_floor_pts_top[index - 1]
        
        # Add material to the ground floor face
        mtl_status = ground_floor_face.material = ground_floor_mtl
        # Position material
        sbg_position_material (ground_floor_face, arr_ground_floor_pts_down[index - 1], arr_ground_floor_pts_down[index], ground_floor_mtl, mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
      end

      # Using floor
      if (floor_ht > 0 && floor_level > 0)
        floor_face = group_ent.add_face arr_floor_pts_down[index - 1], arr_floor_pts_down[index], arr_floor_pts_top[index], arr_floor_pts_top[index - 1]

        # Add material to the ground floor face
        mtl_status = floor_face.material = floor_mtl
        # Position material
        sbg_position_material (floor_face, arr_floor_pts_down[index - 1], arr_floor_pts_down[index], floor_mtl, mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
      end

      # Using rooftop edges
      if (rooftop_edges_ht > 0)
        rooftop_edges_face = group_ent.add_face arr_rooftop_edges_pts_down[index - 1], arr_rooftop_edges_pts_down[index], arr_rooftop_edges_pts_top[index], arr_rooftop_edges_pts_top[index - 1]

        # ===== UPDATE [11/08/2015] V1.04 [START] =====
        # Add material to the outer rooftop edges face
        mtl_status = rooftop_edges_face.material = rooftop_edges_outer_mtl
        # Position material
        sbg_position_material (rooftop_edges_face, arr_rooftop_edges_pts_down[index - 1], arr_rooftop_edges_pts_down[index], rooftop_edges_outer_mtl, mtl_fit)
        # ===== UPDATE [11/08/2015] V1.04 [END] =====
      end
    else
      # Using ground floor
      if (ground_floor_ht > 0)
        ground_floor_face = group_ent.add_face arr_ground_floor_pts_down[index - 1], arr_ground_floor_pts_down[0], arr_ground_floor_pts_top[0], arr_ground_floor_pts_top[index - 1]

        # Add material to the ground floor face
        mtl_status = ground_floor_face.material = ground_floor_mtl
        # Position material
        sbg_position_material (ground_floor_face, arr_ground_floor_pts_down[index - 1], arr_ground_floor_pts_down[0], ground_floor_mtl, mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
      end

      # Using floor
      if (floor_ht > 0 && floor_level > 0)
        floor_face = group_ent.add_face arr_floor_pts_down[index - 1], arr_floor_pts_down[0], arr_floor_pts_top[0], arr_floor_pts_top[index - 1]

        # Add material to the ground floor face
        mtl_status = floor_face.material = floor_mtl
        # Position material
        sbg_position_material (floor_face, arr_floor_pts_down[index - 1], arr_floor_pts_down[0], floor_mtl, mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
      end

      # Using rooftop edges
      if (rooftop_edges_ht > 0)
        rooftop_edges_face = group_ent.add_face arr_rooftop_edges_pts_down[index - 1], arr_rooftop_edges_pts_down[0], arr_rooftop_edges_pts_top[0], arr_rooftop_edges_pts_top[index - 1]

        # ===== UPDATE [11/08/2015] V1.04 [START] =====
        # Add material to the outer rooftop edges face
        mtl_status = rooftop_edges_face.material = rooftop_edges_outer_mtl
        # Position material
        sbg_position_material (rooftop_edges_face, arr_rooftop_edges_pts_down[index - 1], arr_rooftop_edges_pts_down[0], rooftop_edges_outer_mtl, mtl_fit)
        # ===== UPDATE [11/08/2015] V1.04 [END] =====
      end
    end
  end

  # Rooftop floor
  # If using rooftop edges
  if (rooftop_edges_ht > 0)
    # Build rooftop edges top face
    rooftop_edges_top_face = group_ent.add_face arr_rooftop_edges_pts_top
    nor = rooftop_edges_top_face.normal
    if (nor)
      if (nor.z < 0)
        rooftop_edges_top_face.reverse!
      end
    else
      UI.messagebox("Unexpected error : ERR037")
      return
    end
    # ===== UPDATE [11/08/2015] V1.04 [START] =====
    # Apply and position rooftop edges top face material
    mtl_status = rooftop_edges_top_face.material = rooftop_edges_top_mtl
    sbg_position_material (rooftop_edges_top_face, arr_rooftop_edges_pts_top[0], arr_rooftop_edges_pts_top[1], rooftop_edges_top_mtl, false)
    # ===== UPDATE [11/08/2015] V1.04 [END] =====

    # Offset rooftop edges top face for 15 cm
    rooftop_edges_offset_face = rooftop_edges_top_face.offset(-15.cm)
    # Get the vertices from rooftop edges offset face
    arr_rooftop_edges_offset_pts_top = rooftop_edges_offset_face.vertices
    # Delete rooftop edges offset face
    rooftop_edges_offset_face.erase!

    # ===== UPDATE [06/08/2015] V1.01 [START] =====
    # This is to repair some bug that makes the rooftop edges offset to outer
    cnt_arr_rooftop_edges_offset_pts = arr_rooftop_edges_offset_pts_top.size
    # Check if the vertices same or not
    if (cnt_arr_rooftop_edges_offset_pts >= vertex_count*2)
      # -2 so it will not reach the last vertex, since the last vertex
      # is deleted when first edge deleted
      for index in 0..vertex_count-2
        edges = arr_rooftop_edges_offset_pts_top[index].edges
        if (edges)
          edges.each { |edge|
            edge.erase!
          }
        end
      end

      # ===== UPDATE [11/08/2015] V1.04 [START] =====
      # After all outer edges are erased, repeat these step
      # Apply and position rooftop edges top face material
      mtl_status = rooftop_edges_top_face.material = rooftop_edges_top_mtl
      sbg_position_material (rooftop_edges_top_face, arr_rooftop_edges_pts_top[0], arr_rooftop_edges_pts_top[1], rooftop_edges_top_mtl, false)
      # ===== UPDATE [11/08/2015] V1.04 [END] =====

      # Offset rooftop edges top face for 15 cm
      rooftop_edges_offset_face = rooftop_edges_top_face.offset(15.cm)
      # Get the vertices from rooftop edges offset face
      arr_rooftop_edges_offset_pts_top = rooftop_edges_offset_face.vertices
      # Delete rooftop edges offset face
      rooftop_edges_offset_face.erase!
    end
    # ===== UPDATE [06/08/2015] V1.01 [END] =====
    
    # Create bottom vertices
    cnt_arr_rooftop_edges_offset_pts = 0
    arr_rooftop_edges_offset_pts_down = []
    arr_rooftop_edges_offset_pts_top.each { |vertex|
      x = vertex.position[0]
      y = vertex.position[1]
      z = vertex.position[2]

      z = z - rooftop_edges_ht
      arr_rooftop_edges_offset_pts_down[cnt_arr_rooftop_edges_offset_pts] = [x,y,z]

      cnt_arr_rooftop_edges_offset_pts = cnt_arr_rooftop_edges_offset_pts + 1
    }
    # Create rooftop edge inner faces
    for index in 1..cnt_arr_rooftop_edges_offset_pts
      if (index < cnt_arr_rooftop_edges_offset_pts)
        rooftop_edges_inner_face = group_ent.add_face arr_rooftop_edges_offset_pts_down[index - 1], arr_rooftop_edges_offset_pts_down[index], arr_rooftop_edges_offset_pts_top[index], arr_rooftop_edges_offset_pts_top[index - 1]

        # ===== UPDATE [11/08/2015] V1.04 [START] =====
        # Add material to the rooftop edge inner face
        mtl_status = rooftop_edges_inner_face.material = rooftop_edges_inner_mtl
        # Position material
        sbg_position_material (rooftop_edges_inner_face, arr_rooftop_edges_offset_pts_down[index - 1], arr_rooftop_edges_offset_pts_down[index], rooftop_edges_inner_mtl, mtl_fit)
        # ===== UPDATE [11/08/2015] V1.04 [END] =====
      else
        rooftop_edges_inner_face = group_ent.add_face arr_rooftop_edges_offset_pts_down[index - 1], arr_rooftop_edges_offset_pts_down[0], arr_rooftop_edges_offset_pts_top[0], arr_rooftop_edges_offset_pts_top[index - 1]

        # ===== UPDATE [11/08/2015] V1.04 [START] =====
        # Add material to the rooftop edge inner face
        mtl_status = rooftop_edges_inner_face.material = rooftop_edges_inner_mtl
        # Position material
        sbg_position_material (rooftop_edges_inner_face, arr_rooftop_edges_offset_pts_down[index - 1], arr_rooftop_edges_offset_pts_down[0], rooftop_edges_inner_mtl, mtl_fit)
        # ===== UPDATE [11/08/2015] V1.04 [END] =====
      end
    end

    # Create rooftop face
    rooftop_face = group_ent.add_face arr_rooftop_edges_offset_pts_down
    mtl_status = rooftop_face.material = rooftop_mtl

    nor = rooftop_face.normal
    if (nor)
      if (nor.z < 0)
        rooftop_face.reverse!
      end
    else
      UI.messagebox("Unexpected error : ERR038")
      return
    end

    # Position material
    sbg_position_material (rooftop_face, arr_floor_pts_top[0], arr_floor_pts_top[1], rooftop_mtl, false) # ===== UPDATE [08/08/2015] V1.02 =====
  else
    rooftop_face = group_ent.add_face arr_floor_pts_top
    mtl_status = rooftop_face.material = rooftop_mtl

    nor = rooftop_face.normal
    if (nor)
      if (nor.z < 0)
        rooftop_face.reverse!
      end
    else
      UI.messagebox("Unexpected error : ERR039")
      return
    end

    # Position material
    sbg_position_material (rooftop_face, arr_floor_pts_top[0], arr_floor_pts_top[1], rooftop_mtl, false) # ===== UPDATE [08/08/2015] V1.02 =====
  end
  # ===== Generate Building [END] =====
end

def sbg_position_material (face, pt1, pt2, mtl_name, mtl_fit) # ===== UPDATE [08/08/2015] V1.02 =====
  # ===== UPDATE [08/08/2015] V1.02 [START] =====
  # Get the material
  world = Sketchup.active_model
  mtl_lib = world.materials
  mtl = mtl_lib[mtl_name]
  # Check if material has texture
  mtl_isTex = mtl.materialType
  if (mtl_isTex != 0)
    # Material has texture
    sbg_on_front = true
    # Check if SBG should try to fit material
    if (mtl_fit)
      # Try to fit texture to geometry
      # Get geometry width
      point1 = Geom::Point3d.new pt1[0],pt1[1],pt1[2]
      point2 = Geom::Point3d.new pt2[0],pt2[1],pt2[2]
      geom_width = point1.distance point2;
      # Get texture from material
      mtl_tex = mtl.texture
      # Get texture width and height
      mtl_tex_width = mtl_tex.width
      mtl_tex_height = mtl_tex.height
      # Find how many texture should be repeated along geometry
      mtl_tex_repeat = geom_width / mtl_tex_width
      mtl_tex_repeat = mtl_tex_repeat.round
      if (mtl_tex_repeat < 1)
        # Sometime geometry width are so small it doesn't
        # have the half width of the texture making the
        # round function become 0, that's why I need this
        mtl_tex_repeat = 1
      end
      # Get the blue pin position from the ground
      mtl_tex_height = mtl_tex_height + pt1[2]

      # I'm sorry for whoever read this to understand the UV in SU.
      # I get all these point from trial and error that myself can't fully understand.
      # But since you're here I'll try my best to explain :D

      # - Although in SketchUp Ruby API documentation says that the 'q' is not used, it's wrong.
      #   at mtl_uv_pts[0] I use the q location as z location of the red pin.
      # - It's all about placing pins in the right place, like when you position texture in SketchUp.
      # - mtl_uv_pts[0] : This is the red pin position correspond to texture position in world space.
      # - mtl_uv_pts[1] : I don't know this but [0,0,0] give me what I needs :D
      # - mtl_uv_pts[2] : This is the green pin position correspond to texture rotation.
      # - mtl_uv_pts[3] : This x coordinate should have the same value as
      #                   mtl_uv_pts[5] x value, otherwise your texture will be skewed.
      #                   And this will affect the position of your green pin, for example
      #                   when your red pin and green pin distance is 10 when this value is
      #                   set to 0.5 then it will make your green pin distance become 20.
      #                   I think the formula is distance / mtl_uv_pts[3][x].
      #                   The other 0 value is trial and errors :D
      # - mtl_uv_pts[4] : I assume this as the yellow pin location, that's why I position
      #                   this pin above the green pin location. But it also seems to be a controller
      #                   of the blue pin location, that's why I need this :D
      # - mtl_uv_pts[5] : It has the same explanation as mtl_uv_pts[3]

      # That's all I can explain.
      # Please drop by at my thread (http://sketchucation.com/forums/viewtopic.php?f=323&t=62725)
      # or contact me through (mohrizkyk@gmail.com)
      # I'd love to hear any from you ;)
      mtl_uv_pts = []
      mtl_uv_pts[0] = pt1
      mtl_uv_pts[1] = [0,0,0]
      mtl_uv_pts[2] = pt2
      mtl_uv_pts[3] = [mtl_tex_repeat,0,0]
      mtl_uv_pts[4] = [pt2[0],pt2[1],mtl_tex_height]
      mtl_uv_pts[5] = [mtl_tex_repeat,1,1]

      face.position_material(mtl, mtl_uv_pts, sbg_on_front)
    else
      # No fit just reposition the texture
      mtl_uv_pts = []
      mtl_uv_pts[0] = pt1
      mtl_uv_pts[1] = [0,0,0]

      face.position_material(mtl, mtl_uv_pts, sbg_on_front)
    end
  end
  # ===== UPDATE [08/08/2015] V1.02 [END] =====
end

def sbg_count_floorplan (target_mtl_name)
  selection = Sketchup.active_model.selection
  floorplan_count = 0

  selection.each { |obj|
    if obj.is_a? Sketchup::Face
      # ===== UPDATE [12/08/2015] V1.05 [START] =====
      face_is_floorplan = sbg_face_is_floorplan obj, target_mtl_name
      if (face_is_floorplan)
        floorplan_count = floorplan_count + 1
      end
      # ===== UPDATE [12/08/2015] V1.05 [END] =====
    end
  }

  return floorplan_count
end

# ===== UPDATE [12/08/2015] V1.05 [START] =====
def sbg_face_is_floorplan (face, floorplan_mtl_name)
  face_is_floorplan = false
  # Get the face material
  face_mtl      = face.material
  face_back_mtl = face.back_material

  # Check if face have material applied
  # Check the front face
  if (face_mtl)
    # Get face material name and compare
    face_mtl_name = face_mtl.display_name.to_s
    if (face_mtl_name == floorplan_mtl_name)
      face_is_floorplan = true
    end
  end
  # Check the back face
  if (face_back_mtl)
    # Get face material name and compare
    face_back_mtl_name = face_back_mtl.display_name.to_s
    if (face_back_mtl_name == floorplan_mtl_name)
      face_is_floorplan = true
    end
  end

  return face_is_floorplan
end
# ===== UPDATE [12/08/2015] V1.05 [END] =====

def check_material_existence (mtl_name)
  model = Sketchup.active_model
  materials = model.materials
  material = materials[mtl_name]
  if (material)
    return true
  else
    return false
  end
end

=begin credits
The text/code below this point is taken from offset.rb (http://www.smustard.com)
=end

=begin rdoc

= Offset.rb
Copyright 2004,2005,2006,2009 by Rick Wilson - All Rights Reserved

== Disclaimer
THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

== License
This software is distributed under the Smustard End User License Agreement
http://www.smustard.com/eula

== Information
Author:: Rick Wilson
Organization:: Smustard
Name:: offset.rb
Version:: 2.201
SU Version:: 4.0
Date:: 2010-10-15
Description:: Offset edges of a selected face (new method for class Sketchup::Face)

Usage::
* 1:: Intended for developers as a method to call from within a script.  Add a "require 'offset.rb'" line right after the "require 'sketchup.rb'" line.  Developers may distribute with their scripts since not everyone will have this already, but best to link to http://www.smustard.com/script/offset for the most current version.  Returns the face created by the offset, or 'nil' if no face can be created.
* 2:: ArcCurve.offset(dist) -- if dist is (+), offsets outside the curve (larger radius); if dist is (-), offsets inside the curve (smaller radius).
* 3:: Curve.offset(dist) -- if dist is (+), offsets to the right of the curve (relative to the first segment direction and plane); if dist is (-), offsets to the left of the curve.

History::
* 2.201:: 2010-10-15
  * fixed Face.offset(dist) bug that prevented some faces from being offset
* 2.200:: 2009-02-05
  * added point analysis tools and error trapping to the face.offset method
* 2.100:: 2006-06-28
  * changed the face creation to parent.entities.add_face to allow for correct creation regardless of nested status
* 2.000:: 2005-08-12
  * added offset methods for ArcCurve and Curve objects
* 1.000:: 2004-09-07
  * first version

=end

class Sketchup::Face
  def offset(dist)
begin
    pi = Math::PI
    if (not ((dist.class==Fixnum || dist.class==Float || dist.class==Length) && dist!=0))
      return nil
    end
    verts=self.outer_loop.vertices
    pts = []
    
    # CREATE ARRAY pts OF OFFSET POINTS FROM FACE
    
    0.upto(verts.length-1) do |a|
      vec1 = (verts[a].position-verts[a-(verts.length-1)].position).normalize
      vec2 = (verts[a].position-verts[a-1].position).normalize
      vec3 = (vec1+vec2).normalize
      if vec3.valid?
        ang = vec1.angle_between(vec2)/2
        ang = pi/2 if vec1.parallel?(vec2)
        vec3.length = dist/Math::sin(ang)
        t = Geom::Transformation.new(vec3)
        if pts.length > 0
          vec4 = pts.last.vector_to(verts[a].position.transform(t))
          if vec4.valid?
            unless (vec2.parallel?(vec4))
              t = Geom::Transformation.new(vec3.reverse)
            end
          end
        end
        
        pts.push(verts[a].position.transform(t))
      end
    end

    # CHECK FOR DUPLICATE POINTS IN pts ARRAY

    duplicates = []
    pts.each_index do |a|
      pts.each_index do |b|
        next if b==a
        duplicates<<b if pts[a]===pts[b]
      end
      break if a==pts.length-1
    end
    duplicates.reverse.each{|a| pts.delete(pts[a])}

    # CREATE FACE FROM POINTS IN pts ARRAY

    (pts.length > 2) ? (parent.entities.add_face(pts)) : (return nil)

rescue
  puts "#{self} did not offset: #{pts}"
  return nil
end
  end
end

class Array
  def offsetPoints(dist)
#   return nil if dist==0
    pi=Math::PI
    if (not ((dist.class==Fixnum || dist.class==Float || dist.class==Length) && dist!=0))
      return nil
    end
    verts=self
    pts=[]
    0.upto(verts.length-1) do |a|
      if verts[a-(verts.length-1)].class==Sketchup::Vertex
        pt2=verts[a-(verts.length-1)].position
      elsif verts[a-(verts.length-1)].class==Geom::Point3d
        pt2=verts[a-(verts.length-1)]
      else
        return nil
      end
      if verts[a-1].class==Sketchup::Vertex
        pt1=verts[a-1].position
      elsif verts[a-1].class==Geom::Point3d
        pt1=verts[a-1]
      else
        return nil
      end
      if verts[a].class==Sketchup::Vertex
        pt3=verts[a].position
      elsif verts[a].class==Geom::Point3d
        pt3=verts[a]
      else
        return nil
      end
      vec1=(pt3-pt2).normalize
      vec2=(pt3-pt1).normalize
      vec3=(vec1+vec2).normalize
      if vec3.valid?
        ang=vec1.angle_between(vec2)/2
        ang=pi/2 if vec1.parallel?(vec2)
        vec3.length=dist/Math::sin(ang)
        t=Geom::Transformation.new(vec3)
        if pts.length > 0
          if not (vec2.parallel?(pts.last.vector_to(verts[a].position.transform(t))))
            t=Geom::Transformation.new(vec3.reverse)
          end
        end
        pts.push(verts[a].position.transform(t))
      end
    end
    pts
  end

  def offset(dist)
#   return nil if dist==0
    pi=Math::PI
    if (not ((dist.class==Fixnum || dist.class==Float || dist.class==Length) && dist!=0))
      return nil
    end
    verts = self
    pts=[]
    0.upto(verts.length-1) do |a|
      vec1=(verts[a]-verts[a-(verts.length-1)]).normalize
      vec2=(verts[a]-verts[a-1]).normalize
      vec3=(vec1+vec2).normalize
      if vec3.valid?
        ang=vec1.angle_between(vec2)/2
        ang=pi/2 if vec1.parallel?(vec2)
        vec3.length=dist/Math::sin(ang)
        t=Geom::Transformation.new(vec3)
        if pts.length > 0
          if not (vec2.parallel?(pts.last.vector_to(verts[a].transform(t))))
            t=Geom::Transformation.new(vec3.reverse)
          end
        end
        pts.push(verts[a].transform(t))
      end
    end
    return pts
  end
end

class Sketchup::ArcCurve
  def offset(dist)
    return nil if dist==0 || (not (dist.class==Float || dist.class==Fixnum || dist.class==Length))
    radius=self.radius+dist.to_f
    #Sketchup.active_model.active_entities.add_arc self.center, self.xaxis, self.normal, radius, self.start_angle, self.end_angle, self.count_edges
    c=parent.entities.add_arc self.center, self.xaxis, self.normal, radius, self.start_angle, self.end_angle, self.count_edges
    c.first.curve
  end
end

class Sketchup::Curve
  def offset(dist)
    return nil if self.count_edges<2
    pi=Math::PI
    if (not ((dist.class==Fixnum || dist.class==Float || dist.class==Length) && dist!=0))
      return nil
    end
    verts=self.vertices
    pts=[]
    0.upto(verts.length-1) do |a|
      if a==0 #special case for start vertex
        model=self.model
        model.start_operation "offset"
        gp=model.active_entities.add_group
        gpents=gp.entities
        face=gpents.add_face(verts[0].position,verts[1].position,verts[2].position)
        zaxis=face.normal
        v=self.edges[0].line[1]
        f=dist/dist.abs
        t=Geom::Transformation.rotation(verts[0].position,zaxis,(pi/2)*f)
        vec3=v.transform(t)
        vec3.length=dist.abs
        pts.push(verts[0].position.transform(vec3))
        gp.erase!
        model.commit_operation
      elsif a==(verts.length-1) #special case for end vertex
        model=self.model
        model.start_operation "offset"
        gp=model.active_entities.add_group
        gpents=gp.entities
        face=gpents.add_face(verts[a].position,verts[a-1].position,verts[a-2].position)
        zaxis=face.normal
        v=self.edges[a-1].line[1]
        f=dist/dist.abs
        t=Geom::Transformation.rotation(verts[a].position,zaxis,(pi/2)*f)
        vec3=v.transform(t)
        vec3.length=dist.abs
        pts.push(verts[a].position.transform(vec3))
        gp.erase!
        model.commit_operation
      else
        vec1=(verts[a].position-verts[a-(verts.length-1)].position).normalize
        vec2=(verts[a].position-verts[a-1].position).normalize
        vec3=(vec1+vec2).normalize
        if vec3.valid?
          ang=vec1.angle_between(vec2)/2
          ang=pi/2 if vec1.parallel?(vec2)
          vec3.length=dist/Math::sin(ang)
          t=Geom::Transformation.new(vec3)
          if pts.length > 0
            if not (vec2.parallel?(pts.last.vector_to(verts[a].position.transform(t))))
              t=Geom::Transformation.new(vec3.reverse)
            end
          end
          pts.push(verts[a].position.transform(t))
        end
      end
    end
    #Sketchup.active_model.active_entities.add_curve(pts)
    c=parent.entities.add_curve(pts)
    c.first.curve
  end
end