# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolTrainDrive
  #This tool is used to drive train and select train to drive
  
  #List active train for each model so it's still selected when re-activating tool
  #This array is used to set the @train variable on initialization
  @@active_train_for_models ||= []

  #Follow with camera
  @@follow = false
  
  #Use smooth accelerations (user sets train.v_target instead of train.v)
  @@smooth_a = false
  
  #Class variable accessors

  def self.active_train_for_models; @@active_train_for_models; end
  def self.active_train_for_models=(value); @@active_train_for_models = value; end
  
  #Sketchup tool definitions

  def initialize(train = nil)
    @train = train

    #Retrieve active train from last time tool was active in this model, or select first one
    #Can't just be a simple class variable since it's model specific
    #If no train is set, get last one driven in this model
    unless @train
      @@active_train_for_models.each do |train_model_pair|
        next unless train_model_pair[:model] == Sketchup.active_model
        @train = train_model_pair[:train]
        break
      end
    end
    #Unlike track tools (positioning, switch) selection isn't used
    #If train is still not set, get the first for this model (or nil if no trains in model)
    unless @train
      @train = Train.instances.find { |i| i.model == Sketchup.active_model }
    end
    #Create object holding last driven train in this mode, if it doesn't already exist
    #Reference to train is set when tool is deactivated
    unless @@active_train_for_models.any? { |i| i[:model] == Sketchup.active_model}
      @@active_train_for_models << { :model => Sketchup.active_model }
    end
    
    #Follow train with camera if @follow was set to true last time tool was active and there is a train selected
    if @@follow && @train
      ani = Animate.get_from_model
      ani.followed = @train.r_stocks[0]
      self.focus_camera
    end
    
    #Mode is by default select.
    #Select mode is ued to click on a train to drive it.
    #set_destination mode is used to set destination to drive selected train to.
    @mode = "select"
    
    #Path to draw in when looking for destination
    @path = []
    
    #Error message to display when selected train doesn't have tracks for all rolling stocks.
    @error_no_tracks = S.tr "This train doesn't stand on tracks. Add tracks under it or move it onto tracks to be able to drive it."

    #Statusbar texts
    @status_text_select = S.tr "Click on a train to drive it."
    @status_text_set_destination = S.tr "Click to set destination. Esc = Back to select train."
    @status_text = @status_text_select
    
    @cursor_select = UI.create_cursor(File.join(CURSOR_DIR, "train_drive.png"), 2, 2)
    @cursor_destination = UI.create_cursor(File.join(CURSOR_DIR, "train_drive_destination.png"), 13, 23)

  end

  def onSetCursor

    UI.set_cursor(@mode == "select" ? @cursor_select :  @cursor_destination)

  end

  def activate

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper
    
    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    #Prevents select tool from being selected every time this tool is deactivated. This should only happen when dialog is manually closed
    @tool_active = true

    #Mouse position and view from onMouseMove
    #Used in getMenu to check what entity was clicked
    @x = @y = nil
    
    #Open web dialog with tool settings
    self.init_web_dialog

    #Statusbar texts
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    
    Sketchup.active_model.active_view.invalidate
    
    #Notify user if train isn't on tracks
    UI.messagebox(@error_no_tracks) if @train && !@train.on_track?

  end

  def deactivate(view)

    @tool_active = false
    @dialog.close
    view.invalidate

    #Stop following train, individual rolling stocks can still be followed though
    if @@follow
      ani = Animate.get_from_model(Sketchup.active_model)
      ani.followed = nil
    end
    
    #Keep selected train when tool is re-activated
    #active_train_for_models contains train - model pairs, and the one for this model is now updated
    @@active_train_for_models.find { |t| t[:model] == Sketchup.active_model } [:train] = @train

  end

  def draw(view)
    #Print in screen corner if animation is paused
    
    color = Sketchup::Color.new("Red")
    color_border = Sketchup::Color.new("Black")
    color_shadow = Sketchup::Color.new("Gray")
    color_shadow.alpha = 127
    
    unless Animate.get_from_model(view.model).go
      
      view.drawing_color = color_shadow
      view.draw2d GL_QUADS, [8, 8, 0], [18, 8, 0], [18, 38, 0], [8, 38, 0], [28, 8, 0], [38, 8, 0], [38, 38, 0], [28, 38, 0]
      
      view.drawing_color = color
      view.draw2d GL_QUADS, [5, 5, 0], [15, 5, 0], [15, 35, 0], [5, 35, 0], [25, 5, 0], [35, 5, 0], [35, 35, 0], [25, 35, 0]
      
      view.drawing_color = color_border
      view.draw2d GL_LINE_LOOP, [5, 5, 0], [15, 5, 0], [15, 35, 0], [5, 35, 0]
      view.draw2d GL_LINE_LOOP, [25, 5, 0], [35, 5, 0], [35, 35, 0], [25, 35, 0]
      
      view.draw_text([4, 40, 0], S.tr("PAUSED"))
      
    end
    
    #Draw arrow in set destination mode
    if @mode == "set_destination" && !@path.empty?
      view.line_width = 1
      view.drawing_color = Sketchup::Color.new("White")
      MyView.draw_polyline_on_top view, @path
      EneRailroad.MyView view, @path[-1], @path[-1] - @path[-2], 1.m
    end
    
  end
  
  def onMouseMove(flags, x, y, view)
  
    #Save parameters so they can be used in getMenu
    @x = x
    @y = y
    
    if @mode == "set_destination"
      #Get route between train and destination
      @ip.pick view, x, y
      p = @ip.position
      @path = @train.go_to p, true
      view.invalidate
    end

  end
  
  def onLButtonDown(flags, x, y, view)
    #Drive clicked train
    
    
    ani = Animate.get_from_model(Sketchup.active_model)
    
    #Play/pause when clicking upper left corner
    if x < 40 && y < 50
      ani.go = !ani.go
      view.invalidate
      return
    end
    
    if @mode == "select"
      #Select train to drive

      @ph.do_pick(x, y)
      picked = @ph.best_picked

      if picked && RStock.group_is_r_stock?(picked)
        rs = RStock.get_from_group picked
        @train = Train.get_from_r_stock rs

        #Change selected train
        self.update_train_selector
        js = "document.getElementById('autopilot_mode').value='#{@train.autopilot_mode.to_s}';"
        js << "document.getElementById('v').value='#{self.v.to_s}';"
        js << "slider_update_value(0);"
        js << "document.getElementById('reverse_train').disabled = (document.getElementById('v').value != 0)"
        @dialog.execute_script js

        #Follow
        if @@follow
          ani.followed = @train.r_stocks[0]
          self.focus_camera
        end
      
        #Notify user if train isn't on tracks
        UI.messagebox(@error_no_tracks) unless @train.on_track?
      end
      
    elsif @mode == "set_destination"
      #Set destination to drive to
      @ip.pick view, x, y
      p = @ip.position
      @train.go_to p
      @path = []#Stop path from being drawn until mouse moves and a new one is calculated
    end

  end

  def getMenu(menu)
    #Menu with option to follow rolling stock
  
    #Get clicked element
    @ph.do_pick(@x, @y)#NOTE: SU BUG: coordinates doesn't update if right clicking while menu is opened!
    entity = @ph.best_picked
    rs = RStock.get_from_group(entity)
    
    #Animation object
    ani = Animate.get_from_model(Sketchup.active_model)
    
    if rs
      #A rolling stock was clicked
      
      #Check if this rolling stock is followed already
      #If a train is followed this is always false.
      #Even if the rolling stock happens to be the one followed it would change if train changes or is reversed.
      #Only true if rolling stock itself has been right clicked and followed
      rs_selected = ani.followed == rs && !@@follow
      
      item = menu.add_item(S.tr("Follow Rolling Stock With Camera")) {
        if rs_selected
          #Stop following
          ani.followed = nil
        else
          #Follow rolling stock
          ani.followed = rs
          #Train is no longer followed
          @@follow = false
          @dialog.execute_script "document.getElementById('follow').checked=false;"
        end
      }
      menu.set_validation_proc(item) {
        rs_selected ? MF_CHECKED : MF_ENABLED
      }
      
    else
      #No rolling stock was clicked
    
      unless @follow
        menu.add_item(S.tr("Stop Following with Camera")) {
          ani.followed = nil
          @@follow = false
          @dialog.execute_script "document.getElementById('follow').checked=false;"
        }
      end
    end
    
  end
  
  def resume(view)

    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate

  end

  def onCancel(reason, view)
  
    if @mode == "set_destination"
      self.stop_set_destination
    end
  end
  
  #Own definitions
  #Not called from Sketchup itself
  
  def focus_camera
    #Moves the camera so the first rolling stock of the driven train is visible,
    #if not already visible.
    #Called if @@follow when clicking a train, changing train in dropdown, start following a train and when reverse changes
    
    view = Sketchup.active_model.active_view
    
    rs = @train.r_stocks[0]
    group = rs.group
    bb = group.bounds
    point_center = bb.center
    
    point_center_screen = view.screen_coords point_center
    width = view.vpwidth
    height = view.vpheight
    
    if !point_center_screen[0].between?(0, width) || !point_center_screen[1].between?(0, height)
      #Rolling stock center is outside screen, move camera
      
      #camera = view.camera
      #vector = point_center - (view.camera.target)
      #camera.set camera.eye.offset(vector), camera.target.offset(vector), camera.up
      
      #No need to reinvent the wheel, it did exist but was hidden in the docs
      view.zoom rs.group
      
    end
  
  end

  def init_web_dialog
    #Open web dialog with tool settings

    @dialog = UI::WebDialog.new(S.tr("Drive Train"), false, "#{ID}_drive_train", 440, 170, 500, 100, true)
    @dialog.min_width = 440
    @dialog.min_height = 170
    @dialog.navigation_buttons_enabled = false
    @dialog.set_background_color @dialog.get_default_dialog_color
    @dialog.set_file(File.join(DIALOG_DIR, "drive_train", "index.html"))

    #Translate strings
    js = S.tr_dlg
    
    if @train
      #Initialize form

      js << "document.getElementById('no_trains_notifier').style.display='none';"
      js << "document.getElementById('autopilot_mode').value='#{@train.autopilot_mode.to_s}';"
      js << "document.getElementById('follow').checked=#{@@follow.to_s};"
      js << "document.getElementById('smooth_a').checked=#{@@smooth_a.to_s};"
      js << "document.getElementById('v').value='#{self.v.to_s}';"
      #js << "init_checkboxes();"#Checkboxes have their own callbacks.
      js << "init_sliders();"
      #js << "push_changes();"#Sending form data back on form load overrides v_target with the form's (train's) v.
      js << "form = document.forms[0];"
      js << "form.onclick = push_changes;"
      js << "form.onkeyup = push_changes;"

    else
      #Notify user that there are no trains in model

      js << "document.getElementById('trains_content').style.display='none';"

    end

    #Show dialog
    if WIN
      @dialog.show { self.update_train_selector if @train; @dialog.execute_script js }
    else
      @dialog.show_modal { self.update_train_selector if @train; @dialog.execute_script js }
    end
    
    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox

    #Form value changes
    @dialog.add_action_callback("push_changes") {

      #If train changed, update form content
      train = Train.instances[@dialog.get_element_value("train").to_i]
      unless train == @train

        @train = train
        js = "document.getElementById('autopilot_mode').value='#{@train.autopilot_mode.to_s}';"
        js << "document.getElementById('v').value='#{self.v.to_s}';"
        js << "slider_update_value(0);"
        js << "document.getElementById('reverse_train').disabled = (document.getElementById('v').value != 0)"
        @dialog.execute_script js

        #Follow first rolling stock if follow is true
        if @@follow
          ani = Animate.get_from_model(Sketchup.active_model)
          ani.followed = @train.r_stocks[0]
          self.focus_camera
        end
        
        #Notify user if train isn't on tracks
        UI.messagebox(@error_no_tracks) unless @train.on_track?

      end

      #validate and set velocity
      v = @dialog.get_element_value("v")
      v.gsub! ",", "."
      v = v.to_f      
      self.v = v#use wrapper method that either sets train.v or train.v_target depending on smooth acceleration.

    }
    
    #Start or stop following train
    @dialog.add_action_callback("follow_change") { |_, follow|
      @@follow = follow == "true"
      ani = Animate.get_from_model(Sketchup.active_model)
      ani.followed = @@follow ? @train.r_stocks[0] : nil
      self.focus_camera if @@follow
    }
    
    #Focus camera on demand
    @dialog.add_action_callback("focus_camera") {
      self.focus_camera
    }
    
    #Toggle smooth acceleration
    @dialog.add_action_callback("smooth_a") { |_, smooth_a|
      @@smooth_a = smooth_a == "true"
    }
    
    #Set acceleration
    @dialog.add_action_callback("set_a") {
      input = UI.inputbox(["a/(m/s^2)"], [@train.a.to_s], S.tr("Set Acceleration"))
      next unless input
      a = input[0]
      a.gsub! ",", "."
      a = a.to_f
      @train.a = a
    }

    #Rename train
    @dialog.add_action_callback("rename_train") {
      input = UI.inputbox([S.tr("Name")], [@train.name], S.tr("Rename Train"))
      next unless input
      name = input[0]
      next if name == ""
      @train.name = name
      self.update_train_selector
    }
    
    #Change auto pilot mode
    @dialog.add_action_callback("autopilot_mode_change") {
      @train.autopilot_mode = @dialog.get_element_value("autopilot_mode")
      @train.autopilot_mode = nil if @train.autopilot_mode.empty?
      @train.autopilot_mode == "guided" ? self.set_destination : self.stop_set_destination
    }
    @dialog.add_action_callback("set_target") {
      self.set_destination
    }

    #Reverse train
    @dialog.add_action_callback("reverse_train") {
      #NOTE: only enable button when at actual standstill, not when slider is at 0. With some sort of callback from animation/train class when train has stopped it's also possible to update drop down to show what trains are currently moving.
      @train.v = 0
      @train.reverse
      
      #Follow first rolling stock if follow is true
      if @@follow
        ani = Animate.get_from_model(Sketchup.active_model)
        ani.followed = @train.r_stocks[0]
        self.focus_camera
      end
    }

    #Deselect tool when webdialog closes
    @dialog.set_on_close {
      Sketchup.send_action "selectSelectionTool:" if @tool_active
    }

    #Button switching to insert rolling stock tool
    @dialog.add_action_callback("select_insert_rs_tool") {
      #NOTE: SU BUG: whatever I do here drive train tool is highlighted in toolbar, not rolling stock insert tool
      #Sketchup.send_action "selectSelectionTool:"
      #Sketchup.active_model.tools.pop_tool
      #UI.start_timer(1, false) { Sketchup.active_model.select_tool ToolRStockInsert.new }
      Sketchup.active_model.select_tool ToolRStockInsert.new
    }

    #Open link in default browser
    @dialog.add_action_callback("open_in_default_browser") { |_, callbacks|
      UI.openURL callbacks
    }

  end

  def update_train_selector
    #Updates the train selector in the form.
    #Called on when webdialog is created, when train is renamed and when form changes (to update move state).
    html = "<select id=\"train\" style=\"width: 100%;\">"#Since internet explorer is the work of the devil innerHTML doesn't work on select element itself

    Train.instances.sort_by { |t| t.name } .each do |train|
      next unless train.model == Sketchup.active_model
      train_index = Train.instances.index(train)
      selected = train == @train ? " selected=\"selected\"" : ""
      html << "<option value=\"#{train_index}\"#{selected}>#{train.name}</option>"#NOTE: feature: show in dropdown what trains move. this requires the method to be called each time a trains move state changes.
    end#each
    html << "</select>"

    @dialog.execute_script "document.getElementById('train_selecter_wrapper').innerHTML='#{html}';"

  end

  def v
    #Get either v or target_v for train depending on smooth_a.
    #This class uses this method instead of directly looking for @train.v/@train.v_target to avoid unnecessary if statements that are hard to maintain.
    
    @@smooth_a ? @train.v_target : @train.v
    
  end
  
  def v=(v)
    #Set either v or target_v for train depending on smooth_a.
    #This class uses this method instead of directly setting @train.v/@train.v_target to avoid unnecessary if statements that are hard to maintain.
    
    if @@smooth_a
      @train.v_target = v
    else 
      @train.v = v
    end
    
    v
    
  end
  
  def set_destination
    #Changes tool mode to selecting point to go to.
    
    @mode = "set_destination"
    js = "document.getElementById('autopilot_mode').value='guided';"
    @dialog.execute_script js
    @status_text = @status_text_set_destination
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.active_model.active_view.invalidate
    
  end
  
  def stop_set_destination
    #Changes back tool mode to clicking at train to drive it.
    
    @mode = "select"
    @status_text = @status_text_select
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.active_model.active_view.invalidate
    
  end
  
end#class

end#module
