# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolTrackSwitch
  #Used to change what track is used in a switch
  
  #Sketchup tool definitions

  def initialize(t = nil)

    #Track object selected, nil if none is selected
    @t = t

    #Switch state
    @switch_state = [0, 0]

    #Statusbar texts
    @status_text_select_switch = S.tr "Select a switch (dotted) to change."
    @status_text_set_switch = S.tr "Click on neighboring tracks to switch towards them. Enter = Save."

    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "track_switch.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def activate

    #Initialize pick helper to select switch track and available tracks leading from it
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to select switch track and available tracks leading from it
    @ip = Sketchup::InputPoint.new

    #Use track that was selected when tool was activated, unless a track was given as an argument
    unless @t
      @t = Track.get_from_selection
      @t = nil unless @t && @t.is_switch?
    end

    #Load current switch state from track
    if @t
      @switch_state = @t.switch_state.dup
    end

    #Reset statusbar text
    @status_text = @t ? @status_text_set_switch : @status_text_select_switch
    Sketchup::set_status_text(@status_text, SB_PROMPT)

    Sketchup.active_model.active_view.invalidate

  end

  def deactivate(view)

    view.invalidate

    #Save new switch state to track object
    self.save_track if @t

    view.invalidate

  end

  def draw(view)

    color_selected = Sketchup::Color.new("Lime")
    color_not_selected = Sketchup::Color.new("Red")
    view.drawing_color = color_selected
    view.line_width = 3

    #Highlight all switches if none is selected
    unless @t
      all_switches = Track.instances.select { |i| i.is_switch? && i.model == view.model}
      all_switches.each do |i|
      
        view.line_stipple = ""
        
        0.upto(1) do |end_of_switch|
          if i.connections[end_of_switch].length > 1
            point = i.controls[end_of_switch]

            #Switch point
            view.draw_points [point], 8, 2
            view.draw_points [point], 7, 2, color_selected
            
            #Track switch is set towards
            #only draw the half closest to the switch
            used_track = i.connections[end_of_switch][i.switch_state[end_of_switch]]
            path = used_track.path.dup
            path.reverse! if used_track.controls[1] == i.controls[end_of_switch]
            if path.length > 2
              path = path.first(path.length/2)
            else
              path = [path[0], Geom.linear_combination(0.5, path[0], 0.5, path[1])]
            end#if
            view.draw_polyline path
            
            #Arrow head
            MyView.draw_arrow view, path[-1], path[-1] - path[-2], 0.25.m

          end#if
        end#each
        
        #Outline switch track itself
        view.line_stipple = "_"
        view.draw_polyline i.path
        
      end#each
    end

    if @t

      #Highlight selected switch if any is selected
      0.upto(1) do |end_of_switch|
        if @t.connections[end_of_switch].length > 1
          point = @t.controls[end_of_switch]

          view.draw_points [point], 8, 2
          view.draw_points [point], 7, 2, color_selected

        end
      end
      view.draw_polyline @t.path

      #Highlight available tracks to chose from if any track is selected
      0.upto(1) do |end_of_switch|
        if @t.connections[end_of_switch].length > 1
          #Track has multiple connections at this end
          switch_point = @t.controls[end_of_switch]
          used_track = @t.connections[end_of_switch][@switch_state[end_of_switch]]
          @t.connections[end_of_switch].each do |i|
            other_point = i.other_point switch_point
            drawing_color = (used_track == i) ? color_selected : color_not_selected
            view.drawing_color = drawing_color
            view.draw_points [other_point], 8, 2
            view.draw_points [other_point], 7, 2, drawing_color
            view.draw_polyline i.path
          end
        end
      end
    end

  end

  def onReturn(view)

    view.invalidate

    #Save new switch state to track object
    self.save_track if @t

    #Reset statusbar text
    @status_text = @status_text_select_switch
    Sketchup.set_status_text(@status_text, SB_PROMPT)

    @t = nil
    @switch_state[0, 0]

  end

  def onCancel(flag, view)

    view.invalidate

    #Reset statusbar text
    @status_text = @status_text_select_switch
    Sketchup.set_status_text(@status_text, SB_PROMPT)

    @t = nil
    @switch_state[0, 0]

  end

  def onMouseMove(flags, x, y, view)

    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y
    view.model.selection.clear

    if !@t
      #If no switch is selected, temporarily select the hovered in model

      if Track.group_is_track? picked
        t = Track.get_from_group picked
        if t && t.is_switch?
          view.model.selection.add picked
        end#if
      end#if
    else
      #If a switch is selected, temporarily select the hovered connected track

      view.model.selection.add @t.group
      if Track.group_is_track?(picked) && picked != @t.group
        #c_t = Track.get_from_group(picked)
        connected = @t.connections.flatten
        c_t = Track.inspect_point(@ip.position, nil, connected)[:track]#Selects the track with the path closest to the click (among those connected), not the group clicked as it was previosly.
        if c_t
          0.upto(1) do |end_of_switch|
            0.upto(1) do |end_of_connected_track|
              next unless @t.connections[end_of_switch].length > 1
              next if c_t == @t
              point_on_switch = @t.controls[end_of_switch]
              point_on_connected = c_t.controls[end_of_connected_track]
              next unless point_on_switch == point_on_connected
              #view.model.selection.add picked
              view.model.selection.add c_t.group#clicked (or rather hovered) track is not selected, but the one with the path closest to the point clicked
            end
          end
        end
      end

    end

  end

  def onLButtonDown(flags, x, y, view)

    @ph.do_pick(x, y)
    picked = @ph.best_picked

    if !@t
      #use clicked track as switch if no switch is selected

      if Track.group_is_track? picked
        t = Track.get_from_group(picked)
        if t && t.is_switch?
          @t = t
          @switch_state = @t.switch_state.dup
          view.invalidate
          @status_text = @status_text_set_switch
          Sketchup.set_status_text(@status_text, SB_PROMPT)
        end
      end
    else
      #Set switch state to track closest to click if switch is selected and track closest to click is an available option

      if Track.group_is_track?(picked) && picked != @t.group
        #c_t = Track.get_from_group(picked)
        connected = @t.connections.flatten
        c_t = Track.inspect_point(@ip.position, nil, connected)[:track]#Selects the track with the path closest to the click (among those connected), not the group clicked
        if c_t
          0.upto(1) do |end_of_switch|
            0.upto(1) do |end_of_connected_track|
              point_on_switch = @t.controls[end_of_switch]
              point_on_connected = c_t.controls[end_of_connected_track]
              next unless point_on_switch == point_on_connected
              @switch_state[end_of_switch] = @t.connections[end_of_switch].index c_t
            end
          end
          view.invalidate
        end
      end#if
      
    end

  end

  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    view.invalidate
    Sketchup.set_status_text(@status_text, SB_PROMPT)
  end
  
  #Own definitions
  #Not called from Sketchup itself
  
  def save_track
    #Save switch state to track and redraw its switch indicators
    
    @t.switch_state = @switch_state.dup

    @t.model.start_operation(S.tr("Set Switch State"), true)
    @t.save_attributes
    @t.draw_switch_indicators
    @t.model.commit_operation

  end

end#class

end#module
