# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolTrackInsert
  #This tool is used to add one or more new tracks to the model

  #Variables for calculating track
  @@height_offset     = JPOD ? 8.m : 0.m
  @@center_distance   = JPOD ? 3.5.m : 4.5.m
  @@angle             = 15.degrees
  @@formation         = "straight"# Selected in calc mode.
  @@track_formation   = nil# Selected in lib mode. String identifier.
  @@type_of_track     = JPOD ? "JPod" : "default"
  @@type_of_signals   = JPOD ? false : "default"
  @@tracks_parallel   = JPOD ? 2 : 1
  @@radius            = JPOD ? 20.m : 200.m
  @@k                 = JPOD ? (1.0/200.0) : (1.0/5400.0)#Change in curvature/length [m^-2]
  @@structure_type    = JPOD ? "JPod_support_solar_double" : nil
  @@use_structure_data = true
  @@segment_reolution  = 0.25.to_m#[segments/m]
  
  #Getting k for given max centripetal acceleration and velocity
  #(assuming transition curve is an Euler spiral which it isn't)
  
  #curvature = 1/R
  #a = v^2/R = v^2*curvature
  #t = s/v
  #k = curvature/s
  #J = ac/t = (v^2*curvature)/(s/v) = v^3*curvature/s = v^3*k
  
  #k = J/v^3
  #v = 30 m/s, J = 0,5 m/s^3 => k = 1/5400 m^-2
  #v = 50 m/s, J = 0,5 m/s^3 => k = 1/250000 m^-2
  
  #Sketchup tool definitions

  def initialize

    #Only keep tracks in selection
    ss = Sketchup.active_model.selection
    ss.each do |i|
      ss.toggle i unless Track.group_is_track? i
    end

    #Array of 'virtual' tracks insert to model and to display paths of.
    #These are just hashes with the track properties, not track objects.
    @tracks = []
    
    #Point and vector from mouse cursor. snaps to existing tracks
    @point = Geom::Point3d.new
    @vector = Geom::Vector3d.new(10.m, 0, 0)

    #Array of what to base @tracks of.
    #These are too hashes and not track objects.
    #In cont mode this array isn't used, instead tracks are directly added to @tracks based on @nodes.
    #In calc, copy and lib mode these are the tracks to insert, but transformed so their origin is what's following the mouse cursor
    #In offset and connect mode these are the original tracks to base new tracks on (offset sorts the track in this array though).
    @tracks_primitive = []
    
    #Points to connect with track in cont mode.
    @nodes = []
    
    #If first node is at an existing track second node is locked to track end vector.
    @node1_locked_line = nil
    
    #Last existing node can be altered to allow node after it to be placed on existing track.
    #When stop hovering a track this copy replaces the adjusted last node.
    @last_node_copy

    #What user is doing
    #Mode is either cont(continuous), calc (getting switches etc from center distance & angle), lib (predefined from library), copy, offset or connect
    @mode =  if ss.empty?
        "cont"
      elsif ss.length == 2 && !Track.get_from_group(ss[0]).connections.flatten.include?(Track.get_from_group(ss[1]))
        #There are 2 unconnected tracks selected
        "connect"
      else
        "copy"
      end

    #Index of the end of @tracks_primitive is being placed, used for calc, copy and lib mode
    @track_end_index = 0

    #Distance tracks are being offset
    @offset_distance = 0

    #Indexes of ends in selected tracks to connect
    @track_end_indexes = [1, 0]

    #Input point
    @ip = nil

    #Only affects look of tool, not drawn results
    @default_vector_length = 10.m

    #Statusbar texts
    @status_text_cont = S.tr "Click to add a node. Esc = Remove last, Enter = Insert tracks."
    @status_text_place = S.tr "Click to insert track. Tab = Toggle track end, Arrow keys = Along axis."#place is used both when calculating new track(s) and copying
    @status_text_offset = S.tr "Click or press enter to insert tracks."
    @vcb_label_cont = S.tr "Distance"
    @vcb_label_offset = S.tr "Distance"
    @status_text_connect = S.tr "Click anywhere or press enter to insert track. Tab = Toggle ends to connect."

    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "track_insert.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def enableVCB?
  
    true #@mode == "offset"#NOTE: SU ISSUE: only for offset mode. this behavior is only checked by SU after mouse click, not when things changes in web dialog :/.
 
  end

  def activate

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    #Prevents select tool from being selected every time this tool is deactivated. This should only happen when dialog is manually closed
    @tool_active = true

    #Open web dialog with tool settings
    self.init_web_dialog

  end

  def deactivate(view)

    @tool_active = false
    @dlg.close
    view.invalidate

  end

  def draw(view)
  
    tooltip = nil

    color = Sketchup::Color.new(162, 162, 255)
    
    view.line_stipple = "_"
        
    #Connect nodes in cont mode
    if @mode == "cont"
      if @nodes.length > 1
        view.draw_polyline @nodes
      end
    end
    
    #If a height offset is used, draw line between actual ip and point
    if @@height_offset != 0 && @point.on_line?([@ip.position, Z_AXIS])
      view.draw_line @ip.position, @point
    end
    
    view.line_stipple = ""
    
    points = []

    #Track paths
    view.line_width = 1
    view.drawing_color = Sketchup::Color.new("Black")
    @tracks.each_with_index do |t, i|
      controls = t[:controls]
      if i != 0 && 
        controls_prev = @tracks[i-1][:controls]
        if @mode == "cont" && controls_prev[1] == controls[0] && !controls_prev[3].reverse.samedirection?(controls[2])
          #Tracks not connected properly. Probably caused by curves in cont mode being too close to each other.
          # Only check this in continuous mode!
          view.drawing_color = Sketchup::Color.new("Red")
          tooltip = S.tr "Nodes are too close for given radius."
          view.line_stipple = "-"
        end
      end
      path = MyGeom.calc_path controls, t[:curve_algorithm]
      view.draw_polyline path
      #Remember control points so they can be drawn
      points += controls[0..1]
    end#each
    
    view.drawing_color = Sketchup::Color.new("Black")
    view.line_stipple = ""

    #Track end points
    points.uniq!
    points.delete @point if  ["calc","lib","copy"].include?(@mdoe)
    view.draw_points points, 7, 4, Sketchup::Color.new("Black") unless points.empty?

    if @mode == "cont"
      unless @nodes.empty?
        #Line to cursor, shows where next straight will be (minus the curve).
        view.drawing_color = Sketchup::Color.new(128, 128, 128)
        view.draw_line @nodes[-1], @point
                
        #Nodes
        view.draw_points @nodes, 8, 2, color#NOTE: AESTETIC: use smaller points when not moved in position tools for consistency
      end
      
      #Node being placed
      view.draw_points [@point], 10, 2, color
      
    elsif ["calc", "lib", "copy"].include? @mode
      #Point where track is being placed
      view.draw_points [@point], 10, 2, color

      #Vector line
      view.line_width = 3
      view.drawing_color = color
      view.draw_line @point, @point.offset(@vector)
    end

    #Draw input point
    @ip.draw view
    
    #Draw tooltip
    view.tooltip = tooltip if tooltip

  end

  def getExtents
    #Tell Sketchup what bounding box stuff should be drawn to.
    #Not as relevant in other tools since they don't draw outside current model boundaries.
  
    bb = Geom::BoundingBox.new
    @tracks.each { |t| bb.add(t[:controls][0..1]) }
    
    bb
    
 end
 
  def onReturn(view)

    #Insert tracks
    self.insert_tracks

  end
 
  def onCancel(flag, view)

    if @mode == "cont"
      #Remove last node
      unless @nodes.empty?
        @nodes.pop
        @last_node_copy = @nodes[-1]
        self.update_tracks_cont
        view.invalidate
      end
      
      #Update VCB value
      if @nodes.empty?
        Sketchup.set_status_text "", SB_VCB_VALUE
      else
        d = @point.distance @nodes[-1]
        Sketchup.set_status_text d.to_s, SB_VCB_VALUE
      end
    end
    
  end
  
  def onMouseMove(flags, x, y, view)
    #Set control points and vectors from hovered track if any

    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y

    #Get point and vector from mouse location
    t_h = Track.get_from_group(picked)
    if t_h
      #Align to existing track
      if t_h.controls[0].distance(@ip.position) < t_h.controls[1].distance(@ip.position)
        #Cursor is closest to start
        @point = t_h.controls[0]
        @vector = t_h.controls[2].reverse
      else
        #Cursor is closest to end
        @point = t_h.controls[1]
        @vector = t_h.controls[3].reverse
      end
    else
      #Start at cursor
      @point = @ip.position
      @point.offset! Z_AXIS, @@height_offset
    end
    @vector.length = @default_vector_length

    if @mode == "cont"
      #Make continuous railroad line between nodes.
      #If first node is on an existing track second node has to be on that track's line.
    
      #If this is the 1st (second) node, lock to vector if it's defined.
      if @nodes.length == 1 && @node1_locked_line
        @point = @point.project_to_line @node1_locked_line
      end
      
      #If this is 1st node or later and on track, adjust previous node to track's line
      if @nodes.length > 1 && @last_node_copy
        olf_last = @nodes[-1]
        new_last = @nodes[-1] = 
        if t_h
          plane = [@nodes[-2].clone, Z_AXIS*(@nodes[-2]-@nodes[-1])]
          line = [@point, @vector]
          Geom.intersect_line_plane line, plane
        else
          @last_node_copy
        end
        #Calculate tracks again if track hovering started or ended.
        self.update_tracks_cont if olf_last != new_last
      end
      #NOTE: BUG: MINOR: when last node connected to a track and that track is hovered again there's a 0 length vector between the points (same point).
      
      unless @nodes.empty?
        d = @point.distance @nodes[-1]
        Sketchup.set_status_text d.to_s, SB_VCB_VALUE
      end
      
    elsif ["calc", "lib", "copy"].include? @mode
      #Placing predefined tracks
      #Transform @tracks_primitive into @tracks depending on input point

      #Transform @tracks_primitive to cursor location and save to @tracks
      self.update_tracks_place

    elsif @mode == "offset"
      #Offset tracks
      #Get @offset_distance and offset

      #Get end point of selected tracks closest to input point
      #Get distance
      closest_point = nil
      smallest_distance = nil
      end_vector = nil
      @tracks_primitive.each do |i|
        0.upto(1) do |j|
          point = i[:controls][j]
          distance = @point.distance point
          next if smallest_distance && distance > smallest_distance
          smallest_distance = distance
          closest_point = point
          end_vector = i[:controls][j+2].clone
          end_vector.reverse! if j == 1
        end#each
      end#each

      #Project input point to line perpendicular to tracks
      line_perp = [closest_point, end_vector.cross(Z_AXIS)]
      @point = @point.project_to_line line_perp
      @offset_distance = @point.distance closest_point

      #Positive or negative?
      if (MyGeom.flatten_vector(end_vector).cross(closest_point-(@point))).z < 0
        @offset_distance*= -1
      end

      #Write current distance to VCB
      Sketchup.set_status_text(@offset_distance.abs.to_l.to_s, SB_VCB_VALUE)

      self.update_tracks_offset

    end

    view.invalidate

  end

  def onLButtonDown(flags, x, y, view)

    if @mode == "cont"
      #Add node
      picked = @ph.best_picked
      t_h = Track.get_from_group(picked)
                  
      #If this is the 0th node being place, lock or unlock vector 1st node is forced to depending on this node being on track end or not.
      if @nodes.empty?
        @node1_locked_line = t_h ? [@point.clone, @vector.clone] : nil
      end
      
      #If clicking again at the same point, insert tracks.
      #Especially useful when connecting to existing tracks.
      if @nodes[-1] == @point
        self.insert_tracks
        return      
      end
      
      @nodes << @point
      @last_node_copy = t_h ? nil : @point
      
      self.update_tracks_cont
      view.invalidate
      
    else
      #Insert tracks
      self.insert_tracks
    end

  end
  
  def onLButtonDoubleClick(flags, x, y, view)

    if @mode == "cont"
      #Insert tracks (node for clicked point is added in onLButtonDown first)
      self.insert_tracks
    end
    
  end

  def onUserText(text, view)

    if @mode == "cont"
      #Continuous, set length to next node
      
      unless @nodes.empty?
        begin
          distance = text.to_l
        rescue
          UI.messagebox(S.tr("Invalid length."))
          return
        end
        if @point == @nodes[-1]
          #Move last point if cursor hasn't moved since it was placed
          @point = @nodes[-2].offset(@nodes[-1] - @nodes[-2], distance)
          @nodes[-1] = @point
        else
          #Otherwise add new point
          @point = @nodes[-1].offset(@point - @nodes[-1], distance)
          @nodes << @point
        end
        @last_node_copy = @point
        self.update_tracks_cont
        view.invalidate
      end
      
    elsif @mode == "offset"
      #Offset, set parallel distance
      
      begin
        distance = text.to_l
      rescue
        UI.messagebox(S.tr("Invalid length."))
        return
      end
      distance*= -1 if @offset_distance < 0  #for the user the direction mouse is moved in is viewed as positive
      @offset_distance = distance
      self.update_tracks_offset
      self.insert_tracks
      
    end

  end

  def onKeyDown(key, repeat, flags, view)

    case key
    when 9#tab
      if ["calc", "lib", "copy"].include? @mode
        #Toggle what end virtual tracks are being 'hold'
        #Only for copy, lib and calc mode

        self.toggle_end_to_place
        view.invalidate

      elsif @mode == "connect"
        #Toggle what track end are being connected

        self.toggle_ends_to_connect
        view.invalidate

      end

      #Set track direction from arrow keys
      #Up = north, down = south, left = west, right = east
    when VK_RIGHT
      @vector = Geom::Vector3d.new(10.m, 0,0) if ["calc", "lib", "copy"].include?(@mode)
    when VK_LEFT
      @vector = Geom::Vector3d.new(-10.m, 0,0) if ["calc", "lib", "copy"].include?(@mode)
    when VK_UP
      @vector = Geom::Vector3d.new(0, 10.m,0) if ["calc", "lib", "copy"].include?(@mode)
    when VK_DOWN
      @vector = Geom::Vector3d.new(0, -10.m,0) if ["calc", "lib", "copy"].include?(@mode)
    end

    if [VK_RIGHT, VK_LEFT, VK_UP, VK_DOWN].include?(key) && ["calc", "lib", "copy"].include?(@mode)
      self.update_tracks_place
      view.invalidate
    end

  end

  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
    view.invalidate
  end

  #Own definitions
  #Not called from Sketchup itself

  def tracks_primitive_calc
    #calculate track coordinates from formation, angle and parallel distance.
    #These tracks are transformed to mouse position on mouse moved and saved as @tracks.
    
    #Shorter var names for more readable equations
    d = @@center_distance
    a = @@angle

    #Calculate points and vectors

    #Points at track ends
    point_start = Geom::Point3d.new
    point_end = Geom::Point3d.new(d/Math.sin(a), 0, 0)

    point_top_left = Geom::Point3d.new(d/(2*Math.sin(a))-d/(2*Math.tan(a)), d/2, 0)
    point_top_right = Geom::Point3d.new(d/(2*Math.sin(a))+d/(2*Math.tan(a)), d/2, 0)
    point_bottom_left = Geom::Point3d.new(d/(2*Math.sin(a))-d/(2*Math.tan(a)), -d/2, 0)
    point_bottom_right = Geom::Point3d.new(d/(2*Math.sin(a))+d/(2*Math.tan(a)), -d/2, 0)

    #Some reoccurring lengths
    
    #On switches there's a short straight track in each end being the switch track.
    #This used to be 1m but to allow for smaller tracks, such as a model, make it center distance divided by 4,5 m
    length_short_track = (d/(4.5.m)).m
    
    #Adapt length of switch tracks for slips to make either an inside or outside slip.
    if @@formation == "double_slip" || @@formation == "single_slip"
      #Assume standard track gauge.
      g = 1.437.m
      length_inner = d/(2*Math.sin(a))-g/(2*Math.tan(a/2))
      length_outer = d/(2*Math.sin(a))-g/(2*Math.tan(a/4))

      if length_inner > 0 || length_outer > 0
        length_short_track = length_outer > length_short_track ? length_outer : length_inner
      end
    end
    length_vector_long = ((point_top_left-(point_bottom_right)).length-2*length_short_track)/3
    length_vector_short = length_short_track/3

    #Vectors at track ends
    vector_start = Geom::Vector3d.new(1,0,0);
    vector_start.length = length_vector_long
    vector_end = vector_start.reverse

    vector_top_left = point_bottom_right - point_top_left
    vector_top_left.length = length_vector_long
    vector_bottom_right = vector_top_left.reverse
    vector_top_right = point_bottom_left - point_top_right
    vector_top_right.length = length_vector_long
    vector_bottom_left = vector_top_right.reverse

    #Short vectors for short tracks
    vector_start_short = vector_start.clone
    vector_start_short.length = length_vector_short
    vector_end_short = vector_start_short.reverse

    vector_top_left_short = vector_top_left.clone
    vector_top_left_short.length = length_vector_short
    vector_bottom_right_short = vector_top_left_short.reverse

    vector_top_right_short = vector_top_right.clone
    vector_top_right_short.length = length_vector_short
    vector_bottom_left_short = vector_top_right_short.reverse

    #Vectors used to find 'inner' points
    vector_offset_start = vector_start.clone
    vector_offset_start.length = length_short_track
    vector_offset_end = vector_end.clone
    vector_offset_end.length = length_short_track

    vector_offset_top_left = vector_top_left.clone
    vector_offset_top_left.length = length_short_track
    vector_offset_top_right = vector_top_right.clone
    vector_offset_top_right.length = length_short_track
    vector_offset_bottom_left = vector_bottom_left.clone
    vector_offset_bottom_left.length = length_short_track
    vector_offset_bottom_right = vector_bottom_right.clone
    vector_offset_bottom_right.length = length_short_track

    #Inner points, used for switches so they start a small distance from the track end so this part can be used as switch track
    point_inner_start = point_start.offset vector_offset_start
    point_inner_end = point_end.offset vector_offset_end

    point_inner_top_left = point_top_left.offset vector_offset_top_left
    point_inner_top_right = point_top_right.offset vector_offset_top_right
    point_inner_bottom_left = point_bottom_left.offset vector_offset_bottom_left
    point_inner_bottom_right = point_bottom_right.offset vector_offset_bottom_right

    #Points and vectors occurring more than once in a track formation MUST be cloned so transformations doesn't change the same point or vector twice
    @tracks_primitive = 
    if @@formation == "turnout_left"
      [
        {#switch
          :controls => [point_start, point_inner_start, vector_start_short, vector_end_short],
          :curve_algorithm => "arc",
          :switch_state => [0, 1]
        },
        {#straight
          :controls => [point_inner_start.clone, point_end, vector_start, vector_end],
          :curve_algorithm => "arc"
        },
        {#curve
          :controls => [point_inner_start.clone, point_inner_top_right, vector_start.clone, vector_top_right],
          :curve_algorithm => "arc"
        },
        {#short straight after curve
          :controls => [point_inner_top_right.clone, point_top_right, vector_bottom_left_short, vector_top_right_short],
          :curve_algorithm => "arc"
        }
      ]
    elsif @@formation == "turnout_right"
      [
        {#switch
          :controls => [point_start, point_inner_start, vector_start_short, vector_end_short],
          :curve_algorithm => "arc"
        },
        {#straight
          :controls => [point_inner_start.clone, point_end, vector_start, vector_end],
          :curve_algorithm => "arc"
        },
        {#curve
          :controls => [point_inner_start.clone, point_inner_bottom_right, vector_start.clone, vector_bottom_right],
          :curve_algorithm => "arc"
        },
        {#short straight after curve
          :controls => [point_inner_bottom_right.clone, point_bottom_right, vector_top_left_short, vector_bottom_right_short],
          :curve_algorithm => "arc"
        }
      ]
    elsif @@formation == "double_slip"
      [
        {#switch (start)
          :controls => [point_start, point_inner_start, vector_start_short, vector_end_short],
          :curve_algorithm => "arc",
          :switch_state => [0, 1]
        },
        {#switch (top right)
          :controls => [point_inner_top_right, point_top_right, vector_bottom_left_short, vector_top_right_short],
          :curve_algorithm => "arc"
        },
        {#switch (end)
          :controls => [point_end, point_inner_end, vector_end_short.clone, vector_start_short.clone],
          :curve_algorithm => "arc",
          :switch_state => [0, 1]
        },
        {#switch (bottom left)
          :controls => [point_inner_bottom_left, point_bottom_left, vector_top_right_short.clone, vector_bottom_left_short.clone],
          :curve_algorithm => "arc"
        },
        {#straight (start - end)
          :controls => [point_inner_start.clone, point_inner_end.clone, vector_start, vector_end],
          :type_of_signals => "arc"
        },
        {#straight (bottom left - top right)
          :controls => [point_inner_bottom_left.clone, point_inner_top_right.clone, vector_bottom_left, vector_top_right],
          :curve_algorithm => "arc"
        },
        {#curve (start - top right)
          :controls => [point_inner_start.clone, point_inner_top_right.clone, vector_start.clone, vector_top_right.clone],
          :curve_algorithm => "arc"
        },
        {#curve (end - bottom left)
          :controls => [point_inner_end.clone, point_inner_bottom_left.clone, vector_end.clone, vector_bottom_left.clone],
          :curve_algorithm => "arc"
        }
      ]
    elsif @@formation == "single_slip"
      [
        {#switch (start)
          :controls => [point_start, point_inner_start, vector_start_short, vector_end_short],
          :curve_algorithm => "arc",
          :switch_state => [0, 1]
        },
        {#straight (start - end)
          :controls => [point_inner_start.clone, point_end, vector_start, vector_end],
          :curve_algorithm => "arc"
        },
        {#curve
          :controls => [point_inner_start.clone, point_inner_top_right, vector_start.clone, vector_top_right],
          :curve_algorithm => "arc"
        },
        {#switch (top right)
          :controls => [point_inner_top_right.clone, point_top_right, vector_bottom_left_short, vector_top_right_short],
          :curve_algorithm => "arc"
        },
        {#straight (top left - bottom right)
          :controls => [point_inner_top_right.clone, point_bottom_left, vector_top_right.clone, vector_bottom_left],
          :curve_algorithm => "arc"
        }
      ]
    elsif @@formation == "curve"
      [
        {
          :controls => [point_start, point_top_right, vector_start, vector_top_right],
          :curve_algorithm => "arc"
        }
      ]
    elsif @@formation == "straight"
      [
        {
          :controls => [point_start, point_end, vector_start, vector_end],
          :curve_algorithm => "arc"
        }
      ]
    end
    
    #Set track and signal type of newly calculated tracks
    @tracks_primitive.each { |t| t[:type_of_track] = @@type_of_track; t[:type_of_signals] = @@type_of_signals; }
    
    true
        
  end
  
  def tracks_primitive_from_formation
    #Loads track data into @tracks_primitive from text file.
    
    formation = Template.info :track_formation, @@track_formation
    unless formation
      @tracks_primitive = []
      return
    end

    @tracks_primitive = formation[:placement_data]

    nil 
    
  end
  
  def tracks_primitive_from_selection
    #Loads track data into @tracks_primitive from selected tracks
    #This should always be used when loading existing tracks from selection to make sure the right properties are loaded.

    ss = Sketchup.active_model.selection
    @tracks_primitive = []
    ss.each do |i|
      t = Track.get_from_group i
      next unless t
      
      # Clone controls so original track isn't affected by the transformations.
      controls = t.controls.map { |i| i.clone }
      
      # Hard coded here what properties are copied
      # Not dynamic since so many properties are references to groups or other values that should NOT be copied
      # Reference to original added so its group can be copied in copy mode.
      # Other values are still listed and not taken from original because they are needed for the drawing to view, snapping to tracks and other things shared with other modes than copy.
      @tracks_primitive << {
        :controls        => controls,
        :curve_algorithm => t.curve_algorithm,
        :segments        => t.segments,
        :type_of_track   => t.type_of_track,
        :type_of_signals => t.type_of_signals,
        :path_only       => t.path_only,
        :disable_drawing => t.disable_drawing,
        :original        => t
      }
    end

  end

  def tracks_primitive_ends
    #Returns all lose ends of primitive tracks as an array where each element is an array containing a point and corresponding vector

    ends = []
    @tracks_primitive.each do |i|
      0.upto(1) do |j|

        unuiqe = true
        @tracks_primitive.each do |k|
          next if i == k
          0.upto(1) do |l|
            unuiqe = false if i[:controls][j] == k[:controls][l]
          end
        end
        ends << [i[:controls][j], i[:controls][j+2]] if unuiqe

      end
    end

    ends

  end

  def init_web_dialog
    #Open web dialog with tool settings

    @dlg = UI::WebDialog.new(S.tr("Add Track"), false, "#{ID}_insert_track", 410, 210, 500, 100, true)
    @dlg.min_width = 410
    @dlg.min_height = 210
    @dlg.navigation_buttons_enabled = false
    @dlg.set_background_color @dlg.get_default_dialog_color
    @dlg.set_file(File.join(DIALOG_DIR, "insert_track", "index.html"))
    
    #Translate strings
    js = S.tr_dlg
    
    #Create track type dropdown
    track_type_selector = "<select id=\"type_of_track\">"
    Template.list_installed(:track_type).each { |tt| track_type_selector << "<option value=\"#{tt[:id]}\" title=\"#{tt[:name]}\">#{tt[:name]}</option>" }
    track_type_selector << "</select>"
    js << "document.getElementById('track_type_wrapper').innerHTML='#{track_type_selector}';"
    js << "document.getElementById('track_type_slave_wrapper').innerHTML='#{track_type_selector.gsub("type_of_track", "type_of_track_slave")}';"
    
    #Create signal type dropdown
    signal_type_selector = "<select id=\"type_of_signals\">"
    Template.list_installed(:signal_type).each { |st| signal_type_selector << "<option value=\"#{st[:id]}\" title=\"#{st[:name]}\">#{st[:name]}</option>" }
    signal_type_selector << "<option value=\"\">#{S.tr "None"}</option>"
    signal_type_selector << "</select>"
    js << "document.getElementById('type_of_signals_wrapper').innerHTML='#{signal_type_selector}';"
    
    #Create structure type dropdown
    structure_type_selector = "<select id=\"structure_type\">"
    structure_type_selector << "<option value=\"\">#{S.tr "None"}</option>"
    structure_types = Template.list_installed :structure_type
    structure_types.each { |st| structure_type_selector << "<option value=\"#{st[:id]}\" title=\"#{st[:name]}\">#{st[:name]}</option>" }
    structure_type_selector << "</select>"
    js << "document.getElementById('type_of_structure_wrapper').innerHTML='#{structure_type_selector}';"
    
    #Create track formation dropdown
    track_formations_json = Template.list_installed :track_formation, true
    js << "var track_formations=#{track_formations_json};"
    js << "create_track_formation_dropdown();"
    
    #Format k to division of 1
    k_inverse = 1.0/@@k
    k_string = (k_inverse == k_inverse.to_i) ? "1/#{k_inverse.to_i}" : @@k.to_s
    
    #Form data
    js << "document.getElementById('mode').value='#{["cont", "lib", "calc"].include?(@mode) ? "copy" : @mode}';"
    js << "document.getElementById('height_offset').value=#{@@height_offset.to_s.inspect};"
    js << "document.getElementById('center_distance').value=#{@@center_distance.to_s.inspect};"
    js << "document.getElementById('angle').value='#{(Sketchup.format_angle @@angle).to_s.gsub(".",",")}';"
    js << "document.getElementById('formation').value='#{@@formation}';"
    js << "document.getElementById('type_of_track').value='#{@@type_of_track}';"
    js << "document.getElementById('type_of_signals').value='#{@@type_of_signals || ""}';"
    js << "document.getElementById('structure_type').value='#{@@structure_type}';"
    js << "document.getElementById('tracks').value='#{@@tracks_parallel.to_s}';"
    js << "document.getElementById('radius').value=#{@@radius.to_s.inspect};"
    js << "document.getElementById('k').value='#{k_string}';"
    js << "document.getElementById('use_structure_data').value='#{@@use_structure_data.to_s}';"
    js << "document.getElementById('track_formation').value='#{@@track_formation}';" if @@track_formation

    #Enable or disable 'From Selection' depending on existence of selected tracks
    if Sketchup.active_model.selection.any? { |i| Track.group_is_track? i }
      js << "document.getElementById('no_selection_notifier').style.display='none';"
    else
      js << "document.getElementById('from_selection').style.display='none';"
    end
    
    #Init scripts
    js << "document.onkeyup=port_key;"
    js << "init_tabs();"
    js << "change_tab(0,#{(0 if @mode == "cont") || (1 if @mode == "calc") || (2 if @mode == "lib") || 3});"#First tab corresponds to continuous mode, second to calc, third to library and last has radio buttons for mode.
    js << "init_checkboxes();"
    js << "init_radio_buttons();"
    js << "init_input_sync();"
    js << "push_changes();"
    js << "form = document.forms[0];"
    js << "form.onclick = push_changes;"
    js << "form.onkeyup = push_changes;"

    #Show dialog
    if WIN
      @dlg.show { @dlg.execute_script js }
    else
      @dlg.show_modal { @dlg.execute_script js }
    end

    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox
    
    #Form value changes
    @dlg.add_action_callback("push_changes") {

      #First 2 tabs corresponds to a mode each, last tab has a list of modes.
      @mode = @dlg.get_element_value("mode")
      @mode = "cont" if @dlg.get_element_value("tab_opened") == "0"
      @mode = "calc" if @dlg.get_element_value("tab_opened") == "1"
      @mode = "lib" if @dlg.get_element_value("tab_opened") == "2"

      #Only keep tracks in selection
      ss = Sketchup.active_model.selection
      ss.each do |i|
        ss.toggle i unless Track.group_is_track?(i)
      end#each

      #Get values that needs to be validated
      height_offset = @dlg.get_element_value("height_offset")
      center_distance = @dlg.get_element_value("center_distance")
      angle =  @dlg.get_element_value("angle")
      radius = @dlg.get_element_value("radius")
      k = @dlg.get_element_value("k")

      #Validate values
      begin
        height_offset = height_offset.to_l
        center_distance = center_distance.to_l
        radius = radius.to_l
      rescue
        UI.messagebox(S.tr("Invalid length."))
        next
      end
      next if center_distance == 0
      next if angle == ""
      if !angle.match(/\A\d+(.|,)*d*\Z/) or !angle.gsub(",", ".").to_i.between?(1, 89)
        UI.messagebox(S.tr("Angle must be between 1 and 89 degrees."))
        next
      end
      angle.gsub! ",", "."
      angle = angle.to_f.degrees
      ka = k.split "/"
      if ka.length == 2 && ka[1].to_f != 0
        k = ka[0].to_f/(ka[1].to_f)
      else
        k = k.to_f
      end
      k = 1.0/5000.0 if  k <= 0#NOTE: set minimum value. also set max value

      #Save input values as class variables (when passed validation)
      @@height_offset = height_offset
      @@center_distance = center_distance
      @@angle = angle
      @@radius = radius
      @@k = k
      @@tracks_parallel = @dlg.get_element_value("tracks").to_i
      @@tracks_parallel = 1 if  @@tracks_parallel < 1
      @@formation = @dlg.get_element_value("formation")
      @@type_of_track = @dlg.get_element_value("type_of_track")
      @@type_of_signals = @dlg.get_element_value("type_of_signals")
      @@type_of_signals = false if @@type_of_signals == ""
      @@track_formation = @dlg.get_element_value("track_formation")

      if @mode == "cont"
        #Continuous railroad line based on nodes
        
        #Get form data
        old_type_of_structure = @@structure_type
        @@structure_type = @dlg.get_element_value("structure_type")
        @@structure_type = nil if @@structure_type == ""
        old_use_structure_data = @@use_structure_data
        @@use_structure_data = @dlg.get_element_value("use_structure_data") == "true"
        if @@use_structure_data && (old_type_of_structure != @@structure_type || !old_use_structure_data)
          #Structure type changed with checkbox to use structure's values were ticked, use those values.
          str_info = structure_types.find { |str| str[:id] == @@structure_type }
          if str_info
            #Use structure's values
            @@type_of_track   = str_info[:corresponding_track_type]     if str_info[:corresponding_track_type]        && str_info[:corresponding_track_type] != ""
            @@type_of_signals = str_info[:corresponding_signal_type]    if !str_info[:corresponding_signal_type].nil? && str_info[:corresponding_signal_type] != ""# False means that tracks drawn should have no structure.
            @@center_distance = str_info[:track_parallel_distance].to_l if str_info[:track_parallel_distance]         && str_info[:track_parallel_distance] != 0
            @@height_offset   = str_info[:height_offset].to_l           if str_info[:height_offset]
            @@tracks_parallel = str_info[:track_number].to_i            if str_info[:track_number].to_i > 0
            
            #Update web dialog
            js = "document.getElementById('type_of_track').value=document.getElementById('type_of_track_slave').value='#{@@type_of_track}';"
            js << "document.getElementById('type_of_signals').value='#{@@type_of_signals || ""}';"
            js << "document.getElementById('center_distance').value=document.getElementById('center_distance_slave').value=#{@@center_distance.to_s.inspect};"
            js << "document.getElementById('height_offset').value=#{@@height_offset.to_s.inspect};"
            js << "document.getElementById('tracks').value='#{@@tracks_parallel.to_s}';"
            @dlg.execute_script js
          end
        end
      
        #Set statusbar text
        @status_text = @status_text_cont
        @vcb_label = @vcb_label_cont
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE
        
        #Update @tracks array from nodes and form data
        self.update_tracks_cont
        
        Sketchup.active_model.active_view.invalidate
      
      elsif @mode == "calc"
        #Calculate tracks from given data
        #Used for switches etc with given parallel distance and angle

        #Set statusbar text
        @status_text = @status_text_place
        @vcb_label = ""
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE

        #Reset index of track end being hold
        @track_end_index = 0

        #Calculate tracks
        #tracks should e possible to connect in various ways so their center distance is always the same.
        #This can be seen as a grid (non-perpendicular) where every cell is a parallelogram and every intersection is a slip.
        #The intersections can be replaced by normal turnouts, curves or straights but their location are always the same.
        #the height of the parallelogram is the center distance, d. This means each side is d/sin(a), where a is the angle.
        self.tracks_primitive_calc
        
        @tracks = []
        Sketchup.active_model.active_view.invalidate

      elsif @mode == "lib"
        # Load @tracks_primitive from external file.
        
        #Set statusbar text
        @status_text = @status_text_place
        @vcb_label = ""
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE
        
        #Load track formation to @tracks_primitive
        self.tracks_primitive_from_formation
        
        #Hold tracks in end
        @track_end_index = 0
        self.toggle_end_to_place(false)
        @tracks = []
        Sketchup.active_model.active_view.invalidate
      
      elsif @mode == "copy" && !ss.empty?
        #Copy selected tracks in model

        #Set statusbar text
        @status_text = @status_text_place
        @vcb_label = ""
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE

        #Load selected tracks to @tracks_primitive
        self.tracks_primitive_from_selection

        #Hold track copy in its end
        @track_end_index = 0
        self.toggle_end_to_place(false)
        @tracks = []
        Sketchup.active_model.active_view.invalidate

      elsif @mode == "offset" && !ss.empty?
        #Offset selected tracks in model

        #Set statusbar text
        @status_text = @status_text_offset
        @vcb_label = @vcb_label_offset
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE
        
        #Load selected tracks to @tracks_primitive
        self.tracks_primitive_from_selection
        
        #Order tracks as they are connected
        #Start with first track in selection
        sorted = [@tracks_primitive[0]]

        #Don't try to add the same track to tracks_primitive twice
        tracks_used = [@tracks_primitive[0]]

        #Double loop so every track comes up when it can be connected
        @tracks_primitive.each do
          @tracks_primitive.each do |i|
          
            #Don't try to add the same track to tracks_primitive twice
            next if tracks_used.include? i
            
            if [i[:controls][0], i[:controls][1]].include?(sorted[0][:controls][0])
              #tracks is connected to the start of the sorted track array

              controls = i[:controls].dup
              if controls[0] == sorted[0][:controls][0]
                #Not same direction as those in array, reverse
                controls[0], controls[1] = controls[1], controls[0]
                controls[2], controls[3] = controls[3], controls[2]
                i[:controls] = controls
              end

              sorted.unshift i
              tracks_used << i

            elsif [i[:controls][0], i[:controls][1]].include?(sorted[-1][:controls][1])
              #Track is connected to the end of the sorted track array

              controls = i[:controls].dup
              if controls[1] == sorted[-1][:controls][1]
                #Not same direction as those in array, reverse
                controls[0], controls[1] = controls[1], controls[0]
                controls[2], controls[3] = controls[3], controls[2]
                i[:controls] = controls
              end

              sorted << i
              tracks_used << i

            end
          end
        end

        #validate results
        #If not every track has made it into sorted, they aren't selected properly
        if tracks_used.length != @tracks_primitive.length
          UI.messagebox(S.tr("All selected tracks must be connected and there cannot be any branches for offset to work."))
          #Revert to copy mode
          @dlg.execute_script "document.getElementById('mode').value='copy';init_radio_buttons();push_changes();"
          next
        end
        
        @tracks_primitive = sorted
        @tracks = []
        Sketchup.active_model.active_view.invalidate

      elsif @mode == "connect" && !ss.empty?
        #Connect selected tracks in model

        #Set statusbar text
        @status_text = @status_text_connect
        @vcb_label = ""
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
        Sketchup.set_status_text "", SB_VCB_VALUE

        #Load selected tracks to @tracks_primitive
        self.tracks_primitive_from_selection

        #Validate input
        ends = self.tracks_primitive_ends
        if ends.length < 2
          UI.messagebox(S.tr("There must be at least 2 loose ends to connect."))
          #Revert to copy mode
          @dlg.execute_script "document.getElementById('mode').value='copy';init_radio_buttons();push_changes();"
          next
        end

        #Initialize connecting track
        #Find ends closest to each other that doesn't form an existing track
        closest_distance = nil
        closest_ends = []
        0.upto(ends.length-1) do |i|
          i.upto(ends.length-1) do |j|
            next if i == j
            next unless (Track.instances.select { |k| k.uses_point?(ends[i][0]) && k.uses_point?(ends[j][0]) }).empty?#Don't suggest existing track
            distance = ends[i][0].distance(ends[j][0])
            if !closest_distance || distance < closest_distance
              closest_distance = distance
              closest_ends = [i, j]
            end
          end
        end
        @track_end_indexes = closest_ends unless closest_ends.empty?
        self.update_tracks_connect
        Sketchup.active_model.active_view.invalidate

      else
        # In one of the modes that uses the selection but with no valid selection.

        @tracks_primitive = []
        @tracks = []
      
      end
    }

    #Deselct tool when webdialog closes
    @dlg.set_on_close {
      Sketchup.send_action "selectSelectionTool:" if @tool_active
    }

    #Button under selection tab used to switch to select tool
    @dlg.add_action_callback("select_select_tool") {
      Sketchup.send_action "selectSelectionTool:"
    }

    #Open link in default browser
    @dlg.add_action_callback("open_in_default_browser") { |_, callbacks|
      UI.openURL callbacks
    }
    
    #Port key event to tool
    @dlg.add_action_callback("port_key") { |_, callbacks|
      self.onKeyDown callbacks.to_i, false, 0, Sketchup.active_model.active_view#This tool doesn't use flags so it can just be set to 0
    }

  end

  def insert_tracks
    #Inserts @tracks to model
    #Called from mouse click and pressing enter
    
    return if @tracks.empty?
    
    # HACK: Preload component definitions before starting operation not to break
    # it in older SU versions.
    if Sketchup.version < "14"
      @tracks.map { |t| t[:type_of_track] }.compact.uniq.each { |tt| Template.component_def(:track_type, tt) }
      @tracks.map { |t| t[:type_of_signals] }.uniq.each { |st| Template.component_def(:signal_type, st) if st }
      if @@structure_type && @mode == "cont"
        Template.component_def :structure_type, @@structure_type
      end
      if @mode == "lib"
        Template.component_def :track_formation, @@track_formation
      end
    end
      
    Observers.disable
    Sketchup.set_status_text S.tr("Drawing Track"), SB_PROMPT
    Sketchup.active_model.start_operation(S.tr("Add Track"), true)

    added_tracks = []
    
    if @mode == "lib"
      # In lib mode (adding pre-defined formation).
      # Load formation as component, place it, explode it and initialize tracks.
      
      return unless @@track_formation
      
      model = Sketchup.active_model
      entities = model.entities
      definitions = model.definitions
      
      # Load component
      component_def = Template.component_def :track_formation, @@track_formation
      
      # Get transformation.# TODO: is never connected to sloping track. Clicked track must be horizontal for now.
      formation = Template.info :track_formation, @@track_formation
      original_track = formation[:placement_data][0]
      trans_old_from_controls = MyGeom.calc_trans_ends(original_track[:controls])[0]
      trans_new_from_controls = MyGeom.calc_trans_ends(@tracks[0][:controls])[0]
      trans = trans_new_from_controls*(trans_old_from_controls.inverse)
      
      # Place and explode.
      component_inst = entities.add_instance component_def, trans
      old_ents = entities.to_a
      component_inst.explode
      new_ents = entities.to_a - old_ents
      
      # Initialize tracks.
      new_ents.each do |e|
        next unless Track.group_is_track? e
        added_tracks << t = Track.new(e)
        t.update_connections# REVIEW: when run in its separate loop the connections sometimes become un-synced and a tarck can be connected to another track that isn't connected back to the first track. Could be some error in the udpate_conenctiuonss code.
      end
      #added_tracks.each { |t| t.update_connections }
      added_tracks.each { |t| t.draw_endings }
      
      # Initialize balises.
      new_balises = []
      formation[:balises].each do |b|
        point = b[0]
        point.transform! trans
        new_balises << Balise.new(point, nil, b[1])
      end
      Balise.security_check new_balises, :track_formation
    
    else
      # Not in library mode, draw new tracks (or copy existing from within model).
      
      @tracks.each do |i|

        #Create track
        added_tracks << t = Track.new
        # Loop all properties of track data and add to newly created track
        # (controls, type_of_track etc).
        i.each_pair do |key, value|
          next if key == :original# Don't write reference to original track to this track.
          t.instance_variable_set("@" + key.to_s, value)
        end
        
        #Calculate path (not called when assigning controls since instance_variable_set was used)
        t.calc_path!

        #Find connections
        t.update_connections

      end
      
      # If in copy mode, get transformation that differentiate new from original.
      if @mode == "copy" && !@tracks_primitive.empty?
        original_track = @tracks_primitive[0][:original]
        trans_old_from_controls = MyGeom.calc_trans_ends(original_track.controls)[0]
        trans_new_from_controls = MyGeom.calc_trans_ends(added_tracks[0].controls)[0]
        trans_difference = trans_new_from_controls*(trans_old_from_controls.inverse)
      end
      
      # Draw new track, or if in copy mode copy them from within model.
      added_tracks.each_with_index do |t, i|
        Sketchup.set_status_text "#{S.tr("Drawing Track")} #{i+1}/#{added_tracks.length}", SB_PROMPT
        
        if @mode == "copy"
          # If in copy mode, copy group of original track instead of making new
          # group from scratch.
          # This is not used when inserting formations because formation may contain other geometry than tracks.
          original_track = @tracks_primitive[i][:original]
          trans_old_track = original_track.group.transformation
          trans_new_track = trans_difference*trans_old_track
          group_def = original_track.group.entities.parent
          t.group = Sketchup.active_model.entities.add_instance group_def, trans_new_track
          t.draw_endings
          t.save_attributes
        else
          t.draw
        end
        
      end
    
    end# End library/not library mode.
    
    #Draw endings of connected tracks
    (added_tracks.map { |t| t.connections }.flatten.uniq - added_tracks).each do |t|
      t.draw_endings
      #puts "Track new track is connected to is also connected to new tracks: #{t.connections.flatten.any?{|tr| added_tracks.include? tr }}
    end

    #For continuous mode
    if @mode == "cont"
    
      #Empty array of tracks to place and nodes for continuous mode so they are not drawn to view
      @nodes = []
      @tracks = []
      
      #Add structure
      if @@structure_type
        Sketchup.set_status_text S.tr("Drawing Structure"), SB_PROMPT
        rightmost_tracks = added_tracks[0..added_tracks.length/@@tracks_parallel-1]
        path = rightmost_tracks[0..-1].map { |t| t.path[0..-2] } .flatten
        path << rightmost_tracks[-1].path[-1]
        str = Structure.new
        str.path = path
        str.structure_type = @@structure_type
        str.draw
      end
    end

    Sketchup.active_model.commit_operation
    Sketchup.set_status_text S.tr("Done drawing tracks"), SB_PROMPT
    Sketchup.active_model.active_view.invalidate
    Observers.enable

  end

  def toggle_end_to_place(increase = true)
    #Toggles end of @tracks_primitive to use as its origin.
    #Used when placing tracks(copy or calc mode)
    #increase is set to false when initializing copy since end isn't toggled then, just set to the first

    #List unconnected ends of @tracks_primitive
    ends = self.tracks_primitive_ends

    if ends.empty?
      #Closed loop, isn't toggled but just centered

      bb = Geom::BoundingBox.new
      @tracks_primitive.each{ |i| bb.add i[:controls][0] and bb.add i[:controls][1] }
      new_point = bb.center
      new_vector = Geom::Vector3d.new(1, 0, 0)

    else
      #Not closed loop, toggle what end is being placed

      if increase
        #Increase index
        @track_end_index += 1
        @track_end_index = 0 if @track_end_index >= ends.length
      end

      new_point = ends[@track_end_index][0]
      new_vector = ends[@track_end_index][1]

    end

    #Transform virtual tracks so the used point is origin
    translation =  Geom::Transformation.translation Geom::Point3d.new - new_point
    rotation = Geom::Transformation.rotation Geom::Point3d.new, Z_AXIS, MyGeom.angle_in_plane(Geom::Vector3d.new(1,0,0), new_vector)
    transformation = rotation*translation

    @tracks_primitive.each do |i|
      i[:controls].each do |j|
        j.transform! transformation
      end
    end

    #Transform @tracks_primitive to cursor location and save to @tracks
    self.update_tracks_place

  end

  def toggle_ends_to_connect
    #Toggles one of the 2 ends of @tracks_primitive that is about to be connected
    #Called when pressing tab in connect mode

    #Get number of unconnected ends
    ends_num = self.tracks_primitive_ends.length

    #toggle
    @track_end_indexes[0] += 1
    @track_end_indexes[0]+= 1 if @track_end_indexes[0] == @track_end_indexes[1]
    if @track_end_indexes[0] >= ends_num
      @track_end_indexes[0] = 0
      @track_end_indexes[1] += 1
      @track_end_indexes[1] += 1 if @track_end_indexes[0] == @track_end_indexes[1]
      @track_end_indexes[1] = 0 if @track_end_indexes[1] >= ends_num
      @track_end_indexes[0] = 1 if @track_end_indexes[1] == 0
    end

    #Change @tracks so it contains a track connecting these 2 ends
    self.update_tracks_connect

  end

  def update_tracks_cont
    #Create track data for @tracks based on nodes.
    #Nodes should be connected with tracks but with curved tracks at corners.
    
    @tracks = []
    return if @nodes.length < 2
    
    ##Connect nodes with straight tracks
    #0.upto(@nodes.length-2) do |i|
    #  p_start = @nodes[i]
    #  p_end = @nodes[i+1]
    #  v_start = p_end - p_start
    #  v_start.length /= 3
    #  v_end = v_start.reverse
    #  @tracks << {
    #    :controls => [p_start, p_end, v_start, v_end],
    #    :curve_algorithm => "arc"
    #  }
    #end
    
    #Make curves at all nodes but first and last
    prev_curve_end = @nodes[0]
    1.upto(@nodes.length-2) do |i|
      n_prev = @nodes[i-1]
      n = @nodes[i]
      n_next = @nodes[i+1]
      v_prev = n - n_prev
      v_next = n - n_next
      a = MyGeom.angle_in_plane(v_prev, v_next)
      
      if v_prev.parallel? v_next
        #No curve
      
        #Add straight track
        p_start = prev_curve_end
        p_end = n
        v_start = p_end - p_start
        v_start.length /= 4
        v_end = v_start.reverse
        @tracks << {
          :controls => [p_start, p_end, v_start, v_end],
          :curve_algorithm => "arc"
        }
        prev_curve_end = p_end
        
      else
        #Curve
        
        #If turns left, has multiple tracks and is horizontal,
        #let @@radius and @@k apply on innermost curve
        outermost_in_curve = a > 0 && @@tracks_parallel > 1 && v_prev.z == 0 && v_next.z == 0
        
        #If this is the outermost track in a multiple track curve calculate innermost curve with given radius and k and offset from there.
        if outermost_in_curve
          offset_length = (@@tracks_parallel-1)*@@center_distance
          n_prev = n_prev.offset(Z_AXIS*MyGeom::flatten_vector(v_prev), offset_length)
          n_next = n_next.offset(MyGeom::flatten_vector(v_next)*Z_AXIS, offset_length)
          n = Geom.intersect_line_line [n_prev, v_prev], [n_next, v_next]
        end

        #Get curve track data
        curve_data = MyGeom.curve n_prev, n, n_next, @@radius, @@k
        
        if outermost_in_curve
          curve_data.each { |t| t[:controls] = MyGeom.controls_offset(t[:controls], -offset_length) }
        end
        
        #Set segment number for curves. Depends on resolution and linear distance between track ends.
        curve_data.each { |t| t[:segments] = [[(t[:controls][0].distance(t[:controls][1])*@@segment_reolution).round, 3].max, 99].min }
        
        #Get controls for straight track leading from last curve (or start node) to this one
        p_start = prev_curve_end
        p_end = curve_data[0][:controls][0]
        unless p_start == p_end
          v_start = p_end - p_start
          v_start.length /= 4
          v_end = v_start.reverse
          
          #Add straight
          @tracks << {
            :controls => [p_start, p_end, v_start, v_end],
            :curve_algorithm => "arc"
          }
        end
        prev_curve_end = curve_data[-1][:controls][1]
        
        #Add curve tracks
        @tracks += curve_data
      
      end
    
    end#node loop
    
    #Connect to last node
    if @nodes.length > 1
    
      #Get controls for straight leading from last curve to end node
      p_start = prev_curve_end
      p_end = @nodes[-1]
      v_start = p_end - p_start
      v_start.length /= 4
      v_end = v_start.reverse
      
      #Add straight
      @tracks << {
        :controls => [p_start, p_end, v_start, v_end],
        :curve_algorithm => "arc"
      }
    end
    
    #Offset parallel tracks
    tracks_parallel = []
    1.upto(@@tracks_parallel-1) do |p|
      offset_distance = p * @@center_distance
      @tracks.each do |t|
        t_copy = t.dup
        t_copy[:controls] = MyGeom.controls_offset t[:controls], offset_distance
        tracks_parallel << t_copy
      end
    end
    @tracks += tracks_parallel
    
    #Set track and signal type of newly calculated tracks
    @tracks.each { |t| t[:type_of_track] = @@type_of_track; t[:type_of_signals] = @@type_of_signals; }
    
    true
  
  end
  
  def update_tracks_place
    #Transforms tracks_place_cursor to cursor location and save it to @tracks
    #Called in mousemove and when track end being placed is toggled

    transformation = Geom::Transformation.axes @point, @vector, Z_AXIS.cross(@vector), Z_AXIS
    @tracks = []
    @tracks_primitive.each do |i|
      controls = i[:controls]
      controls_transformed = []
      controls.each { |j| controls_transformed << j.transform(transformation) }
      i_transformed = i.dup
      i_transformed[:controls] = controls_transformed
      @tracks << i_transformed
    end#each

  end

  def update_tracks_offset
    #Recreate @tracks based on @tracks_primitives and @offset_distance
    #call from mousemove and when entering exact length in VCB

    @tracks = []
    @tracks_primitive.each do |t|

      t_transformed = t.dup
      t_transformed[:disable_drawing] = false#Offset track is always drawn
      t_transformed[:controls] = MyGeom.controls_offset(t[:controls], -@offset_distance)
      @tracks << t_transformed
      
    end

  end

  def update_tracks_connect
    #Update @tracks based on @tracks_primitive
    #Called when initializing connect mode or pressing tab

    #List unconnected ends of @tracks_primitive
    ends = self.tracks_primitive_ends

    #Get controls
    point_start = ends[@track_end_indexes[0]][0]
    vector_start = ends[@track_end_indexes[0]][1].reverse
    point_end = ends[@track_end_indexes[1]][0]
    vector_end = ends[@track_end_indexes[1]][1].reverse
    
    #Get track and signal type
    track_start = @tracks_primitive.find { |t| t[:controls][0..1].include? point_start }
    type_of_track = track_start[:type_of_track]
    track_types = Template.list_installed(:track_type).map { |tt| tt[:id] }
    type_of_track = track_types.first unless track_types.include? type_of_track
    type_of_signals = track_start[:type_of_signals]
    signal_types = Template.list_installed(:signal_type).map { |st| st[:id] }
    type_of_signals = signal_types.first unless signal_types.include? type_of_signals

    #For a somewhat smooth curve if vectors point somewhat towards each others, use half end distance as vector length
    length = (point_start - point_end).length/2
    vector_start.length = length
    vector_end.length = length

    #Set @tracks
    controls = [point_start, point_end, vector_start, vector_end]
    segments = [[(controls[0].distance(controls[1])*@@segment_reolution).round, 12].max, 99].min#Use 12 as minimum number of segments in connect mode.
    @tracks = [
      {:controls => controls, :segments => segments, :curve_algorithm => "c_bezier", :type_of_track => type_of_track, :type_of_signals => type_of_signals}
    ]

  end

end#class

end#module
