# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class Track
  #A track object is a piece of track that is either straight or a curve.
  #The object is closely linked to a Sketchup::Group in the model root that the track is drawn in.

  #Track objects themselves cannot be saved when model is closed but their data is saved to their groups.
  #On start-up the array and track objects are recreated from group attributes.
  
  # Public: Name of attribute dictionary for data in track group.
  # Ideally this should end with "_track" but that would break old models.
  ATTR_DICT = ID

  #All loaded track objects. In mac these might be in different models
  @@instances ||= []
  
  #Drawing queues
  #Adding tracks that needs to be drawn here and then call Track.draw_queued instead of drawing them all individually prevents track from drawing twice.
  @@drawing_queue ||= []
  @@drawing_queue_ends ||= []

  #Class variable accessors

  def self.instances; @@instances; end
  def self.instances=(value); @@instances = value; end
  
  def self.drawing_queue; @@drawing_queue; end
  def self.drawing_queue=(value); @@drawing_queue = value; end
  
  def self.drawing_queue_ends; @@drawing_queue_ends; end
  def self.drawing_queue_ends=(value); @@drawing_queue_ends = value; end

  #Class methods

  def self.calc_point_along(point, distance, vector, p_center = nil, start_track = nil, mod = nil)
    #Find point at linear length along path (not arc length).

    #Point is where track starts being followed and where distance is measured from unless p_center is set.
    #Distance is the linear distance along track.
    #Vector is approximate(+-90 degrees) and tells in what direction track should be followed.
    #P_center is an optional point to measure distance from instead of start point.
    #Start_track is the track to begin following. If start point is on a switch the other track could be faulty used unless start track is specified.

    #Returns hash with point, track and tangent or false if point wasn't found.
    #Returns last point on track if railroad ends, not the point at the given distance.
    #(you may need to check the result so the returned point has the right distance from start point).

    #Get closest track and project point to it
    q = self.inspect_point point, mod, start_track
    t = q[:track]
    p_start = q[:point]
    segment_start = q[:segment]

    #Center point of sphere. This is the point with a defined distance to
    p_center = p_center ? p_center : p_start

    #Path of first track
    path = t.path.dup

    #In what direction along track should we go?
    #Path is reversed so it ends with the point furthest away from input point
    reverse_path = (path[segment_start+1]-path[segment_start]).angle_between(vector) > Math::PI/2
    #traveling at negative velocity equals traveling backwards
    reverse_path = !reverse_path if distance < 0#Going backwards is currently not used within the plugin.
    #Reverse
    if reverse_path
      path.reverse!
      segment_start = (path.length-2)-segment_start
    end

    #Insert start location as a point in path
    #Later, only check for intersections on the correct side of this point
    unless path.include?(p_start)
      segment_start += 1
      path.insert segment_start, p_start
    end

    #Logically this could be an infinite loop since next track is defined inside it.
    #However, if looped more times than the number of tracks the point will never be found anyway.
    #Therefore execute code at most as many times as the number of tracks.
    #This is usually caused by a a switch changing while there's a train on it.
    self.instances.each do

      #Loop segments
      segment_start.upto(path.length-2) do |i|
        #Segment ends
        p1 = path[i]
        p2 = path[i+1]
        #Go to next segment if start point is at segment end
        next if p_start == p2
        #Segment line
        line = [p1,(p2-p1)]
        #Look for intersections
        intersections = MyGeom.intersect_line_sphere(line, p_center, distance)
        #next unless intersections
        #Check if intersection is on segment
        intersections = intersections.select { |j| p1.distance(j) <= p1.distance(p2) && p2.distance(j) <= p2.distance(p1)}
        unless intersections.empty?
          return {
            :point => intersections.sort_by { |s| p1.distance s } [0],
            :track => t,
            :tangent => p2 - p1
          }
        end
      end#each

      #No intersection found, proceed to next track

      #Get connected track
      t_previous = t
      track_end = reverse_path ? 0 : 1
      if t.connections[track_end].empty?
        #No connected track
        return {
          :point => t.controls[track_end],
          :track => t,
          :tangent => nil
        }
      elsif t.connections[track_end].length == 1
        #One connected track
        t = t.connections[track_end][0]
      else
        #Leaving switch track (facing switch)
        t = t.connections_used[track_end]
      end

      #Path start at track end if the start point isn't within the track
      segment_start = 0

      #Path of next track
      path = t.path.dup

      #Reverse path if connected with end, not start, to previous one
      #reverse_path = t_previous.controls[track_end] == t.controls[1]
      reverse_path = t.connections[1].include?(t_previous)
      path.reverse! if reverse_path

    end

    #Point could not be found, perhaps because a switch changed while train was moving across it
    false

  end

  def self.calc_points_along(point_start, distances, vector, start_track, mod = nil)
    #calculate a number of points along a track from a given start point,
    #an approximate start vector to tell in what direction the track should be followed
    #and an array of the (linear) distances between the points.
    #Returns array of hashes holding informations for the points similar to Track.calc_point_along or Track.inspect_point
    
    #Driving train only with this code would cause end of train to be located at the wrong track if trailing changes with train on it

    point_last = point_start
    track_last = start_track
    vector_last = vector

    distances.map { |distance|

      q = Track.calc_point_along point_last, distance, vector_last, nil, track_last, mod
      #Check if the distance to the new point is correct. calc_point_along returns last point when a railroad ends, even if the distance is smaller
      return unless point_last.distance(q[:point]) == distance
      point_last, track_last, vector_last = q[:point], q[:track], q[:tangent]

      #Check for collision
      #NOTE: code?
      #should have an optional argument in this method
      #if collision is found, return nil
      #Used when placing rolling stocks.
      #nah, put it in calc_point_along and gave options to stop at collision and return nil or return point at collision.

      #Return point
      {:point => point_last, :track => track_last, :tangent => vector_last}

    }

  end
   
  # Public: Draw tracks queued for drawing.
  # When adding multiple tracks at once it's recommended to add them to the draw
  # queue, call queue_changed_connections on them to find connected tracks and
  # add those too to draw queue and then draw all queued tracks.
  #
  # append_operator - If true, make previous operation transparent so these
  #                   changes are appended to it in the undo stack
  #                   (default: false).
  #
  # Returns nothing.
  def self.draw_queued(append_operator = false)
    
    @@drawing_queue.uniq!
    @@drawing_queue_ends.uniq!

    #Don't redraw ends twice for tracks that are completely redrawn anyway.
    @@drawing_queue_ends -= @@drawing_queue
    
    all_to_draw = @@drawing_queue + @@drawing_queue_ends
    
    return if all_to_draw.empty?
    
    sum = all_to_draw.length
    current_one = 1
    
    puts "Draw track queue (#{@@drawing_queue.length} whole, #{@@drawing_queue_ends.length} ends only)"
    
    Observers.disable
    model = all_to_draw[0].model# All in queue should be from the same model.
    model.start_operation("DRAW QUEUE", true, false, true) if append_operator
    
    @@drawing_queue.each do |t|
      Sketchup.set_status_text "#{S.tr("Drawing Track")} #{current_one}/#{sum}", SB_PROMPT
      current_one += 1
      t.draw
    end
    @@drawing_queue_ends.each do |t|
      Sketchup.set_status_text "#{S.tr("Drawing Track")} #{current_one}/#{sum} #{S.tr("(ends only)")}", SB_PROMPT
      current_one += 1
      t.draw_endings
    end
    
    Sketchup.set_status_text(S.tr("Done drawing tracks"), SB_PROMPT)
    
    model.commit_operation if append_operator
    Observers.enable
    
    #Empty queue
    @@drawing_queue = []
    @@drawing_queue_ends = []
    
    nil
  
  end
  
  def self.extract_path(point, distance, vector, start_track = nil, mod = nil)
    #Get array of points defining the track(s).
    
    #Point is where to start extract points.
    #Distance is how long (curve length) to extract.
    #Vector is approximate(+-90 degrees) and tells in what direction from point to go in.
    #Start_track is the track to begin extracting at. If start point is on a switch the other track could be faulty used unless start track is specified.
    
    #Get closest track and project point to it
    q = self.inspect_point point, mod, start_track
    t = q[:track]
    p_start = q[:point]
    segment_start = q[:segment]

    #Path of first track
    path = t.path.dup

    #In what direction along track should we go?
    #Path is reversed so it ends with the point furthest away from input point
    reverse_path = (path[segment_start+1]-path[segment_start]).angle_between(vector) > Math::PI/2
    reverse_path = !reverse_path if distance < 0#Going backwards is currently not used within the plugin.
    #Reverse
    if reverse_path
      path.reverse!
      segment_start = (path.length-2)-segment_start
    end

    #Insert start location as a point in path
    #Later, only count distance on the correct side of this point
    unless path.include?(p_start)
      segment_start += 1
      path.insert segment_start, p_start
    end
    
    path_extracted = [p_start]
    distance_left = distance
    
    #Logically this could be an infinite loop since next track is defined inside it.
    self.instances.each do

      #Loop segments
      segment_start.upto(path.length-2) do |i|
        #Segment ends
        p1 = path[i]
        p2 = path[i+1]
        #Go to next segment if start point is at segment end
        next if p_start == p2
        segment_length = p1.distance p2
        if segment_length > distance_left
          #Extraction ends at this segment
          segment_vector = p2 - p1
          point_last = p1.offset segment_vector, distance_left
          path_extracted << point_last
          return path_extracted
        end
        #NOTE: stop and return if p2 is already in path_extracted array. what if it started on this segment? stop at same point, not segment start.
        path_extracted << p2
        distance_left -= segment_length
      end#each

      #Still some distance to go, proceed to next track

      #Get connected track
      t_previous = t
      track_end = reverse_path ? 0 : 1
      if t.connections[track_end].empty?
        #No connected track
        return path_extracted
      elsif t.connections[track_end].length == 1
        #One connected track
        t = t.connections[track_end][0]
      else
        #Switch
        t = t.connections_used[track_end]
      end

      #Path start at track end if the start point isn't within the track
      segment_start = 0

      #Path of next track
      path = t.path.dup

      #Reverse path if connected with end, not start, to previous one
      #reverse_path = t_previous.controls[track_end] == t.controls[1]
      reverse_path = t.connections[1].include?(t_previous)
      path.reverse! if reverse_path

    end

    #Something must have gone wrong if nothing has been returned by now.
    false
    
  end

  def self.path_between_points(p_start, p_end, direction_vector = nil, t_start = nil, t_end = nil, mod = nil)
    #Get path along track as point array between 2 points.
    #Direction_vector is an approximate vector, if given path will start within 90 degrees from it. Otherwise both directions are checked and shortest path returned.
    
    #Return nil if points cannot be connected.
    #Otherwise return hash:
    #{
    # :path => [point-array],
    # :switch_states => [
    #   {track reference, track end, state (as int, counting from left)}
    # ]
    #}
    return if p_start == p_end
    
    #Get data for start and end
    q = self.inspect_point p_start, mod, t_start
    t_start = q[:track]
    p_start = q[:point]
    segment_start = q[:segment]
    segment_start_vector = t_start.path[segment_start+1] - t_start.path[segment_start]
    
    q = self.inspect_point p_end, mod, t_end
    t_end = q[:track]
    p_end = q[:point]
    segment_end = q[:segment]

    #What direction to check. nil = both, 0 = towards track start, 1 = towards track end
    direction = nil
    if direction_vector
      direction = segment_start_vector.angle_between(direction_vector) < 90.degrees ? 1 : 0
    end
    
    #A branch contains its current length and an array of paired track references and the direction that track is followed in.
    #{
    # :tracks => [
    #   [<track reference>,<direction>(, <switch state if facing switch>)]...
    # ],
    # path => <Point3d array>,
    # length => <length>,
    # finished => <bool>
    #}
    branches = []
    
    #Create a branch in each direction or just one in the given direction
    
    #Go towards track start
    if direction != 1
      if t_start == t_end && segment_start >= segment_end && (segment_start != segment_end || segment_start_vector.samedirection?(p_start-p_end))
        #Within one track
        tracks = [[t_start, 0]]
        path = [p_start]
        path += t_start.path[segment_end+1..segment_start].reverse
        path << p_end
        finished = true
      else
        #Spanning multiple tracks
        tracks = [[t_start, 0]]
        path = [p_start]
        path += t_start.path[1..segment_start].reverse
        finished = false
      end
      length = (path.length == 1 ? 0 : MyGeom.path_length(path))
      branches << {:tracks => tracks, :path => path, :length => length, :finished => finished}
    end
    
    #Go towards track end
    if direction != 0
      if t_start == t_end && segment_start <= segment_end && (segment_start != segment_end || segment_start_vector.samedirection?(p_end-p_start))
        #Within one track
        tracks = [[t_start, 1]]
        path = [p_start]
        path += t_start.path[segment_start+1..segment_end]
        path << p_end
        finished = true
      else
        #Spanning multiple tracks
        tracks = [[t_start, 1]]
        path = [p_start]
        path += t_start.path[segment_start+1..-2]
        finished = false
      end
      length = (path.length == 1 ? 0 : MyGeom.path_length(path))
      branches << {:tracks => tracks, :path => path, :length => length, :finished => finished}
    end
    
    #Shortest distance points can be connected by
    branch_finished = branches.find { |b| b[:finished]}
    shortest_path = branch_finished ? branch_finished[:length] : nil
   
    #Go one track along each branch until the shortest path is found.
    #Limit iterations to twice the number of tracks since path can't possibly be longer.
    (@@instances.length*2).times do
    
      break if branches.length == 0
      break if branches.length == 1 && branches[0][:finished]
    
      marked_for_deltion = []
    
      #Loop all valid branches
      #branches.each do |b|#Don't iterate over elements added inside loop until next time loop is called. Not sure of the behavior of this method but manual loop may be safer.
      0.upto(branches.length-1) do |i|
        b = branches[i]
      
        #Remove this branch if it's longer than a branch that is already finished
        if shortest_path && b[:length] > shortest_path
          marked_for_deltion << b 
          next
        end
        
        #If branch is marked as finished, update shortest_path.
        #Don't keep following branch.
        if b[:finished]
          shortest_path = b[:length] if !shortest_path || shortest_path > b[:length]
          next
        end
        
        #Prevent same track from being followed more than once in each direction.
        #This stops never ending loops when no connection between points can be found.
        #Because of balloon loop a the same track can be followed twice but in different directions.
        last_track = b[:tracks][-1]
        if b[:tracks].select { |t| t[0] == last_track[0] && t[1] == last_track[1] } .length > 1
          marked_for_deltion << b
          next
        end
        
        #Keep following branch
        connected_tracks = last_track[0].connections[last_track[1]]
        
        if connected_tracks.include? t_end#NOTE: ERROR: SOLUTION: set switch state if relevant!
          #Last track is connected to end track.
          
          d = (t_end.connections[0].include?(last_track[0])) ? 1 : 0
          path_append = (d==0 ? t_end.path[segment_end+1..-2].reverse : t_end.path[1..segment_end])
          path_append << p_end
          b[:path] += path_append
          b[:length] += MyGeom.path_length([b[:path][-1]] + path_append)
          b[:finished] = true
          #Set switch state for previous track if relevant.
          if connected_tracks.length > 1
            last_track[2] = connected_tracks.index t_end
          end
          #Comparison to shortest_path is done in next iteration
        
        elsif connected_tracks.length == 0
          #Branch ends without reaching target.
          #Delete it.
          
          marked_for_deltion << b
        
        elsif connected_tracks.length == 1
          #Just one track connected.
          #Follow it.
          
          t_c = connected_tracks[0]
          d = (t_c.connections[0].include?(last_track[0])) ? 1 : 0
          path_append = t_c.path[1..-2]
          unless path_append.empty?
            path_append.reverse! if d == 0
            b[:path] += path_append
            b[:length] += MyGeom.path_length([b[:path][-1]] + path_append)
          end
          
          b[:tracks] << [t_c, d]
          
        else
          #Multiple tracks connected.
          #Create a new branch for each
          
          connected_tracks.each_with_index do |t_c, i|
            d = (t_c.connections[0].include?(last_track[0])) ? 1 : 0
            #Create new branch
            new_b = {:tracks => b[:tracks].dup, :path => b[:path].dup, :length => length, :finished => false}
            branches << new_b
            path_append = t_c.path[1..-2]
            unless path_append.empty?
              path_append.reverse! if d == 0
              new_b[:path] += path_append
              new_b[:length] += MyGeom.path_length([new_b[:path][-1]] + path_append)
            end
            #Switch state
            new_b[:tracks][-1] = new_b[:tracks][-1].dup#Make new object so it's not changed in other branches.
            new_b[:tracks][-1][2] = i
            new_b[:tracks] << [t_c, d]
          end
          
          marked_for_deltion << b
           
        end        
        
      end
      
      #Remove branches that either ended without reaching destination or has already gotten longer than shortest finished branch.
#p branches.length
      branches -= marked_for_deltion
#p marked_for_deltion.length
#p branches.length
#p shortest_path
    end
    
    #Turn branch into route (just another way to represent same values).
    return if branches.empty?
    branch = branches[0]
    return unless branch[:finished]
    switch_states = branch[:tracks].select { |i| i.length == 3}
    route = {:path => branch[:path], :switch_states => switch_states}
    
    route
    
  end
  
  def self.find_all_connections(tracks = nil)
    #Find all connections between tracks.
    #Runs on model load. Can also be called when bunch initializing tracks in importers or other custom scripts.
    
    tracks ||= @@instances
    tracks.each { |t| t.find_connections }
    
    true
    
  end
  
  def self.get_from_group(group)
    #Returns track object for given group, in given model
    #Returns nil if group doesn't represent a track

    @@instances.find { |i| i.group == group }

  end

  def self.get_from_selection
    #Get object for selected track
    #Only one entity should be selected
    #Returns nil if not a track

    ss = Sketchup.active_model.selection
    return unless ss.length == 1

    self.get_from_group ss[0]
    
  end
  
  def self.group_is_track?(entity)

    return false unless entity.is_a?(Sketchup::Group)
    return false unless entity.attribute_dictionary ATTR_DICT
    
    true
    
  end
  
  def self.inspect_point(point, mod = nil, track_limit = nil)
    #Takes a point and optionally model and track array to limit search to as arguments.
    #Returns hash
    # :track => <closest track object>,
    # :point => <closest point along track>,
    # :distance => <distance to track>,
    # :segment => <the number of the segment of the track path that is closest>,
    # :vector => <segment direction (counting from track start to track end, might be 180 degrees wrong from expectation)
    #Used in several other methods and classes, e.g to check what track a rolling stock is one and what track (path) was closest to where user clicked since it's difficult to click a specific track group in turnouts.
    
    #Assume active model
    mod = Sketchup.active_model unless mod

    distance_closest  = nil
    point_closest = nil
    track_closest = nil
    segment_index = nil
    vector_closest = nil

    #Set tracks to search through
    track_limit = [track_limit] if Track === track_limit
    tracks = track_limit ? track_limit : @@instances

    #Loop tracks
    tracks.each do |t|
      next unless t.model == mod
      next unless t.path# REVIEW: don't know why but sometimes path is nil.

      path = t.path.dup

      #Loop all segments
      0.upto(path.length-2) do |i|
        #Segment ends
        p1 = path[i]
        p2 = path[i+1]
        #Segment line
        line = [p1,(p2-p1)]
        #Project argument point to line
        p = point.project_to_line line
        #Check if point is on segment, otherwise put in on closest end
        p = p2 if(p1.distance(p) > p1.distance(p2))
        p = p1 if(p2.distance(p) > p2.distance(p1))

        #Check if this point is closer than the closest one seen so far
        d = point.distance p
        next unless !distance_closest || d < distance_closest

        track_closest = t
        point_closest = p
        distance_closest = d
        segment_index = i
        vector_closest = p2 - p1
      end#each
    end

    #Return
    {
      :track => track_closest,
      :point => point_closest,
      :distance => distance_closest,
      :segment => segment_index,
      :vector => vector_closest
    }

  end

  def self.length_total
    #Get total length of all loaded track objects
    
    @@instances.map { |t| t.length }.inject { |sum, n| sum + n }.to_l
  
  end

  def self.properties_dialog(tracks)
    #Open properties dialog for tracks.
    #Tracks can either be one track or an array of tracks.
    
    tracks = [tracks] if Track === tracks
    
    #Create web dialog
    title = "#{S.tr("Track Properties")} (#{tracks.length} #{tracks.length == 1 ? S.tr("Track") : S.tr("Tracks")})"
    dialog = UI::WebDialog.new title, false, "#{ID}_track_properties", 640, 400, 300, 100, false
    dialog.navigation_buttons_enabled = false
    dialog.set_background_color dialog.get_default_dialog_color
    dialog.set_file(File.join(DIALOG_DIR, "track_properties", "index.html"))
    
    track_types = Template.list_installed :track_type
    signal_types = Template.list_installed :signal_type
    
    # Get values
    # If values differs between these tracks, use nil as Indeterminate value.
    # If any of the used types is missing, warn user and try using another.
    values = {}
    #List all variable names here:
    ["type_of_track", "curve_algorithm", "segments", "type_of_signals", "path_only", "disable_drawing"].each do |v_name|
      a = tracks.map{ |t| t.send v_name }.uniq
      if v_name == "type_of_track" && a != a & track_types.map { |tt| tt[:id] }
        UI.messagebox S.tr("Used track type could not be found. Applying changes will result in redrawing the track as another track type.")
        a = a & track_types.map { |tt| tt[:id] }
        a = a.empty? ? [track_types[0][:id]] : [a.first]
      end
      if v_name == "type_of_signals" && a != [false] && a != a & signal_types.map { |st| st[:id] }
        UI.messagebox S.tr("Used signal type could not be found. Applying changes will result in redrawing the track without signals.")
        a = [false]
      end
      values[v_name] = a.size == 1 ? a.first : nil
    end

    #Create list of track types
    track_types.unshift({:name => "", :id => "", :info => ""}) if values["type_of_track"] == nil
    track_types_json = EneRailroad.object_2_json track_types

    #Create list of signal types
    signal_types.unshift({:name => "", :id => "", :info => ""}) if values["type_of_signals"] == nil
    signal_types << {:name => S.tr("None"), :id => "false"}
    signal_types_json = EneRailroad.object_2_json signal_types
    
    #Translate strings
    js = S.tr_dlg

    #Form data
    js << "var type_of_tracks=#{track_types_json};"
    js << "var type_of_signals=#{signal_types_json};"
    js << "var curve_toggle_indeterminate = #{values["curve_algorithm"] == nil ? "true" : "false"};";
    js << "window.init_dynamic_lists();"
    values.each_pair do |k, v|
      js << "document.getElementById('#{k}').value='#{v.to_s}';"#nil > "", true > "true", false > "false"
    end

    #Init scripts
    js << "init_checkboxes();"
    js << "window.update_type_of_track_info();"
    js << "document.getElementById('type_of_track').onchange = update_type_of_track_info;"
    js << "window.update_curve_algorithm_info();"
    js << "document.getElementById('curve_algorithm').onchange = update_curve_algorithm_info;"
    js << "window.update_type_of_signal_info();"
    js << "document.getElementById('type_of_signals').onchange = update_type_of_signal_info;"

    #Show dialog
    if WIN
      dialog.show { dialog.execute_script js }
    else
      dialog.show_modal { dialog.execute_script js }
    end

    #Apply properties
    dialog.add_action_callback("apply") { |dialog, callbacks|

      #Get values from dialog
      #"reverse" is added as key now since it only exists locally.
      new_values = {}
      (values.keys << "reverse").each do |k|
        i = dialog.get_element_value(k)
        i = nil if i == ""
        new_values[k] = i
      end
      
      #Validate segments and make int
      #3 is minimum for bezier curves
      if new_values["segments"] && (!new_values["segments"].match(/\A\d+\Z/) || !new_values["segments"].to_i.between?(3, 99))
        UI.messagebox(S.tr("Segments must be a number between 3 and 99."))
        next
      end
      new_values["segments"] = new_values["segments"].to_i if new_values["segments"]
      
      #False as type_of_signals means no signals, strings should be kept
      new_values["type_of_signals"] = false if new_values["type_of_signals"] == "false"
      
      #Make boolean
      bool_vars = ["path_only", "disable_drawing", "reverse"]
      bool_vars.each do |bv|
        next if new_values[bv] == nil
        new_values[bv] = new_values[bv] == "true"
      end
      
      #Save values and redraw
      
      # HACK: Preload track and signal type's component definitions before
      # starting operation not to break it in older SU versions.
      if Sketchup.version < "14"
        Template.component_def(:track_type, new_values["type_of_track"]) if new_values["type_of_track"]
        Template.component_def(:signal_type, new_values["type_of_signals"]) if new_values["type_of_signals"]
      end
      
      #Disable observers so tracks aren't drawn twice
      #Start operation
      Observers.disable
      mod = Sketchup.active_model
      mod.start_operation "Track Properties", true
      
      tracks_draw_ends = []
      
      tracks.each do |track|
      
        #Add new values to track
        new_values.each_pair do |k, v|
          next if v == nil#Nil means indeterminate, keep existing.
          next if k == "reverse"#reverse is not saved as a attribute
          #track.send "#{k} = #{v}"#Writing controls, segments and curve algorithm forces path calculation, use attribute_set instead
          track.instance_variable_set("@" + k, v)
        end
        
        track.reverse if new_values["reverse"]
        track.calc_path!
        
        #List connected tracks to be redrawn if type_of_track changed
        tracks_draw_ends << track.connections.flatten if new_values["type_of_track"] && new_values["type_of_track"] != track.type_of_track
      
      end
      
      #Redraw
      tracks_draw_ends.uniq!
      tracks_draw_ends -= tracks
      tracks.each { |t| t.draw}
      tracks_draw_ends.each { |t| t.draw_endings}
    
      mod.commit_operation
      Observers.enable

      #Close if user pressed OK rather than Apply
      dialog.close if callbacks == "close"
    }

    #Close web dialog
    dialog.add_action_callback("close") {
      dialog.close
    }

    #Open link in default browser
    dialog.add_action_callback("open_in_default_browser") { |dialog, callbacks|
      UI.openURL callbacks
    }

  end

  def self.update_from_groups
    #Checks for track objects with controls data that don't match group attributes.
    #Load controls from group attributes (which updates the path and rolling stocks' references to track).
    #Run from undo and redo observer.
    
    @@instances.each do |track|
      next if track.group.deleted?
      ad = track.group.attribute_dictionary(ATTR_DICT)
      next if ad["controls"] == track.controls #Skip when matches
      track.controls = ad["controls"] #This also calculates a new path and checks for rolling stocks on track.
      track.update_connections
    end
    
    true
    
  end
  
  #Instance attribute accessors

  attr_reader :path#Calculated when changing controls, segments or curve algorithm
  attr_reader :controls#Has custom writer that calculates path
  attr_accessor :type_of_track
  attr_reader :curve_algorithm#Has custom writer that calculates path
  attr_reader :segments#Has custom writer that calculates path
  attr_accessor :type_of_signals
  attr_accessor :group
  attr_accessor :switch_state
  attr_accessor :path_only
  attr_accessor :disable_drawing
  attr_accessor :connections

  #Instance methods

  def initialize(group = nil)
    #Create new track object, either default(empty) or load from group.
    #Declare all track's variables.

    if group
      #Create object from existing data saved in group.
      #Used to re-initialize saved tracks with data saved to attributes, NOT create new tracks from any group.

      group.attribute_dictionary(ATTR_DICT).each_pair do |key, value|#ene_railroad_track would be a better name but wouldn't be backward compatible
        self.instance_variable_set("@" + key.to_s, value)
      end

      #Override group reference since group cannot be saved as attribute
      @group = group
      
      #Backward comparability:
      
      #If switch stage isn't an integer set it to 0.
      #switch states in old models will be lost and switches will turn to the leftmost track.
      @switch_state = [0, 0] unless @switch_state[0].class == Fixnum
      
      #Tracks drawn before disable_drawing was implemented will not have it set to false (default).
      @disable_drawing = false if @disable_drawing.nil?
      
      #No signals are now saved as false, not nil. nil is used in track properties dialog for Indeterminate values.
      @type_of_signals = false if @type_of_signals == nil

      #Attach observer
      @group.add_observer(Observers::MyInstanceObserver.new)
      @group.add_observer(Observers::MyEntityObserver.new)

    else
      #Create new object from default settings

      #Control points & vectors
      @controls = []
      #  0  Point where track starts
      #  1  Point where track ends
      #  2  vector from start point
      #  3  vector from end point

      #Track properties
      #Defines in web dialog
      @type_of_track = "default"
      @curve_algorithm = "c_bezier"
      @segments = 12
      @type_of_signals = "default"

      #Group to draw track and save data to
      #Track groups are always on model root.
      #Created and drawn by Track.draw.
      #Observers attached there too.
      @group = nil

      #Switch state
      #Defined in ToolTrackSwitch
      #Tells what connected track is "used" by the index it has in the connection array (group reference cannot be saved as attribute)
      @switch_state = [0, 0]

      #Draw path only
      #This reduces drawing time and is useful when using scale tool
      @path_only = false
      
      #Tells if track shouldn't be redrawn by extension (when changing properties, moving, connecting or disconnected other tracks etc).
      #Can be used when user wants to manually edit track group but without adding a completely new track type.
      #Doesn't prevent attributes from being saved.
      @disable_drawing = false

    end

    #List connections
    #Overrides group attributes since references to other track objects cannot be saved as attribute
    #When initializing from group these are updated, either when model loads or when groups is added (copied in)
    @connections = [[], []]

    #Add track to track list
    @@instances << self
    
    #Check for rolling stocks being on this track (has no affect when initializing whole plugin since rolling stock array is empty then)
    RStock.find_tracks_for_all_not_on_tracks

  end

  def calc_path!
    #Calculate path for track.
    #Called when changing controls, segment number and curve algorithm
    #NOTE: optimize: if several of those are changed at the same time the path and rolling stock relations only need to be calculated once.
    
    @path = MyGeom.calc_path @controls, @curve_algorithm, @segments
    
    #Remove references to this track in rolling stocks (rolling stocks don't stand on this track anymore)
    self.unlink_rstocks
    
    #Check all rolling stocks not on a track if they are now on this track
    RStock.find_tracks_for_all_not_on_tracks
    
    true
  
  end
  
  def connections_used
    #Return 2 element array of the connections that are currently used
    
    [
      @connections[0][@switch_state[0]],
      @connections[1][@switch_state[1]]
    ]
    
  end
  
  def controls=(value)
    #Custom attribute writer for @controls that also calculates path
   
    @controls = value
    calc_path!
    
    value
    
  end
   
  def curve_algorithm=(value)
    #Custom attribute writer for @curve_algorithm that also calculates path
   
    @curve_algorithm = value
    self.calc_path!
    
    value
    
  end
  
  def delete!
    #Removes all connections to track and delete track from instances array
    #Redraws connected tracks! Must be wrapped in operator.

    #Remove connections to this, redraw those tracks no longer connected
    0.upto(1) do |end_of_self|
      self.connections[end_of_self].each do |i|
        next if i.group.deleted?#skip if this is deleted too (e.g. when unloading a model)
        0.upto(1) do |end_of_other|
          #Remove deleted track from connection list of other tracks
          #Connection arrays are still sorted since nothing was added.
          i.connections[end_of_other].delete self
        end#each
        #Redraw formerly connected track
        i.draw_endings unless Observers.undo_redo?
      end#each
    end#each
    
    #Remove references to this track in rolling stocks (those that used to be on it)
    self.unlink_rstocks
    
    #Remove balise object located at this track
    #Cannot be deleted from within loop
    to_delete = []
    Balise.instances.each { |b| to_delete << b if b.track == self }
    to_delete.each { |i| i.delete! }

    #Remove track object
    Track.instances.delete self

    true

  end

  def draw
    #Draws the extrusions and 'along'-components, and call method to draw endings and save attributes
    #Called when the track needs to be completely redrawn in contrast to draw_endings
    #that is called when a connected track is changed.

    #Save attributes but don't draw anything to track group for custom tracks.
    if @disable_drawing
      self.save_attributes
      return
    end
    
    #Create group if there isn't one
    unless @group
      @group = Sketchup.active_model.entities.add_group
      @group.add_observer(Observers::MyInstanceObserver.new)
      @group.add_observer(Observers::MyEntityObserver.new)
    end
    
    #Make track unique
    @group.name += ""#Group.make_unique is deprecated and this seems to work instead
    
    #Standard vars
    defs = model.definitions
    ents_group = @group.entities
    
    #Transform group so bounding box matches general outline of track.
    #otherwise bounding box is aligned to model axis and diagonal tracks look bad.
    x_axis = MyGeom.flatten_vector(@controls[1] - @controls[0])
    y_axis = Z_AXIS*x_axis
    trans = Geom::Transformation.axes @controls[0], x_axis, y_axis, Z_AXIS
    @group.transformation = trans
    
    #Get or create group for extrusions and components to place along track.
    #Clear group content if group exists so new content can replace it.
    group_extrusions = ents_group.find { |e| e.class == Sketchup::Group && e.name == "extrusions"}
    if group_extrusions
      group_extrusions.entities.clear!
    else
      group_extrusions = ents_group.add_group
      group_extrusions.name = "extrusions"
    end
    group_extrusions.transformation = Geom::Transformation.new
    
    ents_ex = group_extrusions.entities
    
    #Transform path to track group's local coordinate system.
    local_path = @path.map { |p| p.transform(@group.transformation.inverse) }

    if @path_only
      #Only draw path
      #Useful when using scale tool on track
      #Don't use add_curve since it doesn't have midpoints tools can inference to
      ents_ex.add_edges local_path
      
    else
      #Draw full track
      
      component_def = Template.component_def :track_type, @type_of_track
      unless component_def
        warn "Could not draw track. Track type '#{@type_of_track}' could not be found.\n#{caller}"
        return
      end

      #Calculate transformations where to place extrusion profile
      #trans_ends is an array where 0th element corresponds to first track end
      trans_ends = MyGeom.calc_trans_ends @controls
      trans_ends.map! { |t| (@group.transformation.inverse)*t }

      # Make extrusion of there's a profile for it.
      profile_template = component_def.entities.find do |e|
        e.get_attribute(Template::ATTR_DICT_PART, "type") == "extrusion_profile"
      end
      if profile_template

        # Place profile at track end.
        profile_component_def = #TODO: get definition from method on Template.
          if profile_template.is_a?(Sketchup::Group)
            profile_template.entities.parent
          else
            profile_template.definition
          end
        component_inst = ents_ex.add_instance profile_component_def, trans_ends[0]

        #Explode profile component. should only contain groups containing one face each.
        ents_exploded = component_inst.explode
        ents_exploded.each do |g|

          #Keep original group in component definition so component doesn't change and have to be reloaded each time a track of this type is drawn
          g.name += ""#Group.make_unique is deprecated and this seems to work instead

          #Move group coordinates to match track group axes.
          MyGeom.move_group_origin g, Geom::Transformation.new

          #Extrude face
          face = g.entities.find { |i| i.class == Sketchup::Face }
          endFace = EneRailroad.extrude(face, local_path, Z_AXIS)

          #Hide ends of extrusion
          face.hidden = endFace.hidden = true
          (face.edges + endFace.edges).each { |i| i.hidden = true }

        end

      end# Extrusion

      #Draw components along path such as ties/sleepers
      array_templates = component_def.entities.select do |e|#TODO: get definition from method on Template.
        e.get_attribute(Template::ATTR_DICT_PART, "type") == "arrayed"
      end
      array_templates.each do |array_template|

        distance = array_template.get_attribute(Template::ATTR_DICT_PART, "distance", 1.m)

        arrayed_component_def =
          if array_template.is_a?(Sketchup::Group)
            array_template.entities.parent
          else
            array_template.definition
          end

        trans_along = MyGeom.calc_trans_along(@controls, @curve_algorithm, distance)
        trans_along.each do |i|
          along_group = ents_ex.add_instance arrayed_component_def, (@group.transformation.inverse)*i
        end
      end
      
    end#if else

    #Call other methods
    self.save_attributes
    self.draw_endings

  end

  def draw_endings
    #Draws the track endings such as buffer stops but also the switch indicators
    #Called from draw and when a connected track has changed so the endings of this needs to be updated

    #Don't draw anything to track group for custom tracks.
    return if @disable_drawing
    
    return if @path_only
    
    #Make track unique
    @group.name += ""#Group.make_unique is deprecated and this seems to work instead
    
    #Standard vars
    defs = model.definitions
    ents_group = @group.entities
    
    component_def = Template.component_def :track_type, @type_of_track
    unless component_def
      warn "Could not draw track endings. Track type '#{@type_of_track}' could not be found.\n#{caller}"
      return
    end
  
    #Get or create group for endings
    group_endings = ents_group.find { |e| e.class == Sketchup::Group && e.name == "endings"}
    if group_endings
      group_endings.entities.clear!
    else
      group_endings = ents_group.add_group
      group_endings.name = "endings"
    end
    group_endings.transformation = Geom::Transformation.new
    
    ents_end = group_endings.entities
    
    group_signals = ents_group.find { |e| e.class == Sketchup::Group && e.name == "signals"}
    if group_signals
      group_signals.entities.clear!
    else
      group_signals = ents_group.add_group
      group_signals.name = "signals"
    end
    group_signals.transformation = Geom::Transformation.new
    
    ents_sig = group_signals.entities

    #Calculate transformations
    trans_ends = MyGeom.calc_trans_ends @controls
    trans_ends.map! { |t| (@group.transformation.inverse)*t }
    
    # Get different endings' definitions.
    endings_component_defs = {}
    component_def.entities.each do |e|
      next unless e.get_attribute(Template::ATTR_DICT_PART, "type") == "ending"
      ending_definition =
        if e.is_a?(Sketchup::Group)
          e.entities.parent
        else
          e.definition
        end
      key = e.get_attribute(Template::ATTR_DICT_PART, "ending_type", "railroad")
      endings_component_defs[key] = ending_definition
    end
    
    # Place endings.
    0.upto(1) do |i|
      trans_end = trans_ends[i + 2]# Mirror back end to match double track (text and signs will be wrong on the other hand).
      
      if @connections[i].empty?
      
        # Whole railroad line (e.g. buffer stop).
        if d = endings_component_defs["railroad"]
          ents_end.add_instance d, trans_ends[i]
        end
        
      else
        # End of track type (e.g. tunnel portal).
        if (d = endings_component_defs["type"]) && @connections[i].first.type_of_track != @type_of_track
          ents_end.add_instance d, trans_ends[i]
        end
      
      end
      
      # End of individual track.
      if d = endings_component_defs["track"]
        ents_end.add_instance d, trans_ends[i]
      end
      
    end
    
    # Place switch indicators
    if @type_of_signals
      0.upto(1) do |end_of_track|
        next if @connections[end_of_track].length < 2
        
        #Define side
        #right by default, left if it's further from other tracks
        switch_point = @controls[end_of_track]
        signal_info = Template.info :signal_type, @type_of_signals
        indicator_distance = signal_info[:distance_to_track] || 1.85.m
        
        vector_left = @controls[end_of_track+2]*Z_AXIS
        vector_left.length = indicator_distance
        indicator_point_left = switch_point.offset vector_left
        indicator_point_right = switch_point.offset(vector_left.reverse)
        distance_from_left_to_track = Track.inspect_point(indicator_point_left, model)[:distance]
        distance_from_right_to_track = Track.inspect_point(indicator_point_right, model)[:distance]
        
        if distance_from_left_to_track >= indicator_distance || distance_from_right_to_track >= indicator_distance
          #There is enough space on one side of the switch to place indicator
          
          side_is_left = distance_from_left_to_track > distance_from_right_to_track
          vector_side = vector_left.clone
          vector_side.reverse! unless side_is_left
          vector_side.transform! @group.transformation.inverse
          
          #Draw signal structure
          trans_signal = trans_ends[end_of_track]
          trans_signal = Geom::Transformation.translation(vector_side) * trans_signal

          component_def = Template.component_def :signal_type, @type_of_signals
          unless component_def
            warn "Could not draw signal. Signal type '#{@type_of_signals}' could not be found.\n#{caller}"
            return
          end
          if component_def
            sig_group = ents_sig.add_group
            sig_group.name = "switch_indicator#{end_of_track}"
            temp_comp = sig_group.entities.add_instance component_def, trans_signal
            temp_comp.explode
          end
          
        end
      end
    end
    
    # Update switch indicators' direction.
    self.draw_switch_indicators
             
  end

  def draw_switch_indicators
    #Update switch indicators.
    #Called from draw endings and when changing switch state.

    #Save attributes but don't draw anything to track group for custom tracks.
    return if @disable_drawing
    
    return if @path_only
    return unless @type_of_signals
    
    #Make track unique
    @group.name += ""#Group.make_unique is deprecated and this seems to work instead
    
    #Standard vars
    ents_group = @group.entities
    
    group_signals = ents_group.find { |e| e.class == Sketchup::Group && e.name == "signals"}
    return unless group_signals
    ents_sig = group_signals.entities
    
    #Loop track ends.
    0.upto(1) do |end_of_track|
        next if @connections[end_of_track].length < 2
        signal = ents_sig.find { |e| e.name == "switch_indicator#{end_of_track}"}
        next unless signal
        
        #Get direction to indicate.
        #Draw as through if track is straight, even if leftmost or rightmost.
        #If there are more than 2 tracks all in the middle will be considered through too.
        #Some signal types have different signals for diverging left or right while some just uses one.
        #The same goes for through signals.
        direction =
        if @connections[end_of_track][@switch_state[end_of_track]].is_straight?
          if @switch_state[end_of_track] == 0
            "through_left"
          elsif @switch_state[end_of_track] == (@connections[end_of_track].length) -1
            "through_right"
          else
            "through_middle"
          end
        elsif @switch_state[end_of_track] == 0
          "diverge_left"
        elsif @switch_state[end_of_track] == (@connections[end_of_track].length) -1
          "diverge_right"
        else
          "through_middle"
        end
        
        #Get direction fallback (just "diverge" or "through", not left or right).
        direction_fallback = direction.split("_")[0]
        
        #Hide all indicators
        signal.entities.each do |e|
          next unless [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class)
          next unless e.name.match(/^indicator_/)
          e.hidden = true
        end
        
        #Show the one currently used
        indicator = signal.entities.find { |e| [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "indicator_#{direction}"}
        indicator ||= signal.entities.find { |e| [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "indicator_#{direction_fallback}"}
        indicator.hidden = false if indicator
        
    end
    
    true

  end

  # Public: Updates this tracks references to connected tracks based on its
  # position. Only updates references in this track, not those it's connected
  # to. For that call update_connections instead.
  #
  # Returns nothing.
  def find_connections
    #Find tracks connected to self.
    #Compare points and vectors

    #Resets arrays
    @connections = [[], []]

    0.upto(1) do |end_of_self|
      point_on_self = @controls[end_of_self]
      vector_on_self = @controls[end_of_self + 2]
      0.upto(1) do |end_of_other|
        Track.instances.each do |i|

          #Must be in same model
          next unless model == i.model
          point_on_other = i.controls[end_of_other]
          #Compare point
          next unless point_on_self == point_on_other
          vectort_on_other = i.controls[end_of_other + 2]
          #Compare vector
          next unless vector_on_self.reverse.samedirection? vectort_on_other

          #Is connected
          @connections[end_of_self] << i

        end#each
      end#each

      #Sort connections from left to right
      self.sort_connections
      
    end#each
    
    nil

  end
 
  def inspect
    #Do not show attributes when inspecting. It clutters the console
    self.to_s
  end

  def length
    #Returns length of track
    
    MyGeom.path_length @path
    
  end
  
  def is_double_switch?
    #Check if track is switch in both ends
    
    @connections.all? { |i| i.length > 1 }
    
  end
  
  def is_straight?
    #Check if track is straight
    
    MyGeom.is_straight? @controls
    
  end
  
  def is_switch?
    #Tells if this track is a switch or not
    #Switch is a track that has multiple other tracks connected to the same end

    @connections.any? { |i| i.length > 1 }

  end

  # Public: Model track is drawn to or active Model if track has not been drawn.
  #
  # Returns a Model.
  def model
  
    @group && @group.valid? ? @group.model : Sketchup.active_model
    
  end
  
  def other_point(point)
    #Returns end if start is given or vice verse.
    #If none of the points is the same as argument, return nil

    return @controls[0] if point == @controls[1]
    return @controls[1] if point == @controls[0]
    nil

  end

 # Public: calls update_connections and add returned track to drawing queue.
 #
 # Returns nothing.
  def queue_changed_connections
  
    @@drawing_queue_ends += update_connections

    nil
    
  end

  # Public: Add this track to drawing queue.
  #
  # Returns nothing.
  def queue_drawing
  
    @@drawing_queue << self
  
    nil
    
  end
  
  # Public: Add this track to drawing queue for ends only.
  # Redrawing ends is significantly faster than redrawing whole track.
  #
  # Returns nothing.
  def queue_end_drawing
  
    @@drawing_queue_ends << self
    
    nil
    
  end
  
  def reverse!
    #Switch start and end of track.
    #Useful for double track tunnels and bridges where the tunnel or bridge is a part of one of the tracks

    @controls[0], @controls[1] = @controls[1], @controls[0]
    @controls[2], @controls[3] = @controls[3], @controls[2]

    @connections.reverse!
    @switch_state.reverse!

    @path.reverse!
    
    true

  end
  alias :reverse :reverse!

  def segments=(value)
    #Custom attribute writer for @segments that also calculates path
   
    @segments = value
    self.calc_path!
    
    value
    
  end
  
  def select
    #Make the track group the only selected entity in model.
    #Used to test what track group a track object corresponds to.
    
    ss = Sketchup.active_model.selection
    ss.clear
    ss.add self.group
    
    true
    
  end
  
  def set_switch(connection_index, end_of_track = nil)
    #Used to set switch from auto pilot and Balise. Not used from switch tool.
    #Conenction_index is the number of the track switch should be set to, counting from left to right.
    #If track is switch in both its ends end_of_track must be defined to tell what end should be changed, otherwise it can be left out.
    
    #Check if track is switch
    raise TypeError.new("Track is not a switch.") unless self.is_switch?
    
    #Determine end of track
    if self.is_double_switch?
      #Is switch in both ends
      raise ArgumentError.new("Track is switch in both ends. Second argument must tell what end to set.") unless end_of_track
    else
      #Is switch in one end, check which
      end_of_track = @connections[0].length > 1 ? 0 : 1
    end
    
    #Don't turn switch with train on it. Puts instead of raise to not clutter console.
    unless self.trains_spanning_end(end_of_track, false).empty?
      puts "Cannot turn switch when a train spans it."
      return
    end
    
    #Check if connection_index is within valid range
    highest_valid_index = @connections[end_of_track].length-1
    unless (0..highest_valid_index).include? connection_index
      raise RangeError.new("connection_index must be between 0 and #{highest_valid_index}")
    end
    
    #Set new switch state and redraw switch indicator
    @switch_state[end_of_track] = connection_index
    
    Observers.disable
    model.start_operation "Set Switch State", true
    save_attributes
    draw_switch_indicators
    model.commit_operation
    Observers.enable
    
    
  end
  alias :turn_switch :set_switch

  def smooth_tangent(point, hard_tangent)
    #Other track methods return hard tangent vectors from the polyline path defining the track.
    #This method calculate the actual tangent from the curve.
    #Used for rotating  bogies and tilting car bodies.
    
    if self.is_straight?
      #Straight track
      
      return hard_tangent
    else
      #Curved track, results depend on curve algorithm
      
      case @curve_algorithm
      when "hard_corner"
      
        return hard_tangent
      
      when "arc"
        
        #Get center axis
        plane_start = [@controls[0], MyGeom.flatten_vector(@controls[2])]
        plane_end = [@controls[1], MyGeom.flatten_vector(@controls[3])]

        if plane_start[1].parallel?(plane_end[1]) && plane_start[0].on_plane?(plane_end)
          #Planes are the same meaning angle is 180 degrees.
          #point_center is between start and end

          point_center = Geom::Point3d.linear_combination 0.5, @controls[0], 0.5, @controls[1]

        elsif plane_start[1].parallel? plane_end[1]
          #Planes are co-planar but not the same
          #Straight track

          return hard_tangent

        else
          #Planes intersect
          #Track turns any angle but +-180 degrees

          line_intersect = Geom.intersect_plane_plane plane_start, plane_end
          point_center = line_intersect[0]

        end
        
        #Move the point on center axis to the same height as input point
        point_center.z = point.z
        
        #Get arc tangent vector
        radius_vector = point_center - point
        smooth_tangent = radius_vector*Z_AXIS
        
        #Adapt vector to have a similar direction to the hard tangent vector
        smooth_tangent.reverse! if smooth_tangent.angle_between(hard_tangent) > Math::PI*0.5
        
        #Set vector z value from hard vector 
        smooth_tangent.z = hard_tangent.z
        
        return smooth_tangent
        
      when "c_bezier"
      
        #Algebraic method
        #Get point's relative distance along track (length along track to track start / track polyline length)
        #Find points defining bezier holograph. 
        #Put that value into the bezier formula. <-- Wont work here since distance along curve from curve start to point isn't the same as the value giving that point when put in bezier formula.
        #Get value and make vector
        
        #Numeric method
        #Get tangents from this segment, previous segment and next segment.
        #Combine linear:
        #on segment start point half previous tangent, half this,
        #on segment middle only this tangent,
        #on segment end point half next tangent, half this.
        #(not perfect but sufficient)
        
        segment_index = Track.inspect_point(point, nil, [self])[:segment]
        segment_start = @path[segment_index]
        segment_end = @path[segment_index + 1]
        tangent = segment_end - segment_start
        position_on_segment = segment_start.distance(point)/segment_start.distance(segment_end)#0..1
        
        #check how far along this segment point is
        #Find tangent of neighboring segment if track doesn't add there.
        if position_on_segment < 0.5
          #Point on first half of segment
          tangent_neighbor = @path[segment_index] - @path[segment_index-1] unless segment_index == 0
        else
          #Point on last half of the segment
          tangent_neighbor = @path[segment_index+2] - @path[segment_index+1] if segment_index < @path.length-2
        end
        
        #Combine neighboring tangent with this' segment's tangent
        smooth_tangent = 
        if tangent_neighbor
          neighbor_influence = (0.5-position_on_segment).abs
          Geom.linear_combination 1-neighbor_influence, tangent, neighbor_influence, tangent_neighbor
        else
          tangent
        end
        
        #Adapt vector to have a similar direction to the hard tangent vector
        smooth_tangent.reverse! if smooth_tangent.angle_between(hard_tangent) > Math::PI*0.5
      
        return smooth_tangent
        
      end
    end
  
  end
  
  def sort_connections
    #Sort the array of connected tracks from left to right (seen from self)
    #Called from find_connections to make sure the arrays are always sorted
    
    #NOTE: keep switch_state referring to same track even if it doesn't have same index (wouldn't this destroy the references when initializing?)
    #NOTE: when deleting a connected track to the left of the used track the track to the left of the used will be used instead.
    
    0.upto(1) do |end_of_self|
      
      switch_point = @controls[end_of_self]
      switch_vector_straight = @controls[end_of_self+2].reverse
            
      next if @connections[end_of_self].empty?
      
      @connections[end_of_self] = @connections[end_of_self].sort_by do |c|
      
        #Sort by angle towards 2nd point in path counting from this switch (starting at 0)
        #That is the first point that isn't located straight ahead.
        #When 2 curved connected tracks are really close and have different number of segments.
        #this point might differ from the general direction of the track causing them to be.
        #sorted in wrong order.
        
        path = c.path.dup
        path.reverse! if c.controls[1] == switch_point
        point_on_connected = path[2] || path[1]#fall back to point[1] on straight tracks
        conenction_vector = point_on_connected - switch_point
      
        MyGeom.angle_in_plane switch_vector_straight, conenction_vector
        
      end#each
    
    end#each
    
  end
 
  def save_attributes
    #Writes all instance variables as attributes to the group so it can be retrieved when the model is reopened.
    #Some variables, such as group references, cannot be saved. Therefore these are recreated when initializing track objects from groups on model load.
    #Called from draw and when changing switch state, either from tool or set_switch method

    #Do not alter model when called from observer triggered undo/redo.
    return if Observers.undo_redo?
    
    #When track.disable_drawing is true this method is NOT stopped unlike draw or draw_ends
    
    #Add all class instance variables as group attributes
    self.instance_variables.each do |i|
      key = i
      value = self.instance_variable_get(key)
      key = key.to_s  #Make string of symbol
      key[0] = ""    #Remove @ sign from key
      next if key == "group"
      next if key == "connections"
      @group.set_attribute ATTR_DICT, key, value
    end#each

    @group.name = "#{S.tr("Track")}, #{length} (#{type_of_track}, #{curve_algorithm})"#NOTE: use human readable strings, not id strings

  end

  def trains(include_buffers = true)
    #Return array of all trains on this track.
    #Train may also span track or have buffer over it but not have an axis on it.
  
    Train.instances.select { |t| t.tracks_all(include_buffers).include? self }
  
  end
 
  def trains_spanning_end(track_ends = nil, include_buffers = true)
    #Check if one or more trains spans an end of this track. Useful on switch tracks to see if a train is already over the switch.
    track_ends = track_ends ? [track_ends] : [0, 1]
    
    trains = self.trains(include_buffers)
    return [] unless trains
    
    track_ends.map { |track_end|
      trains.select{ |t|
        (t.tracks_all(include_buffers) & @connections[track_end]).any?
      }
    }.flatten.uniq
    
  end
  
  def unlink_rstocks
    #Remove references to this track in rolling stocks.
    #Called when removing track (track.delete!) and when track moves (observer).

    RStock.instances.each do |rs|
      rs.tracks.collect! { |i| i == self ? nil : i}
    end

    true

  end

 # Public: Update references for connected tracks based on position, both in
 # this track, those that became connected to it and those that lost connection
 # from it.
 #
 # Returns array of the tracks that changed connection state to this track.
 def update_connections
  
    # Remember old connections and find current connections.
    previous_connections = @connections.clone
    self.find_connections
    current_connections = @connections.clone

    # List tracks whose connection state to this changed.
    changed = []
    (0..1).each do |track_end|
      changed += (previous_connections[track_end] - current_connections[track_end])#Doesn't exist now
      changed += (current_connections[track_end] - previous_connections[track_end])#Didn't exist before
    end
    changed.uniq!

    # Find current connections of the tracks that changed connection state.
    changed.each { |i| i.find_connections }
    
    # Sort connections on all tracks connected to self since self may have moved
    # and connections are order after position.
    (current_connections.flatten. - changed).each { |i| i.sort_connections }
    
    #if @connections.flatten.any? { |i| !i.connections.flatten.include? self }
    #  puts "This track is connected to a track that isn't connected back to this track!!!"
    #end
    
    changed
    
  end
  
  def uses_point?(point)
    #Tells whether or not this is an end or start point of track

    controls[0..1].include? point

  end

end#class

end#module
