# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

# Public: Module handling installed track types, track formations, signal types,
# structure types and rolling stocks.
#
# Each one of these has its own dictionary containing an external sketchup model
# used to load its ComponentDefinition from, a text file containing information
# that can be loaded without loading the component into the active model and,
# for all but track formations, a preview image.
module Template

  # Public: Name of attribute dictionary used to identify what different
  # entities are for in a template component, e.g. what is the track extrusion
  # profile.
  ATTR_DICT_PART = "#{ID}_template_part"
  
  # Public: name prefixes on ComponentDefinitions used by templates indexed by
  # template type identifier.
  COMPONENT_NAME_PREFIXES = {
    :track_type => "#{NAME} Track: ",
    :track_formation => "#{NAME} Track Formation: ",
    :signal_type => "#{NAME} Signal: ",
    :structure_type => "#{NAME} Structure: ",
    :r_stock => "#{NAME} Rolling Stock: "
  }
  
  # Public: Default information hashes for newly created templates indexed by
  # template type identifier.
  DEFAULT_INFO = {
    :track_type =>      {:name => "", :info => ""},
    :track_formation => {:name => "", :description => ""},
    :signal_type =>     {:name => "", :info => "", :distance_to_track => nil},
    :structure_type =>  {:name => "", :corresponding_track_type => "", :corresponding_signal_type => "", :track_parallel_distance => nil, :height_offset => nil, :track_number => 1},
    :r_stock =>         {:name => "", :author => "", :type => "", :country => "", :company => "", :production_start => "", :production_end => "", :retired => ""}
  }
  
  # Public: Basename of template directories indexed by template type
  # identifier.
  DIR_BASENAMES = {
    :track_type => "tracks",
    :track_formation => "track_formations",
    :signal_type => "signals",
    :structure_type => "structures",
    :r_stock => "r_stocks"
  }
  
  # Public: Full path to template directories indexed by template type
  # identifier.
  DIRS = Hash[*DIR_BASENAMES.map { |k, v| [k, File.join(PLUGIN_DIR, "templates", v)] }.flatten]
  
  # Public: vector to point camera in when creating rolling stock preview.
  RSTOCK_CAM_ANGLE = Geom::Vector3d.new 837.559, 480.945, -409.41
  
  # Public: Array of the template type identifiers.
  TYPES = DIR_BASENAMES.keys
  
  # Public: Returns the ComponentDefinition used by a specific template, e.g. a 
  # track or structure type. If component definition isn't loaded, load it.
  #
  # type - Template type identifier.
  # id   - The string identifier of the template.
  #
  # Returns ComponentDefinition or nil when template is not installed.
  def self.component_def(type, id)
  
    raise ArgumentError, "Unknown type '#{type}'." unless TYPES.include? type

    id = id.to_s
    
    dir = DIRS[type]
    name_prefix = COMPONENT_NAME_PREFIXES[type]
    
    defs = Sketchup.active_model.definitions
    name = name_prefix + id
    
    # Check if there's an existing definition by this name.
    component_def = defs.find { |d| d.name == name }
    return component_def if component_def
    
    # Otherwise load component definition.
    old_observer_state = Observers.disabled?# REVIEW: There should be no need to disable observers while loading a component.
    Observers.disable
    path = File.join dir, id, "model.skp"
    return unless File.exists? path
    begin
      component_def = defs.load path
    rescue
      raise IOError, "Couldn't load file #{path}. File exists."
    end
    component_def.name = name
    Observers.disabled = old_observer_state

    component_def
    
  end
  
  # Public: Moves old templates into new folder. Also changes Track and
  # structures to save save all geometry in one model.
  #
  # Returns nothing.
  def self.convert_old
  
    # Sketchup 2013 bugsplats form saving models and FileUtils are only shipped
    # with 2014+. Limit method to SU 2014+.
    if Sketchup.version < "14"
      puts "This method is for technical reasons only available in SU 2014 and later."
      return
    end
  
    require "fileutils"

    Observers.disable
    
    old_locations    = Hash[*DIR_BASENAMES.map { |k, v| [k, File.join(PLUGIN_DIR, v)] }.flatten]
    new_locations    = DIRS
    backup_locations = Hash[*DIR_BASENAMES.map { |k, v| [k, File.join(PLUGIN_DIR, "templates_old", v)] }.flatten]
        
    copied_track_types     = []
    copied_structure_types = []
    
    # Copy files to new location (unless template already exists there).
    puts "Moving files to new location."
    old_locations.each_pair do |type, old_location|
      new_location = new_locations[type]
      Dir.glob(File.join(old_location, "*")) do |old_path|
        next unless File.directory? old_path
        id = File.basename old_path
        new_path = File.join new_location, id
        next if File.exist? new_path
        FileUtils.copy_entry old_path, new_path
        if type == :track_type
          copied_track_types << id
        elsif type == :structure_type
          copied_structure_types << id
        end
      end
    end

    # Move old to keep as backup if conversion fails.
    puts "Backup old files to 'templates_old/'."
    backup_dir = File.join PLUGIN_DIR, "templates_old"
    old_locations.each_pair do |type, old_location|
      backup_location = backup_locations[type]
      
      if File.exist? old_location
        Dir.mkdir backup_dir unless File.exists? backup_dir
        File.rename old_location, backup_location# TODO: merge if exists.
      end
      
    end
    
    # Converting old track type and structure type format to new
    # (merging models into one).
    number_to_convert = copied_track_types.size + copied_structure_types.size
    if number_to_convert > 0
      puts "Converting #{number_to_convert} track/structure types to new format."
      to_convert = copied_track_types.map { |v| [v, :track_type] }
      to_convert += copied_structure_types.map { |v| [v, :structure_type] }
      to_convert.each do |a|
      
        id      = a[0]
        type    = a[1]
        dir = File.join new_locations[type], id
        models = Dir.glob(File.join(dir, "*.skp"))
        next if models.empty?
        
        # Create new empty model.
        Sketchup.file_new
        model    = Sketchup.active_model
        defs     = model.definitions
        entities = model.entities
        
        # Place different models as components.
        models.each do |model_path|
          basename = File.basename model_path
          type =
            case basename
            when "extrusion_profile.skp"
              "extrusion_profile"
            when /\Aalong_(\d*.*\d*)(_.*)?.skp\Z/
              distance = /\Aalong_(\d*.*\d*)(_.*)?.skp\Z/.match(basename)[1].to_f.m
              override_half_end_distance = basename.include? "wholeDistanceIn"
              "arrayed"
            when "ending.skp", "ending_railroad.skp"
              ending_type = nil
              "ending"
            when "ending_track.skp"
              ending_type = "track"
              "ending"
            when "ending_track_type.skp"
              ending_type = "type"
              "ending"
            else
              next
            end
          origin =
            case type
            when "extrusion_profile"
              ORIGIN
            when "arrayed"
              y = distance
              y /= 2 unless override_half_end_distance
              Geom::Point3d.new 0, y, 0
            when "ending"
              Geom::Point3d.new 0, -1.m, 0
            end
          trans = Geom::Transformation.new origin
          group = entities.add_group
          group.transformation = trans
          component_def = defs.load model_path
          component_inst = group.entities.add_instance component_def, Geom::Transformation.new
          component_inst.explode
          group.set_attribute ATTR_DICT_PART, "type", type
          group.set_attribute ATTR_DICT_PART, "distance", distance if type == "arrayed"
          group.set_attribute ATTR_DICT_PART, "override_half_end_distance", true if type == "arrayed" && override_half_end_distance
          group.set_attribute ATTR_DICT_PART, "ending_type", ending_type if type == "ending" && ending_type
        end
        
        # Save active model.
        defs.purge_unused
        model.materials.purge_unused
        new_model_path = File.join dir, "model.skp"
        if Sketchup.version < "15"
          model.save new_model_path
        else
          model.save new_model_path, Sketchup::Model::VERSION_8
        end
        
        Sketchup.file_new
        
      end
    end
    
    puts "Done."
    Observers.enable
    
    nil
    
  end

  # Public: Create/edit template info, e.g. name and description, for the
  # currently opened model if it's used as a template.
  #
  # Returns nothing.
  def self.edit_properties
  
    model = Sketchup.active_model
  
    # Check if model is used for a template and if so, what template.
    not_saved_msg =
      "Cannot find template using this model.\n\n"\
      "Model must be saved by the name model.skp in the a directory named as its ID inside the directory for a specific template type."
    model_use = self.model_use
    unless model_use
      UI.messagebox not_saved_msg
      return
    end
    
    # Load template info if already saved or base on defaults.
    info = self.info model_use[:type], model_use[:id]
    info ||= DEFAULT_INFO[model_use[:type]]
    info.delete :id
    
    # Add info from model content.
    case model_use[:type]
    when :track_formation
      # Get track placement data and balises from model.
      
      tracks = model.entities.map { |e| Track.get_from_group e }.compact
      if tracks.empty?
        UI.messagebox "Cannot save track formation without tracks. Please add some tracks to model."
        return
      end
      info[:placement_data] = tracks.map do |t|
        # Points and vectors need to be saved as arrays of floats.
        # to_a seams to turn lengths into floats, although undocumented, but to be sure, convert to float again.
        controls = t.controls.map { |c| c.to_a.map { |i| i.to_f } }
        {
          :controls        => controls,
          :curve_algorithm => t.curve_algorithm
        }
      end
      balises = Balise.instances.select { |b| b.track.model == model }
      info[:balises] = balises.map { |b| [b.point.to_a.map { |i| i.to_f }, b.code] }
      
    when :r_stock
      # Get placement data from rolling stocks.
      
      model = Sketchup.active_model
      r_stocks = model.entities.map { |e| RStock.get_from_group e }.compact
      r_stocks = r_stocks.sort_by { |rs| rs.train_position }
      info[:placement_data] = self.r_stock_placement_data r_stocks
      
    end
    
    # Prompt user for info that isn't an array or hash.
    # Lengths are saved as floats (inches) internally but written to and parsed
    # from proper strings in inputbox.
    lengths = {
      :signal_type    => [:distance_to_track],
      :structure_type => [:track_parallel_distance, :height_offset]
    }
    no_signal_display_name = "<NO SIGNALS>"
    prompts = []
    info.each_pair do |k, v|
      next if v.is_a? Array
      next if v.is_a? Hash
      is_length = lengths[model_use[:type]] && lengths[model_use[:type]].include?(k)
      if is_length
        # Show length as human readable string.
        # When length is irrelevant its stored as nil and shown as empty string.
        klass = Length
        v = v.nil? ? "" : v.to_l
      elsif model_use[:type] == :structure_type && k == :corresponding_signal_type && v == false
        # When signal type is set to no signal (not unknown) it is saved as false but displayed as a certain string that cannot be an ID. 
        v = no_signal_display_name
      end
      prompts << [k, v, is_length]
    end
    keys_s   = prompts.map { |i| i[0].to_s }
    values_s = prompts.map { |i| i[1].to_s }
    title = "Edit Properties (#{model_use[:type]}, #{model_use[:id]})"
    input = UI.inputbox(keys_s, values_s, title)
    return unless input
    input.each_with_index do |v, i|
      k = prompts[i][0]
      if prompts[i][2]
        v = v == "" ? nil :  v.to_l.to_f# Empty string is turned to nil when field resembles length.
      elsif model_use[:type] == :structure_type && k == :corresponding_signal_type && v == no_signal_display_name
        v = false
      end
      info[k] = v
    end
    
    # Save info.
    file_path = File.join(DIRS[model_use[:type]], model_use[:id], "info")
    EneRailroad.object_2_file file_path, info

    nil
    
  end
  
  # Public: Returns the info hash used by a specific template, e.g. a 
  # track or structure type.
  #
  # type - Template type identifier.
  # id   - The string identifier of the template.
  #
  # Returns info hash.
  def self.info(type, id)
  
    self.list_installed(type).find { |i| i[:id] == id }
  
  end
  
  # Public: lists installed templates by a specific type.
  # 
  # type - The template type identifier.
  # json - If true the list is returned as a JSON string instead of a hash array
  #        (default: false)
  # 
  # Returns array with hashes containing template information, or the
  # corresponding JSON string.
  def self.list_installed(type, json = false)
  
    raise ArgumentError, "Unknown type '#{type}'." unless TYPES.include? type
    
    list = []
    
    dir = DIRS[type]
    Dir.glob(File.join(dir, "*")) do |subdir|
      next unless File.directory? subdir
      info_path = File.join subdir, "info"
      next unless File.exist? info_path
      info = EneRailroad.file_2_object info_path
      info[:id] = File.basename subdir
      
      # Turn track formation points and vectors back into points and vectors.
      if type == :track_formation
        info[:placement_data].each do |t|
          t[:controls] = (
            (t[:controls][0..1].map { |c| Geom::Point3d.new c }) +
            t[:controls][2..3].map { |c| Geom::Vector3d.new c }
          )
        end
      end
      
      list << info
    end
    
    # Sort by name.
    list = list.sort_by { |i| (i[:name] || i["name"]).downcase }
    
    # If any template has the id "default", move it to top.
    default = list.find { |i| i[:id] == "default"}
    if default
      list.delete default
      list.unshift default
    end
    
    return list unless json
    
    EneRailroad.object_2_json list
  
  end

  # Public: If model represents an installed template, get template type and id,
  # otherwise nil, based on model save location.
  #
  # model - Model object or String path to model.
  #
  # Returns hash containing :type and :id, or nil when model isn't a template.
  def self.model_use(model = Sketchup.active_model)

    path = model.is_a?(String) ? model : model.path
    return if path.empty?
    path = File.expand_path path
    
    # In SU 2013 PLUGIN_DIR uses 8.3 filename.
    # Only check path from with plugin's folder instead of full path.
    regex = /#{ID}\/templates\/([^\/]*)\/([^\/]*)\/model.skp/
    matchdata = regex.match path
    return unless matchdata
    
    type = DIR_BASENAMES.invert[matchdata[1]]
    return unless type
    
    {:type => type, :id => matchdata[2]}
    
  end

  # Public: Create preview image.
  # Can either be run for a rolling stock template model or from one of the
  # preview models in the other template types' directories.
  #
  # Rolling stock previews are created directly from the rolling stock model
  # but other templates' model doesn't resemble how they look when drawn, e.g.
  # extruded, so these previews need to be created by assigning the template
  # to an element in an external model.
  #
  # Returns nothing.
  def self.create_preview
  
    model = Sketchup.active_model
  
    # In SU 2013 PLUGIN_DIR uses 8.3 filename.
    # Only check path from with plugin's folder instead of full path.
    regex = /#{ID}\/templates\/([^\/]*)\/preview.skp/
    path = model.path
    path = File.expand_path path
    md = regex.match path

    if md && (type = DIR_BASENAMES.invert[md[1]]) && type != :r_stock
      # In preview model.
      
      dir = DIRS[type]
      
      # Get ID.
      # TODO: suggest ID based on model content.
      # Make list containing all IDs that has a corresponding directory
      # (checking if installed wont include those missing the info file).
      options = Dir.glob(File.join(dir, "*")).select { |d| File.directory? d }.map { |d| File.basename d }
      input = UI.inputbox ["ID"], [options.first], [options.join("|")], "Select ID"
      return unless input
      id = input[0]      

      # Save image.
      img_path = File.join dir, id, "image.png"
      model.active_view.write_image img_path, 150, 100, true
      
    elsif (mu = self.model_use) && mu[:type] == :r_stock
      # In rolling stock model.
      
      id = mu[:id]
      view = model.active_view
      view.camera.set Geom::Point3d.new(ORIGIN), Geom::Point3d.new(ORIGIN.offset RSTOCK_CAM_ANGLE), Z_AXIS
      view.zoom_extents
  
      # Save image
      img_path = File.join DIRS[:r_stock], id, "image.png"
      view.write_image img_path, 300, 300, true
      
    else
      # Not in a model preview can be created from.
      UI.messagebox "Invalid model.\n\n"\
        "Preview image must either be created from a model used by a rolling "\
        "stock template or a preview model in the directory for any other "\
        "template type."
      return
    end
    
    puts "Preview created."
    
    nil
    
  end
  
  # Public: Open a directory containing a specific template type in file
  # browser.
  #
  # type - The template type identifier.
  #
  # Returns nothing.
  def self.open_dir(type)
  
    raise ArgumentError, "Unknown type '#{type}'." unless TYPES.include? type
    EneRailroad.open_dir DIRS[type]
    
    nil
    
  end

  # Public: Save template from selected initialized rolling stock(s).
  # Each template may contain several actual rolling stock, e.g. a steam
  # locomotive along with its tender, however they must be connected when
  # calling this method.
  #
  # Initializing rolling stocks from groups can be done inside any other model
  # with a graphic user interface under the Add Rolling Stock tool. Once
  # rolling stock is initialized it can be saved as a template with this method.
  #
  # r_stocks - Array of RStock objects or single RStock
  # (default: rolling stocks from selection).
  #
  # Yields ID string of newly created rolling stock once its saved.
  #
  # Returns nothing.
  def self.save_selected_r_stocks(r_stocks = nil, &block)

    if r_stocks
      r_stocks = [r_stocks] unless r_stocks.is_a?(Array)
      model = r_stocks.first.model
    else
      model = Sketchup.active_model
      ss = Sketchup.active_model.selection
      
      # Check that selection only contains rolling stocks.
      if !ss.any? { |e| e.class == Sketchup::Group && e.attribute_dictionary(RStock::ATTR_DICT) }
        UI.messagebox(S.tr("At least one initialized rolling stock must be selected."))
        return
      end
      r_stocks = ss.map { |g| RStock.get_from_group g }
      r_stocks.compact!
    end
    
    # Validate that all rollings stocks are connected.
    r_stocks = r_stocks.sort_by { |rs| rs.train_position }
    if r_stocks.map { |rs| rs.train }.uniq.length > 1 ||# Not in same train.
    r_stocks.map { |rs| rs.train_position } != (r_stocks[0].train_position..r_stocks[-1].train_position).to_a# Not next to each others.
      UI.messagebox(S.tr("If several rolling stocks are to be saved as one (e.g. steam locomotive along with tender) they need to be connected."))
      return
    end
    
    # Create web dialog asking for rolling stock info.
    dlg = UI::WebDialog.new(S.tr("Save Rolling Stock to Library"), false, "EneRailroad_r_stock_2_library", 450, 450, 500, 100, false)
    dlg.navigation_buttons_enabled = false
    dlg.set_background_color dlg.get_default_dialog_color
    dlg.set_file(File.join(DIALOG_DIR, "r_stock_2_library", "index.html"))

    # Translate strings.
    js = S.tr_dlg

    # Add default values.
    js << "document.getElementById('name').value='#{r_stocks[0].train_name}';"
    js << "document.getElementById('author').value='#{Sketchup.read_default(ID, "last_r_stock_author") || ENV["USER"] || ENV["USERNAME"] || ""}';"
    author_prefixes = Sketchup.read_default ID, "author_prefixes"
    author_prefixes = author_prefixes ? Hash[*author_prefixes.zip().flatten] : {}
    js << "user_prefixes=#{EneRailroad.object_2_json author_prefixes};"
    js << "guess_id();"
    
    js << "init_radio_buttons();"

    # Show dialog
    if WIN
      dlg.show { dlg.execute_script js }
    else
      dlg.show_modal { dlg.execute_script js }
    end
    
    # Change style of web dialog into a toolbox (windows only).
    WinApi.dialog2toolbox
    
    # Save rolling stock.
    dlg.add_action_callback("save") { |dialog, callbacks|

      # Get and validate data.
      info = {}
      info[:name] = dlg.get_element_value("name")
      if info[:name] !~ /\S/
        UI.messagebox(S.tr("No name given."))
        next
      end
      info[:author] = dlg.get_element_value("author")
      if info[:author] !~ /\S/
        UI.messagebox(S.tr("No author given."))
        next
      end
      info[:id] = dlg.get_element_value("id")
      if info[:id] !~ /\S_\S/
        UI.messagebox(S.tr("Invalid ID. ID must contain author prefix and a name separated by an underscore (_)."))
        next
      end
      if info[:id] =~ /[^A-Za-z0-9_]/
        UI.messagebox(S.tr("Invalid ID. Only A-Z, a-z, 0-9 and _ are allowed characters."))
        next
      end
      dir_path = File.join(Template::DIRS[:r_stock], info[:id])
      if File.exists? dir_path
        UI.messagebox(S.tr("Invalid ID. Already taken."))
        next
      end    
      info[:type] = dlg.get_element_value("type")
      info[:country] = dlg.get_element_value("country")
      info[:company] = dlg.get_element_value("company")
      year_regex = /^(\d{4}|\??)$/
      info[:production_start] = dlg.get_element_value("production_start")
      if info[:production_start] !~ year_regex
        UI.messagebox(S.tr("Production start should be either a 4 digit year, a question mark or be left blank."))
        next
      end
      info[:production_end] = dlg.get_element_value("production_end")
      if info[:production_end] !~ year_regex
        UI.messagebox(S.tr("Production end should be either a 4 digit year, a question mark or be left blank."))
        next
      end
      info[:retired] = dlg.get_element_value("retired")
      if info[:retired] !~ year_regex
        UI.messagebox(S.tr("Retired should be either a 4 digit year, a question mark or be left blank."))
        next
      end
      
      #Get placement_data
      info[:placement_data] = self.r_stock_placement_data r_stocks
      
      Sketchup.set_status_text(S.tr("Saving..."), SB_PROMPT)
    
      Observers.disable
      model.start_operation(S.tr("Save Rolling Stock to Library"))
      
      #Set camera
      view = model.active_view
      cam = view.camera
      cam_backup = [cam.eye, cam.target, cam.up]
      cam.set Geom::Point3d.new(ORIGIN), Geom::Point3d.new(ORIGIN.offset RSTOCK_CAM_ANGLE), Z_AXIS
      
      #Create folder
      Dir.mkdir dir_path unless File.exists? dir_path
      
      # Copy to component definition and save it as file
      cd = model.definitions.add info[:id]
      ents = cd.entities
      point_front = Geom::Point3d.new
      r_stocks.each_with_index do |rs, i|
        rs_group_definition = rs.group.entities.parent
        
        #Group position in this file does not matter other than for the preview image.
        #Insert rs tool relies on train_position attribute of groups and the placement data in the info file.
        trans = Geom::Transformation.axes(point_front.offset([1,0,0], rs.distance_buffers[0]), [-1, 0, 0], [0, -1, 0], [0, 0, 1])
        point_front.offset! [1,0,0], rs.length_over_couplings
        trans = Geom::Transformation.axes(point_front.offset([-1,0,0], rs.distance_buffers[1]), [1, 0, 0], [0, 1, 0], [0, 0, 1]) if rs.reversed
        
        #Copy rolling group to component definition
        rs_group_in_cd = ents.add_instance rs_group_definition, trans
        rs_group_in_cd.material = rs.group.material
        
        #Copy attributes. Points will still be where rolling stocks were located in model but those values are overridden when placed.
        rs.group.attribute_dictionary(RStock::ATTR_DICT).each_pair do |key, value|
          rs_group_in_cd.set_attribute RStock::ATTR_DICT, key, value
        end
        
        #Override position in train with position relative to those selected
        rs_group_in_cd.set_attribute RStock::ATTR_DICT, "train_position", i
        
        #Override reversed value and buffer distances(this may not yet be written to the group)
        rs_group_in_cd.set_attribute RStock::ATTR_DICT, "reversed", rs.reversed
        rs_group_in_cd.set_attribute RStock::ATTR_DICT, "distance_buffers", rs.distance_buffers
        
      end
      
      cd.refresh_thumbnail
      cd.save_as File.join(dir_path, "model.skp")#TODO: save as version 8 when available for components.
      cd.save_thumbnail File.join(dir_path, "image.png")
      
      # Reset camera position.
      cam.set cam_backup[0], cam_backup[1], cam_backup[2]
      
      #Save info
      info_path = File.join(dir_path, "info")
      EneRailroad.object_2_file info_path, info
      
      # Save author name so it can be suggested next time.
      Sketchup.write_default ID, "last_r_stock_author", info[:author]
      
      # Save prefix for author.
      # Author key is turned into lowercase so case insensitive check can be done more easily when getting prefix from username.
      author_prefixes = Sketchup.read_default ID, "author_prefixes"
      author_prefixes = author_prefixes ? Hash[*author_prefixes.zip().flatten] : {}
      author_prefixes[info[:author].downcase] = info[:id].match(/^([A-Za-z0-9]*)/)[0]
      author_prefixes = author_prefixes.to_a
      Sketchup.write_default ID, "author_prefixes", author_prefixes
      
      model.commit_operation
      Observers.enable
      
      Sketchup.set_status_text(S.tr("Saved"), SB_PROMPT)
      
      # Close dialog.
      dlg.close
      
      # Execute block passed to method.
      block.call(info[:id]) if block
    
    }
    
    # Close web dialog.
    dlg.add_action_callback("close") {
      dlg.close
    }
    
    #NOTE: open help file similar to balise code editor?
    
    nil
    
  end

  # Internal: Get placement data for rolling stocks.
  # The placement data is an array containing hashes holding :length_wheelbase
  #(floats) and :distance_buffers (array of 2 floats).
  def self.r_stock_placement_data(r_stocks)
  
    r_stocks.map do |rs|
      {
        :length_wheelbase => rs.points[0].distance(rs.points[1]).to_f,
        :distance_buffers => rs.distance_buffers.map { |i| i.to_f }
      }
    end
    
  end
  
end# Module

end# Module