# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolStructureInsert
  #This tool is used to add a structure along a track, e.g. a bridge or tunnel.
    
  @@structure_type = nil
  
  #Sketchup tool definitions
  
  def initialize
  
    @state = 0
    @path
    @start#-point
    @end#-point
    @ip
    
    #Track tool is locked to
    @track_locked_on

    #Only affects look of tool, not drawn results
    @default_vector_length = 10.m

    #Statusbar texts
    @status_text_0 = S.tr "Click on track to set start position. Shift = Lock on hovered track."
    @status_text_1 = S.tr "Click on track to set end position or type in exact length."
    @vcb_label = S.tr "Length"
    
    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "structure_insert.png"), 2, 2)
    
  end
  
  def onSetCursor

    UI.set_cursor @cursor

  end
  
  def enableVCB?
  
    @state == 1
    
  end
  
  def activate
  
    @ph = Sketchup.active_model.active_view.pick_helper
    @ip = Sketchup::InputPoint.new
    
    #Prevents select tool from being selected every time this tool is deactivated. This should only happen when dialog is manually closed
    @tool_active = true
    
    @status_text = @status_text_0
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
    
    #When dialog is closed the select tool is switched to to deactivate this tool.
    #Deactivating the tool closes the dialog.
    #However selecting another tool, which deactivates this, should not cause the select tool from being selected.
    @tool_active = true

    #Open web dialog with tool settings
    self.init_web_dialog
    
  end

  def deactivate(view)

    @tool_active = false
    @dlg.close
    view.invalidate

  end

  def draw(view)
  
    color = Sketchup::Color.new(255, 194, 42)
    view.line_width = 1
    
    if @state == 1 && @path
    
      #Path
      view.line_width = 1
      view.drawing_color = Sketchup::Color.new("White")
      MyView.draw_polyline_on_top view, @path
      
      #End point
      view.draw_points [@path[-1]], 7, 4, Sketchup::Color.new("White")
      
      #Start vector
      vector = @path[1] - @start
      if vector.valid?
        tip = @start.offset vector, @default_vector_length
        view.line_width = 3
        view.drawing_color = color
        view.draw_line @start, tip
      end
      
    end
  
    if @start
      #Start point
      view.draw_points [@start], 10, 2, color
    end
    
    @ip.draw view if @ip.valid?
  
  end
  
  def onCancel(flag, view)
  
    @state = 0
    @start = nil
    @status_text = @status_text_0
    
    Sketchup.set_status_text @status_text, SB_PROMPT
    Sketchup.set_status_text "", SB_VCB_VALUE
    
    view.invalidate
      
  end
  
  def onMouseMove(flags, x, y, view)
  
    return unless @@structure_type
  
    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y
  
    @start = nil if @state == 0
    @end = nil
    @path = nil
    
    q = Track.inspect_point @ip.position, nil, @track_locked_on
      
    if @state == 0
      #Select start point
      @start = q[:point]
    else
      #Select end point and get path
      @end = q[:point]
      
      @path = Track.path_between_points(@start, @end)
      @path = @path[:path] if @path
      
      #Write current distance to VCB
      l = @path ? MyGeom.path_length(@path) : ""
      Sketchup.set_status_text l.to_s, SB_VCB_VALUE
      
    end
  
    view.invalidate
    
  end
  
  def onLButtonDown(flags, x, y, view)
  
    return unless @@structure_type
    
    if @state == 0
      @state = 1
      @status_text = @status_text_1
    elsif @path
      self.insert
      @state = 0
      @start = nil
      @status_text = @status_text_0
    end
    
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate
    
  end

  def onUserText(text, view)
  
    return if @state == 0
    return unless @path
    return if @path.empty?
    
    begin
      distance = text.to_l
    rescue
      UI.messagebox(S.tr("Invalid length."))
      return
    end
    
    current_length = MyGeom.path_length @path
    
    if distance < current_length
      #Draw shorter than current path, truncate
      @path = MyGeom.path_truncate @path, distance
      
    else
      #Draw longer, start at path end and calculate new
      missing_length = distance - current_length
      @path += Track.extract_path(@path[-1], missing_length, @path[-1] - @path[-2])[1..-1]
    end
    
    self.insert
    @state = 0
    @start = nil
    @status_text = @status_text_0
    
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate

  end
  
  def onKeyDown(key, repeat, flags, view)

    #Lock on track (closest)
    if key == CONSTRAIN_MODIFIER_KEY
      #picked = @ph.best_picked
      #t = Track.get_from_group picked
      q = Track.inspect_point @ip.position, nil
      t = q[:track]
      @track_locked_on = t if t
      return true
    end

  end
  
  def onKeyUp(key, repeat, flags, view)
  
    #Unlock from track
    if key == CONSTRAIN_MODIFIER_KEY
      @track_locked_on = nil
      return true
    end
    
  end
  
  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL)
    view.invalidate
    
  end
  
  def getMenu(menu)
  
    e = Sketchup.active_model.selection[0]#NOTE: doesn't select what's clicked.
    if e && type = e.get_attribute(Structure::ATTR_DICT, "structure_type")
      menu.add_item("Copy Structure Type") {
        @@structure_type = type
        js = "document.getElementById('structure_type').value='#{@@structure_type}';"
        js << "window.update_structure_type_info();"
        @dlg.execute_script js
      }
   end
   
 end
 
  #Own definitions
  #Not called from Sketchup itself
  
  def init_web_dialog
    #Open web dialog with tool settings

    @dlg = UI::WebDialog.new(S.tr("Add Structure"), false, "#{ID}_insert_structure", 300, 180, 500, 100, false)
    @dlg.navigation_buttons_enabled = false
    @dlg.set_background_color @dlg.get_default_dialog_color
    @dlg.set_file(File.join(DIALOG_DIR, "insert_structure", "index.html"))
    
    #Translate strings
    js = S.tr_dlg
    
    #Create list of structure types
    structure_types = Template.list_installed :structure_type
    track_types = Template.list_installed :track_type
    structure_types.each do |st|
      #Make human readable strings from inch measurements
      if !st[:track_parallel_distance] || st[:track_parallel_distance] == 0 || st[:track_parallel_distance] == ""
        st.delete :track_parallel_distance
      else
        st[:track_parallel_distance] = st[:track_parallel_distance].to_l.to_s
      end
      #Show name, not ID of corresponding track
      if !st[:corresponding_track_type] || st[:corresponding_track_type] == ""
        st.delete :corresponding_track_type
      else
        st[:corresponding_track_type] = track_types.find{ |tt| tt[:id] == st[:corresponding_track_type] }[:name]
      end
    end
    structure_types_json = EneRailroad.object_2_json structure_types
    
    #Form data
    js << "var structure_types=#{structure_types_json};"
    js << "window.init_dynamic_lists();"
    js << "document.getElementById('structure_type').value='#{@@structure_type}';" if @@structure_type
    
    #Init scripts
    js << "push_changes();"
    js << "form = document.forms[0];"
    js << "form.onclick = push_changes;"
    js << "form.onkeyup = push_changes;"
    js << "window.update_structure_type_info();"
    js << "document.getElementById('structure_type').onchange = update_structure_type_info;"
    
    #Show dialog
    if WIN
      @dlg.show { @dlg.execute_script js }
    else
      @dlg.show_modal { @dlg.execute_script js }
    end

    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox
    
    #Form value changes
    @dlg.add_action_callback("push_changes") {
      @@structure_type = @dlg.get_element_value("structure_type")
    }

    #Deselct tool when webdialog closes
    @dlg.set_on_close {
      Sketchup.send_action "selectSelectionTool:" if @tool_active
    }
    
  end
  
  def insert
    
    return unless @path
    return unless @@structure_type

    # HACK: Preload component definitions before starting operation not to break
    # it in older SU versions.
    if Sketchup.version < "14"
      Template.component_def :structure_type, @@structure_type
    end
    
    Observers.disable
    Sketchup.set_status_text S.tr("Drawing Structure"), SB_PROMPT
    Sketchup.active_model.start_operation(S.tr("Add Structure"), true)
    
    str = Structure.new
    str.path = @path
    str.structure_type = @@structure_type
    str.draw
    
    Sketchup.active_model.commit_operation
    Sketchup.active_model.active_view.invalidate
    Sketchup.set_status_text(S.tr("Done Drawing Structures"), SB_PROMPT)
    Observers.enable
  
  end
  
end#class

end#module