# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class Animate

  #All loaded animate objects. In mac these are in different models
  @@instances ||= []
  @@dlg = nil
  
  #Class variable accessors

  def self.instances; @@instances; end
  def self.instances=(value); @@instances = value; end

  #Class methods
  
  def self.get_from_model(model = nil)
  
    #Assume active model
    model = Sketchup.active_model unless model

    @@instances.find { |i| i.model == model }

  end

  #Instance attribute accessors

  attr_accessor :model
  attr_reader :go#has custom writer that also starts and stops the animation
  attr_reader :delay#has custom writer that also restarts animation timer
  attr_reader :auto_calibrate#has custom writer that also starts the timer for frequent calibrations
  attr_accessor :followed#Either train or rolling stock to follow
  attr_accessor :use_actual_time_passed#Using the actual time passed between frames to determine how far trains should move. Otherwise use the delay.
  attr_accessor :code_between_frames
  attr_accessor :time_since_last_frame

  #Instance methods

  def initialize(model)
    #Creates a new animation object. Runs when a new model loads

    @model = model
    @delay = 1.0/24
    @delay_callibrate = 5
    @go = false
    @auto_calibrate = true
    @followed = nil
    @use_actual_time_passed = true
    @code_between_frames = ""

    @@instances << self

  end

  def auto_calibrate=(auto_calibrate)
    #Changes whether delay should be automatically calibrated every second

    return if @auto_calibrate == auto_calibrate
    @auto_calibrate = auto_calibrate

    if @auto_calibrate
      self.calibrate
      @timer_calibrate = UI.start_timer(@delay_callibrate, true){ self.calibrate }
    else
      UI.stop_timer(@timer_calibrate) if @timer_calibrate
    end

  end

  def calibrate
    #Updates delay to average drawing time

    self.delay = Sketchup.active_model.active_view.average_refresh_time

  end

  def delay=(delay)
    #Attribute writer to change delay

    @delay = delay
    if @go
      UI.stop_timer(@timer)
      @timer = UI.start_timer(@delay, true){ self.frame }
    end

  end

  def frame
    #Main animation method. Runs every @delay second and moves the trains forward.

    #Stop animation if model isn't active
    unless @model == Sketchup.active_model
      self.stop
      return
    end#unless

    #Get time difference since last frame (delay + actual calculating and drawing time)
    time_now = Time.now

    #Get followed rolling stock
    if Train === @followed
      r_stock_followed = @followed.r_stocks[0]
    else
      r_stock_followed = @followed
    end
    #NOTE: BETTER CODE: adapt code so this can be a train. move focus method from drive train class to animate class
    
    #Copy transformation of r_stock_followed
    ts_old = r_stock_followed.group.transformation if r_stock_followed

    if @use_actual_time_passed
      delta_t = time_now - @time_last
    
      #Limit time difference to 5 times the delay time
      #so train don't move too much when timer has paused due to view being updated by other tool.
      delta_t = 5*@delay if delta_t > 5*@delay
    else
      delta_t = @delay
    end
    
    @time_since_last_frame = delta_t
    
    #Move trains
    Train.instances.select { |i| i.model == @model}.each do |train|
      train.frame delta_t
    end

    #Move/Update camera
    if r_stock_followed
      #Move camera along with followed train

      ts_new = r_stock_followed.group.transformation
      ts = ts_new*(ts_old.inverse)

      cam = @model.active_view.camera
      cam.set(cam.eye.transform(ts), cam.target.transform(ts), cam.up.transform(ts))
    else
      #Update view
      @model.active_view.invalidate
    end
    
    #Run custom code, e.g. for saving animation
    unless @code_between_frames == ""
      begin
        lambda do#Wrap in lamda so return works inuitive.
          eval @code_between_frames
        end.call
      rescue Exception => e
        self.stop
        UI.messagebox "#{S.tr("Error executing code between frames:")}\n\n#{e.message}"
      end
    end

    @time_last = time_now

  end

  def go=(go)
    #Attribute writer that starts or stops animation
    
    #Invalidate view so Drive Tool can change text showing if paused or not
    @model.active_view.invalidate

    return if @go == go
    @go = go

    if @go
      @time_last = Time.now
      @timer = UI.start_timer(@delay, true){ self.frame }
      @timer_calibrate = UI.start_timer(@delay_callibrate, true){ self.calibrate } if @auto_calibrate
    else
      UI.stop_timer(@timer)
      UI.stop_timer(@timer_calibrate) if @auto_calibrate
    end
    
    #Update play/pause in settings dialog if open
    if @@dlg && @@dlg.visible?
      js = "document.getElementById('play').value='#{@go ? "true" : "false"}';"
      js << "update_play_pause_button();"
      @@dlg.execute_script js
    end
    
    true

  end

  def inspect
    #Do not show attributes when inspecting. It clutters the console
    self.to_s
  end
  
  def settings_dialog
    #Show dialog with animation settings
    
    #Create web dialog
    @@dlg = UI::WebDialog.new(S.tr("Advanced Animation Settings"), false, "#{ID}_animation_settings", 440, 430, 300, 100, false)
    @@dlg.navigation_buttons_enabled = false
    @@dlg.set_background_color @@dlg.get_default_dialog_color
    @@dlg.set_file(File.join(DIALOG_DIR, "animation", "index.html"))
    
    #Translate strings
    js = S.tr_dlg
    
    js << "document.onkeyup=check_if_esc_and_close;"
    
    #Add current data to form
    js << "document.getElementById('play').value='#{@go ? "true" : "false"}';"
    js << "update_play_pause_button();"
    
    code = @code_between_frames.gsub("\n", "\\n\\r").gsub("\r", "").gsub("\"", "\\\"")#escape special characters
    js << "document.getElementById('code_editor').value=\"#{code}\";"
    js << "document.getElementById('code_editor').focus();"

    js << "document.getElementById('delay').value=\"#{@delay}\";"
    js << "document.getElementById('auto_calibrate').checked=#{@auto_calibrate ? "true" : "false"};"
    
    js << "document.getElementsByName('use_actual_time_passed')[#{@use_actual_time_passed ? "0" : "1"}].checked=true;"

    #Show dialog
    if WIN
      @@dlg.show { @@dlg.execute_script js }
    else
      @@dlg.show_modal { @@dlg.execute_script js }
    end
    
    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox
    
    #NOTE: Make callbacks affect open model and update dialog when model is changed.
    
    #Play/Pause
    @@dlg.add_action_callback("playPause") {
    
      self.go = @@dlg.get_element_value("play") == "true"
    }
    
    #Set code
    @@dlg.add_action_callback("set_code") {
      self.stop
      @code_between_frames = @@dlg.get_element_value("code_editor")
    }
    
    #Set delay
    @@dlg.add_action_callback("set_delay") { |_, delay|
      delay.gsub!(",", ".")
      next if delay !~ /^\d+\.?\d*$/
      self.auto_calibrate = false
      self.delay = delay.to_f
    }
    
    #Set auto calibrate
    @@dlg.add_action_callback("set_auto_calibrate") { |_, auto_calibrate|
      self.auto_calibrate = auto_calibrate == "true"
      #Calibrate method could be sending this info to form each time it runs but perhaps it would be annoying with constantly changing numbers.
      if @auto_calibrate
        js = "document.getElementById('delay').value=\"#{@delay}\";"
        @@dlg.execute_script js
      end
    }
    
    #Set use_actual_time_passed
    @@dlg.add_action_callback("set_use_actual_time_passed") { |_, use_actual_time_passed|
      @use_actual_time_passed = use_actual_time_passed == "true"
    }
    
    #Close web dialog
    @@dlg.add_action_callback("close") {
      @@dlg.close
    }

    #Show help file
    @@dlg.add_action_callback("show_help") {
      UI.openURL("file://" + File.join(DOCS_DIR, "animate_settings.html"))
    }
    
  end
  
  def start
    #Shortcut for starting animation

    self.go = true

  end

  def stop
    #Shortcut for stopping animation

    self.go = false

  end

end#class

end#module
