/* 
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is the Sablotron XSLT Processor.
 * 
 * The Initial Developer of the Original Code is Ginger Alliance Ltd.
 * Portions created by Ginger Alliance are Copyright (C) 2000 Ginger
 * Alliance Ltd. All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL"), in which case the provisions of the GPL are applicable 
 * instead of those above.  If you wish to allow use of your 
 * version of this file only under the terms of the GPL and not to
 * allow others to use your version of this file under the MPL,
 * indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by
 * the GPL.  If you do not delete the provisions above, a recipient
 * may use your version of this file under either the MPL or the
 * GPL.
 */

#include "tree.h"
#include "proc.h"
#include "guard.h"
#include "key.h"

// GP: clean (no E())

/****************************************
*                                       *
T r e e
*                                       *
****************************************/

/****************************************
T r e e   methods
****************************************/

Tree::Tree(const Str &aname, BOOL aXSLTree)
: theArena(TREE_ARENA_SIZE), 
  theDictionary(&theArena, TREE_DICT_LOGSIZE)
{
    // theDictionary.initialize();
    QName &rootname = (QName&) getTheEmptyQName();
    root = new(&theArena) RootNode(*this, rootname);
    name = aname;
    XSLTree = aXSLTree;
    stackTop = &getRoot();
    pendingTextNode = NULL;
    getRoot().stamp = 0;
    vcount = 1;
    stripped = 0;
    QName dummyName;
    theDummyElement = new(&theArena) Element(*this, dummyName);
    initDict();
};

Tree::~Tree()
{
  // rulesList.freeall(FALSE); - done in ~RuleSList
    aliasesList.freeall(FALSE);
    getRoot().~RootNode();
    // this is just in case there's a processing error:
    pendingNS.freeall(FALSE);
    delete theDummyElement;
}

void Tree::initDict()
{
    theDictionary.initialize();
    theDictionary.insert("", stdPhrases[PHRASE_EMPTY]);
    theDictionary.insert("xsl", stdPhrases[PHRASE_XSL]);
    theDictionary.insert(theXSLTNamespace, stdPhrases[PHRASE_XSL_NAMESPACE]);
    theDictionary.insert(theXMLNamespace, stdPhrases[PHRASE_XML_NAMESPACE]);
    theDictionary.insert("*", stdPhrases[PHRASE_STAR]);
    theDictionary.insert("xmlns", stdPhrases[PHRASE_XMLNS]);
}

Element &Tree::dummyElement() const
{
    return *theDummyElement;
}

void Tree::flushPendingText()
{
    if (pendingTextNode)
        pendingTextNode -> cont = pendingText;
    pendingText.empty();
    pendingTextNode = NULL;
}
    
eFlag Tree::appendVertex(Sit S, Vertex *v)
{
    assert(stackTop && isDaddy(stackTop));
    assert(!isText(v) || !pendingTextNode);
    if (!isText(v))
        flushPendingText();
    E( cast(Daddy*,stackTop) -> newChild(S, v) ); //sets parent too
    if (isDaddy(v))
        stackTop = v;
    v -> stamp = vcount++;
    return OK;
};

void Tree::dropCurrentElement(Vertex *v)
{
    assert(stackTop && isElement(stackTop));
    assert(stackTop == v);
    assert(!pendingTextNode);
    stackTop = v -> parent;
    delete v;
    toE(stackTop) -> contents.deppend();
}

Vertex *Tree::appendText(Sit S, char *string, int len)
{
    Vertex *txt = NULL;
    if (!pendingTextNode)
    {
        // the initializing text does not matter
        txt = new(&getArena()) Text(*this, string, len);
        appendVertex(S, txt);
        pendingTextNode = toText(txt);
    }
    pendingText.nadd(string,len);
    return txt;
}

Vertex *Tree::popVertex()
{
    Vertex *v = NZ( stackTop );
    stackTop = v -> parent;
//    flushPendingText(); - moved to tcEndElement()
    return v;
}

eFlag Tree::parseFinished(Sit S)
{
    flushPendingText();
    return OK;
}

void Tree::setName(const Str& _name)
{
    name = _name;
}

HashTable& Tree::dict() 
{
    return theDictionary;
}

Arena& Tree::getArena()
{
    return theArena;
}

const QName& Tree::getTheEmptyQName() const
{
    return theEmptyQName;
}


Bool Tree::cmpQNames(const QName &first, const QName &second) const
{
/*
    printf("comparing names (%s,%s,%s) and (%s,%s,%s)\n",
        (char*)(((Tree*)this)->expand(first.getPrefix())),
	    (char*)(((Tree*)this)->expand(first.getUri())),
	    (char*)(((Tree*)this)->expand(first.getLocal())),
        (char*)(((Tree*)this)->expand(second.getPrefix())),
	    (char*)(((Tree*)this)->expand(second.getUri())),
	    (char*)(((Tree*)this)->expand(second.getLocal()))
	);
*/	
    if (first.getLocal() == stdPhrase(PHRASE_STAR))
        return (Bool)(first.getPrefix() == UNDEF_PHRASE || 
            first.getUri() == second.getUri());
    else
        return (Bool) (first.getUri() == second.getUri()
	        && first.getLocal() == second.getLocal());
}

Bool Tree::cmpQNamesForeign(const QName &q, const HashTable& dictForeign, const QName &qForeign)
{
/*
    printf("comparing names (%s,%s,%s) and (%s,%s,%s)\n",
        (char*)(((Tree*)this)->expand(q.getPrefix())),
	    (char*)(((Tree*)this)->expand(q.getUri())),
	    (char*)(((Tree*)this)->expand(q.getLocal())),
        (char*)(dictForeign.getKey(qForeign.getPrefix())),
	    (char*)(dictForeign.getKey(qForeign.getUri())),
	    (char*)(dictForeign.getKey(qForeign.getLocal()))
	);
*/	

    if (q.getLocal() == stdPhrase(PHRASE_STAR))
    {
        return (Bool)(q.getPrefix() == UNDEF_PHRASE || 
            (dict().getKey(q.getUri()) == dictForeign.getKey(qForeign.getUri())));
	}
    else
    {
        return (Bool) 
	        (dict().getKey(q.getUri()) == dictForeign.getKey(qForeign.getUri()) &&
            dict().getKey(q.getLocal()) == dictForeign.getKey(qForeign.getLocal()));		
	}
}

Bool Tree::cmpQNameStrings(const QName &q, const Str& uri, const Str& local)
{
    if (q.getLocal() == stdPhrase(PHRASE_STAR))
        return (Bool)(dict().getKey(q.getUri()) == uri);
    else
    {
        return (Bool) 
  	        (dict().getKey(q.getUri()) == uri &&
            dict().getKey(q.getLocal()) == local);		
	}
}

void Tree::expandQ(const QName& q, EQName& expanded)
{
    expanded.setLocal(expand(q.getLocal()));
    expanded.setUri(expand(q.getUri()));
    expanded.setPrefix(expand(q.getPrefix()));
}

void Tree::expandQStr(const QName& q, Str& expName)
{
    EQName expanded;
    expandQ(q, expanded);
    expanded.getname(expName);
}

const Str& Tree::expand(Phrase ph)
{
    return dict().getKey(ph);
}

Phrase Tree::unexpand(const Str& strg)
{
    Phrase result;
    dict().insert(strg, result);
    return result;
}

eFlag Tree::serialize(Sit S, char *& result)
{
    OutputterObj out;
    OutputDefinition def;
    GP( DataLine ) targetLine = new DataLine;
    // set default options for output
    EQName xmlMethod;
    xmlMethod.setLocal((char*)"xml");
    E( def.setItemEQName(S, XSLA_METHOD, xmlMethod, FALSE, NULL) );
    E( def.setDefaults(S) ); 
    E( (*targetLine).open(S, (const char*)"arg:/dummy_", DLMODE_WRITE, NULL) );
    out.setOptions(S, targetLine, &def);
    E( getRoot().serialize(S, out) );
    result = (*targetLine).getOutBuffer() -> compactToBuffer();
    E( (*targetLine).close(S) );
    targetLine.del();
    return OK;
}

eFlag Tree::serializeNode(Sit S, Element *v, char *& result)
{
    OutputterObj out;
    OutputDefinition def;
    GP( DataLine ) targetLine = new DataLine;
    // set default options for output
    EQName xmlMethod;
    xmlMethod.setLocal((char*)"xml");
    E( def.setItemEQName(S, XSLA_METHOD, xmlMethod, FALSE, NULL) );
    E( def.setDefaults(S) ); 
    E( (*targetLine).open(S, (const char*)"arg:/dummy_", DLMODE_WRITE, NULL) );
    out.setOptions(S, targetLine, &def);
    E( (*v).serializeSubtree(S, out) );
    result = (*targetLine).getOutBuffer() -> compactToBuffer();
    E( (*targetLine).close(S) );
    targetLine.del();
    return OK;
}

void Tree::makeStamps()
{
    int stamp_ = 0;
    getRoot().makeStamps(stamp_);
    vcount = stamp_;
}

RuleSList &Tree::rules()
{
    return rulesList;
}

P2List &Tree::aliases()
{
    return aliasesList;
}

Bool getWhDelimString(char *&list, Str& firstPart)
{
    skipWhite(list);
    if (!*list) return FALSE;
    char *list_was = list;
    for(; *list && !isWhite(*list); list++);
    firstPart.nset(list_was, (int)(list - list_was));
    return TRUE;
}

/*****************************************************************
Tree::processVertexAfterParse()
Performs any necessary actions on a vertex immediately after it is
parsed into the tree. The vertex is still 'current'. Typically, this
function calls popVertex.
ARGS:
    v   the vertex to be processed
The function operates only on XSL vertices, namely:
xsl:include - replaces the vertex by the newly constructed tree
xsl:output  - files the information into the tree's structures
...
Other vertices are just popped off the stack.
*****************************************************************/

eFlag Tree::processVertexAfterParse(Sit S, Vertex *v, TreeConstructer* tc)
{
    XSL_OP theOp;
    if (isXSLElement(v))
    {
        XSLElement *x = toX(v);
        switch(theOp = x -> op)
        {
        case XSL_INCLUDE:
            {
                Attribute *a = NZ( x -> atts.find(XSLA_HREF) );
                GP( Tree ) srcTree; 
		        const Str& base = S.findBaseURI(/* this-> */ name);		
		        if (S.getProcessor())
			    {
			        E( S.getProcessor() -> readTreeFromURI(
				        S, srcTree, a -> cont, base, FALSE) );
					srcTree.keep();
			    }
			    else
			    {
    			    Str absolute;
	    		    makeAbsoluteURI(a -> cont, base, absolute);
			        srcTree = new Tree(absolute, FALSE);
    			    DataLine d;
                    E( d.open(S, absolute, DLMODE_READ, /* argList = */ NULL) );
                    E( (*srcTree).parse(S, &d) );
                    E( d.close(S) );
			    }
			    
                Element *theSheet = (*srcTree).findStylesheet(
		            (*srcTree).getRoot());
                if (!theSheet)
                    Warn1(S, W_NO_STYLESHEET, (char*)(a -> cont));
                dropCurrentElement(v);

                OutputterObj source;
		        E( tc -> parseUsingSAXForAWhile(S, source) );
                if (theSheet)
                    E( theSheet -> contents.copy(S, source) );
                E( tc -> parseUsingSAXForAWhileDone(S, source) );
		        
            }; break;
        case XSL_OUTPUT:
            {
                int i, attsNumber = x -> atts.number();
                Attribute *theAtt;
                for (i = 0; i < attsNumber; i++)
                {
                    theAtt = toA(x -> atts[i]);
                    switch(theAtt -> op)
                    {
                    case XSLA_METHOD:
                        {
                            QName q;
			                EQName eq;
                            E( x -> setLogical(S, 
			                    q, theAtt -> cont, FALSE) );
					        expandQ(q, eq);
                            E( outputDef.setItemEQName(S, 
			                    XSLA_METHOD, eq, TRUE, v) );
                        }; break;
                    case XSLA_CDATA_SECT_ELEMS:
                        {
                            QName q;
                            Bool someRemains;
                            Str listPart;
                            char *p = theAtt -> cont;
                            do
                            {
                                someRemains = getWhDelimString(p, listPart);
                                if (someRemains)
                                {
                                    E( x -> setLogical(S, 
				                        q, listPart, TRUE) );
							        EQName expanded;
								    expandQ(q, expanded);
                                    E( outputDef.setItemEQName(S, 
				                        XSLA_CDATA_SECT_ELEMS, 
							            expanded, TRUE, v) );
                                };
                            }
                            while (someRemains);
                        }; break;
                    default:
		                {
                            int code = outputDef.setItemStrCheck(S, 
			                    theAtt -> op, theAtt -> cont, v);
					        if (code)
						    {
						        S.setCurrV(theAtt);
							    S.setCurrFile(name);
							}
					        E( code );
					        
					    };
                    };
                }
            popVertex();
            }; break;
        case XSL_NAMESPACE_ALIAS:
          {
             Phrase style,result;
             dict().insert(NZ( x -> atts.find(XSLA_STYLESHEET_PREFIX)) -> cont,style);
             dict().insert(NZ( x -> atts.find(XSLA_RESULT_PREFIX)) -> cont,result);
             aliases().appendConstruct(style,result);
             popVertex();
          }; break;
        case XSL_TEMPLATE:
            {
                insertRule(S, x);
                popVertex();
            }; break;
        default:
            popVertex();
        }
    }
    else
        popVertex();
    return OK;
}

/*****************************************************************
findStylesheet()
finds a xsl:stylesheet child of the given daddy. Returns NULL 
if not found.
*****************************************************************/

Element *Tree::findStylesheet(Daddy& d)
{
    Vertex *w;
    int dContentsNumber = d.contents.number();
    for (int i = 0; i < dContentsNumber; i++)
    {
        if (isElement(w = d.contents[i]))
        {
            const QName& wName = toE(w) -> name;
	        Tree& owner = w -> getOwner();
		    Str localStr;
//            if (!strcmp(wName.getUri(), theXSLTNamespace) && /* _PH_ */
            if (wName.getUri() == owner.stdPhrase(PHRASE_XSL_NAMESPACE) &&
               ((localStr = owner.expand(wName.getLocal()) == 
	               xslOpNames[XSL_STYLESHEET]) ||
               (localStr == xslOpNames[XSL_TRANSFORM])))
            return toE(w);
        }
    };
    return NULL;
}

void Tree::report(Sit S, MsgType type, MsgCode code, 
    const Str &arg1, const Str &arg2) const
{
    S.message(type, code, arg1, arg2);
}

eFlag Tree::insertRule(Sit S, XSLElement *tmpl)
{
    double prio;
    Attribute *a = tmpl -> atts.find(XSLA_PRIORITY);
    if (!a)
        prio = defaultPriority(tmpl);
    else
    {
        if (a -> cont.toDouble(prio))
            Err(S, ET_BAD_NUMBER);
    };
    QName q; 
    GP( QName ) mode = NULL;
    if (!!(a = tmpl -> atts.find(XSLA_NAME)))
        E( tmpl -> setLogical(S, q, a -> cont, FALSE) );
    if (q.getLocal() != UNDEF_PHRASE && 
        rules().findByName(*this, q))
	    {
	        Str fullName;
		    expandQStr(q, fullName);
            Err1(S, ET_DUPLICATE_RULE_NAME, fullName);
		};
    if (!!(a = tmpl -> atts.find(XSLA_MODE)))
        E( tmpl -> setLogical(S, *(mode = new QName), 
	       a -> cont, FALSE) );
    
    rules().insert(new RuleItem(tmpl,prio,q,mode.keep()));
    return OK;
}

double Tree::defaultPriorityLP(Expression *lpath)
{
    assert(lpath && lpath -> functor == EXF_LOCPATH);
    assert(lpath -> args.number());
    if ((lpath -> args.number() > 1) || lpath -> args[0] -> step -> preds.number())
        return .5;
    else
    {
        switch (lpath -> args[0] -> step -> ntype)
        {
        case EXNODE_COMMENT:
        case EXNODE_TEXT:
        case EXNODE_PI:  //// pi('literal') should have priority 0.5
        case EXNODE_NODE:
            return -0.5;
            break;
        case EXNODE_NONE:
            {
                QName &qn = lpath -> args[0] -> step -> ntest;
                if (qn.getLocal() != lpath -> getOwnerTree().stdPhrase(PHRASE_STAR))
                    return 0.0;
                else
                {   
                    if (qn.getPrefix() == UNDEF_PHRASE)
                        return -0.5;
                    else
                        return -0.25;
                };
            }; break;
        default:
            return 0.5;
        };
    };
    return 0;   // BCC thinks we don't return enough
}

double Tree::defaultPriority(XSLElement *tmpl)
{
    Expression *e = tmpl -> getAttExpr(XSLA_MATCH);
    if (!e) 
        return PRIORITY_NOMATCH;
    switch(e -> functor)
    {
    case EXF_LOCPATH:
        {
            return defaultPriorityLP(e);
        }; break;
    case EXFO_UNION:
        {
            double max=0, priority;
            BOOL first = TRUE;
            int eArgsNumber = e -> args.number();
            for (int i=0; i < eArgsNumber; i++)
            {
                priority = defaultPriorityLP(e -> args[i]);
                if (first || (priority > max)) 
                    max = priority;
                first = FALSE;
            };
            return max;
        }; break;
    default:
        {
            assert(!"expression not a union or LP");
            return 0;   // dummy
        }
    };
    return 0;       // dummy for BCC
}

eFlag Tree::parse(Sit S, DataLine *d)
{
    Log1(S, L1_PARSING, name);
    double time_was = getMillisecs();
    TreeConstructer tc(S);
    eFlag retval = tc.parseDataLineUsingExpat(S, this, d);
    if (!retval)
    {
        Log1(S, L1_PARSE_DONE, getMillisecsDiff(time_was));
    }
    return retval;
}

eFlag Tree::getMatchingList(Sit S, Expression& match, Context& result)
{
    E( getRoot().getMatchingList(S, match, result) );
    return OK;
}


