--[[--
SelectedFusionPolylineToSSF.lua

This tool exports all polylines of selected polymasks to a ssf file

Requires   : Fusion 5.21 Build 23 or later, or 6.40 build 1109 or later if exporting BSplines
Written By : Srecko Zrilic [srecko@eyeonline.com]
Written On : Tuesday 16th of October 2012

Modification on 02. November 2012:
	    - Enabled exporting BSpline polylines by converting it to Bezier polyline while exporting


Modification on 06. February 2013:
	    - Fixed the format of exporting the Shake shape version.

Usage      :


Important  :  - PolyMasks should be rendered before exporting.
--]]--

if tool == nil or tool.ID ~= "PolylineMask"  and tool.ID ~= "BSplineMask" then
	print("Run this script on a Polyline Mask")
	exit()
end

local toollist = comp:GetToolList(true)
local tool2 = nil
local num_shapes = 0
for j, vv in pairs(toollist) do
	if vv:GetID() == "PolylineMask" or vv:GetID() == "BSplineMask" then
		num_shapes = num_shapes + 1
		tool2 = vv
	end
end

if tool2 ~= tool then
	exit()
end

if num_shapes < 1 then
	print("No selected masks\n")
	exit()
end

local width = tool:GetAttrs().TOOLI_ImageWidth
local height = tool:GetAttrs().TOOLI_ImageHeight

if width == nil or height == nil then
	comp:AskUser("You should render first!", {})
	exit()
end

local asp = height / width
local k = 3.1415926536 / 180.0

function GetVerticesCount(polyline)
	local num_ver = 0
	for f, p in pairs(polyline.Value.Points) do
		num_ver = num_ver + 1
	end
	return num_ver
end

function WriteData(toolx, polyline, polyline_edge, center_x, center_y, xangle, yangle, zangle, file, frame)
	if polyline.Value.__ctor == "BSplinePolyline" then
		polyline = toolx:GetBezierPolyline(frame)
	end
	if polyline_edge and polyline_edge.Value.__ctor == "BSplinePolyline" then
		polyline_edge = toolx:GetBezierPolyline(frame, "outter")
	end

	file:write(string.format("key_time %g\n", frame))
	file:write(string.format("center_x %g\n", center_x*width))
	file:write(string.format("center_y %g\n", center_y*height))
	file:write(string.format("color_r %.1f\n", 1.0))
	file:write(string.format("color_g %.1f\n", 1.0))
	file:write(string.format("color_b %.1f\n", 1.0))
	file:write(string.format("color_a %.1f\n", 1.0))

	xangle = xangle * k
	yangle = yangle * k
	zangle = zangle * k
	local sa = math.sin(zangle)
	local ca = math.cos(zangle)
	local sax = math.sin(xangle)
	local cax = math.cos(xangle)
	local say = math.sin(yangle)
	local cay = math.cos(yangle)

	local pv = {}
	local pve = {}
	local pv_count = 0
	local UNDEF = -1000000.0
	for f, p in pairs(polyline.Value.Points) do
		local iv = 1 + (f - 1)*6 
		pv[iv] = p.X
		pv[iv+1] = p.Y
		if p.LX then
			pv[iv+2] = p.LX
		else
			pv[iv+2] = UNDEF
		end
		if p.LY then
			pv[iv+3] = p.LY
		else
			pv[iv+3] = UNDEF
		end
		if p.RX then
			pv[iv+4] = p.RX
		else
			pv[iv+4] = UNDEF
		end
		if p.RY then
			pv[iv+5] = p.RY
		else
			pv[iv+5] = UNDEF
		end
		pv_count = iv + 5
	end
	if polyline_edge then
		for f, p in pairs(polyline_edge.Value.Points) do
			local iv = 1 + (f - 1)*6 
			pve[iv] = p.X
			pve[iv+1] = p.Y
			if p.LX then
				pve[iv+2] = p.LX
			else
				pve[iv+2] = UNDEF
			end
			if p.LY then
				pve[iv+3] = p.LY
			else
				pve[iv+3] = UNDEF
			end
			if p.RX then
				pve[iv+4] = p.RX
			else
				pve[iv+4] = UNDEF
			end
			if p.RY then
				pve[iv+5] = p.RY
			else
				pve[iv+5] = UNDEF
			end
		end
	end
	if pv_count  > 0 then
		for i = 1, pv_count-1, 6 do
			if pv[i+2] == UNDEF then 
				if i > 1  then
					pv[i+2] = (pv[i-6] - pv[i])/3.0
					pv[i+3] = (pv[i-5] - pv[i+1])/3.0
				else
					if closed then
						pv[i+2] = (pv[pv_count-5] - pv[i])/3.0
						pv[i+3] = (pv[pv_count-4] - pv[i+1])/3.0
					else
						pv[i+2] = 0.0
						pv[i+3] = 0.0
					end
				end
			end
			if pv[i+4] == UNDEF then 
				if i < pv_count-5  then
					pv[i+4] = (pv[i+6] - pv[i])/3.0
					pv[i+5] = (pv[i+7] - pv[i+1])/3.0
				else
					if closed then
						pv[i+4] = (pv[1] - pv[i])/3.0
						pv[i+5] = (pv[2] - pv[i+1])/3.0
					else
						pv[i+4] = 0.0
						pv[i+5] = 0.0
					end
				end
			end
			if edge_shape then
				if pve[i+2] == UNDEF then 
					if i > 1  then
						pve[i+2] = (pve[i-6] - pve[i])/3.0
						pve[i+3] = (pve[i-5] - pve[i+1])/3.0
					else
						if closed then
							pve[i+2] = (pve[pv_count-5] - pve[i])/3.0
							pve[i+3] = (pve[pv_count-4] - pve[i+1])/3.0
						else
							pve[i+2] = 0.0
							pve[i+3] = 0.0
						end
					end
				end
				if pve[i+4] == UNDEF then 
					if i < pv_count-5  then
						pve[i+4] = (pve[i+6] - pve[i])/3.0
						pve[i+5] = (pve[i+7] - pve[i+1])/3.0
					else
						if closed then
							pve[i+4] = (pve[1] - pve[i])/3.0
							pve[i+5] = (pve[2] - pve[i+1])/3.0
						else
							pve[i+4] = 0.0
							pve[i+5] = 0.0
						end
					end
				end
			end
		end

		file:write(string.format("vertex_data "))
		for i = 1, pv_count-1, 6 do
			local x = pv[i+2] + pv[i]
			local y = (pv[i+3] + pv[i+1])
			local xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
			local yy = (sa * cax * x + ca * cax * y * asp) / asp
			pv[i+2] = (xx + center_x) * width
			pv[i+3] = (yy + center_y) * height
			x = pv[i+4] + pv[i]
			y = (pv[i+5] + pv[i+1])
			xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
			yy = (sa * cax * x + ca * cax * y * asp) / asp
			pv[i+4] = (xx + center_x) * width
			pv[i+5] = (yy + center_y) * height
			x = pv[i]
			y = pv[i+1]
			xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
			yy = (sa * cax * x + ca * cax * y * asp) / asp
			pv[i] = (xx + center_x) * width
			pv[i+1] = (yy + center_y) * height
			file:write(string.format("%g %g %g %g %g %g ", pv[i], pv[i+1], pv[i+2], pv[i+3], pv[i+4], pv[i+5]))
			if polyline_edge then
				x = pve[i+2] + pve[i]
				y = (pve[i+3] + pve[i+1])
				xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
				yy = (sa * cax * x + ca * cax * y * asp) / asp
				pve[i+2] = (xx + center_x) * width
				pve[i+3] = (yy + center_y) * height
				x = pve[i+4] + pve[i]
				y = (pve[i+5] + pve[i+1])
				xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
				yy = (sa * cax * x + ca * cax * y * asp) / asp
				pve[i+4] = (xx + center_x) * width
				pve[i+5] = (yy + center_y) * height
				x = pve[i]
				y = pve[i+1]
				xx = (ca * cay + sa * sax * say) * x + (ca * sax * say - sa * cay) * y * asp
				yy = (sa * cax * x + ca * cax * y * asp) / asp
				pve[i] = (xx + center_x) * width
				pve[i+1] = (yy + center_y) * height
				file:write(string.format("%g %g %g %g %g %g ", pve[i], pve[i+1], pve[i+2], pve[i+3], pve[i+4], pve[i+5]))
			end
		end
		file:write(string.format("\n"))
	end
end

function ExportShape(tooll, file)
	local polytbl = tooll:SaveSettings()
	for name,t in pairs(polytbl.Tools) do
		if t.__ctor == "PolylineMask" or t.__ctor == "BSplineMask" then
			local ctime = comp.CurrentTime
			local animated = false
			local animated_edge = false
			local polysplinename = t.Inputs.Polyline.SourceOp
			local polysplinename_edge = t.Inputs.Polyline2.SourceOp
			local polyspline = nil
			local polyline = nil
			local polyspline_edge = nil
			local polyline_edge = nil
			local frame = ctime
			local bspline = false
			local closed = 0
			local visible = 1
			local locked = 0
			local num_vertices = 0
			local num_key_times = 0
			if polysplinename then
				polyspline = polytbl.Tools[polysplinename]
				if polyspline and type(polyspline) == "table" then 
					for ff, pl in pairs(polyspline.KeyFrames) do
						if pl and pl.Value then
							num_key_times = num_key_times + 1
							polyline = pl
							frame = ff
--							if polyline.Value.__ctor == "BSplinePolyline" then
--								bspline = true
--							end
							if polyline and polyline.Value then
								num_vertices = GetVerticesCount(polyline)					
								if polyline.Value.Closed and polyline.Value.Closed==true then
									closed = 1
								end								
							end
						end
						if num_key_times > 1 then
							animated = true
							polyline = nil
						end
					end
				end								
			else
				polyline = t.Inputs.Polyline
--				if polyline and polyline.Value.__ctor == "BSplinePolyline" then
--					bspline = true
--				end
				if polyline and polyline.Value then
					num_vertices = GetVerticesCount(polyline)					
					if polyline.Value.Closed and polyline.Value.Closed==true then
						closed = 1
					end
				end
			end
				
			if bspline == true then
				comp:AskUser("You can not export b-spline, you shold convert to Bezier first!", {})
				file:close()
				exit()
			end

			local num_key_times_edge = 0
			local frame_edge = ctime
			local num_vertices_edge = 0
			local closed_edge = 0
			if polysplinename_edge then
				polyspline_edge = polytbl.Tools[polysplinename_edge]
				if polyspline_edge and type(polyspline) == "table" then 
					for ff, pl in pairs(polyspline_edge.KeyFrames) do
						if pl and pl.Value then
							num_key_times_edge = num_key_times_edge + 1
							polyline_edge = pl
							frame_edge = ff
--							if polyline_edge.Value.__ctor == "BSplinePolyline" then
--								bspline = true
--							end
							if polyline_edge and polyline_edge.Value then
								num_vertices_edge = GetVerticesCount(polyline_edge)					
								if polyline_edge.Value.Closed and polyline_edge.Value.Closed==true then
									closed_edge = 1
								end								
							end
						end
						if num_key_times_edge > 1 then
							animated_edge = true
							polyline_edge = nil
						end
					end
				end								
			else
				if t.Inputs.Polyline2.Disabled == nil then
					polyline_edge = t.Inputs.Polyline2
--					if polyline_edge.Value.__ctor == "BSplinePolyline"then
--						bspline = true
--					end
					if polyline_edge and polyline_edge.Value then
						num_vertices_edge = GetVerticesCount(polyline_edge)					
						if polyline_edge.Value.Closed and polyline_edge.Value.Closed==true then
							closed_edge = 1
						end								
					end
				end
			end

			if bspline then
				comp:AskUser("You can not export b-spline, you shold convert to Bezier first!", {})
				file:close()
				exit()
			end

			local edge_shape = 0
			if animated_edge or polyline_edge and polyline_edge.Value then
				edge_shape = 1
			end

			if edge_shape == 1 and num_vertices ~= num_vertices_edge then
				local ret = comp:AskUser("Keys for inner and outer polyline are not the same! Export inner only?", {})
				if ret then				
					edge_shape = 0
					polyline_edge = nil
					animated_edge = false
				else
					file:close()
					exit()
				end
			end

			num_key_times = math.max(num_key_times, num_key_times_edge)
			file:write(string.format("shape_name %s\n", tooll:GetAttrs().TOOLS_Name))
			file:write(string.format("parent_name\n"))
			file:write(string.format("closed %d\n", closed))
			file:write(string.format("visible %d\n", visible))
			file:write(string.format("locked %d\n", locked))
			local tangents = 1
			file:write(string.format("tangents %d\n", tangents))
			file:write(string.format("edge_shape %d\n", edge_shape))
			file:write(string.format("num_vertices %d\n", num_vertices))
			file:write(string.format("num_key_times %d\n", num_key_times))

			if animated and polyline_edge == nil and animated_edge == false then
				polyspline = polytbl.Tools[polysplinename]
				if polysplinename and type(polyspline) == "table" then
					for ff, pl in pairs(polyspline.KeyFrames) do
						if pl and pl.Value then
							local center_x = 0.5
							local center_y = 0.5
							if t.Inputs.Center then
								if t.Inputs.Center.Value then
									center_x = t.Inputs.Center.Value[1]
									center_y = t.Inputs.Center.Value[2]
								else
									local point = tooll:GetInput("Center", ff)	-- REQF_SecondaryTime
									if point then
										center_x = point[1]
										center_y = point[2]
									end
								end
							end
							local zangle = 0
							local xangle = 0
							local yangle = 0
							if t.Inputs.ZRotation then
								zangle = t.Inputs.ZRotation.Value
							else
								zangle = tooll:GetInput("ZRotation", ff)
							end
									
							if t.Inputs.XRotation then
								xangle = t.Inputs.XRotation.Value
							else
								xangle = tooll:GetInput("XRotation", ff)
							end
							if t.Inputs.YRotation then
								yangle = t.Inputs.YRotation.Value
							else
								yangle = tooll:GetInput("YRotation", ff)
							end
							WriteData(tooll, pl, polyline_edge, center_x, center_y, xangle, yangle, zangle, file, ff)
						end
					end
				end
			else
				if polyline and animated_edge then
					polyspline_edge = polytbl.Tools[polysplinename_edge]
					if polysplinename_edge and type(polyspline_edge) == "table" then
						for ff, pl in pairs(polyspline_edge.KeyFrames) do
							if pl and pl.Value then
								local center_x = 0.5
								local center_y = 0.5
								if t.Inputs.Center then
									if t.Inputs.Center.Value then
										center_x = t.Inputs.Center.Value[1]
										center_y = t.Inputs.Center.Value[2]
									else
										local point = tooll:GetInput("Center", ff)	-- REQF_SecondaryTime
										if point then
											center_x = point[1]
											center_y = point[2]
										end
									end
								end
								local zangle = 0
								local xangle = 0
								local yangle = 0
								if t.Inputs.ZRotation then
									zangle = t.Inputs.ZRotation.Value
								else
									zangle = tooll:GetInput("ZRotation", ff)
								end
										
								if t.Inputs.XRotation then
									xangle = t.Inputs.XRotation.Value
								else
									xangle = tooll:GetInput("XRotation", ff)
								end
								if t.Inputs.YRotation then
									yangle = t.Inputs.YRotation.Value
								else
									yangle = tooll:GetInput("YRotation", ff)
								end
								WriteData(tooll, polyline, pl, center_x, center_y, xangle, yangle, zangle, file, ff)
							end
						end
					end
				else
					if animated and animated_edge then
						polyspline = polytbl.Tools[polysplinename]
						if polysplinename_edge and type(polyspline) == "table" then
							for ff, pl in pairs(polyspline.KeyFrames) do
								if pl and pl.Value then
									local polyline_e = nil
									local found = false
									for ee, ple in pairs(polyspline_edge.KeyFrames) do
										if ple and ple.Value then
											if math.abs(ee - ff) < 0.1 then
												polyline_e = ple
												found = true
												break
											end
										end
									end
									if found then
										local center_x = 0.5
										local center_y = 0.5
										if t.Inputs.Center then
											if t.Inputs.Center.Value then
												center_x = t.Inputs.Center.Value[1]
												center_y = t.Inputs.Center.Value[2]
											else
												local point = tooll:GetInput("Center", ff)	-- REQF_SecondaryTime
												if point then
													center_x = point[1]
													center_y = point[2]
												end
											end
										end
										local zangle = 0
										local xangle = 0
										local yangle = 0
										if t.Inputs.ZRotation then
											zangle = t.Inputs.ZRotation.Value
										else
											zangle = tooll:GetInput("ZRotation", ff)
										end
												
										if t.Inputs.XRotation then
											xangle = t.Inputs.XRotation.Value
										else
											xangle = tooll:GetInput("XRotation", ff)
										end
										if t.Inputs.YRotation then
											yangle = t.Inputs.YRotation.Value
										else
											yangle = tooll:GetInput("YRotation", ff)
										end
										WriteData(tooll, pl, polyline_e, center_x, center_y, xangle, yangle, zangle, file, ff)
									end
								end
							end
						end
					else
						if polyline and polyline.Value then
							local center_x = 0.5
							local center_y = 0.5
							if t.Inputs.Center then
								if t.Inputs.Center.Value then
									center_x = t.Inputs.Center.Value[1]
									center_y = t.Inputs.Center.Value[2]
								else
									local point = tooll:GetInput("Center", frame)	-- REQF_SecondaryTime
									if point then
										center_x = point[1]
										center_y = point[2]
									end
								end
							end
							local zangle = 0
							local xangle = 0
							local yangle = 0
							if t.Inputs.ZRotation then
								zangle = t.Inputs.ZRotation.Value
							else
								zangle = tooll:GetInput("ZRotation", frame)
							end
									
							if t.Inputs.XRotation then
								xangle = t.Inputs.XRotation.Value
							else
								xangle = tooll:GetInput("XRotation", frame)
							end
							if t.Inputs.YRotation then
								yangle = t.Inputs.YRotation.Value
							else
								yangle = tooll:GetInput("YRotation", frame)
							end
							WriteData(tooll, polyline, polyline_edge, center_x, center_y, xangle, yangle, zangle, file, frame)
						end
					end
				end
			end
		end
	end
end

local dlg = iup.filedlg{ title = "Select output file", dialogtype = "SAVE", extfilter = "Silhouette Shape Files (*.ssf)|*.ssf|All Files (*.*)te|*.*|" }
print("dialog.")

print(dlg:popup(0,0))
if dlg.status ~= "-1" then
	local filename = dlg.value
	local fileg, err = io.open(filename, "w")

	if fileg then
		local ver = 4.0
		local motion_blur = 0
		local shutter_timing = 0
		local shutter_offset = 0.0

		fileg:write(string.format("shake_shape_data %.1f\n", ver))
		fileg:write(string.format("motion_blur %d\n", motion_blur))
		fileg:write(string.format("shutter_timing %d\n", shutter_timing))
		fileg:write(string.format("shutter_offset %g\n", shutter_offset))
		fileg:write(string.format("num_shapes %g\n", num_shapes))

		for i, v in pairs(toollist) do
			if v:GetID() == "PolylineMask" or v:GetID() == "BSplineMask" then
				ExportShape(v, fileg)
			end
		end

		fileg:close()
		print("Exporting selected polymasks complete\n")
	else
		print("Could not open file "..filename.."\n"..err.."\n")
	end
end