
/* global TelephonySettingHelper */
/**
 * The module loads scripts used by the root panel. In the future these scripts
 * must be converted to AMD modules. Implementation details please refer to
 * {@link Root}.
 *
 * @module root/root
 */
define('panels/root/root',['require','shared/lazy_loader'],function(require) {
  

  var LazyLoader = require('shared/lazy_loader');

  /**
   * @alias module:root/root
   * @class Root
   * @requires module:shared/lazy_loader
   * @returns {Root}
   */
  function Root() {}

  Root.prototype = {
    _initSimItems: function root_initSimItems() {
      // Show proper SIM items.
      if (navigator.mozMobileConnections) {
        if (navigator.mozMobileConnections.length == 1) { // single sim
          document.getElementById('simCardManager-settings').hidden = true;
        } else { // dsds
          document.getElementById('simSecurity-settings').hidden = true;
        }
      } else {
        // hide telephony panels
        var elements = ['call-settings',
                        'data-connectivity',
                        'messaging-settings',
                        'simSecurity-settings',
                        'simCardManager-settings'];
        elements.forEach(function(el) {
          document.getElementById(el).hidden = true;
        });
      }
    },

    _loadScripts: function root_loadScripts() {
      /**
       * Enable or disable the menu items related to the ICC card
       * relying on the card and radio state.
       */
      LazyLoader.load([
        'js/firefox_accounts/menu_loader.js',
        'js/icc_menu.js',
        'js/dsds_settings.js',
        'js/telephony_settings.js',
        'js/telephony_items_handler.js'
      ], function() {
        TelephonySettingHelper
          .init()
          .then(function telephonySettingInitDone() {
            window.dispatchEvent(new CustomEvent('telephony-settings-loaded'));
          });
      });
    },

    init: function root_init() {
      this._initSimItems();
      // Load the necessary scripts after the UI update.
      setTimeout(this._loadScripts);
    }
  };

  return function ctor_root() {
    return new Root();
  };
});

/* global console */

/**
 * VersionDetector:
 *   - VersionDetector is an detector that identify the version of platform
 *     Bluetooth object.
 *   - It has only one method: getVersion.
 * VersionDetector only identify version and does not involve in any UI logic.
 *
 * @module modules/bluetooth/version_detector
 */
define('modules/bluetooth/version_detector',['require','modules/navigator/mozBluetooth'],function(require) {
  

  var NavigatorBluetooth = require('modules/navigator/mozBluetooth');

  var VersionDetector = {
    /**
     * The value indicates whether the API version is responding.
     *
     * @access public
     * @memberOf VersionDetector
     * @return {number}
     */
    getVersion: function() {
      if (!NavigatorBluetooth) {
        console.error('[VersionDetector]: ' +
                      'navigator.mozBluetooth is not existed!!');
        // Since there is no navigator.mozBluetooth on B2G Desktop,
        // we workaround to return version 1 for Gaia UI test case.
        return 1;
      } else if
        (typeof(NavigatorBluetooth.onattributechanged) !== 'undefined') {
        return 2;
      } else {
        return 1;
      }
    }
  };

  return VersionDetector;
});

/**
 * The moudle supports displaying bluetooth information on an element.
 *
 * @module panels/root/bluetooth_item
 */
define('panels/root/bluetooth_item',['require','modules/settings_service','modules/bluetooth/version_detector'],function(require) {
  

  var SettingsService = require('modules/settings_service');
  var APIVersionDetector = require('modules/bluetooth/version_detector');
  
  var APIVersion = APIVersionDetector.getVersion();
  /**
   * @alias module:panels/root/bluetooth_item
   * @class BluetoothItem
   * @requires module:modules/bluetooth
   * @param {HTMLElement} element
                          The element displaying the bluetooth information
   * @return {BluetoothItem}
   */
  function BluetoothItem(element) {
    this._enabled = false;
    this._element = element;
    this._boundRefreshMenuDescription =
      this._refreshMenuDescription.bind(this, element);

    // The decision of navigation panel will be removed while we are no longer
    // to use Bluetooth API v1.
    element.parentNode.onclick = this._navigatePanelWithVersionCheck.bind(this);
  }

  BluetoothItem.prototype = {
    /**
     * Return Bluetooth API version via APIVersionDetector module.
     *
     * @access private
     * @memberOf BluetoothItem.prototype
     * @type {Number}
     */
    _APIVersion: function bt__APIVersion() {
      return APIVersion;
    },

    /**
     * An instance to maintain that we have created a promise to get Bluetooth
     * module.
     *
     * @access private
     * @memberOf BluetoothItem.prototype
     * @type {Promise}
     */
    _getBluetoothPromise: null,

    /**
     * A promise function to get Bluetooth module.
     *
     * @access private
     * @memberOf BluetoothItem.prototype
     * @type {Promise}
     */
    _getBluetooth: function bt__getBluetooth() {
      if (!this._getBluetoothPromise) {
        this._getBluetoothPromise = new Promise(function(resolve) {
          var bluetoothModulePath;
          if (this._APIVersion() === 1) {
            bluetoothModulePath = 'modules/bluetooth/bluetooth_v1';
          } else if (this._APIVersion() === 2) {
            bluetoothModulePath = 'modules/bluetooth/bluetooth';
          }

          require([bluetoothModulePath], resolve);
        }.bind(this));
      }
      return this._getBluetoothPromise;
    },

    /**
     * Refresh the text based on the Bluetooth module enabled/disabled,
     * paired devices information.
     *
     * @access private
     * @memberOf BluetoothItem.prototype
     * @param {HTMLElement} element
                            The element displaying the bluetooth information
     */
    _refreshMenuDescription: function bt__refreshMenuDescription(element) {
      if (!navigator.mozL10n) {
        return;
      }

      this._getBluetooth().then(function(bluetooth) {
        if (bluetooth.enabled) {
          if (bluetooth.numberOfPairedDevices === 0) {
            element.setAttribute('data-l10n-id', 'bt-status-nopaired');
          } else {
            navigator.mozL10n.setAttributes(element, 'bt-status-paired',
              {
                name: bluetooth.firstPairedDeviceName,
                n: bluetooth.numberOfPairedDevices - 1
              });
          }
        } else {
          element.setAttribute('data-l10n-id', 'bt-status-turnoff');
        }
      });
    },

    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf BluetoothItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      if (this._enabled === value) {
        return;
      }

      this._enabled = value;
      this._getBluetooth().then(function(bluetooth) {
        if (this._enabled) {
          bluetooth.observe('enabled', this._boundRefreshMenuDescription);
          bluetooth.observe('numberOfPairedDevices',
            this._boundRefreshMenuDescription);
          this._boundRefreshMenuDescription();
        } else {
          bluetooth.unobserve('enabled', this._boundRefreshMenuDescription);
          bluetooth.unobserve('numberOfPairedDevices',
            this._boundRefreshMenuDescription);
        }
      }.bind(this));
    },

    /**
     * Navigate new/old Bluetooth panel via version of mozBluetooth API.
     *
     * @access private
     * @memberOf BluetoothItem.prototype
     * @type {Function}
     */
    _navigatePanelWithVersionCheck:
    function bt__navigatePanelWithVersionCheck() {
      if (this._APIVersion() === 1) {
        // navigate old bluetooth panel..
        SettingsService.navigate('bluetooth');
      } else if (this._APIVersion() === 2) {
        // navigate new bluetooth panel..
        SettingsService.navigate('bluetooth_v2');
      }
    }
  };

  return function ctor_bluetoothItem(element) {
    return new BluetoothItem(element);
  };
});

/**
 * The moudle supports displaying nfc toggle on an element.
 *
 * @module panels/root/nfc_item
 */
define('panels/root/nfc_item',['require'],function(require) {
  

  /**
   * @alias module:panels/root/nfc_item
   * @class NFCItem
   * @param {HTMLElement} element
                          The element displaying the nfc toggle
   * @returns {NFCItem}
   */
  function NFCItem(element) {
    this._render(element);
  }

  NFCItem.prototype = {
    /**
     * Display the NFC item if mozNfc API exist.
     *
     * @access private
     * @memberOf NFCItem.prototype
     * @param {HTMLElement} element
                            The element displaying the NFC toggle
     */
    _render: function nfc_render(element) {
      // Check if NFC is available on platform, and update UI
      if (element) {
        element.hidden = !navigator.mozNfc;
      }
    }
  };

  return function ctor_nfcItem(element) {
    return new NFCItem(element);
  };
});

/**
 * The moudle supports displaying language information on an element.
 *
 * @module panels/root/language_item
 */
define('panels/root/language_item',['require','shared/language_list'],function(require) {
  

  var LanguageList = require('shared/language_list');

  /**
   * @alias module:panels/root/language_item
   * @class LanguageItem
   * @param {HTMLElement} element
                          The element displaying the language information
   * @returns {LanguageItem}
   */
  function LanguageItem(element) {
    this._enabled = false;
    this._boundRefreshText = this._refreshText.bind(this, element);
  }

  LanguageItem.prototype = {
    /**
     * Refresh the text based on the language setting.
     *
     * @access private
     * @memberOf LanguageItem.prototype
     * @param {HTMLElement} element
                            The element displaying the language information
     */
    _refreshText: function l_refeshText(element) {
      // display the current locale in the main panel
      LanguageList.get(function displayLang(languages, currentLanguage) {
        element.textContent = LanguageList.wrapBidi(
          currentLanguage, languages[currentLanguage]);
      });
    },

    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf LanguageItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      if (this._enabled === value || !navigator.mozL10n) {
        return;
      }
      
      this._enabled = value;
      if (this._enabled) {
        window.addEventListener('localized', this._boundRefreshText);
        this._boundRefreshText();
      } else {
        window.removeEventListener('localized', this._boundRefreshText);
      }
    }
  };

  return function ctor_languageItem(element) {
    return new LanguageItem(element);
  };
});

/**
 * The moudle supports displaying battery information on an element.
 *
 * @module panels/root/battery_item
 */
define('panels/root/battery_item',['require','modules/battery'],function(require) {
  

  var Battery = require('modules/battery');

  /**
   * @alias module:panels/root/battery_item
   * @class BatteryItem
   * @requires module:modules/battery
   * @param {HTMLElement} element
                          The element displaying the battery information
   * @returns {BatteryItem}
   */
  function BatteryItem(element) {
    this._enabled = false;
    this._element = element;
    this._boundRefreshText = this._refreshText.bind(this, element);
  }

  BatteryItem.prototype = {
    /**
     * Refresh the text based on the Battery module.
     *
     * @access private
     * @memberOf BatteryItem.prototype
     * @param {HTMLElement} element
                            The element displaying the battery information
     */
    _refreshText: function b_refreshText(element) {
      if (!navigator.mozL10n) {
        return;
      }

      navigator.mozL10n.setAttributes(element,
        'batteryLevel-percent-' + Battery.state, { level: Battery.level });
      if (element.hidden) {
        element.hidden = false;
      }
    },

    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf BatteryItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      if (this._enabled === value) {
        return;
      }
      
      this._enabled = value;
      if (this._enabled) {
        Battery.observe('level', this._boundRefreshText);
        Battery.observe('state', this._boundRefreshText);
        this._boundRefreshText();
      } else {
        Battery.unobserve('level', this._boundRefreshText);
        Battery.unobserve('state', this._boundRefreshText);
      }
    }
  };

  return function ctor_batteryItem(element) {
    return new BatteryItem(element);
  };
});

/* global SettingsListener */

/**
 * This module display Find My Device's enabled/disabled state on an
 * element.
 *
 * @module panels/root/findmydevice_item
 */
define('panels/root/findmydevice_item',['require'],function(require) {
  

  /**
   * @alias module:panels/root/findmydevice_item
   * @class FindMyDeviceItem
   * @param {HTMLElement} element
                          The element displaying Find My Device's enabled state
   * @returns {FindMyDeviceItem}
   */
  function FindMyDeviceItem(element) {
    this._itemEnabled = false;
    this._FMDEnabled = false;
    this._boundRefreshText = this._refreshText.bind(this, element);
    this._boundFMDEnabledChanged = this._onFMDEnabledChanged.bind(this);
  }

  FindMyDeviceItem.prototype = {
    /**
     * Refresh the text based on Find My Device's enabled state
     *
     * @access private
     * @memberOf FindMyDeviceItem.prototype
     * @param {HTMLElement} element
                            The element showing Find My Device's enabled state
     */
    _refreshText: function fmd_refresh_text(element) {
      if (!navigator.mozL10n) {
        return;
      }

      element.setAttribute('data-l10n-id',
                           this._FMDEnabled ? 'enabled' : 'disabled');
      element.hidden = false;
    },

    /**
     * Listener for changes in Find My Device's enabled state. Updates the
     * text if this item is enabled.
     *
     * @access private
     * @memberOf FindMyDeviceItem.prototype
     * @param {Boolean} value
                        The current enabled state for Find My Device
     */
    _onFMDEnabledChanged: function fmd_enabled_changed(value) {
      this._FMDEnabled = value;
      if (this._itemEnabled) {
        this._boundRefreshText();
      }
    },

    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf FindMyDeviceItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._itemEnabled;
    },

    set enabled(value) {
      if (this._itemEnabled === value) {
        return;
      }

      this._itemEnabled = value;
      if (this._itemEnabled) {
        SettingsListener.observe('findmydevice.enabled', false,
          this._boundFMDEnabledChanged);
      } else {
        SettingsListener.unobserve('findmydevice.enabled',
          this._boundFMDEnabledChanged);
      }
    }
  };

  return function ctor_findMyDeviceItem(element) {
    return new FindMyDeviceItem(element);
  };
});

/* global DeviceStorageHelper, openIncompatibleSettingsDialog */
/**
 * Links the root panel list item with USB Storage.
 *
 * XXX bug 973451 will remove media storage part
 */
define('panels/root/storage_usb_item',['require','shared/settings_listener','shared/async_storage','modules/settings_cache','modules/settings_service'],function(require) {
  

  var SettingsListener = require('shared/settings_listener');
  var AsyncStorage = require('shared/async_storage');
  var SettingsCache = require('modules/settings_cache');
  var SettingsService = require('modules/settings_service');

  /**
   * @alias module:panels/root/storage_usb_item
   * @class USBStorageItem
   * @param {Object} elements
                     elements displaying the usb and media storage information
   * @returns {USBStorageItem}
   */
  function USBStorageItem(elements) {
    this._enabled = false;
    this._elements = elements;
    this._umsSettingKey = 'ums.enabled';
    // XXX media related attributes
    this._defaultMediaVolume = null;
    this._defaultVolumeState = 'available';
    this._defaultMediaVolumeKey = 'device.storage.writable.name';
    this._boundUmsSettingHandler = this._umsSettingHandler.bind(this);
    this._boundMediaVolumeChangeHandler =
      this._mediaVolumeChangeHandler.bind(this);
  }

  USBStorageItem.prototype = {
    /**
     * The value indicates whether the module is responding. If it is false, the
     * UI stops reflecting the updates from the root panel context.
     *
     * @access public
     * @memberOf USBStorageItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      if (this._enabled === value) {
        return;
      } else {
        this._enabled = value;
      }
      if (value) { //observe
        // ums master switch on root panel
        this._elements.usbEnabledCheckBox.addEventListener('change', this);

        SettingsListener.observe(this._umsSettingKey, false,
          this._boundUmsSettingHandler);

        // media storage
        // Show default media volume state on root panel
        SettingsListener.observe(this._defaultMediaVolumeKey, 'sdcard',
          this._boundMediaVolumeChangeHandler);
        window.addEventListener('localized', this);

        // register USB storage split click handler
        this._elements.usbStorage.addEventListener('click', this._onItemClick);
      } else { //unobserve
        this._elements.usbEnabledCheckBox.removeEventListener('change', this);

        SettingsListener.unobserve(this._umsSettingKey,
          this._boundUmsSettingHandler);

        // media storage
        SettingsListener.unobserve(this._defaultMediaVolumeKey,
          this._boundMediaVolumeChangeHandler);
        window.removeEventListener('localized', this);

        this._elements.usbStorage.removeEventListener('click',
          this._onItemClick);
      }
    },

    _umsSettingHandler: function storage_umsSettingHandler(enabled) {
      this._elements.usbEnabledCheckBox.checked = enabled;
      this._updateUmsDesc();
    },

    // navigate to USB Storage panel
    _onItemClick: function storage_onItemClick(evt) {
      SettingsService.navigate('usbStorage');
    },

    handleEvent: function storage_handleEvent(evt) {
      switch (evt.type) {
        case 'localized':
          this._updateMediaStorageInfo();
          break;
        case 'change':
          if (evt.target === this._elements.usbEnabledCheckBox) {
            this._umsMasterSettingChanged(evt);
          } else {
            // we are handling storage state changes
            // possible state: available, unavailable, shared
            this._updateMediaStorageInfo();
          }
          break;
      }
    },

    // ums description
    _updateUmsDesc: function storage_updateUmsDesc() {
      var key;
      if (this._elements.usbEnabledCheckBox.checked) {
        //TODO list all enabled volume name
        key = 'enabled';
      } else if (this._defaultVolumeState === 'shared') {
        key = 'umsUnplugToDisable';
      } else {
        key = 'disabled';
      }
      this._elements.usbEnabledInfoBlock.setAttribute('data-l10n-id', key);
    },

    _umsMasterSettingChanged: function storage_umsMasterSettingChanged(evt) {
      var checkbox = evt.target;
      var cset = {};
      var warningKey = 'ums-turn-on-warning';

      if (checkbox.checked) {
        AsyncStorage.getItem(warningKey, function(showed) {
          if (!showed) {
            this._elements.umsWarningDialog.hidden = false;

            this._elements.umsConfirmButton.onclick = function() {
              AsyncStorage.setItem(warningKey, true);
              this._elements.umsWarningDialog.hidden = true;

              SettingsCache.getSettings(
                this._openIncompatibleSettingsDialogIfNeeded.bind(this));
            }.bind(this);

            this._elements.umsCancelButton.onclick = function() {
              cset[this._umsSettingKey] = false;
              Settings.mozSettings.createLock().set(cset);

              checkbox.checked = false;
              this._elements.umsWarningDialog.hidden = true;
            }.bind(this);
          } else {
            SettingsCache.getSettings(
              this._openIncompatibleSettingsDialogIfNeeded.bind(this));
          }
        }.bind(this));
      } else {
        cset[this._umsSettingKey] = false;
        Settings.mozSettings.createLock().set(cset);
      }
    },

    _openIncompatibleSettingsDialogIfNeeded:
      function storage_openIncompatibleSettingsDialogIfNeeded(settings) {
        var cset = {};
        var umsSettingKey = this._umsSettingKey;
        var usbTetheringSetting = settings['tethering.usb.enabled'];

        if (!usbTetheringSetting) {
          cset[umsSettingKey] = true;
          Settings.mozSettings.createLock().set(cset);
        } else {
          var oldSetting = 'tethering.usb.enabled';
          openIncompatibleSettingsDialog('incompatible-settings-warning',
            umsSettingKey, oldSetting, null);
        }
    },

    // XXX media related functions
    _mediaVolumeChangeHandler:
      function storage_mediaVolumeChangeHandler(defaultName) {
      if (this._defaultMediaVolume) {
        this._defaultMediaVolume.removeEventListener('change', this);
      }
      this._defaultMediaVolume = this._getDefaultVolume(defaultName);
      this._defaultMediaVolume.addEventListener('change', this);
      this._updateMediaStorageInfo();
    },

    // Media Storage
    _updateMediaStorageInfo: function storage_updateMediaStorageInfo() {
      if (!this._defaultMediaVolume) {
        return;
      }

      var self = this;
      this._defaultMediaVolume.available().onsuccess = function(evt) {
        var state = evt.target.result;
        var firstVolume = navigator.getDeviceStorages('sdcard')[0];
        // if the default storage is unavailable, and it's not the
        // internal storage, we show the internal storage status instead.
        if (state === 'unavailable' &&
          self._defaultMediaVolume.storageName !== firstVolume.storageName) {
          firstVolume.available().onsuccess = function(e) {
            self._updateVolumeState(firstVolume, e.target.result);
          };
        } else {
          self._updateVolumeState(self._defaultMediaVolume, state);
        }
      };
    },

    _updateVolumeState: function storage_updateVolumeState(volume, state) {
      this._defaultVolumeState = state;
      this._updateUmsDesc();
      switch (state) {
        case 'available':
          this._updateMediaFreeSpace(volume);
          this._lockMediaStorageMenu(false);
          break;

        case 'shared':
          this._elements.mediaStorageDesc.removeAttribute('data-l10n-id');
          this._elements.mediaStorageDesc.textContent = '';
          this._lockMediaStorageMenu(false);
          break;

        case 'unavailable':
          this._elements.mediaStorageDesc.setAttribute('data-l10n-id',
                                                       'no-storage');
          this._lockMediaStorageMenu(true);
          break;
      }
    },

    _updateMediaFreeSpace: function storage_updateMediaFreeSpace(volume) {
      var self = this;
      volume.freeSpace().onsuccess = function(e) {
        DeviceStorageHelper.showFormatedSize(self._elements.mediaStorageDesc,
          'availableSize', e.target.result);
      };
    },

    _lockMediaStorageMenu: function storage_setMediaMenuState(lock) {
      if (lock) {
        this._elements.mediaStorageSection.setAttribute('aria-disabled', true);
      } else {
        this._elements.mediaStorageSection.removeAttribute('aria-disabled');
      }
    },

    // util function
    _getDefaultVolume: function storage_getDefaultVolume(name) {
      // Per API design, all media type return the same volumes.
      // So we use 'sdcard' here for no reason.
      // https://bugzilla.mozilla.org/show_bug.cgi?id=856782#c10
      var volumes = navigator.getDeviceStorages('sdcard');
      if (!name || name === '') {
        return volumes[0];
      }
      for (var i = 0; i < volumes.length; ++i) {
        if (volumes[i].storageName === name) {
          return volumes[i];
        }
      }
      return volumes[0];
    }
  };

  return function ctor_usb_storage_item(elements) {
    return new USBStorageItem(elements);
  };
});

/**
 * AppStorage is a singleton that caches app storage values for
 * app storage and root panel fast access
 */
define('modules/app_storage',['require','modules/mvvm/observable'],function(require) {
  

  var Observable = require('modules/mvvm/observable');

  var AppStorage = function() {
    this._enabled = false;
    this._appStorage = navigator.getDeviceStorage('apps');

    this.storage = Observable({
      usedPercentage: 0,
      totalSize: 0,
      usedSize: 0,
      freeSize: 0
    });
  };

  AppStorage.prototype = {
    /**
     * The value indicates whether the module is responding. If it is false, the
     * UI stops reflecting the updates from the app storage.
     *
     * @access public
     * @memberOf AppStorage.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      // early return if the value is not changed
      if (this._enabled === value) {
        return;
      } else {
        this._enabled = value;
      }
      if (value) {
        this._attachListeners();
        this._getSpaceInfo();
      } else {
        this._detachListeners();
      }
    },

    _attachListeners: function as_attachListeners() {
      this._appStorage.addEventListener('change', this);
    },

    _detachListeners: function as_detachListeners() {
      this._appStorage.removeEventListener('change', this);
    },

    handleEvent: function as_handler(evt) {
      switch (evt.type) {
        case 'change':
          this._getSpaceInfo();
          break;
      }
    },

    _getSpaceInfo: function as_getSpaceInfo() {
      var deviceStorage = this._appStorage;

      if (!deviceStorage) {
        console.error('Cannot get DeviceStorage for: app');
        return;
      }
      deviceStorage.freeSpace().onsuccess = function(e) {
        this.storage.freeSize = e.target.result;
        deviceStorage.usedSpace().onsuccess = function(e) {
          this.storage.usedSize = e.target.result;
          // calculate the percentage to show a space usage bar
          this.storage.totalSize =
            this.storage.usedSize + this.storage.freeSize;
          var usedPercentage = (this.storage.totalSize === 0) ?
            0 : (this.storage.usedSize * 100 / this.storage.totalSize);
          if (usedPercentage > 100) {
            usedPercentage = 100;
          }
          this.storage.usedPercentage = usedPercentage;
        }.bind(this);
      }.bind(this);
    }
  };

  // return singleton
  var instance = new AppStorage();
  instance.enabled = true;
  return instance;
});


/* global DeviceStorageHelper */
/**
 * Links the root panel list item with AppStorage.
 */
define('panels/root/storage_app_item',['require','modules/app_storage'],function(require) {
  

  var AppStorage = require('modules/app_storage');

  /**
   * @alias module:panels/root/storage_app_item
   * @class AppStorageItem
   * @requires module:modules/app_storage
   * @param {HTMLElement} element
                          The element displaying the app storage information
   * @returns {AppStorageItem}
   */
  function AppStorageItem(element) {
    this._enabled = false;
    this._element = element;
    this._boundUpdateAppFreeSpace = this._updateAppFreeSpace.bind(this);
  }

  AppStorageItem.prototype = {
    /**
     * The value indicates whether the module is responding. If it is false, the
     * UI stops reflecting the updates from the root panel context.
     *
     * @access public
     * @memberOf AppStorageItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      if (this._enabled === value) {
        return;
      } else {
        this._enabled = value;
      }
      if (value) { //observe
        AppStorage.storage.observe('freeSize', this._boundUpdateAppFreeSpace);
        this._updateAppFreeSpace();
        window.addEventListener('localized', this);
      } else { //unobserve
        AppStorage.storage.unobserve('freeSize', this._boundUpdateAppFreeSpace);
        window.removeEventListener('localized', this);
      }
    },

    // Application Storage
    _updateAppFreeSpace: function storage_updateAppFreeSpace() {
      DeviceStorageHelper.showFormatedSize(this._element,
        'availableSize', AppStorage.storage.freeSize);
    },

    handleEvent: function storage_handleEvent(evt) {
      switch (evt.type) {
        case 'localized':
          this._updateAppFreeSpace();
          break;
      }
    }
  };

  return function ctor_app_storage_item(element) {
    return new AppStorageItem(element);
  };
});

define('panels/root/wifi_item',['require','modules/wifi_context'],function(require) {
  

  var WifiContext = require('modules/wifi_context');
  var wifiManager = navigator.mozWifiManager;

  function WifiItem(element) {
    this._enabled = false;
    this._boundUpdateWifiDesc = this._updateWifiDesc.bind(this, element);
  }

  WifiItem.prototype = {
    set enabled(value) {
      if (value === this._enabled || !wifiManager) {
        return;
      }

      this._enabled = value;
      if (this._enabled) {
        this._boundUpdateWifiDesc();
        WifiContext.addEventListener('wifiStatusTextChange',
          this._boundUpdateWifiDesc);
      } else {
        WifiContext.removeEventListener('wifiStatusTextChange',
          this._boundUpdateWifiDesc);
      }
    },

    get enabled() {
      return this._enabled;
    },

    _updateWifiDesc: function root_updateWifiDesc(element) {
      element.setAttribute('data-l10n-id', WifiContext.wifiStatusText.id);
      if (WifiContext.wifiStatusText.args) {
        element.setAttribute('data-l10n-args',
          JSON.stringify(WifiContext.wifiStatusText.args));
      } else {
        element.removeAttribute('data-l10n-args');
      }
    }
  };

  return function ctor_wifiItem(element) {
    return new WifiItem(element);
  };
});

define('panels/root/screen_lock_item',['require','shared/settings_listener'],function(require) {
  

  var SettingsListener = require('shared/settings_listener');

  function ScreenLockItem(element) {
    this._itemEnabled = false;
    this._observedKey = 'lockscreen.enabled';
    this._element = element;
    this._boundUpdateUI = this._updateUI.bind(this);
  }
  
  ScreenLockItem.prototype = {
    set enabled(value) {
      if (value === this._itemEnabled) {
        return;
      } else {
        this._itemEnabled = value;
        if (this._itemEnabled) {
          SettingsListener.observe(this._observedKey, false,
            this._boundUpdateUI);
        } else {
          SettingsListener.unobserve(this._observedKey, this._boundUpdateUI);
        }
      }
    },

    get enabled() {
      return this._itemEnabled;
    },

    _updateUI: function sl_updateUI(enabled) {
      var l10nId = enabled ? 'enabled' : 'disabled';
      this._element.setAttribute('data-l10n-id', l10nId);
    }
  };

  return function ctor_screen_lock_item(element) {
    return new ScreenLockItem(element);
  };
});

/**
 * SimSecurityItem is manily used in Single Sim device because this would
 * be integrated into Sim Manager > Sim Security in DSDS devices.
 *
 * @module SimSecurityItem
 */
define('panels/root/sim_security_item',['require','shared/simslot_manager','shared/airplane_mode_helper'],function(require) {
  

  var SIMSlotManager = require('shared/simslot_manager');
  var AirplaneModeHelper = require('shared/airplane_mode_helper');

  function SimSecurityItem(element) {
    this._element = element;
    this._itemEnabled = false;
    this._activeSlot = this._getActiveSlot();
    this._boundUpdateUI = this._updateUI.bind(this);
  }

  SimSecurityItem.prototype = {
    /**
     * Set the current status of SimSecurityItem
     *
     * @access public
     * @param {Boolean} enabled
     * @memberOf SimSecurityItem
     */
    set enabled(enabled) {
      // 1. SimSecurityItem only shows up on Single SIM devices
      // 2. If there is no activeSlot, it means we don't have to do anything
      // 3. If internal variable is enabled and we still want to enable,
      // we don't have to do anything and vice versa.
      if (SIMSlotManager.isMultiSIM() ||
        !this._activeSlot || enabled === this._itemEnabled) {
          return;
      }

      this._itemEnabled = enabled;
      if (this._itemEnabled) {
        this._boundUpdateUI();
        this._activeSlot.conn.addEventListener('cardstatechange',
          this._boundUpdateUI);
        AirplaneModeHelper.addEventListener('statechange',
          this._boundUpdateUI);
      } else {
        this._activeSlot.conn.removeEventListener('cardstatechange',
          this._boundUpdateUI);
        AirplaneModeHelper.removeEventListener('statechange',
          this._boundUpdateUI);
      }
    },

    /**
     * Get the current status of SimSecurityItem
     *
     * @access public
     * @memberOf SimSecurityItem
     */
    get enabled() {
      return this._itemEnabled;
    },

    /**
     * This method is used to update UI based on statuses of SIM / APM
     *
     * @access private
     * @memberOf SimSecurityItem
     */
    _updateUI: function() {
      var self = this;
      AirplaneModeHelper.ready(function() {
        // if disabled
        self._element.style.fontStyle = 'italic';

        // if APM is enabled
        var airplaneModeStatus = AirplaneModeHelper.getStatus();
        if (airplaneModeStatus === 'enabled') {
          self._element.setAttribute('data-l10n-id', 'simCardNotReady');
          return;
        }

        var cardState = self._activeSlot.simCard.cardState;
        switch(cardState) {
          case null:
            self._element.setAttribute('data-l10n-id', 'noSimCard');
            return;
          case 'unknown':
            self._element.setAttribute('data-l10n-id', 'unknownSimCardState');
            return;
        }

        // enabled instead
        self._element.style.fontStyle = 'normal';

        // with SIM card, query its status
        var icc = self._activeSlot.simCard;
        var req = icc.getCardLock('pin');
        req.onsuccess = function spl_checkSuccess() {
          var enabled = req.result.enabled;
          self._element.setAttribute('data-l10n-id',
            enabled ? 'enabled' : 'disabled');
        };
      });
    },

    /**
     * We use this to get active Sim slot.
     *
     * @access private
     * @memberOf SimSecurityItem
     */
    _getActiveSlot: function() {
      var activeSlot;
      SIMSlotManager.getSlots().forEach(function(SIMSlot) {
        if (!SIMSlot.isAbsent()) {
          activeSlot = SIMSlot;
        }
      });
      return activeSlot;
    }
  };

  return function ctor_sim_security_item(element) {
    return new SimSecurityItem(element);
  };
});

/**
 * This module is used to control the background stuff when users
 * toggle on/off airplane mode checkbox.
 *
 * @module panels/root/airplane_mode_item
 */
define('panels/root/airplane_mode_item',['require','shared/airplane_mode_helper'],function(require) {
  

  var AirplaneModeHelper = require('shared/airplane_mode_helper');

  /**
   * @alias module:panels/root/airplane_mode_item
   * @class AirplaneModeItem
   * @param {HTMLElement} element the checkbox for airplane mode
   * @returns {AirplaneModeItem}
   */
  function AirplaneModeItem(element) {
    this._itemEnabled = false;
    this._element = element;
    this.init();
    this._boundAPMStateChange = this._onAPMStateChange.bind(this);
  }

  AirplaneModeItem.prototype = {
    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf AirplaneModeItem.prototype
     * @type {Boolean}
     */
    set enabled(value) {
      if (this._itemEnabled === value) {
        return;
      } else {
        this._itemEnabled = value;
        if (this._itemEnabled) {
          AirplaneModeHelper.addEventListener('statechange',
            this._boundAPMStateChange);
        } else {
          AirplaneModeHelper.removeEventListener('statechange',
            this._boundAPMStateChange);
        }
      }
    },

    /**
     * The value indicates whether the module is responding.
     *
     * @access public
     * @memberOf AirplaneModeItem.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._itemEnabled;
    },

    /**
     * This function is used to reflect current status of APM to checkbox
     *
     * @access private
     * @memberOf AirplaneModeItem.prototype
     * @param {String} status current status of APM
     * @type {Function}
     */
    _onAPMStateChange: function ami_onAPMStateChange(status) {
      if (status === 'enabled' || status === 'disabled') {
        this._element.checked = (status === 'enabled') ? true : false;
        this._element.disabled = false;
      } else {
        this._element.disabled = true;
      }
    },

    /**
     * Initialize function
     *
     * @access public
     * @memberOf AirplaneModeItem.prototype
     * @type {Function}
     */
    init: function ami_init() {
      AirplaneModeHelper.ready(function() {
        // handle change on radio
        this._element.addEventListener('change', function(e) {
          this.disabled = true;
          AirplaneModeHelper.setEnabled(this.checked);
        });

        // initial status
        var status = AirplaneModeHelper.getStatus();
        this._element.checked = (status === 'enabled') ? true : false;
        this._element.disabled = false;
      }.bind(this));
    }
  };

  return function ctor_airplane_mode_item(element) {
    return new AirplaneModeItem(element);
  };
});

/**
 * This module is used to show/hide themes menuItem based on the number of
 * current installed themes.
 *
 * @module ThemesItem
 */
define('panels/root/themes_item',['require','modules/apps_cache'],function(require) {
  

  var AppsCache = require('modules/apps_cache');

  function ThemesItem(element) {
    this._enabled = false;
    this._element = element; 
    this._themeCount = 0;
    this._boundUpdateThemes = this._updateThemes.bind(this);
    this.init();
  }

  ThemesItem.prototype = {
    /**
     * Set current status of themesItem
     *
     * @access public
     * @param {Boolean} enabled
     * @memberOf ThemesItem
     */
    set enabled(enabled) {
      if (this._enabled === enabled) {
        return;
      } else {
        this._enabled = enabled;
        if (this._enabled) {
          this._updateThemeSectionVisibility();
        }
      }
    },

    /**
     * Get current status of themesItem
     *
     * @access public
     * @memberOf ThemesItem
     */
    get enabled() {
      return this._enabled;
    },

    /**
     * Initialization
     *
     * @access private
     * @memberOf ThemesItem
     * @return {Promise}
     */
    init: function() {
      var self = this;
      AppsCache.addEventListener('install', this._boundUpdateThemes);
      AppsCache.addEventListener('uninstall', this._boundUpdateThemes);
      return AppsCache.apps().then(function(apps) {
        apps.some(function(app) {
          if (self._isThemeApp(app)) {
            self._themeCount += 1;
          }
        });
        self._updateThemeSectionVisibility();
      });
    },

    /**
     * Check whether this app is theme app
     *
     * @param {Object} app
     * @returns {Boolean}
     * @memberOf ThemesItem
     */
    _isThemeApp: function(app) {
      var manifest = app.manifest || app.updateManifest;
      return manifest.role === 'theme';
    },

    /**
     * We have to update theme count based on incoming evt and
     * decide to show/hide or not.
     *
     * @param {Object} evt
     * @memberOf ThemesItem
     */
    _updateThemes: function(evt) {
      var app = evt && evt.application;
      var type = evt.type;

      if (this._isThemeApp(app)) {
        if (type === 'install') {
          this._themeCount += 1;
        } else if (type === 'uninstall') {
          this._themeCount -= 1;
        }
        this._updateThemeSectionVisibility();
      }
    },

    /**
     * Update theme section visibility based on _themeCount
     *
     * @memberOf ThemesItem
     */
    _updateThemeSectionVisibility: function() {
      this._element.hidden = (this._themeCount < 2);
    }
  };

  return function ctor_themesItem(element) {
    return new ThemesItem(element);
  };
});

/**
 * HomescreenItem is used to handle the visibility of this menuItem
 *
 * @module HomescreenItem
 */
define('panels/root/homescreen_item',['require','modules/apps_cache'],function(require) {
  

  var AppsCache = require('modules/apps_cache');

  var HomescreenItem = function(element) {
    this._itemEnabled = false;
    this._element = element;
    this._boundToggleHomescreenSection =
      this._updateHomescreenSection.bind(this);
  };

  HomescreenItem.prototype = {
    /**
     * Set the current status of HomescreenItem
     * 
     * @access public
     * @param {Boolean} value
     * @memberOf HomescreenItem
     */
    set enabled(value) {
      if (this._itemEnabled === value) {
        return;
      } else {
        this._itemEnabled = value;
        if (this._itemEnabled) {
          this._boundToggleHomescreenSection();
          AppsCache.addEventListener('oninstall',
            this._boundToggleHomescreenSection);
          AppsCache.addEventListener('onuninstall',
            this._boundToggleHomescreenSection);
        } else {
          AppsCache.removeEventListener('oninstall',
            this._boundToggleHomescreenSection);
          AppsCache.removeEventListener('onuninstall',
            this._boundToggleHomescreenSection);
        }
      }
    },

    /**
     * Get the current status of HomescreenItem
     *
     * @access public
     * @memberOf HomescreenItem
     */
    get enabled() {
      return this._itemEnabled;
    },

    /**
     * Toggle the visibility of homescreen menuItem
     *
     * @access private
     * @memberOf HomescreenItem
     * @return {Promise}
     */
    _updateHomescreenSection: function h__updateHomescreenSection() {
      var self = this;
      return AppsCache.apps().then(function(apps) {
        var homescreenApps = self._getHomescreenApps(apps);
        if (homescreenApps.length < 2) {
          self._element.hidden = true;
        } else {
          self._element.hidden = false;
        }
      });
    },

    /**
     * Get homescreen related apps
     *
     * @access private
     * @param {Array.<Object>} apps - all installed apps
     * @memberOf HomescreenItem
     * @return {Array.<Object>} homescreen apps
     */
    _getHomescreenApps: function h__getHomescreenApps(apps) {
      return apps.filter(function(app) {
        var manifest = app.manifest || app.updateManifest;
        var role = manifest && manifest.role;
        return role === 'homescreen';
      });
    }
  };
  
  return function ctor_homescreenItem(element) {
    return new HomescreenItem(element);
  };
});

/**
 * PrivacyPanelItem provides the transition to Privacy Panel app.
 *
 * @module PrivacyPanelItem
 */

define('panels/root/privacy_panel_item',['require','modules/apps_cache'],function(require) {
  

  var AppsCache = require('modules/apps_cache');

  function PrivacyPanelItem(args) {
    this._element = args.element;
    this._link = args.link;
    this._app = null;

    this._privacyPanelManifestURL = document.location.protocol +
      '//privacy-panel.gaiamobile.org' +
      (location.port ? (':' + location.port) : '') + '/manifest.webapp';

    this._getApp();

    this._element.addEventListener('click', this._launch.bind(this));
  }

  PrivacyPanelItem.prototype = {

    /**
     * Set current status of privacyPanelItem
     *
     * @access public
     * @param {Boolean} enabled
     * @memberOf PrivacyPanelItem
     */
    set enabled(enabled) {
      if (this._enabled === enabled) {
        return;
      } else {
        this._enabled = enabled;
        if (this._enabled) {
          this._blurLink();
        }
      }
    },

    /**
     * Get current status of privacyPanelItem
     *
     * @access public
     * @memberOf PrivacyPanelItem
     */
    get enabled() {
      return this._enabled;
    },

    /**
     * Search from privacy-panel app and grab it's instance.
     * @memberOf PrivacyPanelItem
     */
    _getApp: function pp_getApp() {
      return AppsCache.apps().then(function(apps) {
        var i, app;
        for (i = 0; i < apps.length; i++) {
          app = apps[i];
          if (app.manifestURL === this._privacyPanelManifestURL) {
            this._app = app;
            this._element.removeAttribute('hidden');
            return;
          }
        }
      }.bind(this));
    },

    /**
     * Launch Privacy Panel app.
     *
     * @param {Event} event
     * @memberOf PrivacyPanelItem
     */
    _launch: function pp_launch(event) {
      // Stop propagation & prevent default not to block other settings events.
      event.stopImmediatePropagation();
      event.preventDefault();
      
      if (this._app) {
        // Let privacy-panel app know that we launched it from settings
        // so the app can show us a back button pointing to settings app.
        var flag = navigator.mozSettings.createLock().set({
          'privacypanel.launched.by.settings': true
        });
        flag.onsuccess = function() {
          this._app.launch();
        }.bind(this);
        flag.onerror = function() {
          console.error('Problem with launching Privacy Panel');
          alert('Problem with launching Privacy Panel');
        };
      } else {
        alert(navigator.mozL10n.get('no-privacy-panel'));
      }
    },

    /**
     * Blur link.
     *
     * @memberOf PrivacyPanelItem
     */
    _blurLink: function pp_blurLink() {
      this._link.blur();
    }
  };

  return function ctor_privacyPanelItem(element) {
    return new PrivacyPanelItem(element);
  };
});

define('panels/root/panel',['require','modules/settings_panel','panels/root/root','panels/root/bluetooth_item','panels/root/nfc_item','panels/root/language_item','panels/root/battery_item','panels/root/findmydevice_item','panels/root/storage_usb_item','panels/root/storage_app_item','panels/root/wifi_item','panels/root/screen_lock_item','panels/root/sim_security_item','panels/root/airplane_mode_item','panels/root/themes_item','panels/root/homescreen_item','panels/root/privacy_panel_item'],function(require) {
  

  var SettingsPanel = require('modules/settings_panel');
  var Root = require('panels/root/root');
  var BluetoothItem = require('panels/root/bluetooth_item');
  var NFCItem = require('panels/root/nfc_item');
  var LanguageItem = require('panels/root/language_item');
  var BatteryItem = require('panels/root/battery_item');
  var FindMyDeviceItem = require('panels/root/findmydevice_item');
  var StorageUSBItem = require('panels/root/storage_usb_item');
  var StorageAppItem = require('panels/root/storage_app_item');
  var WifiItem = require('panels/root/wifi_item');
  var ScreenLockItem = require('panels/root/screen_lock_item');
  var SimSecurityItem = require('panels/root/sim_security_item');
  var AirplaneModeItem = require('panels/root/airplane_mode_item');
  var ThemesItem = require('panels/root/themes_item');
  var HomescreenItem = require('panels/root/homescreen_item');
  var PrivacyPanelItem = require('panels/root/privacy_panel_item');

  return function ctor_root_panel() {
    var root = Root();
    var bluetoothItem;
    var nfcItem;
    var languageItem;
    var batteryItem;
    var findMyDeviceItem;
    var storageUsbItem;
    var storageAppItem;
    var wifiItem;
    var screenLockItem;
    var simSecurityItem;
    var airplaneModeItem;
    var themesItem;
    var homescreenItem;
    var privacyPanelItem;

    return SettingsPanel({
      onInit: function rp_onInit(panel) {
        root.init();
        bluetoothItem = BluetoothItem(panel.querySelector('.bluetooth-desc'));
        nfcItem = NFCItem(panel.querySelector('.nfc-settings'));
        languageItem = LanguageItem(panel.querySelector('.language-desc'));
        batteryItem = BatteryItem(panel.querySelector('.battery-desc'));
        findMyDeviceItem = FindMyDeviceItem(
          panel.querySelector('.findmydevice-desc'));

        var storageDialog = document.querySelector('.turn-on-ums-dialog');
        storageUsbItem = StorageUSBItem({
          mediaStorageDesc: panel.querySelector('.media-storage-desc'),
          usbEnabledCheckBox: panel.querySelector('.usb-switch'),
          usbStorage: panel.querySelector('#menuItem-enableStorage'),
          usbEnabledInfoBlock: panel.querySelector('.usb-desc'),
          umsWarningDialog: storageDialog,
          umsConfirmButton: storageDialog.querySelector('.ums-confirm-option'),
          umsCancelButton: storageDialog.querySelector('.ums-cancel-option'),
          mediaStorageSection: panel.querySelector('.media-storage-section')
        });
        storageAppItem = StorageAppItem(
          panel.querySelector('.application-storage-desc'));
        wifiItem = WifiItem(panel.querySelector('#wifi-desc'));
        screenLockItem =
          ScreenLockItem(panel.querySelector('.screenLock-desc'));
        airplaneModeItem =
          AirplaneModeItem(panel.querySelector('.airplaneMode-input'));
        simSecurityItem =
          SimSecurityItem(panel.querySelector('.simCardLock-desc'));
        themesItem =
          ThemesItem(panel.querySelector('.themes-section'));
        homescreenItem =
          HomescreenItem(panel.querySelector('#homescreens-section'));
        privacyPanelItem = PrivacyPanelItem({
          element: panel.querySelector('.privacy-panel-item'),
          link: panel.querySelector('.privacy-panel-item a')
        });
      },
      onBeforeShow: function rp_onBeforeShow() {
        bluetoothItem.enabled = true;
        languageItem.enabled = true;
        batteryItem.enabled = true;
        findMyDeviceItem.enabled = true;
        storageUsbItem.enabled = true;
        storageAppItem.enabled = true;
        wifiItem.enabled = true;
        screenLockItem.enabled = true;
        simSecurityItem.enabled = true;
        airplaneModeItem.enabled = true;
        themesItem.enabled = true;
        privacyPanelItem.enabled = true;
      },
      onShow: function rp_onShow() {
        homescreenItem.enabled = true;
      },
      onHide: function rp_onHide() {
        bluetoothItem.enabled = false;
        languageItem.enabled = false;
        batteryItem.enabled = false;
        findMyDeviceItem.enabled = false;
        storageUsbItem.enabled = false;
        storageAppItem.enabled = false;
        wifiItem.enabled = false;
        screenLockItem.enabled = false;
        simSecurityItem.enabled = false;
        airplaneModeItem.enabled = false;
        themesItem.enabled = false;
        homescreenItem.enabled = false;
        privacyPanelItem.enabled = false;
      }
    });
  };
});
