/* exported BlobView */
'use strict';

var BlobView = (function() {
  function fail(msg) {
    throw Error(msg);
  }

  // This constructor is for internal use only.
  // Use the BlobView.get() factory function or the getMore instance method
  // to obtain a BlobView object.
  function BlobView(blob, sliceOffset, sliceLength, slice,
                    viewOffset, viewLength, littleEndian)
  {
    this.blob = blob;                  // The parent blob that the data is from
    this.sliceOffset = sliceOffset;    // The start address within the blob
    this.sliceLength = sliceLength;    // How long the slice is
    this.slice = slice;                // The ArrayBuffer of slice data
    this.viewOffset = viewOffset;      // The start of the view within the slice
    this.viewLength = viewLength;      // The length of the view
    this.littleEndian = littleEndian;  // Read little endian by default?

    // DataView wrapper around the ArrayBuffer
    this.view = new DataView(slice, viewOffset, viewLength);

    // These fields mirror those of DataView
    this.buffer = slice;
    this.byteLength = viewLength;
    this.byteOffset = viewOffset;

    this.index = 0;   // The read methods keep track of the read position
  }

  // Async factory function
  BlobView.get = function(blob, offset, length, callback, littleEndian) {
    if (offset < 0) {
      fail('negative offset');
    }
    if (length < 0) {
      fail('negative length');
    }
    if (offset > blob.size) {
      fail('offset larger than blob size');
    }
    // Don't fail if the length is too big; just reduce the length
    if (offset + length > blob.size) {
      length = blob.size - offset;
    }
    var slice = blob.slice(offset, offset + length);
    var reader = new FileReader();
    reader.readAsArrayBuffer(slice);
    reader.onloadend = function() {
      var result = null;
      if (reader.result) {
        result = new BlobView(blob, offset, length, reader.result,
                              0, length, littleEndian || false);
      }
      callback(result, reader.error);
    };
  };

  // Synchronous factory function for use when you have an array buffer and want
  // to treat it as a BlobView (e.g. to use the readXYZ functions). We need
  // this for the music app, which uses an array buffer to hold
  // de-unsynchronized ID3 frames.
  BlobView.getFromArrayBuffer = function(buffer, offset, length, littleEndian) {
    return new BlobView(null, offset, length, buffer, offset, length,
                        littleEndian);
  };

  BlobView.prototype = {
    constructor: BlobView,

    // This instance method is like the BlobView.get() factory method,
    // but it is here because if the current buffer includes the requested
    // range of bytes, they can be passed directly to the callback without
    // going back to the blob to read them
    getMore: function(offset, length, callback) {
      // If we made this BlobView from an array buffer, there's no blob backing
      // it, and so it's impossible to get more data.
      if (!this.blob)
        fail('no blob backing this BlobView');

      if (offset >= this.sliceOffset &&
          offset + length <= this.sliceOffset + this.sliceLength) {
        // The quick case: we already have that region of the blob
        callback(new BlobView(this.blob,
                              this.sliceOffset, this.sliceLength, this.slice,
                              offset - this.sliceOffset, length,
                              this.littleEndian));
      }
      else {
        // Otherwise, we have to do an async read to get more bytes
        BlobView.get(this.blob, offset, length, callback, this.littleEndian);
      }
    },

    // Set the default endianness for the other methods
    littleEndian: function() {
      this.littleEndian = true;
    },
    bigEndian: function() {
      this.littleEndian = false;
    },

    // These "get" methods are just copies of the DataView methods, except
    // that they honor the default endianness
    getUint8: function(offset) {
      return this.view.getUint8(offset);
    },
    getInt8: function(offset) {
      return this.view.getInt8(offset);
    },
    getUint16: function(offset, le) {
      return this.view.getUint16(offset,
                                 le !== undefined ? le : this.littleEndian);
    },
    getInt16: function(offset, le) {
      return this.view.getInt16(offset,
                                le !== undefined ? le : this.littleEndian);
    },
    getUint32: function(offset, le) {
      return this.view.getUint32(offset,
                                 le !== undefined ? le : this.littleEndian);
    },
    getInt32: function(offset, le) {
      return this.view.getInt32(offset,
                                le !== undefined ? le : this.littleEndian);
    },
    getFloat32: function(offset, le) {
      return this.view.getFloat32(offset,
                                  le !== undefined ? le : this.littleEndian);
    },
    getFloat64: function(offset, le) {
      return this.view.getFloat64(offset,
                                  le !== undefined ? le : this.littleEndian);
    },

    // These "read" methods read from the current position in the view and
    // update that position accordingly
    readByte: function() {
      return this.view.getInt8(this.index++);
    },
    readUnsignedByte: function() {
      return this.view.getUint8(this.index++);
    },
    readShort: function(le) {
      var val = this.view.getInt16(this.index,
                                   le !== undefined ? le : this.littleEndian);
      this.index += 2;
      return val;
    },
    readUnsignedShort: function(le) {
      var val = this.view.getUint16(this.index,
                                    le !== undefined ? le : this.littleEndian);
      this.index += 2;
      return val;
    },
    readInt: function(le) {
      var val = this.view.getInt32(this.index,
                                   le !== undefined ? le : this.littleEndian);
      this.index += 4;
      return val;
    },
    readUnsignedInt: function(le) {
      var val = this.view.getUint32(this.index,
                                    le !== undefined ? le : this.littleEndian);
      this.index += 4;
      return val;
    },
    readFloat: function(le) {
      var val = this.view.getFloat32(this.index,
                                     le !== undefined ? le : this.littleEndian);
      this.index += 4;
      return val;
    },
    readDouble: function(le) {
      var val = this.view.getFloat64(this.index,
                                     le !== undefined ? le : this.littleEndian);
      this.index += 8;
      return val;
    },

    // Methods to get and set the current position
    tell: function() {
      return this.index;
    },
    remaining: function() {
      return this.byteLength - this.index;
    },
    seek: function(index) {
      if (index < 0) {
        fail('negative index');
      }
      if (index > this.byteLength) {
        fail('index greater than buffer size');
      }
      this.index = index;
    },
    advance: function(n) {
      var index = this.index + n;
      if (index < 0) {
        fail('advance past beginning of buffer');
      }
      // It's usual that when we finished reading one target view,
      // the index is advanced to the start(previous end + 1) of next view,
      // and the new index will be equal to byte length(the last index + 1),
      // we will not fail on it because it means the reading is finished,
      // or do we have to warn here?
      if (index > this.byteLength) {
        fail('advance past end of buffer');
      }
      this.index = index;
    },

    // Additional methods to read other useful things
    getUnsignedByteArray: function(offset, n) {
      return new Uint8Array(this.buffer, offset + this.viewOffset, n);
    },

    // Additional methods to read other useful things
    readUnsignedByteArray: function(n) {
      var val = new Uint8Array(this.buffer, this.index + this.viewOffset, n);
      this.index += n;
      return val;
    },

    getBit: function(offset, bit) {
      var byte = this.view.getUint8(offset);
      return (byte & (1 << bit)) !== 0;
    },

    getUint24: function(offset, le) {
      var b1, b2, b3;
      if (le !== undefined ? le : this.littleEndian) {
        b1 = this.view.getUint8(offset);
        b2 = this.view.getUint8(offset + 1);
        b3 = this.view.getUint8(offset + 2);
      }
      else {    // big end first
        b3 = this.view.getUint8(offset);
        b2 = this.view.getUint8(offset + 1);
        b1 = this.view.getUint8(offset + 2);
      }

      return (b3 << 16) + (b2 << 8) + b1;
    },

    readUint24: function(le) {
      var value = this.getUint24(this.index, le);
      this.index += 3;
      return value;
    },

    // There are lots of ways to read strings.
    // ASCII, UTF-8, UTF-16.
    // null-terminated, character length, byte length
    // I'll implement string reading methods as needed

    getASCIIText: function(offset, len) {
      var bytes = new Uint8Array(this.buffer, offset + this.viewOffset, len);
      return String.fromCharCode.apply(String, bytes);
    },

    readASCIIText: function(len) {
      var bytes = new Uint8Array(this.buffer,
                                 this.index + this.viewOffset,
                                 len);
      this.index += len;
      return String.fromCharCode.apply(String, bytes);
    },

    // Replace this with the StringEncoding API when we've got it.
    // See https://bugzilla.mozilla.org/show_bug.cgi?id=764234
    getUTF8Text: function(offset, len) {
      function fail() { throw new Error('Illegal UTF-8'); }

      var pos = offset;         // Current position in this.view
      var end = offset + len;   // Last position
      var charcode;             // Current charcode
      var s = '';               // Accumulate the string
      var b1, b2, b3, b4;       // Up to 4 bytes per charcode

      // See http://en.wikipedia.org/wiki/UTF-8
      while (pos < end) {
        var b1 = this.view.getUint8(pos);
        if (b1 < 128) {
          s += String.fromCharCode(b1);
          pos += 1;
        }
        else if (b1 < 194) {
          // unexpected continuation character...
          fail();
        }
        else if (b1 < 224) {
          // 2-byte sequence
          if (pos + 1 >= end) {
            fail();
          }
          b2 = this.view.getUint8(pos + 1);
          if (b2 < 128 || b2 > 191) {
            fail();
          }
          charcode = ((b1 & 0x1f) << 6) + (b2 & 0x3f);
          s += String.fromCharCode(charcode);
          pos += 2;
        }
        else if (b1 < 240) {
          // 3-byte sequence
          if (pos + 2 >= end) {
            fail();
          }
          b2 = this.view.getUint8(pos + 1);
          if (b2 < 128 || b2 > 191) {
            fail();
          }
          b3 = this.view.getUint8(pos + 2);
          if (b3 < 128 || b3 > 191) {
            fail();
          }
          charcode = ((b1 & 0x0f) << 12) + ((b2 & 0x3f) << 6) + (b3 & 0x3f);
          s += String.fromCharCode(charcode);
          pos += 3;
        }
        else if (b1 < 245) {
          // 4-byte sequence
          if (pos + 3 >= end) {
            fail();
          }
          b2 = this.view.getUint8(pos + 1);
          if (b2 < 128 || b2 > 191) {
            fail();
          }
          b3 = this.view.getUint8(pos + 2);
          if (b3 < 128 || b3 > 191) {
            fail();
          }
          b4 = this.view.getUint8(pos + 3);
          if (b4 < 128 || b4 > 191) {
            fail();
          }
          charcode = ((b1 & 0x07) << 18) +
            ((b2 & 0x3f) << 12) +
            ((b3 & 0x3f) << 6) +
            (b4 & 0x3f);

          // Now turn this code point into two surrogate pairs
          charcode -= 0x10000;
          s += String.fromCharCode(0xd800 + ((charcode & 0x0FFC00) >>> 10));
          s += String.fromCharCode(0xdc00 + (charcode & 0x0003FF));

          pos += 4;
        }
        else {
          // Illegal byte
          fail();
        }
      }

      return s;
    },

    readUTF8Text: function(len) {
      try {
        return this.getUTF8Text(this.index, len);
      }
      finally {
        this.index += len;
      }
    },

    // Get UTF16 text.  If le is not specified, expect a BOM to define
    // endianness.  If le is true, read UTF16LE, if false, UTF16BE.
    getUTF16Text: function(offset, len, le) {
      if (len % 2) {
        fail('len must be a multiple of two');
      }
      if (le === null || le === undefined) {
        var BOM = this.getUint16(offset);
        len -= 2;
        offset += 2;
        if (BOM === 0xFEFF)
          le = false;
        else
          le = true;
      }

      // We need to support unaligned reads, so we can't use a Uint16Array here.
      var s = '';
      for (var i = 0; i < len; i += 2)
        s += String.fromCharCode(this.getUint16(offset + i, le));
      return s;
    },

    readUTF16Text: function(len, le) {
      var value = this.getUTF16Text(this.index, len, le);
      this.index += len;
      return value;
    },

    // Read 4 bytes, ignore the high bit and combine them into a 28-bit
    // big-endian unsigned integer.
    // This format is used by the ID3v2 metadata.
    getID3Uint28BE: function(offset) {
      var b1 = this.view.getUint8(offset) & 0x7f;
      var b2 = this.view.getUint8(offset + 1) & 0x7f;
      var b3 = this.view.getUint8(offset + 2) & 0x7f;
      var b4 = this.view.getUint8(offset + 3) & 0x7f;
      return (b1 << 21) | (b2 << 14) | (b3 << 7) | b4;
    },

    readID3Uint28BE: function() {
      var value = this.getID3Uint28BE(this.index);
      this.index += 4;
      return value;
    },

    // Read bytes up to and including a null terminator, but never
    // more than size bytes.  And return as a Latin1 string
    readNullTerminatedLatin1Text: function(size) {
      var s = '';
      for (var i = 0; i < size; i++) {
        var charcode = this.view.getUint8(this.index + i);
        if (charcode === 0) {
          i++;
          break;
        }
        s += String.fromCharCode(charcode);
      }
      this.index += i;
      return s;
    },

    // Read bytes up to and including a null terminator, but never
    // more than size bytes.  And return as a UTF8 string
    readNullTerminatedUTF8Text: function(size) {
      for (var len = 0; len < size; len++) {
        if (this.view.getUint8(this.index + len) === 0) {
          break;
        }
      }
      var s = this.readUTF8Text(len);
      if (len < size) {    // skip the null terminator if we found one
        this.advance(1);
      }
      return s;
    },

    // Read UTF16 text.  If le is not specified, expect a BOM to define
    // endianness.  If le is true, read UTF16LE, if false, UTF16BE
    // Read until we find a null-terminator, but never more than size bytes.
    readNullTerminatedUTF16Text: function(size, le) {
      if (size % 2) {
        fail('size must be a multiple of two');
      }
      if (le === null || le === undefined) {
        var BOM = this.readUnsignedShort();
        size -= 2;
        if (BOM === 0xFEFF) {
          le = false;
        } else {
          le = true;
        }
      }

      var s = '';
      for (var i = 0; i < size; i += 2) {
        var charcode = this.getUint16(this.index + i, le);
        if (charcode === 0) {
          i += 2;
          break;
        }
        s += String.fromCharCode(charcode);
      }
      this.index += i;
      return s;
    }
  };

  // We don't want users of this library to accidentally call the constructor
  // instead of using one of the factory functions, so we return a dummy object
  // instead of the real constructor. If someone really needs to get at the
  // real constructor, the contructor property of the prototype refers to it.
  return {
    get: BlobView.get,
    getFromArrayBuffer: BlobView.getFromArrayBuffer
  };
}());

/* global ForwardLockMetadata, ID3v1Metadata, ID3v2Metadata, LazyLoader,
   MP4Metadata, OggMetadata */
/* exported MetadataFormats */
'use strict';

/**
 * Delegates metadata parsing to the appropriate parser based on magic header
 * values.
 */
var MetadataFormats = (function() {

  /*
   * This is the list of formats that we know how to parse. Each format has
   * three properties:
   *
   * @property {String} file The path to the file for parsing this metadata
   *   format.
   * @property {Object} module A getter that returns the module object for this
   *   parser. Note: It *must* be a getter because it needs to be evaluated
   *   *after* the file is loaded.
   * @property {Function} match A function that takes a BlobView of the file
   *   and returns true if the file uses this metadata format.
   */
  var formats = [
    {
      file: 'js/metadata/forward_lock.js',
      get module() { return ForwardLockMetadata; },
      match: function(header) {
        return header.getASCIIText(0, 9) === 'LOCKED 1 ';
      }
    },
    {
      file: 'js/metadata/id3v2.js',
      get module() { return ID3v2Metadata; },
      match: function(header) {
        return header.getASCIIText(0, 3) === 'ID3';
      }
    },
    {
      file: 'js/metadata/ogg.js',
      get module() { return OggMetadata; },
      match: function(header) {
        return header.getASCIIText(0, 4) === 'OggS';
      }
    },
    {
      file: 'js/metadata/mp4.js',
      get module() { return MP4Metadata; },
      match: function(header) {
        return header.getASCIIText(4, 4) === 'ftyp';
      }
    },
    {
      file: 'js/metadata/id3v1.js',
      get module() { return ID3v1Metadata; },
      match: function(header) {
        return (header.getUint16(0, false) & 0xFFFE) === 0xFFFA;
      }
    }
  ];

  /**
   * Create a new metadata parser for a particular file format.
   *
   * @param {Object} formatInfo A description of the file format containing a
   *   `file` attribute for the file to load, and a `module` attribute returning
   *   the module that contains the parse() method we should call.
   */
  function MetadataParser(formatInfo) {
    this._formatInfo = formatInfo;
  }

  MetadataParser.prototype = {
    /**
     * Parse a file and return a Promise with the metadata.
     *
     * @param {BlobView} header The file in question.
     * @param {Metadata} metadata The (partially filled-in) metadata object.
     * @return {Promise} A Promise that resolves with the completed metadata
     *   object.
     */
    parse: function(header, metadata) {
      var info = this._formatInfo;
      return new Promise(function(resolve, reject) {
        LazyLoader.load(info.file, function() {
          resolve(info.module.parse(header, metadata));
        });
      });
    }
  };

  /**
   * Find the appropriate metadata parser for a given file.
   *
   * @param {BlobView} header The file in question.
   * @return {MetadataParser} The metadata parser to use for this file.
   */
  function findParser(header) {
    for (var i = 0; i < formats.length; i++) {
      if (formats[i].match(header)) {
        return new MetadataParser(formats[i]);
      }
    }
    return null;
  }

  return {
    findParser: findParser
  };

})();

/* global BlobView, MetadataFormats */
/* exported AudioMetadata */
'use strict';

// XXX We're hiding the fact that these are JSdoc comments because our linter
// refuses to accept "property" as a valid tag. Grumble grumble.

/*
 * Metadata for a track.
 *
 * @typedef {Object} Metadata
 * @property {String} tag_format The format of the tag (e.g. id3v2.4).
 * @property {string} artist The track's artist.
 * @property {string} album The track's album.
 * @property {number} tracknum The number of the track on the album.
 * @property {string} title The track's title.
 * @property {Picture} [picture] The cover art, if any.
 * @property {number} rated The track's rating; starts at 0.
 * @property {number} played The track's play count; starts at 0.
 */

/*
 * Metadata describing the cover art for a track. Cover art comes in one of
 * three flavors: embedded (stored in the audio track's metadata), unsynced
 * (like embedded, but stored in an unsynchronized block of ID3 data; we
 * ultimately save it in an external file), or external (stored as a separate
 * file in the same directory as the audio track).
 *
 * @typedef {Object} Picture
 * @property {string} flavor How the art was stored; one of "embedded",
 *   "unsynced", or "external".
 * @property {number} [start] The offset in bytes to where the picture is
 *   stored in the audio file; only applies when flavor="embedded".
 * @property {number} [end] The offset in bytes to the end of where the
 *   picture is stored in the audio file; only applies when flavor="embedded".
 * @property {number} [type] The mimetype of the picture; only applies when
 *   flavor="embedded".
 * @property {string} [filename] The path on the filesystem to the original
 *   (full-size) picture; only applies when flavor="external" or "unsynced".
 */

/**
 * Parse the metadata for an audio file.
 */
var AudioMetadata = (function() {
  var pictureStorage = navigator.getDeviceStorage('pictures');

  // A cache of metadata Picture objects for albums with external cover art. The
  // key is the directory name for the song in question.
  var externalCoverCache = {};

  // A cache of filenames for saved cover art (generated from unsynced ID3
  // tags). The value is the absolute path name.
  var savedCoverCache = new Set();

  /**
   * Parse the specified blob and return a Promise with the metadata.
   *
   * @param {Blob} blob The audio file to parse.
   * @return {Promise} A Promise returning the parsed metadata object.
   */
  function parse(blob) {
    var filename = blob.name;

    // If blob.name exists, it should be an audio file from system
    // otherwise it should be an audio blob that probably from network/process
    // we can still parse it but we don't need to care about the filename
    if (filename) {
      // If the file is in the DCIM/ directory and has a .3gp extension
      // then it is a video, not a music file and we ignore it
      if (filename.slice(0, 5) === 'DCIM/' &&
          filename.slice(-4).toLowerCase() === '.3gp') {
        return Promise.reject('skipping 3gp video file');
      }
    }

    // If the file is too small to be a music file then ignore it
    if (blob.size < 128) {
      return Promise.reject('file is empty or too small');
    }

    // Start off with some default metadata
    var metadata = {};
    metadata.artist = metadata.album = metadata.title = '';
    metadata.rated = metadata.played = 0;

    // If the blob has a name, use that as a default title in case
    // we can't find one in the file
    if (filename) {
      var p1 = filename.lastIndexOf('/');
      var p2 = filename.lastIndexOf('.');
      if (p2 <= p1) {
        p2 = filename.length;
      }
      metadata.title = filename.substring(p1 + 1, p2);
    }

    // Read the start of the file, figure out what kind it is, and call
    // the appropriate parser.  Start off with an 64kb chunk of data.
    // If the metadata is in that initial chunk we won't have to read again.
    return new Promise(function(resolve, reject) {
      var headersize = Math.min(64 * 1024, blob.size);
      BlobView.get(blob, 0, headersize, function(header, error) {
        if (error) {
          reject(error);
          return;
        }

        try {
          var parser = MetadataFormats.findParser(header);
          if (parser) {
            resolve(parser.parse(header, metadata).then(function(metadata) {
              return handleCoverArt(blob, metadata);
            }));
          } else {
            // This is some kind of file that we don't know about.
            // Let's see if we can play it.
            resolve(checkPlayability(blob).then(function() {
              return metadata;
            }));
          }
        } catch (e) {
          console.error('AudioMetadata.parse:', e, e.stack);
          reject(e);
        }
      });
    });
  }

  /**
   * Check if a blob can be played as an audio file.
   *
   * @param {Blob} blob The file to test.
   * @return {Promise} A promise that resolves successfully if the file is
   *   playable.
   */
  function checkPlayability(blob) {
    var player = new Audio();
    player.mozAudioChannelType = 'content';
    var canplay = blob.type && player.canPlayType(blob.type);
    if (canplay === 'probably') {
      return Promise.resolve();
    } else {
      return new Promise(function(resolve, reject) {
        var url = URL.createObjectURL(blob);
        player.src = url;

        player.onerror = function() {
          URL.revokeObjectURL(url);
          player.removeAttribute('src');
          player.load();
          reject('Unplayable music file');
        };

        player.oncanplay = function() {
          URL.revokeObjectURL(url);
          player.removeAttribute('src');
          player.load();
          resolve();
        };
      });
    }
  }

  /**
   * Perform additional processing for cover art, such as checking for external
   * artwork and saving temporary image blobs (for unsynchronized ID3 data).
   *
   * @param {Blob} blob The audio file.
   * @param {Metadata} metadata The metadata for the file.
   * @return {Promise} A Promise resolving to the metadata object with any
   *   additional cover art fields added as necessary.
   */
  function handleCoverArt(blob, metadata) {
    // If a blob isn't backed by an actual file, we can't do anything here
    // except return what we have.
    var filename = blob.name;
    if (!filename) {
      return Promise.resolve(metadata);
    }

    if (!metadata.picture) {
      // We don't have any embedded album art; see if there's any external album
      // art in the same directory.
      var lastSlash = filename.lastIndexOf('/');
      var dirName = filename.substring(0, lastSlash + 1);

      // First, check if we've recently fetched this cover; if so, just use the
      // cached results so we're not going out to the disk more than necessary.
      if (dirName in externalCoverCache) {
        metadata.picture = externalCoverCache[dirName];
        return Promise.resolve(metadata);
      }

      // Try to find external album art using a handful of common names. The
      // possibilities listed here appear to be the most common, based on a
      // brief survey of what people use in the wild.
      return new Promise(function(resolve, reject) {
        var possibleFilenames = ['folder.jpg', 'cover.jpg', 'front.jpg'];
        var tryFetchExternalCover = function(index) {
          if (index === possibleFilenames.length) {
            // We couldn't find any external album art.
            externalCoverCache[dirName] = null;
            resolve(metadata);
            return;
          }

          var externalCoverFilename = dirName + possibleFilenames[index];
          var getcoverrequest = pictureStorage.get(externalCoverFilename);
          getcoverrequest.onsuccess = function() {
            metadata.picture = { flavor: 'external',
                                 filename: externalCoverFilename };
            // Cache the picture object we generated to make things faster.
            externalCoverCache[dirName] = metadata.picture;
            resolve(metadata);
          };
          getcoverrequest.onerror = function() {
            tryFetchExternalCover(index + 1);
          };
        };
        tryFetchExternalCover(0);
      });
    } else if (metadata.picture.blob) {
      // We have album art in a separate blob that we need to save somewhere;
      // generally, this is because we had an unsynced ID3 frame with the art
      // and needed to de-unsync it.
      //
      // First, try to make a unique filename for the image based on the artist,
      // album, and file size. The file size is essentially a poor man's
      // checksum to make sure we aren't reusing an inappropriate image, such as
      // if the user updated the embedded art.
      var albumKey;
      if (metadata.artist || metadata.album) {
        // Don't let the artist or album strings be too long; VFAT has a maximum
        // of 255 characters for the path, and we need some room for the rest of
        // it!
        var artist = (metadata.artist || '').substr(0, 64);
        var album = (metadata.album || '').substr(0, 64);
        albumKey = artist + '.' + album;
      } else {
        // If there's no album or artist, use the path to the file instead; we
        // truncate from the end to help ensure uniqueness.
        albumKey = filename.substr(-128);
      }

      var coverBlob = metadata.picture.blob;
      delete metadata.picture.blob;

      // coverBlob is always a JPEG or PNG.
      var extension = coverBlob.type === 'image/jpeg' ? '.jpg' : '.png';
      var imageFilename = vfatEscape(albumKey) + '.' + coverBlob.size +
                          extension;
      var storageName = getStorageName(filename);
      return checkSaveCover(coverBlob, imageFilename, storageName).then(
        function(savedFile) {
          metadata.picture.filename = savedFile;
          return metadata;
        }, function(err) {
          delete metadata.picture;
          return metadata;
        }
      );
    } else {
      // We have embedded album art! All the processing is already done, so we
      // can just return.
      return Promise.resolve(metadata);
    }
  }

  /**
   * Escape any characters that are illegal on the VFAT filesystem.
   *
   * @param {string} str The string to escape.
   * @return {string} The escaped string.
   */
  function vfatEscape(str) {
    return str.replace(/["*\/:<>?\|]/g, '_');
  }

  /**
   * Get the name of the storage area for an absolute filename, with leading
   * and trailing slashes (e.g. '/sdcard/'). If the filename is relative,
   * return the empty string (i.e. use the default storage area.)
   *
   * @param {string} filename The path to the file.
   * @return {string} The file's storage area, or the empty string if the path
   *   was relative.
   */
  function getStorageName(filename) {
    // `filename` is usually a fully qualified name (perhaps something like
    // /sdcard/Music/file.mp3). On desktop, it's a relative name, but desktop
    // only has one storage area anyway.
    if (filename[0] === '/') {
      var slashIndex = filename.indexOf('/', 1);
      if (slashIndex < 0) {
        var err = Error('handleCoverArt: Bad filename: "' + filename + '"');
        console.error(err);
        return Promise.reject(err);
      }

      // Get the storage name, e.g. /sdcard/
      return filename.substring(0, slashIndex + 1);
    }

    return '';
  }

  /**
   * Check for the existence of an image on the audio file's storage area,
   * and if it's not present, try to save it.
   *
   * @param {Blob} coverBlob The blob for the full-size cover art.
   * @param {string} imageFilename A relative filename for the image.
   * @param {string} storageName The name of the storage area too use for
   *   saving the image. May be an empty string or a path with trailing '/'.
   *   For instance, to store on the SD card, pass '/sdcard/'.
   * @return {Promise} A Promise that resolves when finished, providing the
   *   filename of the saved image.
   */
  function checkSaveCover(coverBlob, imageFilename, storageName) {
    // We want to put the image in the same storage area as the audio track it's
    // for. Since the audio track could be in any storage area, we examine its
    // filename to get the storage name; the storage name is always the first
    // part of the (absolute) filename, so we grab that and build an absolute
    // path for the image. This will ensure that the generic deviceStorage we
    // use for pictures ("pictureStorage") puts the image where we want it.
    var imageAbsPath = storageName + '.music/covers/' + imageFilename;
    if (savedCoverCache.has(imageAbsPath)) {
      return Promise.resolve(imageAbsPath);
    }

    return new Promise(function(resolve, reject) {
      var getrequest = pictureStorage.get(imageAbsPath);

      // We already have the image. We're done!
      getrequest.onsuccess = function() {
        savedCoverCache.add(imageAbsPath);
        resolve(imageAbsPath);
      };

      // We don't have the image yet. Let's save it.
      getrequest.onerror = function() {
        var saverequest = pictureStorage.addNamed(coverBlob, imageAbsPath);
        saverequest.onerror = function() {
          console.error('Could not save cover image', imageFilename);
        };

        // Don't bother waiting for saving to finish. Just return.
        savedCoverCache.add(imageAbsPath);
        resolve(imageAbsPath);
      };
    });
  }

  return {
    parse: parse,
    pictureStorage: pictureStorage
  };
})();
