
define('view',['require','exports','module'],function(require, exports, module) {


var DEFAULT_ERROR_ID = 'error-default';
const INVALID_CSS = /([^a-zA-Z\-\_0-9])/g;

/**
 * Very simple base class for views.
 * Provides functionality for active/inactive.
 *
 * The first time the view is activated
 * the onactive function/event will fire.
 *
 * The .seen property is added to each object
 * with view in its prototype. .seen can be used
 * to detect if the view has ever been activated.
 *
 * @param {String|Object} options options or a selector for element.
 */
function View(options) {
  if (typeof(options) === 'undefined') {
    options = {};
  }

  if (typeof(options) === 'string') {
    this.selectors = { element: options };
  } else {
    var key;

    if (typeof(options) === 'undefined') {
      options = {};
    }

    for (key in options) {
      if (options.hasOwnProperty(key)) {
        this[key] = options[key];
      }
    }
  }

  this.hideErrors = this.hideErrors.bind(this);
}
module.exports = View;

View.ACTIVE = 'active';

View.prototype = {
  seen: false,
  activeClass: View.ACTIVE,
  errorVisible: false,

  get element() {
    return this._findElement('element');
  },

  get status() {
    return this._findElement('status');
  },

  get errors() {
    return this._findElement('errors');
  },

  /**
   * Creates a string id for a given model.
   *
   *    view.idForModel('foo-', { _id: 1 }); // => foo-1
   *    view.idForModel('foo-', '2'); // => foo-2
   *
   * @param {String} prefix of string.
   * @param {Object|String|Numeric} objectOrString representation of model.
   */
  idForModel: function(prefix, objectOrString) {
    prefix += (typeof(objectOrString) === 'object') ?
      objectOrString._id :
      objectOrString;

    return prefix;
  },

  calendarId: function(input) {
    if (typeof(input) !== 'string') {
      input = input.calendarId;
    }

    input = this.cssClean(input);
    return 'calendar-id-' + input;
  },

  /**
   * Delegate pattern event listener.
   *
   * @param {HTMLElement} element parent element.
   * @param {String} type type of dom event.
   * @param {String} selector css selector element should match
   *                          _note_ there is no magic here this
   *                          is determined from the root of the document.
   * @param {Function|Object} handler event handler.
   *                                  first argument is the raw
   *                                  event second is the element
   *                                  matching the pattern.
   */
  delegate: function(element, type, selector, handler) {
    if (typeof(handler) === 'object') {
      var context = handler;
      handler = function() {
        context.handleEvent.apply(context, arguments);
      };
    }

    element.addEventListener(type, function(e) {
      var target = e.target;
      while (target !== element) {
        if ('mozMatchesSelector' in target &&
            target.mozMatchesSelector(selector)) {
          return handler(e, target);
        }
        target = target.parentNode;
      }
    });
  },

  /**
   * Clean a string for use with css.
   * Converts illegal chars to legal ones.
   */
  cssClean: function(string) {
    if (typeof(string) !== 'string') {
      return string;
    }

    //TODO: I am worried about the performance
    //of using this all over the place =/
    //consider sanitizing all keys to ensure
    //they don't blow up when used as a selector?
    return string.replace(INVALID_CSS, '-');
  },

  /**
   * Finds a caches a element defined
   * by selectors
   *
   * @param {String} selector name as defined in selectors.
   * @param {Boolean} all true when to find all elements. (default false).
   */
  _findElement: function(name, all, element) {
    if (typeof(all) === 'object') {
      element = all;
      all = false;
    }

    element = element || document;

    var cacheName;
    var selector;

    if (typeof(all) === 'undefined') {
      all = false;
    }

    if (name in this.selectors) {
      cacheName = '_' + name + 'Element';
      selector = this.selectors[name];

      if (!this[cacheName]) {
        if (all) {
          this[cacheName] = element.querySelectorAll(selector);
        } else {
          this[cacheName] = element.querySelector(selector);
        }
      }

      return this[cacheName];
    }

    return null;
  },

 /**
   * Displays a list of errors
   *
   * @param {Array} list error list
   *  (see Event.validaitonErrors) or Error object.
   */
  showErrors: function(list) {
    var _ = navigator.mozL10n.get;
    var errors = '';

    // We can pass Error objects or
    // Array of {name: foo} objects
    if (!Array.isArray(list)) {
        list = [list];
    }

    var i = 0;
    var len = list.length;

    for (; i < len; i++) {
      var name = list[i].l10nID || list[i].name;
      errors += _('error-' + name) || _(DEFAULT_ERROR_ID);
    }

    // populate error and display it.
    this.errors.textContent = errors;
    this.errorVisible = true;
    this.status.classList.add(this.activeClass);

    this.status.addEventListener('animationend', this.hideErrors);
  },

  hideErrors: function() {
    this.status.classList.remove(this.activeClass);
    this.status.removeEventListener('animationend', this.hideErrors);
    this.errorVisible = false;
  },

  onactive: function() {
    if (this.errorVisible) {
      this.hideErrors();
    }

    // seen can be set to anything other than false to override this behaviour
    if (this.seen === false) {
      this.onfirstseen();
    }

    // intentionally using 'in'
    if ('dispatch' in this) {
      this.dispatch.apply(this, arguments);
    }

    this.seen = true;
    if (this.element) {
      this.element.classList.add(this.activeClass);
    }
  },

  oninactive: function() {
    if (this.element) {
      this.element.classList.remove(this.activeClass);
    }
  },

  onfirstseen: function() {}
};

});

define('template',['require','exports','module'],function(require, exports, module) {


var POSSIBLE_HTML = /[&<>"'`]/;

var span = document.createElement('span');

function create(templates) {
  var key, result = {};

  for (key in templates) {
    if (templates.hasOwnProperty(key)) {
      result[key] = new Template(templates[key]);
    }
  }

  return result;
}

function Template(fn) {
  this.template = fn;
}
module.exports = Template;

Template.prototype = {
  arg: function(key) {
    if (typeof(this.data) === 'undefined') {
      return '';
    } else if (typeof(this.data) !== 'object') {
      return this.data;
    }

    return this.data[key];
  },

  h: function(a) {
    var arg = this.arg(a);
    // accept anything that can be converted into a string and we make sure
    // the only falsy values that are converted into empty strings are
    // null/undefined to avoid mistakes
    arg = arg == null ? '' : String(arg);

    //only escape bad looking stuff saves
    //a ton of time
    if (POSSIBLE_HTML.test(arg)) {
      span.textContent = arg;
      return span.innerHTML.replace(/"/g, '&quot;').replace(/'/g, '&#x27;');
    } else {
      return arg;
    }
  },

  s: function(a) {
    var arg = this.arg(a);
    return String((arg || ''));
  },

  bool: function(key, onTrue) {
    if (this.data[key]) {
      return onTrue;
    } else {
      return '';
    }
  },

  l10n: function(key, prefix) {
    var value = this.arg(key);

    if (prefix) {
      value = prefix + value;
    }
    return navigator.mozL10n.get(value);
  },

  l10nId: function(a) {
    return this.s(a).replace(/\s/g, '-');
  },

  /**
   * Renders template with given slots.
   *
   * @param {Object} object key, value pairs for template.
   */
  render: function(data) {
    this.data = data;
    return this.template();
  },

  /**
   * Renders template multiple times
   *
   * @param {Array} objects object details to render.
   * @param {String} [join] optional join argument will join the array.
   * @return {String|Array} String if join argument is given array otherwise.
   */
  renderEach: function(objects, join) {
    var i = 0, len = objects.length,
        result = [];

    for (; i < len; i++) {
      result.push(this.render(objects[i]));
    }

    if (typeof(join) !== 'undefined') {
      return result.join(join);
    }

    return result;
  }
};

Template.create = create;

});

define('templates/month',['require','exports','module','template'],function(require, exports, module) {


var create = require('template').create;

module.exports = create({
  busy: function() {
    return '<span class="' +
              'busytime-' + this.h('_id') +
              ' busy-length-' + this.h('length') +
              ' busy-' + this.h('start') +
              ' calendar-id-' + this.h('calendarId') + '">' +
            '&nbsp;' +
          '</span>';
  },

  weekDaysHeader: function() {
    return '<header id="month-days" role="presentation">' +
        '<ol role="row">' +
          this.s('value') +
        '</ol>' +
      '</header>';
  },

  weekDaysHeaderDay: function() {
    return '<li data-l10n-id="' + this.h('l10n') + '" role="columnheader">' +
        this.h('dayName') +
      '</li>';
  },

  week: function() {
    return '<ol role="row">' +
        this.s('value') +
      '</ol>';
  },

  day: function() {
    var date = this.h('date');
    var dateString = this.s('dateString');
    var id = this.s('id');
    var l10nStateId = this.l10nId('state');
    var state = this.s('state');

    return `<li role="gridcell" tabindex="0" id="${id}"
      aria-describedby="${id}-busy-indicator ${id}-description"
      data-date="${dateString}" class="${state}">
        <span class="day" role="button">${date}</span>
        <div id="${id}-busy-indicator"
          class="busy-indicator" aria-hidden="true"></div>
        <span id="${id}-description" aria-hidden="true"
          data-l10n-id="${l10nStateId}"></span>
      </li>`;
  }
});

});

define('views/month_child',['require','exports','module','calc','view','debug','next_tick','performance','templates/month'],function(require, exports, module) {


var Calc = require('calc');
var View = require('view');
var debug = require('debug')('views/month_child');
var nextTick = require('next_tick');
var performance = require('performance');
var template = require('templates/month');

// horrible hack to clear cache when we re-localize
window.addEventListener('localized', function clearHeaderCache() {
  Child._dayHeaders = null;
});

function Child() {
  View.apply(this, arguments);

  this.id = this.date.valueOf();
  this.controller = this.app.timeController;

  this._days = Object.create(null);
  this._dayToBusyCount = Object.create(null);
  this.timespan = Calc.spanOfMonth(this.date);
}
module.exports = Child;

Child.prototype = {
  __proto__: View.prototype,

  ACTIVE: 'active',

  hasBeenActive: false,

  //Override parent view...
  get element() {
    return this._element;
  },

  set element(val) {
    this._element = val;
    return val;
  },

  _dayId: function(date) {
    if (date instanceof Date) {
      date = Calc.getDayId(date);
    }

    return 'month-view-' + this.id + '-' + date;
  },

  _initEvents: function() {
    this.controller.observeTime(this.timespan, this);
  },

  _destroyEvents: function() {
    this.controller.removeTimeObserver(this.timespan, this);
  },

  handleEvent: function(event) {
    var added = [], removed = [];
    switch (event.type) {
      case 'add':
        added.push(event.data);
        break;
      case 'remove':
        removed.push(event.data);
        break;
    }

    this._updateBusytimes({ added: added, removed: removed });
  },

  _updateBusytimes: function(options) {
    if ('added' in options) {
      options.added.forEach(function(busytime) {
        this._updateBusyCount(busytime, 1);
      }, this);
    }

    if ('removed' in options) {
      options.removed.forEach(function(busytime) {
        this._updateBusyCount(busytime, -1);
      }, this);
    }
  },

  _updateBusyCount: function(busytime, difference) {
    var {startDate, endDate} = busytime;
    var dates = [];
    // Use the last second of previous day as the base for endDate
    // (e.g., 1991-09-14T23:59:59 insteads of 1991-09-15T00:00:00).
    // IMPORTANT: yahoo uses same start/end date for recurring all day events!
    if (Number(startDate) !== Number(endDate) &&
        endDate.getHours() === 0 &&
        endDate.getMinutes() === 0 &&
        endDate.getSeconds() === 0) {
      endDate = new Date(endDate.getTime() - 1000);
    }

    dates = Calc.daysBetween(
      busytime.startDate,
      endDate
    );

    dates.forEach(function(date) {
      var dayId = Calc.getDayId(date);
      var count = this._dayToBusyCount[dayId];
      this._setBusyCount(dayId, count + difference);
    }, this);
  },

  _setBusyCount: function(dayId, count) {
    this._dayToBusyCount[dayId] = count;

    // Now redraw the busytime dots.
    var element = this._busyElement(dayId);
    if (!element) {
      return debug('Could not find container for ' + dayId + '!');
    }

    var difference = Math.min(3, count) - element.childNodes.length;

    if (count > 0) {
      element.setAttribute('aria-label', navigator.mozL10n.get('busy', {
        n: count
      }));
    } else {
      element.removeAttribute('aria-label');
    }

    if (difference === 0) {
      return;
    }

    var i = 0;
    if (difference > 0) {
      var dot;
      for (; i < difference; i++) {
        dot = document.createElement('div');
        dot.className = 'gaia-icon icon-calendar-dot';
        element.appendChild(dot);
      }

      return;
    }

    // difference < 0
    for (; i > difference; i--) {
      element.removeChild(element.firstChild);
    }
  },

  /**
   * Finds day element busytime container.
   * Caches over time.
   *
   * @param {String} dayId date id.
   */
  _busyElement: function(dayId) {
    var id = this._dayId(dayId);
    var found = this.element.querySelector('#' + id + ' .busy-indicator');
    this._days[dayId] = found;
    return found;
  },


  /**
   * Renders out a day with busy times.
   *
   * @param {Date} date representing a date.
   */
  _renderDay: function _renderDay(date) {
    var id = Calc.getDayId(date);
    var state = Calc.relativeState(
      date,
      this.date
    );

    // register instance in map
    this._days[id] = null;
    this._dayToBusyCount[id] = 0;

    return template.day.render({
      id: this._dayId(id),
      dateString: id,
      state: state,
      date: date.getDate()
    });
  },

  /**
   * Renders a week from weekdays Array
   *
   */
  _renderWeek: function _renderWeek(days) {
    var output = '';

    for (var i = 0, iLen = days.length; i < iLen; i++) {
      output += this._renderDay(days[i]);
    }

    return template.week.render(output);
  },

  /**
   * Renders out the calendar headers.
   *
   * @return {String} returns a list of headers.
   */
  _renderDayHeaders: function _renderDayHeaders() {
    if (!Child._dayHeaders) {
      var i = 0;
      var days = 7;
      var name;
      var html = '';

      for (; i < days; i++) {
        var day = i;
        // localization updates this value
        if (Calc.startsOnMonday) {
          // 0 is monday which is 1 in l10n (based on js engine's getDay)
          day += 1;

          // 6th day of the week which Sunday (and 0 in js engine).
          if (day === 7) {
            day = 0;
          }
        }
        var l10n = 'weekday-' + day + '-single-char';

        name = navigator.mozL10n.get(l10n);
        html += template.weekDaysHeaderDay.render({
          l10n: l10n,
          dayName: name
        });
      }

      Child._dayHeaders = template.weekDaysHeader.render(html);
      return Child._dayHeaders;
    }

    return Child._dayHeaders;
  },

  /**
   * Renders out an entire month.
   *
   * @param {Date} date date which month resides in.
   * @return {String} return value.
   */
  _renderMonth: function _renderMonth() {
    var week = 0;
    var slice;
    var days = Calc.daysBetween(this.timespan);
    var daysInWeek = Calc.daysInWeek();
    var numberOfWeeks = days.length / daysInWeek;
    var html = '';

    this.weeks = numberOfWeeks;

    for (week; week <= numberOfWeeks; week++) {
      slice = days.splice(
        0,
        daysInWeek
      );

      if (slice.length) {
        html += this._renderWeek(slice);
      }
    }

    return this._renderDayHeaders() + html;
  },

  /**
   * Activate this child view visually.
   */
  activate: function() {
    this.element.classList.add(this.ACTIVE);

    /**
     * The first time we "activate" a view we initialize its
     * events and query th cache for related records.
     * We do this async so to minimally effect swipes.
     */
    if (this.hasBeenActive) {
      return;
    }

    nextTick(function() {
      var busytimes = this.controller.queryCache(this.timespan);
      this._updateBusytimes({ added: busytimes });
      this._initEvents();
      // at this point the month view should be ready
      performance.monthReady();
    }.bind(this));

    this.hasBeenActive = true;
  },

  /**
   * Deactivate this child view visually.
   */
  deactivate: function() {
    this.element.classList.remove(this.ACTIVE);
  },

  /**
   * Attaches child view to dom node
   * or object that has a .element.
   *
   * Sets the .element
   *
   * @return {DOMElement} inserted dom node.
   */
  create: function() {
    var html = this._renderMonth();
    var element = document.createElement('section');

    element.classList.add('month');
    element.setAttribute('role', 'grid');
    element.setAttribute('aria-labelledby', 'current-month-year');
    element.setAttribute('aria-readonly', true);
    element.innerHTML = html;

    this.element = element;

    return element;
  },

  destroy: function() {
    this._destroyEvents();
    this._days = Object.create(null);
    this._dayToBusyCount = Object.create(null);

    if (this.element && this.element.parentNode) {
      this.element.parentNode.removeChild(this.element);
      this.element = undefined;
    }
  },

  getScrollTop: function() {},

  setScrollTop: function(scrollTop) {}
};

});

/* exported GestureDetector */



/**
 * GestureDetector.js: generate events for one and two finger gestures.
 *
 * A GestureDetector object listens for touch events on a specified
 * element and generates higher-level events that describe one and two finger
 * gestures on the element.
 *
 * Supported events:
 *
 *  tap        like a click event
 *  dbltap     like dblclick
 *  pan        one finger motion
 *  swipe      when a finger is released following pan events
 *  holdstart  touch and hold. Must set an option to get these.
 *  holdmove   motion after a holdstart event
 *  holdend    when the finger goes up after holdstart/holdmove
 *  transform  2-finger pinch and twist gestures for scaling and rotation
 *
 * Each of these events is a bubbling CustomEvent with important details in the
 * event.detail field. The event details are not yet stable and are not yet
 * documented. See the calls to emitEvent() for details.
 *
 * To use this library, create a GestureDetector object by passing an element to
 * the GestureDetector() constructor and then calling startDetecting() on it.
 * The element will be the target of all the emitted gesture events. You can
 * also pass an optional object as the second constructor argument. If you're
 * interested in holdstart/holdmove/holdend events, pass {holdEvents:true} as
 * this second argument. Otherwise they will not be generated.
 * If you want to customize the pan threshold, pass {panThreshold:X} 
 * (X and Y in pixels) in the options argument.
 *
 * Implementation note: event processing is done with a simple finite-state
 * machine. This means that in general, the various kinds of gestures are
 * mutually exclusive. You won't get pan events until your finger has
 * moved more than a minimum threshold, for example, but it does, the FSM enters
 * a new state in which it can emit pan and swipe events and cannot emit hold
 * events. Similarly, if you've started a 1 finger pan/swipe gesture and
 * accidentally touch with a second finger, you'll continue to get pan events,
 * and won't suddenly start getting 2-finger transform events.
 *
 * This library never calls preventDefault() or stopPropagation on any of the
 * events it processes, so the raw touch events should still be
 * available for other code to process. It is not clear to me whether this is a
 * feature or a bug.
 */

var GestureDetector = (function() {

  //
  // Constructor
  //
  function GD(e, options) {
    this.element = e;
    this.options = options || {};
    this.options.panThreshold = this.options.panThreshold || GD.PAN_THRESHOLD;
    this.state = initialState;
    this.timers = {};
  }

  //
  // Public methods
  //

  GD.prototype.startDetecting = function() {
    var self = this;
    eventtypes.forEach(function(t) {
      self.element.addEventListener(t, self);
    });
  };

  GD.prototype.stopDetecting = function() {
    var self = this;
    eventtypes.forEach(function(t) {
      self.element.removeEventListener(t, self);
    });
  };

  //
  // Internal methods
  //

  GD.prototype.handleEvent = function(e) {
    var handler = this.state[e.type];
    if (!handler) {
      return;
    }
    // If this is a touch event handle each changed touch separately
    if (e.changedTouches) {
      // XXX https://bugzilla.mozilla.org/show_bug.cgi?id=785554
      // causes touchend events to list all touches as changed, so
      // warn if we see that bug
      if (e.type === 'touchend' && e.changedTouches.length > 1) {
        console.warn('gesture_detector.js: spurious extra changed touch on ' +
                     'touchend. See ' +
                     'https://bugzilla.mozilla.org/show_bug.cgi?id=785554');
      }

      for (var i = 0; i < e.changedTouches.length; i++) {
        handler(this, e, e.changedTouches[i]);
        // The first changed touch might have changed the state of the
        // FSM. We need this line to workaround the bug 785554, but it is
        // probably the right thing to have here, even once that bug is fixed.
        handler = this.state[e.type];
      }
    }
    else {    // Otherwise, just dispatch the event to the handler
      handler(this, e);
    }
  };

  GD.prototype.startTimer = function(type, time) {
    this.clearTimer(type);
    var self = this;
    this.timers[type] = setTimeout(function() {
      self.timers[type] = null;
      var handler = self.state[type];
      if (handler) {
        handler(self, type);
      }
    }, time);
  };

  GD.prototype.clearTimer = function(type) {
    if (this.timers[type]) {
      clearTimeout(this.timers[type]);
      this.timers[type] = null;
    }
  };

  // Switch to a new FSM state, and call the init() function of that
  // state, if it has one.  The event and touch arguments are optional
  // and are just passed through to the state init function.
  GD.prototype.switchTo = function(state, event, touch) {
    this.state = state;
    if (state.init) {
      state.init(this, event, touch);
    }
  };

  GD.prototype.emitEvent = function(type, detail) {
    if (!this.target) {
      console.error('Attempt to emit event with no target');
      return;
    }

    var event = this.element.ownerDocument.createEvent('CustomEvent');
    event.initCustomEvent(type, true, true, detail);
    this.target.dispatchEvent(event);
  };

  //
  // Tuneable parameters
  //
  GD.HOLD_INTERVAL = 1000;     // Hold events after 1000 ms
  GD.PAN_THRESHOLD = 20;       // 20 pixels movement before touch panning
  GD.DOUBLE_TAP_DISTANCE = 50;
  GD.DOUBLE_TAP_TIME = 500;
  GD.VELOCITY_SMOOTHING = 0.5;

  // Don't start sending transform events until the gesture exceeds a threshold
  GD.SCALE_THRESHOLD = 20;     // pixels
  GD.ROTATE_THRESHOLD = 22.5;  // degrees

  // For pans and zooms, we compute new starting coordinates that are part way
  // between the initial event and the event that crossed the threshold so that
  // the first event we send doesn't cause a big lurch. This constant must be
  // between 0 and 1 and says how far along the line between the initial value
  // and the new value we pick
  GD.THRESHOLD_SMOOTHING = 0.9;

  //
  // Helpful shortcuts and utility functions
  //

  var abs = Math.abs, floor = Math.floor, sqrt = Math.sqrt, atan2 = Math.atan2;
  var PI = Math.PI;

  // The names of events that we need to register handlers for
  var eventtypes = [
    'touchstart',
    'touchmove',
    'touchend'
  ];

  // Return the event's timestamp in ms
  function eventTime(e) {
    // In gecko, synthetic events seem to be in microseconds rather than ms.
    // So if the timestamp is much larger than the current time, assue it is
    // in microseconds and divide by 1000
    var ts = e.timeStamp;
    if (ts > 2 * Date.now()) {
      return Math.floor(ts / 1000);
    } else {
      return ts;
    }
  }


  // Return an object containg the space and time coordinates of
  // and event and touch. We freeze the object to make it immutable so
  // we can pass it in events and not worry about values being changed.
  function coordinates(e, t) {
    return Object.freeze({
      screenX: t.screenX,
      screenY: t.screenY,
      clientX: t.clientX,
      clientY: t.clientY,
      timeStamp: eventTime(e)
    });
  }

  // Like coordinates(), but return the midpoint between two touches
  function midpoints(e, t1, t2) {
    return Object.freeze({
      screenX: floor((t1.screenX + t2.screenX) / 2),
      screenY: floor((t1.screenY + t2.screenY) / 2),
      clientX: floor((t1.clientX + t2.clientX) / 2),
      clientY: floor((t1.clientY + t2.clientY) / 2),
      timeStamp: eventTime(e)
    });
  }

  // Given coordinates objects c1 and c2, return a new coordinates object
  // representing a point and time along the line between those points.
  // The position of the point is controlled by the THRESHOLD_SMOOTHING constant
  function between(c1, c2) {
    var r = GD.THRESHOLD_SMOOTHING;
    return Object.freeze({
      screenX: floor(c1.screenX + r * (c2.screenX - c1.screenX)),
      screenY: floor(c1.screenY + r * (c2.screenY - c1.screenY)),
      clientX: floor(c1.clientX + r * (c2.clientX - c1.clientX)),
      clientY: floor(c1.clientY + r * (c2.clientY - c1.clientY)),
      timeStamp: floor(c1.timeStamp + r * (c2.timeStamp - c1.timeStamp))
    });
  }

  // Compute the distance between two touches
  function touchDistance(t1, t2) {
    var dx = t2.screenX - t1.screenX;
    var dy = t2.screenY - t1.screenY;
    return sqrt(dx * dx + dy * dy);
  }

  // Compute the direction (as an angle) of the line between two touches
  // Returns a number d, -180 < d <= 180
  function touchDirection(t1, t2) {
    return atan2(t2.screenY - t1.screenY,
                 t2.screenX - t1.screenX) * 180 / PI;
  }

  // Compute the clockwise angle between direction d1 and direction d2.
  // Returns an angle a -180 < a <= 180.
  function touchRotation(d1, d2) {
    var angle = d2 - d1;
    if (angle > 180) {
      angle -= 360;
    } else if (angle <= -180) {
      angle += 360;
    }
    return angle;
  }

  // Determine if two taps are close enough in time and space to
  // trigger a dbltap event. The arguments are objects returned
  // by the coordinates() function.
  function isDoubleTap(lastTap, thisTap) {
    var dx = abs(thisTap.screenX - lastTap.screenX);
    var dy = abs(thisTap.screenY - lastTap.screenY);
    var dt = thisTap.timeStamp - lastTap.timeStamp;
    return (dx < GD.DOUBLE_TAP_DISTANCE &&
            dy < GD.DOUBLE_TAP_DISTANCE &&
            dt < GD.DOUBLE_TAP_TIME);
  }

  //
  // The following objects are the states of our Finite State Machine
  //

  // In this state we're not processing any gestures, just waiting
  // for an event to start a gesture and ignoring others
  var initialState = {
    name: 'initialState',
    init: function(d) {
      // When we enter or return to the initial state, clear
      // the detector properties that were tracking gestures
      // Don't clear d.lastTap here, though. We need it for dbltap events
      d.target = null;
      d.start = d.last = null;
      d.touch1 = d.touch2 = null;
      d.vx = d.vy = null;
      d.startDistance = d.lastDistance = null;
      d.startDirection = d.lastDirection = null;
      d.lastMidpoint = null;
      d.scaled = d.rotated = null;
    },

    // Switch to the touchstarted state and process the touch event there
    touchstart: function(d, e, t) {
      d.switchTo(touchStartedState, e, t);
    }
  };

  // One finger is down but we haven't generated any event yet. We're
  // waiting to see...  If the finger goes up soon, its a tap. If the finger
  // stays down and still, its a hold. If the finger moves its a pan/swipe.
  // And if a second finger goes down, its a transform
  var touchStartedState = {
    name: 'touchStartedState',
    init: function(d, e, t) {
      // Remember the target of the event
      d.target = e.target;
      // Remember the id of the touch that started
      d.touch1 = t.identifier;
      // Get the coordinates of the touch
      d.start = d.last = coordinates(e, t);
      // Start a timer for a hold
      // If we're doing hold events, start a timer for them
      if (d.options.holdEvents) {
        d.startTimer('holdtimeout', GD.HOLD_INTERVAL);
      }
    },

    touchstart: function(d, e, t) {
      // If another finger goes down in this state, then
      // go to transform state to start 2-finger gestures.
      d.clearTimer('holdtimeout');
      d.switchTo(transformState, e, t);
    },
    touchmove: function(d, e, t) {
      // Ignore any touches but the initial one
      // This could happen if there was still a finger down after
      // the end of a previous 2-finger gesture, e.g.
      if (t.identifier !== d.touch1) {
        return;
      }

      if (abs(t.screenX - d.start.screenX) > d.options.panThreshold ||
          abs(t.screenY - d.start.screenY) > d.options.panThreshold) {
        d.clearTimer('holdtimeout');
        d.switchTo(panStartedState, e, t);
      }
    },
    touchend: function(d, e, t) {
      // Ignore any touches but the initial one
      if (t.identifier !== d.touch1) {
        return;
      }

      // If there was a previous tap that was close enough in time
      // and space, then emit a 'dbltap' event
      if (d.lastTap && isDoubleTap(d.lastTap, d.start)) {
        d.emitEvent('tap', d.start);
        d.emitEvent('dbltap', d.start);
        // clear the lastTap property, so we don't get another one
        d.lastTap = null;
      }
      else {
        // Emit a 'tap' event using the starting coordinates
        // as the event details
        d.emitEvent('tap', d.start);

        // Remember the coordinates of this tap so we can detect double taps
        d.lastTap = coordinates(e, t);
      }

      // In either case clear the timer and go back to the initial state
      d.clearTimer('holdtimeout');
      d.switchTo(initialState);
    },

    holdtimeout: function(d) {
      d.switchTo(holdState);
    }

  };

  // A single touch has moved enough to exceed the pan threshold and now
  // we're going to generate pan events after each move and a swipe event
  // when the touch ends. We ignore any other touches that occur while this
  // pan/swipe gesture is in progress.
  var panStartedState = {
    name: 'panStartedState',
    init: function(d, e, t) {
      // Panning doesn't start until the touch has moved more than a
      // certain threshold. But we don't want the pan to have a jerky
      // start where the first event is a big distance. So proceed as
      // pan actually started at a point along the path between the
      // first touch and this current touch.
      d.start = d.last = between(d.start, coordinates(e, t));

      // If we transition into this state with a touchmove event,
      // then process it with that handler. If we don't do this then
      // we can end up with swipe events that don't know their velocity
      if (e.type === 'touchmove') {
        panStartedState.touchmove(d, e, t);
      }
    },

    touchmove: function(d, e, t) {
      // Ignore any fingers other than the one we're tracking
      if (t.identifier !== d.touch1) {
        return;
      }

      // Each time the touch moves, emit a pan event but stay in this state
      var current = coordinates(e, t);
      d.emitEvent('pan', {
        absolute: {
          dx: current.screenX - d.start.screenX,
          dy: current.screenY - d.start.screenY
        },
        relative: {
          dx: current.screenX - d.last.screenX,
          dy: current.screenY - d.last.screenY
        },
        position: current
      });

      // Track the pan velocity so we can report this with the swipe
      // Use a exponential moving average for a bit of smoothing
      // on the velocity
      var dt = current.timeStamp - d.last.timeStamp;
      var vx = (current.screenX - d.last.screenX) / dt;
      var vy = (current.screenY - d.last.screenY) / dt;

      if (d.vx == null) { // first time; no average
        d.vx = vx;
        d.vy = vy;
      }
      else {
        d.vx = d.vx * GD.VELOCITY_SMOOTHING +
          vx * (1 - GD.VELOCITY_SMOOTHING);
        d.vy = d.vy * GD.VELOCITY_SMOOTHING +
          vy * (1 - GD.VELOCITY_SMOOTHING);
      }

      d.last = current;
    },
    touchend: function(d, e, t) {
      // Ignore any fingers other than the one we're tracking
      if (t.identifier !== d.touch1) {
        return;
      }

      // Emit a swipe event when the finger goes up.
      // Report start and end point, dx, dy, dt, velocity and direction
      var current = coordinates(e, t);
      var dx = current.screenX - d.start.screenX;
      var dy = current.screenY - d.start.screenY;
      // angle is a positive number of degrees, starting at 0 on the
      // positive x axis and increasing clockwise.
      var angle = atan2(dy, dx) * 180 / PI;
      if (angle < 0) {
        angle += 360;
      }

      // Direction is 'right', 'down', 'left' or 'up'
      var direction;
      if (angle >= 315 || angle < 45) {
        direction = 'right';
      } else if (angle >= 45 && angle < 135) {
        direction = 'down';
      } else if (angle >= 135 && angle < 225) {
        direction = 'left';
      } else if (angle >= 225 && angle < 315) {
        direction = 'up';
      }

      d.emitEvent('swipe', {
        start: d.start,
        end: current,
        dx: dx,
        dy: dy,
        dt: e.timeStamp - d.start.timeStamp,
        vx: d.vx,
        vy: d.vy,
        direction: direction,
        angle: angle
      });

      // Go back to the initial state
      d.switchTo(initialState);
    }
  };

  // We enter this state if the user touches and holds for long enough
  // without moving much.  When we enter we emit a holdstart event. Motion
  // after the holdstart generates holdmove events. And when the touch ends
  // we generate a holdend event. holdmove and holdend events can be used
  // kind of like drag and drop events in a mouse-based UI. Currently,
  // these events just report the coordinates of the touch.  Do we need
  // other details?
  var holdState = {
    name: 'holdState',
    init: function(d) {
      d.emitEvent('holdstart', d.start);
    },

    touchmove: function(d, e, t) {
      var current = coordinates(e, t);
      d.emitEvent('holdmove', {
        absolute: {
          dx: current.screenX - d.start.screenX,
          dy: current.screenY - d.start.screenY
        },
        relative: {
          dx: current.screenX - d.last.screenX,
          dy: current.screenY - d.last.screenY
        },
        position: current
      });

      d.last = current;
    },

    touchend: function(d, e, t) {
      var current = coordinates(e, t);
      d.emitEvent('holdend', {
        start: d.start,
        end: current,
        dx: current.screenX - d.start.screenX,
        dy: current.screenY - d.start.screenY
      });
      d.switchTo(initialState);
    }
  };

  // We enter this state if a second touch starts before we start
  // recoginzing any other gesture.  As the touches move we track the
  // distance and angle between them to report scale and rotation values
  // in transform events.
  var transformState = {
    name: 'transformState',
    init: function(d, e, t) {
      // Remember the id of the second touch
      d.touch2 = t.identifier;

      // Get the two Touch objects
      var t1 = e.touches.identifiedTouch(d.touch1);
      var t2 = e.touches.identifiedTouch(d.touch2);

      // Compute and remember the initial distance and angle
      d.startDistance = d.lastDistance = touchDistance(t1, t2);
      d.startDirection = d.lastDirection = touchDirection(t1, t2);

      // Don't start emitting events until we're past a threshold
      d.scaled = d.rotated = false;
    },

    touchmove: function(d, e, t) {
      // Ignore touches we're not tracking
      if (t.identifier !== d.touch1 && t.identifier !== d.touch2) {
        return;
      }

      // Get the two Touch objects
      var t1 = e.touches.identifiedTouch(d.touch1);
      var t2 = e.touches.identifiedTouch(d.touch2);

      // Compute the new midpoints, distance and direction
      var midpoint = midpoints(e, t1, t2);
      var distance = touchDistance(t1, t2);
      var direction = touchDirection(t1, t2);
      var rotation = touchRotation(d.startDirection, direction);

      // Check all of these numbers against the thresholds. Otherwise
      // the transforms are too jittery even when you try to hold your
      // fingers still.
      if (!d.scaled) {
        if (abs(distance - d.startDistance) > GD.SCALE_THRESHOLD) {
          d.scaled = true;
          d.startDistance = d.lastDistance =
            floor(d.startDistance +
                  GD.THRESHOLD_SMOOTHING * (distance - d.startDistance));
        } else {
          distance = d.startDistance;
        }
      }
      if (!d.rotated) {
        if (abs(rotation) > GD.ROTATE_THRESHOLD) {
          d.rotated = true;
        } else {
          direction = d.startDirection;
        }
      }

      // If nothing has exceeded the threshold yet, then we
      // don't even have to fire an event.
      if (d.scaled || d.rotated) {
        // The detail field for the transform gesture event includes
        // 'absolute' transformations against the initial values and
        // 'relative' transformations against the values from the last
        // transformgesture event.
        d.emitEvent('transform', {
          absolute: { // transform details since gesture start
            scale: distance / d.startDistance,
            rotate: touchRotation(d.startDirection, direction)
          },
          relative: { // transform since last gesture change
            scale: distance / d.lastDistance,
            rotate: touchRotation(d.lastDirection, direction)
          },
          midpoint: midpoint
        });

        d.lastDistance = distance;
        d.lastDirection = direction;
        d.lastMidpoint = midpoint;
      }
    },

    touchend: function(d, e, t) {
      // If either finger goes up, we're done with the gesture.
      // The user might move that finger and put it right back down
      // again to begin another 2-finger gesture, so we can't go
      // back to the initial state while one of the fingers remains up.
      // On the other hand, we can't go back to touchStartedState because
      // that would mean that the finger left down could cause a tap or
      // pan event. So we need an afterTransform state that waits for
      // a finger to come back down or the other finger to go up.
      if (t.identifier === d.touch2) {
        d.touch2 = null;
      } else if (t.identifier === d.touch1) {
        d.touch1 = d.touch2;
        d.touch2 = null;
      } else {
        return; // It was a touch we weren't tracking
      }

      // If we emitted any transform events, now we need to emit
      // a transformend event to end the series.  The details of this
      // event use the values from the last touchmove, and the
      // relative amounts will 1 and 0, but they are included for
      // completeness even though they are not useful.
      if (d.scaled || d.rotated) {
        d.emitEvent('transformend', {
          absolute: { // transform details since gesture start
            scale: d.lastDistance / d.startDistance,
            rotate: touchRotation(d.startDirection, d.lastDirection)
          },
          relative: { // nothing has changed relative to the last touchmove
            scale: 1,
            rotate: 0
          },
          midpoint: d.lastMidpoint
        });
      }

      d.switchTo(afterTransformState);
    }
  };

  // We did a tranform and one finger went up. Wait for that finger to
  // come back down or the other finger to go up too.
  var afterTransformState = {
    name: 'afterTransformState',
    touchstart: function(d, e, t) {
      d.switchTo(transformState, e, t);
    },

    touchend: function(d, e, t) {
      if (t.identifier === d.touch1) {
        d.switchTo(initialState);
      }
    }
  };

  return GD;
}());

define("shared/gesture_detector", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.GestureDetector;
    };
}(this)));

define('utils/ordered_map',['require','exports','module','compare','binsearch'],function(require, exports, module) {


/**
 * Module dependencies
 */
var baseCompare = require('compare');
var binsearch = require('binsearch');

function OrderedMap(list, compare) {
  if (!compare) {
    compare = baseCompare;
  }

  this.compare = function(a, b) {
    return compare(a[0], b[0]);
  };

  if (list) {
    this.items = list.sort(this.compare);
  } else {
    this.items = [];
  }
}
module.exports = OrderedMap;

OrderedMap.prototype = {
  has: function(value) {
    return this.indexOf(value) !== null;
  },

  insertIndexOf: function(value) {
    return binsearch.insert(
      this.items,
      [value],
      this.compare
    );
  },

  previous: function(key) {
    var idx = this.indexOf(key);
    if (idx !== null && this.items[idx - 1]) {
      return this.items[idx - 1][1];
    }
    return null;
  },

  next: function(key) {
    var idx = this.indexOf(key);
    if (idx !== null && this.items[idx + 1]) {
      return this.items[idx + 1][1];
    }
    return null;
  },

  indexOf: function(value) {
    return binsearch.find(
      this.items,
      [value],
      this.compare
    );
  },

  set: function(key, value) {
    var arr = [key, value];

    var idx = binsearch.insert(
      this.items,
      arr,
      this.compare
    );

    var prev = this.items[idx];
    var remove = 0;

    if (prev && prev[0] === key) {
      remove = 1;
    }

    this.items.splice(idx, remove, arr);

    return value;
  },

  get: function(item) {
    if (typeof(item) === 'undefined') {
      throw new Error('cannot search "undefined" values');
    }

    var idx = this.indexOf(item);
    if (idx !== null) {
      return this.items[idx][1];
    }
    return null;
  },

  remove: function(key) {
    var idx = this.indexOf(key);

    if (idx !== null) {
      this.items.splice(idx, 1);
    }
  },

  get length() {
    return this.items.length;
  }
};

});

define('views/time_parent',['require','exports','module','shared/gesture_detector','utils/ordered_map','view','debug'],function(require, exports, module) {


var GestureDetector = require('shared/gesture_detector');
var OrderedMap = require('utils/ordered_map');
var View = require('view');
var debug = require('debug')('time_parent');

var XSWIPE_OFFSET = window.innerWidth / 10;

/**
 * Parent view for busytime-based views
 * (month, week, day) contains basic
 * handlers for purging frames, panning, etc...
 *
 * Each "child" must be added to frames.
 * Each child must be identified by some id.
 *
 * Child classes are expected to have "create"
 * method and "destroy" methods for adding &
 * removing them from the dom.
 */
function TimeParent() {
  View.apply(this, arguments);
  this.frames = new OrderedMap();
  this._initEvents();
}
module.exports = TimeParent;

TimeParent.prototype = {
  __proto__: View.prototype,

  /**
   * Maximum number of child elements to keep
   * around until we start removing them.
   */
  maxFrames: 5,

  get frameContainer() {
    return this.element;
  },

  _initEvents: function() {
    this.app.timeController.on('purge', this);
    this.element.addEventListener('swipe', this);
    this.element.addEventListener('wheel', this);

    this.gd = new GestureDetector(this.element);
    this.gd.startDetecting();
  },

  _onswipe: function(data) {
    if (Math.abs(data.dy) > (Math.abs(data.dx) - XSWIPE_OFFSET)) {
      return false;
    }

    this._move(data.direction === 'left');
    return true;
  },

  _onwheel: function(event) {
    if (event.deltaMode !== event.DOM_DELTA_PAGE || event.deltaX === 0) {
      return false;
    }

    this._move(event.deltaX > 0);
    return true;
  },

  _move: function(next) {
    var controller = this.app.timeController;

    // TODO: RTL
    if (next) {
      controller.move(this._nextTime(this.date));
    } else {
      controller.move(this._previousTime(this.date));
    }
  },

  handleEvent: function(e) {
    switch (e.type) {
      case 'swipe':
        debug('Detected swipe!');
        this._onswipe(e.detail);
        break;
      case 'purge':
        this.purgeFrames(e.data[0]);
        break;
      case 'wheel':
        this._onwheel(e);
        break;
    }
  },

  _onCalendarVisibilityChange: function() {
    // we need to restore previous scroll position otherwise it would move
    // back to top every time the calendar visibility is toggled which would
    // be very confusing for the user
    var scrollTop = this.currentFrame.getScrollTop() || 0;
    this.purgeFrames(this.app.timeController.timespan);
    this.changeDate(this.date);
    this.currentFrame.setScrollTop(scrollTop);
  },

  /**
   * Creates a single 'frame' for the parent.
   * A frame can be any object with the following capabilities;
   *
   *    - element: property that contains a dom element
   *               that has yet to be inserted into the document.
   *
   *    - timespan: a timespan object for purge events.
   *
   *    - activate: a method to activate the frame.
   *
   *    - deactivate: a method to deactivate the frame.
   *
   *    - destroy: a method to destroy the frame.
   *
   *
   * The default behaviour of this method is to use
   * the 'childClass' property to create an object
   * to use as the frame. In day/month cases the frame
   * can be the child class directly.
   *
   * @param {Date} date frame time.
   */
  _createFrame: function(date) {
    /** default childClass implementation */
    var child = new this.childClass({
      app: this.app,
      date: date
    });
    child.create();
    return child;
  },

  _nextTime: function() {},
  _previousTime: function() {},

  _getId: function(date) {
    return date.valueOf();
  },

  /**
   * Removes extra frames when frames.length > maxFrames.
   */
  _trimFrames: function() {
    var frames = this.frames;
    var keep;

    if (frames.length > this.maxFrames) {
      // determine splice size
      var idx = frames.indexOf(this.currentFrame.id);
      idx = (idx - 1) || 0;

      // remove the ones we want to keep from the original list.
      // 3 here is not a magic number but the original + prev + next (3)
      keep = frames.items.splice(idx, 3);
      var deleteFrames = frames.items;

      // destroy the rest
      idx = 0;
      var len = deleteFrames.length;
      for (; idx < len; idx++) {
        deleteFrames[idx][1].destroy();
      }

      // replace the .items array with the ones we kept.
      frames.items = keep;
    }
  },

  /**
   * Adds a frame for the given time.
   *
   * @param {Date} date time to add frame for.
   * @return {Object} existing or newly added frame.
   */
  addFrame: function(date) {
    var id = this._getId(date);
    var frame = this.frames.get(id);
    if (!frame) {
      frame = this._createFrame(date);
      this.frames.set(id, frame);

      // XXX: look into correctly positioning
      //      elements by their viewing order.
      this.frameContainer.appendChild(
        frame.element
      );
    }

    return frame;
  },

  /**
   * Changes date of the parent frame.
   *
   * @param {Date} time center point to activate.
   */
  changeDate: function(time) {
    var prevScrollTop = 0;

    // deactivate previous frame
    if (this.currentFrame) {
      prevScrollTop = this.currentFrame.getScrollTop();
      this.currentFrame.deactivate();
    }

    this.date = time;

    // setup & find all ids
    var next = this._nextTime(time);
    var prev = this._previousTime(time);

    // add previous frame
    this.addFrame(prev);

    // create & activate current frame
    var cur = this.currentFrame = this.addFrame(time);
    cur.activate();
    cur.setScrollTop(prevScrollTop);

    // add next frame
    this.addFrame(next);

    // ensure we don't have too many extra frames.
    this._trimFrames();
  },

  /**
   *
   * @param {Calendar.Timespan} timespan span of time.
   */
  purgeFrames: function(span) {
    var child;
    var i = 0;
    var len = this.frames.length;

    var offset = 0;

    for (; i < len; i++) {
      child = this.frames.items[i - offset][1];
      if (span.contains(child.timespan)) {
        // Bug 827249 - remove current frame when its purged.
        if (this.currentFrame === child) {
          this.currentFrame = null;
        }

        child.destroy();
        this.frames.items.splice(i - offset, 1);
        offset += 1;
      }
    }
  },

  onactive: function() {
    View.prototype.onactive.apply(this, arguments);
    if (this.app && this.scale) {
      this.app.timeController.scale = this.scale;
    }
  }
};

});

define('views/month',['require','exports','module','calc','./month_child','./time_parent'],function(require, exports, module) {


var Calc = require('calc');
var MonthChild = require('./month_child');
var Parent = require('./time_parent');

function Month(options) {
  Parent.apply(this, arguments);
  // default to today
  this._selectedDay = new Date();
}
module.exports = Month;

Month.prototype = {
  __proto__: Parent.prototype,

  scale: 'month',

  selectors: {
    element: '#month-view',
    selectedDay: 'li.selected'
  },

  childClass: MonthChild,

  SELECTED: 'selected',

  _onswipe: function() {
    var didSwipe = Parent.prototype._onswipe.apply(this, arguments);

    // If we changed months, set the selected day to the 1st
    if (didSwipe) {
      this.controller.selectedDay = this.date;
    }
  },

  _onwheel: function() {
    var didWheel = Parent.prototype._onwheel.apply(this, arguments);

    // If we changed months, set the selected day to the 1st
    if (didWheel) {
      this.controller.selectedDay = this.date;
    }
  },

  _clearSelectedDay: function() {
    var day = this.element.querySelector(
      this.selectors.selectedDay
    );

    if (day) {
      day.classList.remove(this.SELECTED);
      day.removeAttribute('aria-selected');
    }
  },

  _selectDay: function(date) {
    var el, id;
    this._clearSelectedDay();

    id = Calc.getDayId(date);
    id = this.currentFrame._dayId(id);

    el = document.getElementById(id);

    if (el) {
      el.classList.add(this.SELECTED);
      el.setAttribute('aria-selected', true);
      // Put the screen reader cursor onto the selected day.
      el.focus();
      this._selectedDay = date;
    }
  },

  _initEvents: function() {
    this.controller = this.app.timeController;

    Parent.prototype._initEvents.apply(this, arguments);

    this.controller.on('selectedDayChange', this);
    this.controller.on('monthChange', this);
    this.delegate(this.element, 'click', '[data-date]', this);
    this.delegate(this.element, 'dbltap', '[data-date]', this);
  },

  handleEvent: function(e, target) {
    Parent.prototype.handleEvent.apply(this, arguments);

    switch (e.type) {
      case 'click':
        var date = Calc.dateFromId(target.dataset.date);
        this.controller.selectedDay = date;
        break;
      case 'dbltap':
        this.app.go('/day/');
        break;
      case 'selectedDayChange':
        this._selectDay(e.data[0]);
        break;
      case 'monthChange':
        this._clearSelectedDay();
        this.changeDate(e.data[0]);
        break;
    }
  },

  _createChild: function(time) {
    return new MonthChild({ app: this.app, date: time });
  },

  _getId: function(date) {
    return date.valueOf();
  },

  /**
   * Moves calendar to the next month.
   */
  _nextTime: function(time) {
    return new Date(
      time.getFullYear(),
      time.getMonth() + 1,
      time.getDate()
    );
  },

  /**
   * Moves calendar to the next month.
   */
  _previousTime: function(time) {
    return new Date(
      time.getFullYear(),
      time.getMonth() - 1,
      time.getDate()
    );
  },

  /**
   * Render current month
   */
  render: function() {
    var time = this.controller.month;
    this.changeDate(time);
  }

};

Month.prototype.onfirstseen = Month.prototype.render;

});
