
define('template',['require','exports','module'],function(require, exports, module) {


var POSSIBLE_HTML = /[&<>"'`]/;

var span = document.createElement('span');

function create(templates) {
  var key, result = {};

  for (key in templates) {
    if (templates.hasOwnProperty(key)) {
      result[key] = new Template(templates[key]);
    }
  }

  return result;
}

function Template(fn) {
  this.template = fn;
}
module.exports = Template;

Template.prototype = {
  arg: function(key) {
    if (typeof(this.data) === 'undefined') {
      return '';
    } else if (typeof(this.data) !== 'object') {
      return this.data;
    }

    return this.data[key];
  },

  h: function(a) {
    var arg = this.arg(a);
    // accept anything that can be converted into a string and we make sure
    // the only falsy values that are converted into empty strings are
    // null/undefined to avoid mistakes
    arg = arg == null ? '' : String(arg);

    //only escape bad looking stuff saves
    //a ton of time
    if (POSSIBLE_HTML.test(arg)) {
      span.textContent = arg;
      return span.innerHTML.replace(/"/g, '&quot;').replace(/'/g, '&#x27;');
    } else {
      return arg;
    }
  },

  s: function(a) {
    var arg = this.arg(a);
    return String((arg || ''));
  },

  bool: function(key, onTrue) {
    if (this.data[key]) {
      return onTrue;
    } else {
      return '';
    }
  },

  l10n: function(key, prefix) {
    var value = this.arg(key);

    if (prefix) {
      value = prefix + value;
    }
    return navigator.mozL10n.get(value);
  },

  l10nId: function(a) {
    return this.s(a).replace(/\s/g, '-');
  },

  /**
   * Renders template with given slots.
   *
   * @param {Object} object key, value pairs for template.
   */
  render: function(data) {
    this.data = data;
    return this.template();
  },

  /**
   * Renders template multiple times
   *
   * @param {Array} objects object details to render.
   * @param {String} [join] optional join argument will join the array.
   * @return {String|Array} String if join argument is given array otherwise.
   */
  renderEach: function(objects, join) {
    var i = 0, len = objects.length,
        result = [];

    for (; i < len; i++) {
      result.push(this.render(objects[i]));
    }

    if (typeof(join) !== 'undefined') {
      return result.join(join);
    }

    return result;
  }
};

Template.create = create;

});

define('templates/alarm',['require','exports','module','template','date_format'],function(require, exports, module) {


var create = require('template').create;
var dateFormat = require('date_format');

var MINUTE = 60;
var HOUR = 3600;
var DAY = 86400;
var WEEK = 604800;
var MORNING = HOUR * 9;
var layouts = {
  standard: [
    'none',
    0,
    0 - MINUTE * 5,
    0 - MINUTE * 15,
    0 - MINUTE * 30,
    0 - HOUR,
    0 - HOUR * 2,
    0 - DAY
  ],
  allday: [
    'none',
    0 + MORNING,
    0 - DAY + MORNING,
    0 - DAY * 2 + MORNING,
    0 - WEEK + MORNING,
    0 - WEEK * 2 + MORNING
  ]
};

var Alarm = create({
  reminder: function() {
    var alarmContent = '';
    var alarms = this.arg('alarms');
    var isAllDay = this.arg('isAllDay');

    var i = 0;
    var alarm;
    while ((alarm = alarms[i])) {
      i++;
      alarmContent += Alarm.description.render({
        trigger: alarm.trigger,
        layout: isAllDay ? 'allday' : 'standard'
      });
    }

    return alarmContent;
  },

  description: function() {
    var {id, data} = getL10n(this.arg('trigger'), this.arg('layout'));
    var args = JSON.stringify(data);
    var description = navigator.mozL10n.get(id, data);

    return `<div role="listitem" data-l10n-id="${id}"
      data-l10n-args=\'${args}\'>
        ${description}
      </div>`;
  },

  // builds a list of <option>
  options: function() {
    var content = '';
    var selected;
    var foundSelected = false;

    var trigger = this.arg('trigger');
    var layout = this.arg('layout') || 'standard';
    var options = layouts[layout];

    var i = 0;
    var iLen = options.length;

    for (; i < iLen; i++) {
      selected = false;

      // trigger option 'selected' by normalizing imported dates
      if (layout === 'allday') {
        if (options[i] === (trigger + MORNING)) {
          trigger += MORNING;
        }
      }

      if (!selected && trigger && options[i] === trigger) {
        selected = true;
        foundSelected = true;
      }

      content += Alarm.option.render({
        selected: selected,
        layout: layout,
        value: options[i]
      });
    }

    // foundSelected is used in cases where user is editing an event that has
    // a custom reminder value (X minutes/hours/days before event) and that
    // is an option that we don't support internally on the calendar app.
    // we always add a new <option> using the custom value and mark it as
    // selected.
    if (!foundSelected && /^-?\d+$/.test(trigger)) {
      content += Alarm.option.render({
        selected: true,
        layout: layout,
        value: trigger
      });
    }

    return content;
  },

  option: function() {
    var _ = navigator.mozL10n.get;

    var layout = this.arg('layout');
    var value = this.arg('value');
    var selected = this.arg('selected');

    var l10n = getL10n(value, layout);

    var content = [
      '<option',
      'value="' + value + '"',
      (selected ? 'selected' : ''),
      'data-l10n-id="' + l10n.id + '"',
      'data-l10n-args=\'' + JSON.stringify(l10n.data) + '\'>',
      _(l10n.id, l10n.data) + '</option>'
    ].join(' ');

    return content;
  },

  picker: function() {
    return '<span class="button icon icon-dialog">' +
      '<select name="alarm[]">' +
        Alarm.options.render(this.data) +
      '</select>' +
    '</span>';
  }
});

function getL10n(trigger, layout) {
  if (trigger === 'none') {
    return {
      id: trigger,
      data: {}
    };
  }

  // Format the display text based on a zero-offset trigger
  if (layout === 'allday') {
    var options = layouts.allday;
    if (options.indexOf(trigger) !== -1) {
      trigger -= MORNING;
    }
  }

  if (trigger === 0) {
    return {
      id: 'alarm-at-event-' + layout,
      data: {}
    };
  }

  var affix = trigger > 0 ? 'after' : 'before';
  var parts = dateFormat.relativeParts(trigger);

  for (var i in parts) {
    // we only use the first part (biggest value)
    return {
      id: i + '-' + affix,
      data: {
        value: parts[i]
      }
    };
  }
}
module.exports = Alarm;

});

define('models/event',['require','exports','module','calc','probably_parse_int'],function(require, exports, module) {


var Calc = require('calc');
var probablyParseInt = require('probably_parse_int');

/**
 * Creates a wrapper around a event instance from the db
 */
function Event(data) {
  var isNew = false;

  if (typeof(data) === 'undefined') {
    isNew = true;
    data = Object.create(null);
    data.remote = {};
  }

  this.data = data;
  /** shortcut */
  var remote = this.remote = this.data.remote;

  if ('start' in remote && !('startDate' in remote)) {
    remote.startDate = Calc.dateFromTransport(
      remote.start
    );
  }

  if ('end' in remote && !('endDate' in remote)) {
    remote.endDate = Calc.dateFromTransport(
      remote.end
    );
  }

  if (isNew) {
    this.resetToDefaults();
  }

  var start = this.remote.startDate;
  var end = this.remote.endDate;

  // the typeof check is to see if we have already
  // set the value in resetToDefaults (or prior)
  if (
      typeof(this._isAllDay) === 'undefined' &&
      Calc.isOnlyDate(start) &&
      Calc.isOnlyDate(end)
  ) {
    // mostly to handle the case before the time
    // where we actually managed isAllDay as a setter.
    this.isAllDay = true;
  } else {
    // not on prototype intentionally because
    // we need to either need to resetToDefaults
    // or check startDate/endDate in the constructor.
    this.isAllDay = false;
  }
}
module.exports = Event;

Event.prototype = {

  /**
   * Sets default values of an event.
   */
  resetToDefaults: function() {
    var now = new Date();

    this.isAllDay = false;

    this.startDate = new Date(
      now.getFullYear(),
      now.getMonth(),
      now.getDate()
    );

    this.endDate = new Date(
      now.getFullYear(),
      now.getMonth(),
      now.getDate() + 1
    );
  },

  get _id() {
    return this.data._id;
  },

  _setDate: function(date, field) {
    if (!(date instanceof Date)) {
      throw new TypeError('must pass instance of Date');
    }

    var allDay = this.isAllDay;

    if (allDay) {
      // clone the existing date
      date = new Date(date.valueOf());

      // filter out the stuff we don't care about
      date.setHours(0);
      date.setMinutes(0);
      date.setSeconds(0);
      date.setMilliseconds(0);
    }

    this.remote[field] = Calc.dateToTransport(
      date,
      null, // TODO allow setting tzid
      allDay
    );

    this.remote[field + 'Date'] = date;
  },

  /* start date */

  get startDate() {
    return this.remote.startDate;
  },

  set startDate(value) {
    this._setDate(value, 'start');
  },

  /* end date */

  get endDate() {
    return this.remote.endDate;
  },

  set endDate(value) {
    this._setDate(value, 'end');
  },

  set isAllDay(value) {
    this._isAllDay = value;

    // send values through the their setter.
    if (this.endDate) {
      this.endDate = this.endDate;
    }

    if (this.startDate) {
      this.startDate = this.startDate;
    }
  },

  get isAllDay() {
    return this._isAllDay;
  },

  /* associated records */

  get calendarId() {
    return this.data.calendarId;
  },

  set calendarId(value) {
    if (value && typeof(value) !== 'number') {
      value = probablyParseInt(value);
    }

    this.data.calendarId = value;
  },

  /* simple setters */

  get syncToken() {
    return this.remote.syncToken;
  },

  set syncToken(value) {
    this.remote.syncToken = value;
  },

  get title() {
    return this.remote.title || '';
  },

  set title(value) {
    this.remote.title = value;
  },

  get description() {
    return this.remote.description || '';
  },

  set description(value) {
    this.remote.description = value;
  },

  get location() {
    return this.remote.location || '';
  },

  set location(value) {
    this.remote.location = value;
    return this.remote.location;
  },

  get alarms() {
    return this.remote.alarms || [];
  },

  set alarms(value) {
    this.remote.alarms = value;
    return this.remote.alarms;
  },

  /**
   * If data doesn't have any errors, the event
   * takes on the attributes of data.
   *
   * @param {Object} data, object that contains
   *  at least some attributes of the event object.
   *
   * @return {Object} errors if validationErrors returns erros,
   *  true otherwise.
   */
  updateAttributes: function(data) {
    var errors = this.validationErrors(data);
    if (errors) {
      return errors;
    }
    for (var field in data) {
      this[field] = data[field];
    }
    return true;
  },

  /**
   * Validates the contents of the model.
   *
   * Output example:
   *
   *   [
   *     {
   *       name: 'invalidDate',
   *       properties: ['startDate', 'endDate']
   *     }
   *     //...
   *   ]
   *
   * @param {Object} data, optional object that contains
   *  at least some attributes of the event object.
   * @return {Array|False} see above.
   */
  validationErrors: function(data) {
    var obj = data || this;
    var end = obj.endDate.valueOf();
    var start = obj.startDate.valueOf();
    var errors = [];

    if (start >= end) {
      errors.push({
        name: 'start-after-end'
      });
    }

    if (errors.length) {
      return errors;
    }

    return false;
  }
};

});

define('view',['require','exports','module'],function(require, exports, module) {


var DEFAULT_ERROR_ID = 'error-default';
const INVALID_CSS = /([^a-zA-Z\-\_0-9])/g;

/**
 * Very simple base class for views.
 * Provides functionality for active/inactive.
 *
 * The first time the view is activated
 * the onactive function/event will fire.
 *
 * The .seen property is added to each object
 * with view in its prototype. .seen can be used
 * to detect if the view has ever been activated.
 *
 * @param {String|Object} options options or a selector for element.
 */
function View(options) {
  if (typeof(options) === 'undefined') {
    options = {};
  }

  if (typeof(options) === 'string') {
    this.selectors = { element: options };
  } else {
    var key;

    if (typeof(options) === 'undefined') {
      options = {};
    }

    for (key in options) {
      if (options.hasOwnProperty(key)) {
        this[key] = options[key];
      }
    }
  }

  this.hideErrors = this.hideErrors.bind(this);
}
module.exports = View;

View.ACTIVE = 'active';

View.prototype = {
  seen: false,
  activeClass: View.ACTIVE,
  errorVisible: false,

  get element() {
    return this._findElement('element');
  },

  get status() {
    return this._findElement('status');
  },

  get errors() {
    return this._findElement('errors');
  },

  /**
   * Creates a string id for a given model.
   *
   *    view.idForModel('foo-', { _id: 1 }); // => foo-1
   *    view.idForModel('foo-', '2'); // => foo-2
   *
   * @param {String} prefix of string.
   * @param {Object|String|Numeric} objectOrString representation of model.
   */
  idForModel: function(prefix, objectOrString) {
    prefix += (typeof(objectOrString) === 'object') ?
      objectOrString._id :
      objectOrString;

    return prefix;
  },

  calendarId: function(input) {
    if (typeof(input) !== 'string') {
      input = input.calendarId;
    }

    input = this.cssClean(input);
    return 'calendar-id-' + input;
  },

  /**
   * Delegate pattern event listener.
   *
   * @param {HTMLElement} element parent element.
   * @param {String} type type of dom event.
   * @param {String} selector css selector element should match
   *                          _note_ there is no magic here this
   *                          is determined from the root of the document.
   * @param {Function|Object} handler event handler.
   *                                  first argument is the raw
   *                                  event second is the element
   *                                  matching the pattern.
   */
  delegate: function(element, type, selector, handler) {
    if (typeof(handler) === 'object') {
      var context = handler;
      handler = function() {
        context.handleEvent.apply(context, arguments);
      };
    }

    element.addEventListener(type, function(e) {
      var target = e.target;
      while (target !== element) {
        if ('mozMatchesSelector' in target &&
            target.mozMatchesSelector(selector)) {
          return handler(e, target);
        }
        target = target.parentNode;
      }
    });
  },

  /**
   * Clean a string for use with css.
   * Converts illegal chars to legal ones.
   */
  cssClean: function(string) {
    if (typeof(string) !== 'string') {
      return string;
    }

    //TODO: I am worried about the performance
    //of using this all over the place =/
    //consider sanitizing all keys to ensure
    //they don't blow up when used as a selector?
    return string.replace(INVALID_CSS, '-');
  },

  /**
   * Finds a caches a element defined
   * by selectors
   *
   * @param {String} selector name as defined in selectors.
   * @param {Boolean} all true when to find all elements. (default false).
   */
  _findElement: function(name, all, element) {
    if (typeof(all) === 'object') {
      element = all;
      all = false;
    }

    element = element || document;

    var cacheName;
    var selector;

    if (typeof(all) === 'undefined') {
      all = false;
    }

    if (name in this.selectors) {
      cacheName = '_' + name + 'Element';
      selector = this.selectors[name];

      if (!this[cacheName]) {
        if (all) {
          this[cacheName] = element.querySelectorAll(selector);
        } else {
          this[cacheName] = element.querySelector(selector);
        }
      }

      return this[cacheName];
    }

    return null;
  },

 /**
   * Displays a list of errors
   *
   * @param {Array} list error list
   *  (see Event.validaitonErrors) or Error object.
   */
  showErrors: function(list) {
    var _ = navigator.mozL10n.get;
    var errors = '';

    // We can pass Error objects or
    // Array of {name: foo} objects
    if (!Array.isArray(list)) {
        list = [list];
    }

    var i = 0;
    var len = list.length;

    for (; i < len; i++) {
      var name = list[i].l10nID || list[i].name;
      errors += _('error-' + name) || _(DEFAULT_ERROR_ID);
    }

    // populate error and display it.
    this.errors.textContent = errors;
    this.errorVisible = true;
    this.status.classList.add(this.activeClass);

    this.status.addEventListener('animationend', this.hideErrors);
  },

  hideErrors: function() {
    this.status.classList.remove(this.activeClass);
    this.status.removeEventListener('animationend', this.hideErrors);
    this.errorVisible = false;
  },

  onactive: function() {
    if (this.errorVisible) {
      this.hideErrors();
    }

    // seen can be set to anything other than false to override this behaviour
    if (this.seen === false) {
      this.onfirstseen();
    }

    // intentionally using 'in'
    if ('dispatch' in this) {
      this.dispatch.apply(this, arguments);
    }

    this.seen = true;
    if (this.element) {
      this.element.classList.add(this.activeClass);
    }
  },

  oninactive: function() {
    if (this.element) {
      this.element.classList.remove(this.activeClass);
    }
  },

  onfirstseen: function() {}
};

});

define('views/event_base',['require','exports','module','models/event','view','next_tick','provider/provider_factory'],function(require, exports, module) {


var Event = require('models/event');
var View = require('view');
var nextTick = require('next_tick');
var providerFactory = require('provider/provider_factory');

function EventBase(options) {
  View.apply(this, arguments);

  this.store = this.app.store('Event');

  this._els = Object.create(null);
  this._changeToken = 0;

  this.cancel = this.cancel.bind(this);
  this.primary = this.primary.bind(this);
  this._initEvents();
}
module.exports = EventBase;

EventBase.prototype = {
  __proto__: View.prototype,

  READONLY: 'readonly',
  CREATE: 'create',
  UPDATE: 'update',
  PROGRESS: 'in-progress',
  ALLDAY: 'allday',
  LOADING: 'loading',

  DEFAULT_VIEW: '/month/',

  _initEvents: function() {
    this.header.addEventListener('action', this.cancel);
    this.primaryButton.addEventListener('click', this.primary);
  },

  uiSelector: '.%',

  get header() {
    return this._findElement('header');
  },

  get primaryButton() {
    return this._findElement('primaryButton');
  },

  get fieldRoot() {
    return this.element;
  },

  /**
   * Returns the url the view will "redirect" to
   * after completing the current add/edit/delete operation.
   *
   * @return {String} redirect url.
   */
  returnTo: function() {
    var path = this._returnTo || this.DEFAULT_VIEW;
    return path;
  },

  /**
   * Returns the top level URL, or returnTo()
   * Resets the returnTop variable so we can override on next visit
   */
  returnTop: function() {
    var path = this._returnTop || this.returnTo();
    delete this._returnTop;
    return path;
  },

  /**
   * Dismiss modification and go back to previous screen.
   */
  cancel: function() {
    window.history.back();
  },

  /**
   * This method is overridden
   */
  primary: function() {},

  /**
   * This method is overridden
   */
  _markReadonly: function() {},

  /**
   * When the event is something like this:
   * 2012-01-02 and we detect this is an all day event
   * we want to display the end date like this 2012-01-02.
   */
  formatEndDate: function(endDate) {
    if (
      endDate.getHours() === 0 &&
      endDate.getSeconds() === 0 &&
      endDate.getMinutes() === 0
    ) {
      // subtract the date to give the user a better
      // idea of which dates the event spans...
      endDate = new Date(
        endDate.getFullYear(),
        endDate.getMonth(),
        endDate.getDate() - 1
      );
    }

    return endDate;
  },

  /**
   * Assigns and displays event & busytime information.
   * Marks view as "loading"
   *
   * @param {Object} busytime for view.
   * @param {Object} event for view.
   * @param {Function} [callback] optional callback.
   */
  useModel: function(busytime, event, callback) {
    // mark view with loading class
    var classList = this.element.classList;
    classList.add(this.LOADING);

    this.event = new Event(event);
    this.busytime = busytime;

    var changeToken = ++this._changeToken;

    var self = this;

    this.store.ownersOf(event, fetchOwners);

    function fetchOwners(err, owners) {
      self.originalCalendar = owners.calendar;
      self.provider = providerFactory.get(owners.account.providerType);
      self.provider.eventCapabilities(
        self.event,
        fetchEventCaps
      );
    }

    function fetchEventCaps(err, caps) {
      if (self._changeToken !== changeToken) {
        return;
      }

      if (err) {
        console.error('Failed to fetch events capabilities', err);

        if (callback) {
          classList.remove(self.LOADING);
          callback(err);
        }

        return;
      }

      if (!caps.canUpdate) {
        self._markReadonly(true);
        self.element.classList.add(self.READONLY);
      }

      // inheritance hook...
      self._updateUI();

      // we only remove the loading class after the UI is rendered just to
      // avoid potential race conditions during marionette tests (trying to
      // read the data before it's on the DOM)
      classList.remove(self.LOADING);

      if (callback) {
        callback();
      }
    }
  },

  /** override me! **/
  _updateUI: function() {},

  /**
   * Loads event and triggers form update.
   * Gracefully will handle race conditions
   * if rapidly switching between events.
   * TODO: This token may no longer be needed
   *   as we have an aria-disabled guard now.
   *
   * @param {String} id busytime id.
   */
  _loadModel: function(id, callback) {
    var self = this;
    var token = ++this._changeToken;
    var time = this.app.timeController;
    var classList = this.element.classList;

    classList.add(this.LOADING);

    time.findAssociated(id, function(err, list) {
      if (err) {
        classList.remove(this.LOADING);
        console.error('Error looking up records for id: ', id);
      }

      var records = list[0];
      if (token === self._changeToken) {
        self.useModel(
          records.busytime,
          records.event,
          callback
        );
      } else {
        // ensure loading is removed
        classList.remove(this.LOADING);
      }
    });
  },

  /**
   * Builds and sets defaults for a new model.
   *
   * @return {Calendar.Models.Model} new model.
   */
  _createModel: function(time) {
    var now = new Date();

    if (time < now) {
      time = now;
      now.setHours(now.getHours() + 1);
      now.setMinutes(0);
      now.setSeconds(0);
      now.setMilliseconds(0);
    }

    var model = new Event();
    model.startDate = time;

    var end = new Date(time.valueOf());
    end.setHours(end.getHours() + 1);

    model.endDate = end;

    return model;
  },

  /**
   * Gets and caches an element by selector
   */
  getEl: function(name) {
    if (!(name in this._els)) {
      var el = this.fieldRoot.querySelector(
        this.uiSelector.replace('%', name)
      );
      if (el) {
        this._els[name] = el;
      }
    }
    return this._els[name];
  },

  oninactive: function() {
    View.prototype.oninactive.apply(this, arguments);
  },

  /**
   * Handles the url parameters for when this view
   * comes into focus.
   *
   * When the (busytime) id parameter is given the event will
   * be found via the time controller.
   */
  dispatch: function(data) {
    // always remove loading initially (to prevent worst case)
    this.element.classList.remove(this.LOADING);

    var id = data.params.id;
    var classList = this.element.classList;
    var last = this.app.router.last;

    if (last && last.path) {
      if (!(/^\/(day|event|month|week)/.test(last.path))) {
        // We came from some place suspicious so fall back to default.
        this._returnTo = this.DEFAULT_VIEW;
      } else {
        // Return to the default view if we just added an event.
        // Else go back to where we came from.
        this._returnTo = /^\/event\/add\//.test(last.path) ?
            this.DEFAULT_VIEW : last.path;
      }
    }

    if (!this._returnTop && this._returnTo) {
      this._returnTop = this._returnTo;
    }

    var self = this;
    function completeDispatch() {
      if (self.ondispatch) {
        self.ondispatch();
      }
    }

    if (id) {
      classList.add(this.UPDATE);

      this._loadModel(id, completeDispatch);
    } else {
      classList.add(this.CREATE);

      var controller = this.app.timeController;
      this.event = this._createModel(controller.mostRecentDay);
      this._updateUI();

      nextTick(completeDispatch);
    }

    this.primaryButton.removeAttribute('aria-disabled');
  },

  onfirstseen: function() {}

};

});

/* exported InputParser */


/**
 * Stateless object for input parser functions..
 * The intent is the methods here will only relate to the parsing
 * of input[type="date|time"]
 */
var InputParser = (function() {

  var InputParser = {
    _dateParts: ['year', 'month', 'date'],
    _timeParts: ['hours', 'minutes', 'seconds'],

    /**
     * Import HTML5 input[type="time"] string value
     *
     * @param {String} value 23:20:50.52, 17:39:57.
     * @return {Object} { hours: 23, minutes: 20, seconds: 50 }.
     */
    importTime: function(value) {
      var result = {
        hours: 0,
        minutes: 0,
        seconds: 0
      };

      if (typeof(value) !== 'string') {
        return result;
      }

      var parts = value.split(':');
      var part;
      var partName;

      var i = 0;
      var len = InputParser._timeParts.length;

      for (; i < len; i++) {
        partName = InputParser._timeParts[i];
        part = parts[i];
        if (part) {
          result[partName] = parseInt(part.slice(0, 2), 10) || 0;
        }
      }

      return result;
    },

    /**
     * Export date to HTML5 input[type="time"]
     *
     * @param {Date} value export value.
     * @return {String} 17:39:57.
     */
    exportTime: function(value) {
      var hour = value.getHours();
      var minute = value.getMinutes();
      var second = value.getSeconds();

      var result = '';

      result += InputParser.padNumber(hour) + ':';
      result += InputParser.padNumber(minute) + ':';
      result += InputParser.padNumber(second);

      return result;
    },

    /**
     * Import HTML5 input[type="time"] to object.
     *
     * @param {String} value 1997-12-19.
     * @return {Object} { year: 1997, month: 12, date: 19 }.
     */
    importDate: function(value) {
      var result = {
        year: 0,
        month: 0,
        date: 0
      };

      var parts = value.split('-');
      var part;
      var partName;

      var i = 0;
      var len = InputParser._dateParts.length;

      for (; i < len; i++) {
        partName = InputParser._dateParts[i];
        part = parts[i];
        if (part) {
          result[partName] = parseInt(part, 10);
        }
      }

      if (result.month > 0) {
        result.month = result.month - 1;
      }

      result.date = result.date || 1;

      return result;
    },

    /**
     * Export js date to HTML5 input[type="date"]
     *
     * @param {Date} value export value.
     * @return {String} date string (1997-12-19).
     */
    exportDate: function(value) {
      var year = value.getFullYear();
      var month = value.getMonth() + 1;
      var date = value.getDate();

      var result = '';

      result += InputParser.padNumber(year) + '-';
      result += InputParser.padNumber(month) + '-';
      result += InputParser.padNumber(date);

      return result;
    },

    /**
     * Designed to take a date & time value from
     * html5 input types and returns a JS Date.
     *
     * @param {String} date input date.
     * @param {String} time input time.
     *
     * @return {Date} full date object from date/time.
     */
    formatInputDate: function(date, time) {
      time = InputParser.importTime(time);
      date = InputParser.importDate(date);

      return new Date(
        date.year,
        date.month,
        date.date,
        time.hours,
        time.minutes,
        time.seconds
      );
    },

    /**
     * @param {Numeric} numeric value.
     * @return {String} Pad the numeric with a leading zero if < 10.
     */
    padNumber: function(numeric) {
      var value = String(numeric);
      if (numeric < 10) {
        return '0' + value;
      }

      return value;
    }
  };

  return InputParser;
}());

define("shared/input_parser", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.InputParser;
    };
}(this)));

/* global Buffer */
define('querystring',['require','exports','module'],function(require, exports) {


// Copyright Joyent, Inc. and other Node contributors.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to permit
// persons to whom the Software is furnished to do so, subject to the
// following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
// USE OR OTHER DEALINGS IN THE SOFTWARE.

// If obj.hasOwnProperty has been overridden, then calling
// obj.hasOwnProperty(prop) will break.
// See: https://github.com/joyent/node/issues/1707
function hasOwnProperty(obj, prop) {
  return Object.prototype.hasOwnProperty.call(obj, prop);
}


function charCode(c) {
  return c.charCodeAt(0);
}


// a safe fast alternative to decodeURIComponent
exports.unescapeBuffer = function(s, decodeSpaces) {
  var out = new Buffer(s.length);
  var state = 'CHAR'; // states: CHAR, HEX0, HEX1
  var n, m, hexchar;

  for (var inIndex = 0, outIndex = 0; inIndex <= s.length; inIndex++) {
    var c = s.charCodeAt(inIndex);
    switch (state) {
      case 'CHAR':
        switch (c) {
          case charCode('%'):
            n = 0;
            m = 0;
            state = 'HEX0';
            break;
          case charCode('+'):
            if (decodeSpaces) {
              c = charCode(' ');
            }
            out[outIndex++] = c;
            break;
          default:
            out[outIndex++] = c;
        }
        break;

      case 'HEX0':
        state = 'HEX1';
        hexchar = c;
        if (charCode('0') <= c && c <= charCode('9')) {
          n = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          n = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          n = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = c;
          state = 'CHAR';
          break;
        }
        break;

      case 'HEX1':
        state = 'CHAR';
        if (charCode('0') <= c && c <= charCode('9')) {
          m = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          m = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          m = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = hexchar;
          out[outIndex++] = c;
          break;
        }
        out[outIndex++] = 16 * n + m;
        break;
    }
  }

  // TODO support returning arbitrary buffers.

  return out.slice(0, outIndex - 1);
};


exports.unescape = function(s, decodeSpaces) {
  return exports.unescapeBuffer(s, decodeSpaces).toString();
};


exports.escape = function(str) {
  return encodeURIComponent(str);
};

var stringifyPrimitive = function(v) {
  switch (typeof v) {
    case 'string':
      return v;

    case 'boolean':
      return v ? 'true' : 'false';

    case 'number':
      return isFinite(v) ? v : '';

    default:
      return '';
  }
};


exports.stringify = exports.encode = function(obj, sep, eq, name) {
  sep = sep || '&';
  eq = eq || '=';
  if (obj === null) {
    obj = undefined;
  }

  if (typeof obj === 'object') {
    return Object.keys(obj).map(function(k) {
      var ks = exports.escape(stringifyPrimitive(k)) + eq;
      if (Array.isArray(obj[k])) {
        return obj[k].map(function(v) {
          return ks + exports.escape(stringifyPrimitive(v));
        }).join(sep);
      } else {
        return ks + exports.escape(stringifyPrimitive(obj[k]));
      }
    }).join(sep);

  }

  if (!name) {
    return '';
  }
  return exports.escape(stringifyPrimitive(name)) + eq +
         exports.escape(stringifyPrimitive(obj));
};

// Parse a key=val string.
exports.parse = exports.decode = function(qs, sep, eq, options) {
  sep = sep || '&';
  eq = eq || '=';
  var obj = {};

  if (typeof qs !== 'string' || qs.length === 0) {
    return obj;
  }

  var regexp = /\+/g;
  qs = qs.split(sep);

  var maxKeys = 1000;
  if (options && typeof options.maxKeys === 'number') {
    maxKeys = options.maxKeys;
  }

  var len = qs.length;
  // maxKeys <= 0 means that we should not limit keys count
  if (maxKeys > 0 && len > maxKeys) {
    len = maxKeys;
  }

  for (var i = 0; i < len; ++i) {
    var x = qs[i].replace(regexp, '%20'),
        idx = x.indexOf(eq),
        kstr, vstr, k, v;

    if (idx >= 0) {
      kstr = x.substr(0, idx);
      vstr = x.substr(idx + 1);
    } else {
      kstr = x;
      vstr = '';
    }

    try {
      k = decodeURIComponent(kstr);
      v = decodeURIComponent(vstr);
    } catch (e) {
      k = exports.unescape(kstr, true);
      v = exports.unescape(vstr, true);
    }

    if (!hasOwnProperty(obj, k)) {
      obj[k] = v;
    } else if (Array.isArray(obj[k])) {
      obj[k].push(v);
    } else {
      obj[k] = [obj[k], v];
    }
  }

  return obj;
};

});

// Loader plugin for loading CSS. Does not guarantee loading via onload
// watching, just inserts link tag.
// PS: borrowed from email app and edited it to include the baseUrl
define('css',['require','exports','module'],function(require, exports) {


exports.baseUrl = '/style/';

exports.load = function(id, require, onload, config) {
  if (config.isBuild) {
    return onload();
  }

  var style = document.createElement('link');
  style.type = 'text/css';
  style.rel = 'stylesheet';
  style.href = require.toUrl(exports.baseUrl + id + '.css');
  style.addEventListener('load', onload, false);
  document.head.appendChild(style);
};

});

/* exported LazyLoader */
/* globals HtmlImports, Promise */


/**
 * This contains a simple LazyLoader implementation
 * To use:
 *
 *   LazyLoader.load(
 *    ['/path/to/file.js', '/path/to/file.css', 'domNode'], callback
 *   );
 */
var LazyLoader = (function() {

  function LazyLoader() {
    this._loaded = {};
    this._isLoading = {};
  }

  LazyLoader.prototype = {

    _js: function(file, callback) {
      var script = document.createElement('script');
      script.src = file;
      // until bug 916255 lands async is the default so
      // we must disable it so scripts load in the order they where
      // required.
      script.async = false;
      script.addEventListener('load', callback);
      document.head.appendChild(script);
      this._isLoading[file] = script;
    },

    _css: function(file, callback) {
      var style = document.createElement('link');
      style.type = 'text/css';
      style.rel = 'stylesheet';
      style.href = file;
      document.head.appendChild(style);
      callback();
    },

    _html: function(domNode, callback) {

      // The next few lines are for loading html imports in DEBUG mode
      if (domNode.getAttribute('is')) {
        this.load(['/shared/js/html_imports.js'], function() {
          HtmlImports.populate(callback);
        }.bind(this));
        return;
      }

      for (var i = 0; i < domNode.childNodes.length; i++) {
        if (domNode.childNodes[i].nodeType == document.COMMENT_NODE) {
          domNode.innerHTML = domNode.childNodes[i].nodeValue;
          break;
        }
      }

      window.dispatchEvent(new CustomEvent('lazyload', {
        detail: domNode
      }));

      callback();
    },

    /**
     * Retrieves content of JSON file.
     *
     * @param {String} file Path to JSON file
     * @return {Promise} A promise that resolves to the JSON content
     * or null in case of invalid path. Rejects if an error occurs.
     */
    getJSON: function(file) {
      return new Promise(function(resolve, reject) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', file, true);
        xhr.responseType = 'json';

        xhr.onerror = function(error) {
          reject(error);
        };
        xhr.onload = function() {
          if (xhr.response !== null) {
            resolve(xhr.response);
          } else {
            reject(new Error('No valid JSON object was found (' + 
			     xhr.status + ' ' + xhr.statusText + ')'));
          }
        };

        xhr.send();
      });
    },

    load: function(files, callback) {
      var deferred = {};
      deferred.promise = new Promise(resolve => {
        deferred.resolve = resolve;
      });

      if (!Array.isArray(files)) {
        files = [files];
      }

      var loadsRemaining = files.length, self = this;
      function perFileCallback(file) {
        if (self._isLoading[file]) {
          delete self._isLoading[file];
        }
        self._loaded[file] = true;

        if (--loadsRemaining === 0) {
          deferred.resolve();
          if (callback) {
            callback();
          }
        }
      }

      for (var i = 0; i < files.length; i++) {
        var file = files[i];

        if (this._loaded[file.id || file]) {
          perFileCallback(file);
        } else if (this._isLoading[file]) {
          this._isLoading[file].addEventListener(
            'load', perFileCallback.bind(null, file));
        } else {
          var method, idx;
          if (typeof file === 'string') {
            method = file.match(/\.([^.]+)$/)[1];
            idx = file;
          } else {
            method = 'html';
            idx = file.id;
          }

          this['_' + method](file, perFileCallback.bind(null, idx));
        }
      }

      return deferred.promise;
    }
  };

  return new LazyLoader();
}());

define("shared/lazy_loader", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.LazyLoader;
    };
}(this)));

define('dom',['require','exports','module','shared/lazy_loader'],function(require, exports) {


var LazyLoader = require('shared/lazy_loader');

exports.load = function(id, require, onLoad, config) {
  if (config.isBuild) {
    return onLoad();
  }

  var node = document.getElementById(id);
  if (!node) {
    onLoad.error('can\'t find element with id #' + id);
    return;
  }

  LazyLoader.load(node, function() {
    onLoad(node);
  });
};

});

define('views/modify_event',['require','exports','module','templates/alarm','./event_base','shared/input_parser','provider/local','querystring','date_format','calc','next_tick','css!modify_event_view','dom!modify-event-view'],function(require, exports, module) {


var AlarmTemplate = require('templates/alarm');
var EventBase = require('./event_base');
var InputParser = require('shared/input_parser');
var Local = require('provider/local');
var QueryString = require('querystring');
var dateFormat = require('date_format');
var getTimeL10nLabel = require('calc').getTimeL10nLabel;
var nextTick = require('next_tick');

require('css!modify_event_view');
require('dom!modify-event-view');

function ModifyEvent(options) {
  this.deleteRecord = this.deleteRecord.bind(this);
  this._toggleAllDay = this._toggleAllDay.bind(this);
  EventBase.apply(this, arguments);
}
module.exports = ModifyEvent;

ModifyEvent.prototype = {
  __proto__: EventBase.prototype,

  ERROR_PREFIX: 'event-error-',

  MAX_ALARMS: 5,

  formats: {
    date: 'dateTimeFormat_%x',
    time: 'shortTimeFormat'
  },

  selectors: {
    element: '#modify-event-view',
    alarmList: '#modify-event-view .alarms',
    form: '#modify-event-view form',
    endDateLocale: '#end-date-locale',
    endTimeLocale: '#end-time-locale',
    status: '#modify-event-view section[role="status"]',
    errors: '#modify-event-view .errors',
    primaryButton: '#modify-event-view .save',
    deleteButton: '#modify-event-view .delete-record',
    header: '#modify-event-header'
  },

  uiSelector: '[name="%"]',

  _duration: 0, // The duration between start and end dates.

  _initEvents: function() {
    EventBase.prototype._initEvents.apply(this, arguments);

    var calendars = this.app.store('Calendar');

    calendars.on('add', this._addCalendarId.bind(this));
    calendars.on('preRemove', this._removeCalendarId.bind(this));
    calendars.on('remove', this._removeCalendarId.bind(this));
    calendars.on('update', this._updateCalendarId.bind(this));

    this.deleteButton.addEventListener('click', this.deleteRecord);
    this.form.addEventListener('click', this.focusHandler);
    this.form.addEventListener('submit', this.primary);

    var allday = this.getEl('allday');
    allday.addEventListener('change', this._toggleAllDay);

    this.alarmList.addEventListener('change', this._changeAlarm.bind(this));
  },

  /**
   * Fired when the allday checkbox changes.
   */
  _toggleAllDay: function(e) {
    var allday = this.getEl('allday').checked;

    if (allday) {
      // enable case
      this.element.classList.add(this.ALLDAY);
    } else {
      // disable case
      this.element.classList.remove(this.ALLDAY);
    }

    // because of race conditions it is theoretically possible
    // for the user to check/uncheck this value
    // when we don't actually have a model loaded.
    if (this.event) {
      this.event.isAllDay = !!allday;
    }

    // Reset alarms if we come from a user event
    if (e) {
      this.event.alarms = [];
      this.updateAlarms(allday);
    }
  },

  /**
   * Called when any alarm is changed
   */
  _changeAlarm: function(e) {
    var template = AlarmTemplate;
    if (e.target.value == 'none') {
      var parent = e.target.parentNode;
      parent.parentNode.removeChild(parent);
      return;
    }

    // Append a new alarm select only if we don't have an empty one or if we
    // didn't reach the maximum number of alarms
    var alarms = this.queryAlarms();
    if (alarms.length >= this.MAX_ALARMS ||
        alarms.some(el => el.value === 'none')) {
      return;
    }

    var newAlarm = document.createElement('div');
    newAlarm.innerHTML = template.picker.render({
      layout: this.event.isAllDay ? 'allday' : 'standard'
    });
    this.alarmList.appendChild(newAlarm);
  },

  /**
   * Check if current event has been stored in the database
   */
  isSaved: function() {
      return !!this.provider;
  },

  /**
   * Build the initial list of calendar ids.
   */
  onfirstseen: function() {
    // we need to notify users (specially automation tests) somehow that the
    // options are still being loaded from DB, this is very important to
    // avoid race conditions (eg.  trying to set calendar before list is
    // built) notice that we also add the class to the markup because on some
    // really rare occasions "onfirstseen" is called after the EventBase
    // removed the "loading" class from the root element (seen it happen less
    // than 1% of the time)
    this.getEl('calendarId').classList.add(self.LOADING);

    var calendarStore = this.app.store('Calendar');
    calendarStore.all(function(err, calendars) {
      if (err) {
        return console.error('Could not build list of calendars');
      }

      var pending = 0;
      var self = this;

      function next() {
        if (!--pending) {
          self.getEl('calendarId').classList.remove(self.LOADING);

          if (self.onafteronfirstseen) {
            self.onafteronfirstseen();
          }
        }
      }

      for (var id in calendars) {
        pending++;
        this._addCalendarId(id, calendars[id], next);
      }

    }.bind(this));
  },

  /**
   * Updates a calendar id option.
   *
   * @param {String} id calendar id.
   * @param {Calendar.Model.Calendar} calendar model.
   */
  _updateCalendarId: function(id, calendar) {
    var element = this.getEl('calendarId');
    var option = element.querySelector('[value="' + id + '"]');
    var store = this.app.store('Calendar');

    store.providerFor(calendar, function(err, provider) {
      var caps = provider.calendarCapabilities(
        calendar
      );

      if (!caps.canCreateEvent) {
        this._removeCalendarId(id);
        return;
      }

      if (option) {
        option.text = calendar.remote.name;
      }


      if (this.oncalendarupdate) {
        this.oncalendarupdate(calendar);
      }
    }.bind(this));
  },

  /**
   * Add a single calendar id.
   *
   * @param {String} id calendar id.
   * @param {Calendar.Model.Calendar} calendar calendar to add.
   */
  _addCalendarId: function(id, calendar, callback) {
    var store = this.app.store('Calendar');
    store.providerFor(calendar, function(err, provider) {
      var caps = provider.calendarCapabilities(
        calendar
      );

      if (!caps.canCreateEvent) {
        if (callback) {
          nextTick(callback);
        }
        return;
      }

      var option;
      var element = this.getEl('calendarId');

      option = document.createElement('option');

      if (id === Local.calendarId) {
        option.text = navigator.mozL10n.get('calendar-local');
        option.setAttribute('data-l10n-id', 'calendar-local');
      } else {
        option.text = calendar.remote.name;
      }

      option.value = id;
      element.add(option);

      if (callback) {
        nextTick(callback);
      }

      if (this.onaddcalendar) {
        this.onaddcalendar(calendar);
      }
    }.bind(this));
  },

  /**
   * Remove a single calendar id.
   *
   * @param {String} id to remove.
   */
  _removeCalendarId: function(id) {
    var element = this.getEl('calendarId');

    var option = element.querySelector('[value="' + id + '"]');
    if (option) {
      element.removeChild(option);
    }

    if (this.onremovecalendar) {
      this.onremovecalendar(id);
    }
  },

  /**
   * Mark all field's readOnly flag.
   *
   * @param {Boolean} boolean true/false.
   */
  _markReadonly: function(boolean) {
    var i = 0;
    var fields = this.form.querySelectorAll('[name]');
    var len = fields.length;

    for (; i < len; i++) {
      fields[i].readOnly = boolean;
    }
  },

  queryAlarms: function() {
    return Array.from(document.querySelectorAll('[name="alarm[]"]'));
  },

  get alarmList() {
    return this._findElement('alarmList');
  },

  get form() {
    return this._findElement('form');
  },

  get deleteButton() {
    return this._findElement('deleteButton');
  },

  get fieldRoot() {
    return this.form;
  },

  /**
   * Ask the provider to persist an event:
   *
   *  1. update the model with form data
   *
   *  2. send it to the provider if it has the capability
   *
   *  3. set the position of the calendar to startDate of new/edited event.
   *
   *  4. redirect to last view.
   *
   * For now both update & create share the same
   * behaviour (redirect) in the future we may change this.
   */
  _persistEvent: function(method, capability) {
    // create model data
    var data = this.formData();
    var errors;

    // we check explicitly for true, because the alternative
    // is an error object.
    if ((errors = this.event.updateAttributes(data)) !== true) {
      this.showErrors(errors);
      return;
    }

    // can't create without a calendar id
    // because of defaults this should be impossible.
    if (!data.calendarId) {
      return;
    }

    var self = this;
    var provider;

    this.store.providerFor(this.event, fetchProvider);

    function fetchProvider(err, result) {
      provider = result;
      provider.eventCapabilities(
        self.event.data,
        verifyCaps
      );
    }

    function verifyCaps(err, caps) {
      if (err) {
        return console.error('Error fetching capabilities for', self.event);
      }

      // safe-guard but should not ever happen.
      if (caps[capability]) {
        persistEvent();
      }
    }

    function persistEvent() {
      var list = self.element.classList;

      // mark view as 'in progress' so we can style
      // it via css during that time period
      list.add(self.PROGRESS);

      var moveDate = self.event.startDate;

      provider[method](self.event.data, function(err) {
        list.remove(self.PROGRESS);

        if (err) {
          self.showErrors(err);
          return;
        }

        // move the position in the calendar to the added/edited day
        self.app.timeController.move(moveDate);
        // order is important the above method triggers the building
        // of the dom elements so selectedDay must come after.
        self.app.timeController.selectedDay = moveDate;

        if (method === 'updateEvent') {
          // If we edit a view our history stack looks like:
          //   /week -> /event/view -> /event/save -> /event/view
          // We need to return all the way to the top of the stack
          // We can remove this once we have a history stack
          self.app.view('ViewEvent', function(view) {
            self.app.go(view.returnTop());
          });

          return;
        }

        self.app.go(self.returnTo());
      });
    }
  },

  /**
   * Deletes current record if provider is present and has the capability.
   */
  deleteRecord: function(event) {
    if (event) {
      event.preventDefault();
    }

    if (this.isSaved()) {
      var self = this;
      var handleDelete = function me_handleDelete() {
        self.provider.deleteEvent(self.event.data, function(err) {
          if (err) {
            self.showErrors(err);
            return;
          }

          // If we edit a view our history stack looks like:
          //   /week -> /event/view -> /event/save -> /event/view
          // We need to return all the way to the top of the stack
          // We can remove this once we have a history stack
          self.app.view('ViewEvent', function(view) {
            self.app.go(view.returnTop());
          });
        });
      };

      this.provider.eventCapabilities(this.event.data, function(err, caps) {
        if (err) {
          return console.error('Error fetching event capabilities', this.event);
        }

        if (caps.canDelete) {
          handleDelete();
        }
      });
    }
  },

  /**
   * Persist current model.
   */
  primary: function(event) {
    if (event) {
      event.preventDefault();
    }

    // Disable the button on primary event to avoid race conditions
    this.disablePrimary();

    if (this.isSaved()) {
      this._persistEvent('updateEvent', 'canUpdate');
    } else {
      this._persistEvent('createEvent', 'canCreate');
    }
  },

  /**
   * Enlarges focus areas for .button controls
   */
  focusHandler: function(e) {
    var input = e.target.querySelector('input, select');
    if (input && e.target.classList.contains('button')) {
      input.focus();
    }
  },

  /**
   * Export form information into a format
   * the model can understand.
   *
   * @return {Object} formatted data suitable
   *                  for use with Calendar.Model.Event.
   */
  formData: function() {
    var fields = {
      title: this.getEl('title').value,
      location: this.getEl('location').value,
      description: this.getEl('description').value,
      calendarId: this.getEl('calendarId').value
    };

    var startTime;
    var endTime;
    var allday = this.getEl('allday').checked;

    if (allday) {
      startTime = null;
      endTime = null;
    } else {
      startTime = this.getEl('startTime').value;
      endTime = this.getEl('endTime').value;
    }

    fields.startDate = InputParser.formatInputDate(
      this.getEl('startDate').value,
      startTime
    );

    fields.endDate = InputParser.formatInputDate(
      this.getEl('endDate').value,
      endTime
    );

    if (allday) {
      // when the event is all day we display the same
      // day that the entire event spans but we must actually
      // end the event at the first second, minute hour of the next
      // day. This will ensure the server handles it as an all day event.
      fields.endDate.setDate(
        fields.endDate.getDate() + 1
      );
    }

    fields.alarms = [];
    var triggers = ['none'];
    this.queryAlarms().forEach(alarm => {
      if (triggers.indexOf(alarm.value) !== -1) {
        return;
      }

      triggers.push(alarm.value);

      fields.alarms.push({
        action: 'DISPLAY',
        trigger: parseInt(alarm.value, 10)
      });
    });

    return fields;
  },

  enablePrimary: function() {
    this.primaryButton.removeAttribute('aria-disabled');
  },

  disablePrimary: function() {
    this.primaryButton.setAttribute('aria-disabled', 'true');
  },

  /**
   * Re-enable the primary button when we show errors
   */
  showErrors: function() {
    this.enablePrimary();
    EventBase.prototype.showErrors.apply(this, arguments);
  },

  /**
   * Read the urlparams and override stuff on our event model.
   * @param {string} search Optional string of the form ?foo=bar&cat=dog.
   * @private
   */
  _overrideEvent: function(search) {
    search = search || window.location.search;
    if (!search || search.length === 0) {
      return;
    }

    // Remove the question mark that begins the search.
    if (search.substr(0, 1) === '?') {
      search = search.substr(1, search.length - 1);
    }

    var field, value;
    // Parse the urlparams.
    var params = QueryString.parse(search);
    for (field in params) {
      value = params[field];
      switch (field) {
        case ModifyEvent.OverrideableField.START_DATE:
        case ModifyEvent.OverrideableField.END_DATE:
          params[field] = new Date(value);
          break;
        default:
          params[field] = value;
          break;
      }
    }

    // Override fields on our event.
    var model = this.event;
    for (field in ModifyEvent.OverrideableField) {
      value = ModifyEvent.OverrideableField[field];
      model[value] = params[value] || model[value];
    }
  },

  /**
   * Updates form to use values from the current model.
   *
   * Does not handle readonly flags or calenarId associations.
   * Suitable for use in pre-populating values for both new and
   * existing events.
   *
   * Resets any value on the current form.
   */
  _updateUI: function() {
    this._overrideEvent();
    this.form.reset();

    var model = this.event;
    this.getEl('title').value = model.title;
    this.getEl('location').value = model.location;
    var dateSrc = model;
    if (model.remote.isRecurring && this.busytime) {
      dateSrc = this.busytime;
    }

    var startDate = dateSrc.startDate;
    var endDate = dateSrc.endDate;
    this._duration = endDate.getTime() - startDate.getTime();

    // update the allday status of the view
    var allday = this.getEl('allday');
    if (allday && (allday.checked = model.isAllDay)) {
      this._toggleAllDay();
      endDate = this.formatEndDate(endDate);
    }

    this.getEl('startDate').value = InputParser.exportDate(startDate);
    this._setupDateTimeSync(
      'date', 'startDate', 'start-date-locale', startDate);

    this.getEl('endDate').value = InputParser.exportDate(endDate);
    this._setupDateTimeSync(
      'date', 'endDate', 'end-date-locale', endDate);

    this.getEl('startTime').value = InputParser.exportTime(startDate);
    this._setupDateTimeSync(
      'time', 'startTime', 'start-time-locale', startDate);

    this.getEl('endTime').value = InputParser.exportTime(endDate);
    this._setupDateTimeSync(
      'time', 'endTime', 'end-time-locale', endDate);

    this.getEl('description').textContent = model.description;

    // update calendar id
    this.getEl('calendarId').value = model.calendarId;

    // calendar display
    var currentCalendar = this.getEl('currentCalendar');

    if (this.originalCalendar) {
      currentCalendar.value =
        this.originalCalendar.remote.name;

      currentCalendar.readOnly = true;
    }

    this.updateAlarms(model.isAllDay);
  },

  /**
   * Handling a layer over <input> to have localized
   * date/time
   */
  _setupDateTimeSync: function(type, src, target, value) {
    var targetElement = document.getElementById(target);
    if (!targetElement) {
      return;
    }
    this._renderDateTimeLocale(type, targetElement, value);

    var callback = type === 'date' ?
      this._updateDateLocaleOnInput : this._updateTimeLocaleOnInput;

    this.getEl(src)
      .addEventListener('input', function(e) {
        callback.call(this, targetElement, e);

        // We only auto change the end date and end time
        // when user changes start date or start time,
        // or end datetime is NOT after start datetime
        // after changing end date or end time.
        // Otherwise, we don't auto change end date and end time.
        if (targetElement.id === 'start-date-locale' ||
            targetElement.id === 'start-time-locale') {
          this._setEndDateTimeWithCurrentDuration();
        } else if (this._getEndDateTime() <= this._getStartDateTime()) {
          this._setEndDateTimeWithCurrentDuration();
          this.showErrors({
            name: type === 'date' ?
              'start-date-after-end-date' :
              'start-time-after-end-time'
          });
        }

        this._duration = this._getEndDateTime() - this._getStartDateTime();
      }.bind(this));
  },

  _setEndDateTimeWithCurrentDuration: function() {
    var date = new Date(this._getStartDateTime() + this._duration);
    var endDateLocale = this._findElement('endDateLocale');
    var endTimeLocale = this._findElement('endTimeLocale');
    this.getEl('endDate').value = date.toLocaleFormat('%Y-%m-%d');
    this.getEl('endTime').value = date.toLocaleFormat('%H:%M:%S');
    this._renderDateTimeLocale('date', endDateLocale, date);
    this._renderDateTimeLocale('time', endTimeLocale, date);
  },

  _getStartDateTime: function() {
    return new Date(this.getEl('startDate').value + 'T' +
      this.getEl('startTime').value).getTime();
  },

  _getEndDateTime: function() {
    return new Date(this.getEl('endDate').value + 'T' +
      this.getEl('endTime').value).getTime();
  },

  _renderDateTimeLocale: function(type, targetElement, value) {
    // we inject the targetElement to make it easier to test
    var localeFormat = dateFormat.localeFormat;
    var formatKey = this.formats[type];
    if (type === 'time') {
      formatKey = getTimeL10nLabel(formatKey);
    }
    var format = navigator.mozL10n.get(formatKey);
    targetElement.textContent = localeFormat(value, format);
    // we need to store the format and date for l10n
    targetElement.setAttribute('data-l10n-date-format', formatKey);
    targetElement.dataset.date = value;
  },

  _updateDateLocaleOnInput: function(targetElement, e) {
    var selected = InputParser.importDate(e.target.value);
    // use date constructor to avoid issues, see Bug 966516
    var date = new Date(selected.year, selected.month, selected.date);
    this._renderDateTimeLocale('date', targetElement, date);
  },

  _updateTimeLocaleOnInput: function(targetElement, e) {
    var selected = InputParser.importTime(e.target.value);
    var date = new Date();
    date.setHours(selected.hours);
    date.setMinutes(selected.minutes);
    date.setSeconds(0);
    this._renderDateTimeLocale('time', targetElement, date);
  },

  /**
   * Called on render or when toggling an all-day event
   */
  updateAlarms: function(isAllDay, callback) {
    var template = AlarmTemplate;
    var alarms = [];

    // Used to make sure we don't duplicate alarms
    var alarmMap = {};

    if (this.event.alarms) {
      //jshint boss:true
      for (var i = 0, alarm; alarm = this.event.alarms[i]; i++) {
        alarmMap[alarm.trigger] = true;
        alarm.layout = isAllDay ? 'allday' : 'standard';
        alarms.push(alarm);
      }
    }

    var settings = this.app.store('Setting');
    var layout = isAllDay ? 'allday' : 'standard';
    settings.getValue(layout + 'AlarmDefault', next.bind(this));

    function next(err, value) {
      //jshint -W040
      if (!this.isSaved() && !alarmMap[value] && !this.event.alarms.length) {
        alarms.push({
          layout: layout,
          trigger: value
        });
      }

      // Bug_898242 to show an event when default is 'none',
      // we check if the event is not saved, if so, we push
      // the default alarm on to the list.
      if ((value === 'none' && this.isSaved()) || value !== 'none') {
        alarms.push({
          layout: layout
        });
      }

      this.alarmList.innerHTML = template.picker.renderEach(alarms).join('');

      if (callback) {
        callback();
      }
    }
  },

  reset: function() {
    var list = this.element.classList;

    list.remove(this.UPDATE);
    list.remove(this.CREATE);
    list.remove(this.READONLY);
    list.remove(this.ALLDAY);

    var allday = this.getEl('allday');

    if (allday) {
      allday.checked = false;
    }

    this._returnTo = null;
    this._markReadonly(false);
    this.provider = null;
    this.event = null;
    this.busytime = null;

    this.alarmList.innerHTML = '';

    this.form.reset();
  },

  oninactive: function() {
    EventBase.prototype.oninactive.apply(this, arguments);
    this.reset();
  }
};

/**
 * The fields on our event model which urlparams may override.
 * @enum {string}
 */
ModifyEvent.OverrideableField = {
  CALENDAR_ID: 'calendarId',
  DESCRIPTION: 'description',
  END_DATE: 'endDate',
  IS_ALL_DAY: 'isAllDay',
  LOCATION: 'location',
  START_DATE: 'startDate',
  TITLE: 'title'
};

});
