
define('utils/account_creation',['require','exports','module','responder','app','promise'],function(require, exports, module) {


var Responder = require('responder');
var app = require('app');
var denodeifyAll = require('promise').denodeifyAll;

/**
 * Helper class to create accounts.
 * Emits events during the process of
 * creation to allow views to hook into
 * the full cycle while further separating
 * this logic from their own.
 *
 *
 * Events:
 *
 *    - authorize
 *    - calendar sync
 *
 *
 * @param {Calendar.App} app instance of app.
 */
function AccountCreation() {
  this.app = app;
  Responder.call(this);

  denodeifyAll(this, [ 'send' ]);
}
module.exports = AccountCreation;

AccountCreation.prototype = {
  __proto__: Responder.prototype,

  /**
   * Sends a request to create an account.
   *
   * @param {Calendar.Models.Account} model account details.
   * @param {Function} callback fired when entire transaction is complete.
   */
  send: function(model, callback) {
    var self = this;
    var accountStore = this.app.store('Account');
    var calendarStore = this.app.store('Calendar');

    // begin by persisting the account
    accountStore.verifyAndPersist(model, function(accErr, id, result) {

      if (accErr) {
        // we bail when we cannot create the account
        // but also give custom error events.
        self.emit('authorizeError', accErr);
        callback(accErr);
        return;
      }


      self.emit('authorize', result);

      // finally sync the account so when
      // we exit the request the user actually
      // has some calendars. This should not take
      // too long (compared to event sync).
      accountStore.sync(result, function(syncErr) {
        if (syncErr) {
          self.emit('calendarSyncError', syncErr);
          callback(syncErr);
          return;
        }

        function syncCalendars(err, calendars) {
          if (err) {
            console.error('Error fetch calendar list in account creation');
            return callback(err);
          }

          self.emit('calendarSync');

          // note we don't wait for any of this to complete
          // we just begin the sync and let the event handlers
          // on the sync controller do the work.
          for (var key in calendars) {
            self.app.syncController.calendar(
              result,
              calendars[key]
            );
          }

          callback(null, result);
        }

        // begin sync of calendars
        calendarStore.remotesByAccount(
          result._id,
          syncCalendars
        );
      });
    });
  }
};

});

/* global Buffer */
define('querystring',['require','exports','module'],function(require, exports) {


// Copyright Joyent, Inc. and other Node contributors.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to permit
// persons to whom the Software is furnished to do so, subject to the
// following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
// USE OR OTHER DEALINGS IN THE SOFTWARE.

// If obj.hasOwnProperty has been overridden, then calling
// obj.hasOwnProperty(prop) will break.
// See: https://github.com/joyent/node/issues/1707
function hasOwnProperty(obj, prop) {
  return Object.prototype.hasOwnProperty.call(obj, prop);
}


function charCode(c) {
  return c.charCodeAt(0);
}


// a safe fast alternative to decodeURIComponent
exports.unescapeBuffer = function(s, decodeSpaces) {
  var out = new Buffer(s.length);
  var state = 'CHAR'; // states: CHAR, HEX0, HEX1
  var n, m, hexchar;

  for (var inIndex = 0, outIndex = 0; inIndex <= s.length; inIndex++) {
    var c = s.charCodeAt(inIndex);
    switch (state) {
      case 'CHAR':
        switch (c) {
          case charCode('%'):
            n = 0;
            m = 0;
            state = 'HEX0';
            break;
          case charCode('+'):
            if (decodeSpaces) {
              c = charCode(' ');
            }
            out[outIndex++] = c;
            break;
          default:
            out[outIndex++] = c;
        }
        break;

      case 'HEX0':
        state = 'HEX1';
        hexchar = c;
        if (charCode('0') <= c && c <= charCode('9')) {
          n = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          n = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          n = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = c;
          state = 'CHAR';
          break;
        }
        break;

      case 'HEX1':
        state = 'CHAR';
        if (charCode('0') <= c && c <= charCode('9')) {
          m = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          m = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          m = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = hexchar;
          out[outIndex++] = c;
          break;
        }
        out[outIndex++] = 16 * n + m;
        break;
    }
  }

  // TODO support returning arbitrary buffers.

  return out.slice(0, outIndex - 1);
};


exports.unescape = function(s, decodeSpaces) {
  return exports.unescapeBuffer(s, decodeSpaces).toString();
};


exports.escape = function(str) {
  return encodeURIComponent(str);
};

var stringifyPrimitive = function(v) {
  switch (typeof v) {
    case 'string':
      return v;

    case 'boolean':
      return v ? 'true' : 'false';

    case 'number':
      return isFinite(v) ? v : '';

    default:
      return '';
  }
};


exports.stringify = exports.encode = function(obj, sep, eq, name) {
  sep = sep || '&';
  eq = eq || '=';
  if (obj === null) {
    obj = undefined;
  }

  if (typeof obj === 'object') {
    return Object.keys(obj).map(function(k) {
      var ks = exports.escape(stringifyPrimitive(k)) + eq;
      if (Array.isArray(obj[k])) {
        return obj[k].map(function(v) {
          return ks + exports.escape(stringifyPrimitive(v));
        }).join(sep);
      } else {
        return ks + exports.escape(stringifyPrimitive(obj[k]));
      }
    }).join(sep);

  }

  if (!name) {
    return '';
  }
  return exports.escape(stringifyPrimitive(name)) + eq +
         exports.escape(stringifyPrimitive(obj));
};

// Parse a key=val string.
exports.parse = exports.decode = function(qs, sep, eq, options) {
  sep = sep || '&';
  eq = eq || '=';
  var obj = {};

  if (typeof qs !== 'string' || qs.length === 0) {
    return obj;
  }

  var regexp = /\+/g;
  qs = qs.split(sep);

  var maxKeys = 1000;
  if (options && typeof options.maxKeys === 'number') {
    maxKeys = options.maxKeys;
  }

  var len = qs.length;
  // maxKeys <= 0 means that we should not limit keys count
  if (maxKeys > 0 && len > maxKeys) {
    len = maxKeys;
  }

  for (var i = 0; i < len; ++i) {
    var x = qs[i].replace(regexp, '%20'),
        idx = x.indexOf(eq),
        kstr, vstr, k, v;

    if (idx >= 0) {
      kstr = x.substr(0, idx);
      vstr = x.substr(idx + 1);
    } else {
      kstr = x;
      vstr = '';
    }

    try {
      k = decodeURIComponent(kstr);
      v = decodeURIComponent(vstr);
    } catch (e) {
      k = exports.unescape(kstr, true);
      v = exports.unescape(vstr, true);
    }

    if (!hasOwnProperty(obj, k)) {
      obj[k] = v;
    } else if (Array.isArray(obj[k])) {
      obj[k].push(v);
    } else {
      obj[k] = [obj[k], v];
    }
  }

  return obj;
};

});

define('view',['require','exports','module'],function(require, exports, module) {


var DEFAULT_ERROR_ID = 'error-default';
const INVALID_CSS = /([^a-zA-Z\-\_0-9])/g;

/**
 * Very simple base class for views.
 * Provides functionality for active/inactive.
 *
 * The first time the view is activated
 * the onactive function/event will fire.
 *
 * The .seen property is added to each object
 * with view in its prototype. .seen can be used
 * to detect if the view has ever been activated.
 *
 * @param {String|Object} options options or a selector for element.
 */
function View(options) {
  if (typeof(options) === 'undefined') {
    options = {};
  }

  if (typeof(options) === 'string') {
    this.selectors = { element: options };
  } else {
    var key;

    if (typeof(options) === 'undefined') {
      options = {};
    }

    for (key in options) {
      if (options.hasOwnProperty(key)) {
        this[key] = options[key];
      }
    }
  }

  this.hideErrors = this.hideErrors.bind(this);
}
module.exports = View;

View.ACTIVE = 'active';

View.prototype = {
  seen: false,
  activeClass: View.ACTIVE,
  errorVisible: false,

  get element() {
    return this._findElement('element');
  },

  get status() {
    return this._findElement('status');
  },

  get errors() {
    return this._findElement('errors');
  },

  /**
   * Creates a string id for a given model.
   *
   *    view.idForModel('foo-', { _id: 1 }); // => foo-1
   *    view.idForModel('foo-', '2'); // => foo-2
   *
   * @param {String} prefix of string.
   * @param {Object|String|Numeric} objectOrString representation of model.
   */
  idForModel: function(prefix, objectOrString) {
    prefix += (typeof(objectOrString) === 'object') ?
      objectOrString._id :
      objectOrString;

    return prefix;
  },

  calendarId: function(input) {
    if (typeof(input) !== 'string') {
      input = input.calendarId;
    }

    input = this.cssClean(input);
    return 'calendar-id-' + input;
  },

  /**
   * Delegate pattern event listener.
   *
   * @param {HTMLElement} element parent element.
   * @param {String} type type of dom event.
   * @param {String} selector css selector element should match
   *                          _note_ there is no magic here this
   *                          is determined from the root of the document.
   * @param {Function|Object} handler event handler.
   *                                  first argument is the raw
   *                                  event second is the element
   *                                  matching the pattern.
   */
  delegate: function(element, type, selector, handler) {
    if (typeof(handler) === 'object') {
      var context = handler;
      handler = function() {
        context.handleEvent.apply(context, arguments);
      };
    }

    element.addEventListener(type, function(e) {
      var target = e.target;
      while (target !== element) {
        if ('mozMatchesSelector' in target &&
            target.mozMatchesSelector(selector)) {
          return handler(e, target);
        }
        target = target.parentNode;
      }
    });
  },

  /**
   * Clean a string for use with css.
   * Converts illegal chars to legal ones.
   */
  cssClean: function(string) {
    if (typeof(string) !== 'string') {
      return string;
    }

    //TODO: I am worried about the performance
    //of using this all over the place =/
    //consider sanitizing all keys to ensure
    //they don't blow up when used as a selector?
    return string.replace(INVALID_CSS, '-');
  },

  /**
   * Finds a caches a element defined
   * by selectors
   *
   * @param {String} selector name as defined in selectors.
   * @param {Boolean} all true when to find all elements. (default false).
   */
  _findElement: function(name, all, element) {
    if (typeof(all) === 'object') {
      element = all;
      all = false;
    }

    element = element || document;

    var cacheName;
    var selector;

    if (typeof(all) === 'undefined') {
      all = false;
    }

    if (name in this.selectors) {
      cacheName = '_' + name + 'Element';
      selector = this.selectors[name];

      if (!this[cacheName]) {
        if (all) {
          this[cacheName] = element.querySelectorAll(selector);
        } else {
          this[cacheName] = element.querySelector(selector);
        }
      }

      return this[cacheName];
    }

    return null;
  },

 /**
   * Displays a list of errors
   *
   * @param {Array} list error list
   *  (see Event.validaitonErrors) or Error object.
   */
  showErrors: function(list) {
    var _ = navigator.mozL10n.get;
    var errors = '';

    // We can pass Error objects or
    // Array of {name: foo} objects
    if (!Array.isArray(list)) {
        list = [list];
    }

    var i = 0;
    var len = list.length;

    for (; i < len; i++) {
      var name = list[i].l10nID || list[i].name;
      errors += _('error-' + name) || _(DEFAULT_ERROR_ID);
    }

    // populate error and display it.
    this.errors.textContent = errors;
    this.errorVisible = true;
    this.status.classList.add(this.activeClass);

    this.status.addEventListener('animationend', this.hideErrors);
  },

  hideErrors: function() {
    this.status.classList.remove(this.activeClass);
    this.status.removeEventListener('animationend', this.hideErrors);
    this.errorVisible = false;
  },

  onactive: function() {
    if (this.errorVisible) {
      this.hideErrors();
    }

    // seen can be set to anything other than false to override this behaviour
    if (this.seen === false) {
      this.onfirstseen();
    }

    // intentionally using 'in'
    if ('dispatch' in this) {
      this.dispatch.apply(this, arguments);
    }

    this.seen = true;
    if (this.element) {
      this.element.classList.add(this.activeClass);
    }
  },

  oninactive: function() {
    if (this.element) {
      this.element.classList.remove(this.activeClass);
    }
  },

  onfirstseen: function() {}
};

});

/* exported LazyLoader */
/* globals HtmlImports, Promise */


/**
 * This contains a simple LazyLoader implementation
 * To use:
 *
 *   LazyLoader.load(
 *    ['/path/to/file.js', '/path/to/file.css', 'domNode'], callback
 *   );
 */
var LazyLoader = (function() {

  function LazyLoader() {
    this._loaded = {};
    this._isLoading = {};
  }

  LazyLoader.prototype = {

    _js: function(file, callback) {
      var script = document.createElement('script');
      script.src = file;
      // until bug 916255 lands async is the default so
      // we must disable it so scripts load in the order they where
      // required.
      script.async = false;
      script.addEventListener('load', callback);
      document.head.appendChild(script);
      this._isLoading[file] = script;
    },

    _css: function(file, callback) {
      var style = document.createElement('link');
      style.type = 'text/css';
      style.rel = 'stylesheet';
      style.href = file;
      document.head.appendChild(style);
      callback();
    },

    _html: function(domNode, callback) {

      // The next few lines are for loading html imports in DEBUG mode
      if (domNode.getAttribute('is')) {
        this.load(['/shared/js/html_imports.js'], function() {
          HtmlImports.populate(callback);
        }.bind(this));
        return;
      }

      for (var i = 0; i < domNode.childNodes.length; i++) {
        if (domNode.childNodes[i].nodeType == document.COMMENT_NODE) {
          domNode.innerHTML = domNode.childNodes[i].nodeValue;
          break;
        }
      }

      window.dispatchEvent(new CustomEvent('lazyload', {
        detail: domNode
      }));

      callback();
    },

    /**
     * Retrieves content of JSON file.
     *
     * @param {String} file Path to JSON file
     * @return {Promise} A promise that resolves to the JSON content
     * or null in case of invalid path. Rejects if an error occurs.
     */
    getJSON: function(file) {
      return new Promise(function(resolve, reject) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', file, true);
        xhr.responseType = 'json';

        xhr.onerror = function(error) {
          reject(error);
        };
        xhr.onload = function() {
          if (xhr.response !== null) {
            resolve(xhr.response);
          } else {
            reject(new Error('No valid JSON object was found (' + 
			     xhr.status + ' ' + xhr.statusText + ')'));
          }
        };

        xhr.send();
      });
    },

    load: function(files, callback) {
      var deferred = {};
      deferred.promise = new Promise(resolve => {
        deferred.resolve = resolve;
      });

      if (!Array.isArray(files)) {
        files = [files];
      }

      var loadsRemaining = files.length, self = this;
      function perFileCallback(file) {
        if (self._isLoading[file]) {
          delete self._isLoading[file];
        }
        self._loaded[file] = true;

        if (--loadsRemaining === 0) {
          deferred.resolve();
          if (callback) {
            callback();
          }
        }
      }

      for (var i = 0; i < files.length; i++) {
        var file = files[i];

        if (this._loaded[file.id || file]) {
          perFileCallback(file);
        } else if (this._isLoading[file]) {
          this._isLoading[file].addEventListener(
            'load', perFileCallback.bind(null, file));
        } else {
          var method, idx;
          if (typeof file === 'string') {
            method = file.match(/\.([^.]+)$/)[1];
            idx = file;
          } else {
            method = 'html';
            idx = file.id;
          }

          this['_' + method](file, perFileCallback.bind(null, idx));
        }
      }

      return deferred.promise;
    }
  };

  return new LazyLoader();
}());

define("shared/lazy_loader", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.LazyLoader;
    };
}(this)));

define('dom',['require','exports','module','shared/lazy_loader'],function(require, exports) {


var LazyLoader = require('shared/lazy_loader');

exports.load = function(id, require, onLoad, config) {
  if (config.isBuild) {
    return onLoad();
  }

  var node = document.getElementById(id);
  if (!node) {
    onLoad.error('can\'t find element with id #' + id);
    return;
  }

  LazyLoader.load(node, function() {
    onLoad(node);
  });
};

});

define('oauth_window',['require','exports','module','querystring','view','dom!oauth2'],function(require, exports, module) {


var QueryString = require('querystring');
var View = require('view');

require('dom!oauth2');

/**
 * Creates a oAuth dialog given a set of parameters.
 *
 *    var oauth = new OAuthWindow(
 *      elementContainer,
 *      'https://accounts.google.com/o/oauth2/auth',
 *      {
 *        response_type: 'code',
 *        client_id: 'xxx',
 *        scope: 'https://www.googleapis.com/auth/calendar',
 *        redirect_uri: 'xxx',
 *        state: 'foobar',
 *        access_type: 'offline',
 *        approval_prompt: 'force'
 *      }
 *    );
 *
 *    oauth.oncomplete = function(evt) {
 *      if (evt.detail.code) {
 *        // success
 *      }
 *    };
 *
 *    oauth.onabort = function() {
 *      // oauth was aborted
 *    };
 *
 *
 */
function OAuthWindow(container, server, params) {
  if (!params.redirect_uri) {
    throw new Error(
      'must provide params.redirect_uri so oauth flow can complete'
    );
  }

  this.params = {};
  for (var key in params) {
    this.params[key] = params[key];
  }

  this._element = container;

  View.call(this);
  this.target = server + '?' + QueryString.stringify(params);

  this._handleUserTriggeredClose =
    this._handleUserTriggeredClose.bind(this);
}
module.exports = OAuthWindow;

OAuthWindow.prototype = {
  __proto__: View.prototype,

  get element() {
    return this._element;
  },

  get isOpen() {
    return !!this.browserFrame;
  },

  selectors: {
    browserHeader: 'gaia-header',
    browserTitle: 'gaia-header > h1',
    browserContainer: '.browser-container'
  },

  get browserContainer() {
    return this._findElement('browserContainer', this.element);
  },

  get browserTitle() {
    return this._findElement('browserTitle', this.element);
  },

  get browserHeader() {
    return this._findElement('browserHeader', this.element);
  },

  _handleFinalRedirect: function(url) {
    this.close();

    if (this.oncomplete) {
      var params;

      // find query string
      var queryStringIdx = url.indexOf('?');
      if (queryStringIdx !== -1) {
        params = QueryString.parse(url.slice(queryStringIdx + 1));
      }

      this.oncomplete(params || {});
    }
  },

  _handleLocationChange: function(url) {
    this.browserTitle.textContent = url;
  },

  _handleUserTriggeredClose: function() {
    // close the oauth flow
    this.close();

    // trigger an event so others can cleanup
    if (this.onabort) {
      this.onabort();
    }
  },

  handleEvent: function(event) {
    switch (event.type) {
      case 'mozbrowserlocationchange':
        var url = event.detail;
        if (url.indexOf(this.params.redirect_uri) === 0) {
          return this._handleFinalRedirect(url);
        }
        this._handleLocationChange(url);
        break;
    }
  },

  open: function() {
    if (this.browserFrame) {
      throw new Error('attempting to open frame while another is open');
    }

    // add the active class
    this.element.classList.add(View.ACTIVE);

    // handle cancel events
    this.browserHeader.addEventListener(
      'action', this._handleUserTriggeredClose
    );

    // setup browser iframe
    var iframe = this.browserFrame =
      document.createElement('iframe');

    iframe.setAttribute('mozbrowser', true);
    iframe.setAttribute('src', this.target);

    this.browserContainer.appendChild(iframe);

    iframe.addEventListener('mozbrowserlocationchange', this);
  },

  close: function() {
    if (!this.isOpen) {
      return;
    }

    this.browserFrame.removeEventListener(
      'mozbrowserlocationchange', this
    );

    this.browserHeader.removeEventListener(
      'action', this._handleUserTriggeredClose
    );

    this.element.classList.remove(View.ACTIVE);

    this.browserFrame.parentNode.removeChild(
      this.browserFrame
    );

    this.browserFrame = undefined;
  }
};

});

define('utils/uri',['require','exports','module'],function(require, exports) {


/**
 * Get the port of the url.
 * @param {string} url full url.
 * @return {?number} port number, null if none found.
 */
exports.getPort = function(url) {
  var parts = url.split(':');
  if (parts.length < 2) {
    return null;
  }

  // If we found a port and path, it's the last part
  var candidate = parts[parts.length - 1];
  parts = candidate.split('/');

  // If we found a port, it's the first part
  candidate = parts[0];
  if (!isInteger(candidate)) {
    return null;
  }

  return parseInt(candidate, 10);
};

/**
 * Get the scheme of the url. Note that this only detects http and https.
 * @param {string} url full url.
 * @return {?string} uri scheme (ie http), null if none found.
 */
exports.getScheme = function(url) {
  var parts = url.split(':');
  if (parts.length < 2) {
    return null;
  }


  // If we found a scheme, it's the first part
  var candidate = parts[0];
  if (candidate !== 'http' && candidate !== 'https') {
    return null;
  }

  return candidate;
};

/**
 * Decide whether or not this string represents an integer.
 * @param {string} str some string.
 * @param {boolean} whether or not str represents an integer.
 */
function isInteger(str) {
  return (/^\d+$/).test(str);
}

});

// Loader plugin for loading CSS. Does not guarantee loading via onload
// watching, just inserts link tag.
// PS: borrowed from email app and edited it to include the baseUrl
define('css',['require','exports','module'],function(require, exports) {


exports.baseUrl = '/style/';

exports.load = function(id, require, onload, config) {
  if (config.isBuild) {
    return onload();
  }

  var style = document.createElement('link');
  style.type = 'text/css';
  style.rel = 'stylesheet';
  style.href = require.toUrl(exports.baseUrl + id + '.css');
  style.addEventListener('load', onload, false);
  document.head.appendChild(style);
};

});

define('views/modify_account',['require','exports','module','models/account','utils/account_creation','oauth_window','presets','utils/uri','view','css!modify_account_view','dom!modify-account-view'],function(require, exports, module) {


var Account = require('models/account');
var AccountCreation = require('utils/account_creation');
var OAuthWindow = require('oauth_window');
var Presets = require('presets');
var URI = require('utils/uri');
var View = require('view');

require('css!modify_account_view');
require('dom!modify-account-view');

var DEFAULT_AUTH_TYPE = 'basic';
var OAUTH_AUTH_CREDENTIALS = [
  'client_id',
  'scope',
  'redirect_uri',
  'state'
];

function ModifyAccount(options) {
  View.apply(this, arguments);

  this.deleteRecord = this.deleteRecord.bind(this);
  this.cancel = this.cancel.bind(this);
  this.displayOAuth2 = this.displayOAuth2.bind(this);
  this.hideHeaderAndForm = this.hideHeaderAndForm.bind(this);
  this.cancelDelete = this.cancelDelete.bind(this);

  this.accountHandler = new AccountCreation(this.app);
  this.accountHandler.on('authorizeError', this);

  // bound so we can add remove listeners
  this._boundSaveUpdateModel = this.save.bind(this, { updateModel: true });
}
module.exports = ModifyAccount;

ModifyAccount.prototype = {
  __proto__: View.prototype,

  _changeToken: 0,

  selectors: {
    element: '#modify-account-view',
    form: '.modify-account-form',
    fields: '*[name]',
    saveButton: '#modify-account-view .save',
    deleteButton: '#modify-account-view .delete-confirm',
    deleteRecordButton: '.delete-record',
    cancelDeleteButton: '#modify-account-view .delete-cancel',
    header: '#modify-account-header',
    status: '#modify-account-view section[role="status"]',
    errors: '#modify-account-view .errors',
    oauth2Window: '#oauth2',
    oauth2SignIn: '#modify-account-view .force-oauth2'
  },

  progressClass: 'in-progress',
  removeDialogClass: 'remove-dialog',

  get authenticationType() {
    if (this.preset && this.preset.authenticationType) {
      return this.preset.authenticationType;
    }

    return DEFAULT_AUTH_TYPE;
  },

  get oauth2Window() {
    return this._findElement('oauth2Window');
  },

  get oauth2SignIn() {
    return this._findElement('oauth2SignIn');
  },

  get deleteRecordButton() {
    return this._findElement('deleteRecordButton');
  },

  get deleteButton() {
    return this._findElement('deleteButton');
  },

  get cancelDeleteButton() {
    return this._findElement('cancelDeleteButton');
  },

  get header() {
    return this._findElement('header');
  },

  get saveButton() {
    return this._findElement('saveButton');
  },

  get form() {
    return this._findElement('form');
  },

  get fields() {
    if (!this._fields) {
      var result = this._fields = {};
      var elements = this.element.querySelectorAll(
        this.selectors.fields
      );

      var i = 0;
      var len = elements.length;

      for (i; i < len; i++) {
        var el = elements[i];
        result[el.getAttribute('name')] = el;
      }
    }

    return this._fields;
  },

  handleEvent: function(event) {
    var type = event.type;
    var data = event.data;

    switch (type) {
      case 'authorizeError':
        // we only expect one argument an error object.
        this.showErrors(data[0]);
        break;
    }
  },

  updateForm: function() {
    var update = ['user', 'fullUrl'];

    update.forEach(function(name) {
      var field = this.fields[name];
      field.value = this.model[name];
    }, this);
  },

  updateModel: function() {
    var update = ['user', 'password', 'fullUrl'];

    update.forEach(function(name) {
      var field = this.fields[name];
      var value = field.value;
      if (name === 'fullUrl') {
        // Prepend a scheme if url has neither port nor scheme
        var port = URI.getPort(value);
        var scheme = URI.getScheme(value);
        if (!port && !scheme) {
          value = 'https://' + value;
        }
      }

      this.model[name] = value;
    }, this);
  },

  deleteRecord: function(e) {
    if (e) {
      e.preventDefault();
    }

    var app = this.app;
    var id = this.model._id;
    var store = app.store('Account');

    // begin the removal (which will emit the preRemove event) but don't wait
    // for it to complete...
    store.remove(id);

    // semi-hack clear the :target - harmless in tests
    // but important in the current UI because css :target
    // does not get cleared (for some reason)
    window.location.replace('#');

    // TODO: in the future we may want to store the entry
    // url of this view and use that instead of this
    // hard coded value...
    app.router.show('/advanced-settings/');
  },

  cancel: function(event) {
    if (event) {
      event.preventDefault();
    }

    window.history.back();
  },

  cancelDelete: function() {
    this.element.classList.remove(this.removeDialogClass);
    this.cancel();
  },

  save: function(options, e) {

    if (e) {
      e.preventDefault();
    }

    var list = this.element.classList;
    var self = this;

    if (this.app.offline()) {
      this.showErrors([{name: 'offline'}]);
      return;
    }

    list.add(this.progressClass);

    this.errors.textContent = '';

    if (options && options.updateModel) {
      this.updateModel();
    }

    this.accountHandler.send(this.model, function(err) {
      list.remove(self.progressClass);
      if (!err) {
        self.app.go(self.completeUrl);
      }
    });
  },

  hideHeaderAndForm: function() {
    this.element.classList.add(this.removeDialogClass);
  },

  displayOAuth2: function(event) {
    if (event) {
      event.preventDefault();
    }

    var self = this;
    this.oauth2Window.classList.add(View.ACTIVE);

    navigator.mozApps.getSelf().onsuccess = function(e) {
      var app = e.target.result;
      app.clearBrowserData().onsuccess = function() {
        self._redirectToOAuthFlow();
      };
    };
  },

  /**
   * @param {String} preset name of value in Calendar.Presets.
   */
  _createModel: function(preset, callback) {
    var settings = Presets[preset];
    var model = new Account(settings.options);
    model.preset = preset;
    return model;
  },

  _redirectToOAuthFlow: function() {

    var apiCredentials = this.preset.apiCredentials;
    var params = {
      /*
       * code response type for now might change when we can use window.open
       */
      response_type: 'code',
      /* offline so we get refresh_token[s] */
      access_type: 'offline',
      /* we us force so we always get a refresh_token */
      approval_prompt: 'force'
    };

    OAUTH_AUTH_CREDENTIALS.forEach(function(key) {
      if (key in apiCredentials) {
        params[key] = apiCredentials[key];
      }
    });

    var oauth = this._oauthDialog = new OAuthWindow(
      this.oauth2Window,
      apiCredentials.authorizationUrl,
      params
    );

    var self = this;

    oauth.open();
    oauth.onabort = function() {
      self.cancel();
    };

    oauth.oncomplete = function(params) {
      if ('error' in params) {
        // Ruh roh
        return self.cancel();
      }

      if (!params.code) {
        return console.error('authentication error');
      }

      // Fistpump!
      self.model.oauth = { code: params.code };
      self.save();
    };
  },

  render: function() {
    if (!this.model) {
      throw new Error('must provider model to ModifyAccount');
    }

    this.form.addEventListener('submit', this._boundSaveUpdateModel);
    this.saveButton.addEventListener('click', this._boundSaveUpdateModel);
    this.header.addEventListener('action', this.cancel);
    this.deleteRecordButton.addEventListener('click', this.hideHeaderAndForm);

    if (this.model._id) {
      this.type = 'update';
      this.deleteButton.addEventListener('click', this.deleteRecord);
      this.cancelDeleteButton.addEventListener('click', this.cancelDelete);
    } else {
      this.type = 'create';
    }

    var list = this.element.classList;
    list.add(this.type);
    list.add('preset-' + this.model.preset);
    list.add('provider-' + this.model.providerType);
    list.add('auth-' + this.authenticationType);

    if (this.model.error) {
      list.add('error');
    }

    if (this.authenticationType === 'oauth2') {
      this.oauth2SignIn.addEventListener('click', this.displayOAuth2);

      if (this.type === 'create') {
        this.displayOAuth2();
      }

      this.fields.user.disabled = true;
      this.saveButton.disabled = true;
    }

    this.form.reset();
    this.updateForm();

    var usernameType = this.model.usernameType;
    this.fields.user.type = (usernameType === undefined) ?
        'text' : usernameType;
 },

  destroy: function() {
    var list = this.element.classList;

    list.remove(this.type);

    list.remove('preset-' + this.model.preset);
    list.remove('provider-' + this.model.providerType);
    list.remove('auth-' + this.authenticationType);
    list.remove('error');

    this.fields.user.disabled = false;
    this.saveButton.disabled = false;

    this._fields = null;
    this.form.reset();

    this.deleteRecordButton.removeEventListener('click',
      this.hideHeaderAndForm);
    this.oauth2SignIn.removeEventListener('click', this.displayOAuth2);
    this.saveButton.removeEventListener('click', this._boundSaveUpdateModel);
    this.deleteButton.removeEventListener('click', this.deleteRecord);
    this.cancelDeleteButton.removeEventListener('click', this.cancelDelete);
    this.header.removeEventListener('action',
                                    this.cancel);
    this.form.removeEventListener('submit', this._boundSaveUpdateModel);
  },

  dispatch: function(data) {
    if (this.model) {
      this.destroy();
    }

    var params = data.params;
    var changeToken = ++this._changeToken;

    this.completeUrl = '/settings/';

    var self = this;
    function displayModel(err, model) {
      self.preset = Presets[model.preset];

      // race condition another dispatch has queued
      // while we where waiting for an async event.
      if (self._changeToken !== changeToken) {
        return;
      }

      if (err) {
        return console.error('Error displaying model in ModifyAccount', data);
      }

      self.model = model;
      self.render();

      if (self.ondispatch) {
        self.ondispatch();
      }
    }

    if (params.id) {
      this.app.store('Account').get(params.id, displayModel);
    } else if (params.preset) {
      displayModel(null, this._createModel(params.preset));
    }
  },

  oninactive: function() {
    View.prototype.oninactive.apply(this, arguments);

    if (this._oauthDialog) {
      this._oauthDialog.close();
      this._oauthDialog = null;
    }
  }
};

});
