
define('view',['require','exports','module'],function(require, exports, module) {


var DEFAULT_ERROR_ID = 'error-default';
const INVALID_CSS = /([^a-zA-Z\-\_0-9])/g;

/**
 * Very simple base class for views.
 * Provides functionality for active/inactive.
 *
 * The first time the view is activated
 * the onactive function/event will fire.
 *
 * The .seen property is added to each object
 * with view in its prototype. .seen can be used
 * to detect if the view has ever been activated.
 *
 * @param {String|Object} options options or a selector for element.
 */
function View(options) {
  if (typeof(options) === 'undefined') {
    options = {};
  }

  if (typeof(options) === 'string') {
    this.selectors = { element: options };
  } else {
    var key;

    if (typeof(options) === 'undefined') {
      options = {};
    }

    for (key in options) {
      if (options.hasOwnProperty(key)) {
        this[key] = options[key];
      }
    }
  }

  this.hideErrors = this.hideErrors.bind(this);
}
module.exports = View;

View.ACTIVE = 'active';

View.prototype = {
  seen: false,
  activeClass: View.ACTIVE,
  errorVisible: false,

  get element() {
    return this._findElement('element');
  },

  get status() {
    return this._findElement('status');
  },

  get errors() {
    return this._findElement('errors');
  },

  /**
   * Creates a string id for a given model.
   *
   *    view.idForModel('foo-', { _id: 1 }); // => foo-1
   *    view.idForModel('foo-', '2'); // => foo-2
   *
   * @param {String} prefix of string.
   * @param {Object|String|Numeric} objectOrString representation of model.
   */
  idForModel: function(prefix, objectOrString) {
    prefix += (typeof(objectOrString) === 'object') ?
      objectOrString._id :
      objectOrString;

    return prefix;
  },

  calendarId: function(input) {
    if (typeof(input) !== 'string') {
      input = input.calendarId;
    }

    input = this.cssClean(input);
    return 'calendar-id-' + input;
  },

  /**
   * Delegate pattern event listener.
   *
   * @param {HTMLElement} element parent element.
   * @param {String} type type of dom event.
   * @param {String} selector css selector element should match
   *                          _note_ there is no magic here this
   *                          is determined from the root of the document.
   * @param {Function|Object} handler event handler.
   *                                  first argument is the raw
   *                                  event second is the element
   *                                  matching the pattern.
   */
  delegate: function(element, type, selector, handler) {
    if (typeof(handler) === 'object') {
      var context = handler;
      handler = function() {
        context.handleEvent.apply(context, arguments);
      };
    }

    element.addEventListener(type, function(e) {
      var target = e.target;
      while (target !== element) {
        if ('mozMatchesSelector' in target &&
            target.mozMatchesSelector(selector)) {
          return handler(e, target);
        }
        target = target.parentNode;
      }
    });
  },

  /**
   * Clean a string for use with css.
   * Converts illegal chars to legal ones.
   */
  cssClean: function(string) {
    if (typeof(string) !== 'string') {
      return string;
    }

    //TODO: I am worried about the performance
    //of using this all over the place =/
    //consider sanitizing all keys to ensure
    //they don't blow up when used as a selector?
    return string.replace(INVALID_CSS, '-');
  },

  /**
   * Finds a caches a element defined
   * by selectors
   *
   * @param {String} selector name as defined in selectors.
   * @param {Boolean} all true when to find all elements. (default false).
   */
  _findElement: function(name, all, element) {
    if (typeof(all) === 'object') {
      element = all;
      all = false;
    }

    element = element || document;

    var cacheName;
    var selector;

    if (typeof(all) === 'undefined') {
      all = false;
    }

    if (name in this.selectors) {
      cacheName = '_' + name + 'Element';
      selector = this.selectors[name];

      if (!this[cacheName]) {
        if (all) {
          this[cacheName] = element.querySelectorAll(selector);
        } else {
          this[cacheName] = element.querySelector(selector);
        }
      }

      return this[cacheName];
    }

    return null;
  },

 /**
   * Displays a list of errors
   *
   * @param {Array} list error list
   *  (see Event.validaitonErrors) or Error object.
   */
  showErrors: function(list) {
    var _ = navigator.mozL10n.get;
    var errors = '';

    // We can pass Error objects or
    // Array of {name: foo} objects
    if (!Array.isArray(list)) {
        list = [list];
    }

    var i = 0;
    var len = list.length;

    for (; i < len; i++) {
      var name = list[i].l10nID || list[i].name;
      errors += _('error-' + name) || _(DEFAULT_ERROR_ID);
    }

    // populate error and display it.
    this.errors.textContent = errors;
    this.errorVisible = true;
    this.status.classList.add(this.activeClass);

    this.status.addEventListener('animationend', this.hideErrors);
  },

  hideErrors: function() {
    this.status.classList.remove(this.activeClass);
    this.status.removeEventListener('animationend', this.hideErrors);
    this.errorVisible = false;
  },

  onactive: function() {
    if (this.errorVisible) {
      this.hideErrors();
    }

    // seen can be set to anything other than false to override this behaviour
    if (this.seen === false) {
      this.onfirstseen();
    }

    // intentionally using 'in'
    if ('dispatch' in this) {
      this.dispatch.apply(this, arguments);
    }

    this.seen = true;
    if (this.element) {
      this.element.classList.add(this.activeClass);
    }
  },

  oninactive: function() {
    if (this.element) {
      this.element.classList.remove(this.activeClass);
    }
  },

  onfirstseen: function() {}
};

});

define('views/current_time',['require','exports','module','view','calc','date_format','calc'],function(require, exports, module) {


var View = require('view');
var createDay = require('calc').createDay;
var dateFormat = require('date_format');
var getTimeL10nLabel = require('calc').getTimeL10nLabel;

var activeClass = View.ACTIVE;

function CurrentTime(options) {
  this._container = options.container;
  // timespan can be injected later! this is just a convenience
  this.timespan = options.timespan;
  this._sticky = options.sticky;
}
module.exports = CurrentTime;

CurrentTime.prototype = {
  _create: function() {
    if (this.element) {
      return;
    }

    this.element = document.createElement('div');
    this.element.classList.add('md__current-time');
    this._container.appendChild(this.element);
  },

  refresh: function() {
    this._clearInterval();

    if (this._previousOverlap) {
      this._previousOverlap.classList.remove('is-hidden');
    }

    this._unmarkCurrentDay();
    this._hide();
    this.activate();
  },

  activate: function() {
    if (!this.timespan.containsNumeric(Date.now())) {
      this._maybeActivateInTheFuture();
      return;
    }

    this._create();
    this.element.classList.add(activeClass);
    this._tick();
  },

  _maybeActivateInTheFuture: function() {
    var now = Date.now();
    var diff = this.timespan.start - now;
    if (diff >= 0) {
      // if timespan is in the "future" we make sure it will start rendering
      // the current time as soon as it reaches 00:00:00 of the first day
      // inside timespan (eg. current time is 2014-05-22T23:59:50 and user is
      // viewing 2014-05-23 until past midnight)
      this._clearInterval();
      this._interval = setTimeout(this.activate.bind(this), diff);
    }
  },

  deactivate: function() {
    this._clearInterval();
    this._hide();
  },

  _hide: function() {
    if (this.element) {
      this.element.classList.remove(activeClass);
    }
  },

  destroy: function() {
    this.deactivate();
    if (this.element) {
      this._container.removeChild(this.element);
    }
  },

  _clearInterval: function() {
    if (this._interval) {
      clearTimeout(this._interval);
      this._interval = null;
    }
  },

  _tick: function() {
    this._clearInterval();
    var now = new Date();

    if (!this.timespan.contains(now)) {
      this.deactivate();
      this._unmarkCurrentDay();
      return;
    }
    this._render();

    // will call tick once per minute
    var nextTick = (60 - now.getSeconds()) * 1000;
    this._interval = setTimeout(this._tick.bind(this), nextTick);
  },

  _render: function() {
    var now = new Date();
    var format = getTimeL10nLabel('current-time');

    this.element.textContent = dateFormat.localeFormat(
      now,
      navigator.mozL10n.get('current-time')
    );

    this.element.textContent = dateFormat.localeFormat(
      now,
      navigator.mozL10n.get(format)
    );
    this.element.dataset.date = now;
    this.element.dataset.l10nDateFormat = format;

    var hour = now.getHours();
    var elapsedMinutes = (hour * 60) + now.getMinutes();
    var totalMinutes = 24 * 60;
    var percentage = ((elapsedMinutes / totalMinutes) * 100);
    // we limit the position between 0.5-99.5% to avoid cropping the text
    this.element.style.top = Math.max(Math.min(percentage, 99.5), 0.5) + '%';

    this._checkOverlap(hour);
    this._markCurrentDay(now);
  },

  _checkOverlap: function(hour) {
    // we only need to check the current hour (with current design there is
    // no way to overlap previous/next hours)
    var displayHour = this._container.querySelector(
      `.md__hour-${hour} .md__display-hour`
    );

    displayHour.classList.toggle('is-hidden', this._intersect(displayHour));

    // just in case last time it checked was against a different hour
    if (this._previousOverlap && this._previousOverlap !== displayHour) {
      this._previousOverlap.classList.remove('is-hidden');
    }

    this._previousOverlap = displayHour;
  },

  _intersect: function(displayHour) {
    var b1 = this.element.getBoundingClientRect();
    var b2 = displayHour.getBoundingClientRect();

    return (
      b1.left <= b2.right &&
      b2.left <= b1.right &&
      b1.top <= b2.bottom &&
      b2.top <= b1.bottom
    );
  },

  _markCurrentDay: function(date) {
    if (!this._sticky) {
      return;
    }

    var day = createDay(date);
    var selector = `.md__allday[data-date="${day}"] .md__day-name`;
    var header = this._sticky.querySelector(selector);

    if (header) {
      header.classList.add('is-today');
    }

    if (this._previousHeader !== header) {
      this._unmarkCurrentDay();
    }

    this._previousHeader = header;
  },

  _unmarkCurrentDay: function() {
    if (this._previousHeader) {
      this._previousHeader.classList.remove('is-today');
    }
  }
};

});

define('template',['require','exports','module'],function(require, exports, module) {


var POSSIBLE_HTML = /[&<>"'`]/;

var span = document.createElement('span');

function create(templates) {
  var key, result = {};

  for (key in templates) {
    if (templates.hasOwnProperty(key)) {
      result[key] = new Template(templates[key]);
    }
  }

  return result;
}

function Template(fn) {
  this.template = fn;
}
module.exports = Template;

Template.prototype = {
  arg: function(key) {
    if (typeof(this.data) === 'undefined') {
      return '';
    } else if (typeof(this.data) !== 'object') {
      return this.data;
    }

    return this.data[key];
  },

  h: function(a) {
    var arg = this.arg(a);
    // accept anything that can be converted into a string and we make sure
    // the only falsy values that are converted into empty strings are
    // null/undefined to avoid mistakes
    arg = arg == null ? '' : String(arg);

    //only escape bad looking stuff saves
    //a ton of time
    if (POSSIBLE_HTML.test(arg)) {
      span.textContent = arg;
      return span.innerHTML.replace(/"/g, '&quot;').replace(/'/g, '&#x27;');
    } else {
      return arg;
    }
  },

  s: function(a) {
    var arg = this.arg(a);
    return String((arg || ''));
  },

  bool: function(key, onTrue) {
    if (this.data[key]) {
      return onTrue;
    } else {
      return '';
    }
  },

  l10n: function(key, prefix) {
    var value = this.arg(key);

    if (prefix) {
      value = prefix + value;
    }
    return navigator.mozL10n.get(value);
  },

  l10nId: function(a) {
    return this.s(a).replace(/\s/g, '-');
  },

  /**
   * Renders template with given slots.
   *
   * @param {Object} object key, value pairs for template.
   */
  render: function(data) {
    this.data = data;
    return this.template();
  },

  /**
   * Renders template multiple times
   *
   * @param {Array} objects object details to render.
   * @param {String} [join] optional join argument will join the array.
   * @return {String|Array} String if join argument is given array otherwise.
   */
  renderEach: function(objects, join) {
    var i = 0, len = objects.length,
        result = [];

    for (; i < len; i++) {
      result.push(this.render(objects[i]));
    }

    if (typeof(join) !== 'undefined') {
      return result.join(join);
    }

    return result;
  }
};

Template.create = create;

});

define('templates/date_span',['require','exports','module','calc','template','date_format'],function(require, exports, module) {


var Calc = require('calc');
var create = require('template').create;
var dateFormat = require('date_format');

var l10n = navigator.mozL10n;

module.exports = create({
  time: function() {
    var time = this.arg('time');
    var format = Calc.getTimeL10nLabel(this.h('format'));
    var displayTime = dateFormat.localeFormat(time, l10n.get(format));

    return `<span data-l10n-date-format="${format}"
                  data-date="${time}">${displayTime}</span>`;
  },

  hour: function() {
    var hour = this.h('hour');
    var format = Calc.getTimeL10nLabel(this.h('format'));
    var className = this.h('className');
    var date = new Date();
    date.setHours(hour, 0, 0, 0);

    var l10nLabel = l10n.get(format);
    if (this.arg('addAmPmClass')) {
      l10nLabel = l10nLabel.replace(
        /\s*%p\s*/,
        '<span class="ampm">%p</span>'
      );
    }

    var displayHour = dateFormat.localeFormat(date, l10nLabel);
    // remove leading zero
    displayHour = displayHour.replace(/^0/, '');
    var l10nAttr = (hour === Calc.ALLDAY) ?
      'data-l10n-id="hour-allday"' :
      `data-l10n-date-format="${format}"`;
    return `<span class="${className}" data-date="${date}" ${l10nAttr}>
              ${displayHour}
            </span>`;
  }
});

});

/* global Buffer */
define('querystring',['require','exports','module'],function(require, exports) {


// Copyright Joyent, Inc. and other Node contributors.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to permit
// persons to whom the Software is furnished to do so, subject to the
// following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
// USE OR OTHER DEALINGS IN THE SOFTWARE.

// If obj.hasOwnProperty has been overridden, then calling
// obj.hasOwnProperty(prop) will break.
// See: https://github.com/joyent/node/issues/1707
function hasOwnProperty(obj, prop) {
  return Object.prototype.hasOwnProperty.call(obj, prop);
}


function charCode(c) {
  return c.charCodeAt(0);
}


// a safe fast alternative to decodeURIComponent
exports.unescapeBuffer = function(s, decodeSpaces) {
  var out = new Buffer(s.length);
  var state = 'CHAR'; // states: CHAR, HEX0, HEX1
  var n, m, hexchar;

  for (var inIndex = 0, outIndex = 0; inIndex <= s.length; inIndex++) {
    var c = s.charCodeAt(inIndex);
    switch (state) {
      case 'CHAR':
        switch (c) {
          case charCode('%'):
            n = 0;
            m = 0;
            state = 'HEX0';
            break;
          case charCode('+'):
            if (decodeSpaces) {
              c = charCode(' ');
            }
            out[outIndex++] = c;
            break;
          default:
            out[outIndex++] = c;
        }
        break;

      case 'HEX0':
        state = 'HEX1';
        hexchar = c;
        if (charCode('0') <= c && c <= charCode('9')) {
          n = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          n = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          n = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = c;
          state = 'CHAR';
          break;
        }
        break;

      case 'HEX1':
        state = 'CHAR';
        if (charCode('0') <= c && c <= charCode('9')) {
          m = c - charCode('0');
        } else if (charCode('a') <= c && c <= charCode('f')) {
          m = c - charCode('a') + 10;
        } else if (charCode('A') <= c && c <= charCode('F')) {
          m = c - charCode('A') + 10;
        } else {
          out[outIndex++] = charCode('%');
          out[outIndex++] = hexchar;
          out[outIndex++] = c;
          break;
        }
        out[outIndex++] = 16 * n + m;
        break;
    }
  }

  // TODO support returning arbitrary buffers.

  return out.slice(0, outIndex - 1);
};


exports.unescape = function(s, decodeSpaces) {
  return exports.unescapeBuffer(s, decodeSpaces).toString();
};


exports.escape = function(str) {
  return encodeURIComponent(str);
};

var stringifyPrimitive = function(v) {
  switch (typeof v) {
    case 'string':
      return v;

    case 'boolean':
      return v ? 'true' : 'false';

    case 'number':
      return isFinite(v) ? v : '';

    default:
      return '';
  }
};


exports.stringify = exports.encode = function(obj, sep, eq, name) {
  sep = sep || '&';
  eq = eq || '=';
  if (obj === null) {
    obj = undefined;
  }

  if (typeof obj === 'object') {
    return Object.keys(obj).map(function(k) {
      var ks = exports.escape(stringifyPrimitive(k)) + eq;
      if (Array.isArray(obj[k])) {
        return obj[k].map(function(v) {
          return ks + exports.escape(stringifyPrimitive(v));
        }).join(sep);
      } else {
        return ks + exports.escape(stringifyPrimitive(obj[k]));
      }
    }).join(sep);

  }

  if (!name) {
    return '';
  }
  return exports.escape(stringifyPrimitive(name)) + eq +
         exports.escape(stringifyPrimitive(obj));
};

// Parse a key=val string.
exports.parse = exports.decode = function(qs, sep, eq, options) {
  sep = sep || '&';
  eq = eq || '=';
  var obj = {};

  if (typeof qs !== 'string' || qs.length === 0) {
    return obj;
  }

  var regexp = /\+/g;
  qs = qs.split(sep);

  var maxKeys = 1000;
  if (options && typeof options.maxKeys === 'number') {
    maxKeys = options.maxKeys;
  }

  var len = qs.length;
  // maxKeys <= 0 means that we should not limit keys count
  if (maxKeys > 0 && len > maxKeys) {
    len = maxKeys;
  }

  for (var i = 0; i < len; ++i) {
    var x = qs[i].replace(regexp, '%20'),
        idx = x.indexOf(eq),
        kstr, vstr, k, v;

    if (idx >= 0) {
      kstr = x.substr(0, idx);
      vstr = x.substr(idx + 1);
    } else {
      kstr = x;
      vstr = '';
    }

    try {
      k = decodeURIComponent(kstr);
      v = decodeURIComponent(vstr);
    } catch (e) {
      k = exports.unescape(kstr, true);
      v = exports.unescape(vstr, true);
    }

    if (!hasOwnProperty(obj, k)) {
      obj[k] = v;
    } else if (Array.isArray(obj[k])) {
      obj[k].push(v);
    } else {
      obj[k] = [obj[k], v];
    }
  }

  return obj;
};

});

define('utils/dom',['require','exports','module'],function(require, exports) {


/**
 * Gets the element absolute offset top relative to the window top.
 */
exports.absoluteOffsetTop = function(el) {
  var top = 0;
  while (el) {
    var pos = window.getComputedStyle(el).position;
    if (pos === 'absolute' || pos === 'relative') {
      top += el.offsetTop;
    }
    el = el.parentElement;
  }
  return top;
};

/**
 * Gets the closest (parent) element that matches the given selector.
 */
exports.closest = function(el, selector) {
  while (el) {
    if (matches(el, selector)) {
      return el;
    }
    el = el.parentElement;
  }
};

function matches(el, selector) {
  // "matches" is only unprefixed on Fx 34
  return 'matches' in el ?
    el.matches(selector) :
    el.mozMatchesSelector(selector);
}

});

define('views/hour_double_tap',['require','exports','module','querystring','utils/dom','utils/dom','calc'],function(require, exports, module) {


// this will be replaced later for a better logic (see Bug 992728) but it was
// too much to do in a single patch, so for now we do a simple double tap
// without any visual feedback (similar to the old day view behavior)

var QueryString = require('querystring');
var absoluteOffsetTop = require('utils/dom').absoluteOffsetTop;
var closest = require('utils/dom').closest;
var createDay = require('calc').createDay;

function HourDoubleTap(options) {
  this.app = options.app;
  this.main = options.main;
  this.daysHolder = options.daysHolder;
  this.alldaysHolder = options.alldaysHolder;
  this.hourHeight = options.hourHeight;

  this._onDayTap = this._onDayTap.bind(this);
  this._onAllDayTap = this._onAllDayTap.bind(this);
  this.removeAddEventLink = this.removeAddEventLink.bind(this);
}
module.exports = HourDoubleTap;

HourDoubleTap.prototype = {

  _isActive: false,

  _addEventLink: null,

  setup: function() {
    this._mainOffset = absoluteOffsetTop(this.main);
    this.daysHolder.addEventListener('click', this._onDayTap);
    this.alldaysHolder.addEventListener('click', this._onAllDayTap);
  },

  destroy: function() {
    this.removeAddEventLink();
    this.daysHolder.removeEventListener('click', this._onDayTap);
    this.alldaysHolder.removeEventListener('click', this._onAllDayTap);
  },

  _onDayTap: function(evt) {
    var target = evt.target;
    if (!target.classList.contains('md__day')) {
      return;
    }

    var y = evt.clientY + this.main.scrollTop - this._mainOffset;
    var hour = Math.floor(y / this.hourHeight);
    var baseDate = new Date(target.dataset.date);

    this._onTap(target, {
      startDate: addHours(baseDate, hour).toString(),
      endDate: addHours(baseDate, hour + 1).toString()
    }, hour);
  },

  _onAllDayTap: function(evt) {
    var target = evt.target;
    if (!target.classList.contains('md__allday-events')) {
      return;
    }

    var startDate = new Date(closest(target, '.md__allday').dataset.date);

    this._onTap(target, {
      isAllDay: true,
      startDate: startDate.toString(),
      endDate: createDay(startDate, startDate.getDate() + 1).toString()
    });
  },

  _onTap: function(container, data, hour) {
    hour = hour || 0;

    if (this._addEventLink) {
      this.removeAddEventLink();
      return;
    }

    var link = document.createElement('a');
    link.href = '/event/add/?' + QueryString.stringify(data);
    link.className = 'md__add-event gaia-icon icon-newadd';
    link.dataset.l10nId = 'multi-day-new-event-link';
    link.style.top = (hour * this.hourHeight) + 'px';
    link.style.opacity = 0;

    link.addEventListener('click', this.removeAddEventLink);

    container.appendChild(link);
    this._addEventLink = link;

    // opacity will trigger transition, needs to happen after nextTick
    setTimeout(() => {
      this._addEventLink && (this._addEventLink.style.opacity = 1);
    });
  },

  removeAddEventLink: function() {
    var link = this._addEventLink;
    if (!link) {
      return;
    }

    link.removeEventListener('click', this.removeAddEventLink);

    link.addEventListener('transitionend', function onTransitionEnd() {
      link.removeEventListener('transitionend', onTransitionEnd);
      link.parentNode && link.parentNode.removeChild(link);
    });
    link.style.opacity = 0;

    this._addEventLink = null;
  }

};

function addHours(date, hourDiff) {
  var result = new Date(date);
  result.setHours(result.getHours() + hourDiff);
  return result;
}

});

// this module is responsible for the touch/panning of MultiDay views
define('views/pan',['require','exports','module','ext/eventemitter2','utils/mout','utils/mout','utils/mout'],function(require, exports, module) {


var EventEmitter2 = require('ext/eventemitter2');
var clamp = require('utils/mout').clamp;
var lerp = require('utils/mout').lerp;
var norm = require('utils/mout').norm;

function Pan(options) {
  EventEmitter2.call(this);

  this.eventTarget = options.eventTarget;
  this.targets = options.targets;
  this.overflows = options.overflows || [];

  var size = Math.max(options.gridSize || 0, 1);
  var cells = Math.max(options.visibleCells || 0, 1);

  this._gridSize = size;
  this._visibleCells = cells;
  this._minX = size * cells * -2;
  this._origX = this._startX = this._curX = this._destX = this._minX / 2;
  this._startMouseX = this._startMouseY = 0;
  this._isVertical = false;
  this._startTime = 0;
  this._touchStart = 0;
  this._dx = 0;

  // _lockedAxis is used to control if we detected if the movement is
  // vertical/horizontal, very important for ignoring clicks and also to be
  // able to set a threshold for the axis detection
  this._lockedAxis = false;

  this._onTouchStart = this._onTouchStart.bind(this);
  this._onTouchMove = this._onTouchMove.bind(this);
  this._onTouchEnd = this._onTouchEnd.bind(this);
  this._tick = this._tick.bind(this);
  this._onTweenEnd = null;
}
module.exports = Pan;

Pan.prototype = {
  __proto__: EventEmitter2.prototype,

  TRANSITION_DURATION: 800,

  setup: function() {
    var element = this.eventTarget;
    element.addEventListener('touchstart', this._onTouchStart);
    element.addEventListener('touchmove', this._onTouchMove);
    element.addEventListener('touchend', this._onTouchEnd);
    element.addEventListener('touchcancel', this._onTouchEnd);

    this._set(this._origX);
  },

  _onTouchStart: function(evt) {
    this._startMouseX = evt.touches[0].clientX;
    this._startMouseY = evt.touches[0].clientY;
    this._isVertical = false;
    this._lockedAxis = false;
    this._touchStart = Date.now();
    // we need to reset the tween callback because we should only call it
    // once and only if user did not trigger a new touch
    this._onTweenEnd = null;
  },

  _onTouchMove: function(evt) {
    if (this._isVertical) {
      return;
    }

    var dx = this._startMouseX - evt.touches[0].clientX;
    var dy = this._startMouseY - evt.touches[0].clientY;
    this._dx = dx;

    if (!this._lockedAxis) {
      var adx = Math.abs(dx);
      var ady = Math.abs(dy);

      // we wait until we are sure movement is horizontal before we do anything.
      // if absolute difference between x/y movement is over a threshold (10px)
      // we assume drag follows a single axis.
      if (Math.abs(adx - ady) < 10) {
        return;
      }

      this._isVertical = adx < ady;
      this._lockedAxis = true;

      this.emit('start');

      if (this._isVertical) {
        return;
      }

      // we should only lock scroll once and only if dragging horizontally
      this._lockScroll();
    }

    this._updateDestination(this._origX - dx, 0);
  },

  _lockScroll: function() {
    this.overflows.forEach(el => el.style.overflowY = 'hidden');
  },

  _unlockScroll: function() {
    this.overflows.forEach(el => el.style.overflowY = 'scroll');
  },

  _updateDestination: function(x, duration) {
    duration = duration != null ? duration : this.TRANSITION_DURATION;

    this._startX = this._curX;
    this._destX = clamp(x, this._minX, 0);

    var now = Date.now();
    this._endTime = now + duration;

    if (!this._requestId) {
      this._startTime = now;
      this._tween();
    }
  },

  _tween: function() {
    this._requestId = window.requestAnimationFrame(this._tick);
  },

  _tick: function() {
    var t = norm(Date.now(), this._startTime, this._endTime);

    if (t >= 1 || this._curX === this._destX) {
      this._killTween();
      this._set(this._destX);
      return;
    }

    var x = lerp(ease(t), this._startX, this._destX);
    this._set(x);
    this._tween();
  },

  _killTween: function() {
    if (this._requestId) {
      window.cancelAnimationFrame(this._requestId);
      this._requestId = null;
    }
    this._onTweenEnd && this._onTweenEnd.call(this);
  },

  _onTouchEnd: function() {
    // if touch is very fast momentum would be bigger than our threshold,
    // this is very important for click events otherwise they wouldn't open
    // the event details
    if (this._isVertical || !this._lockedAxis) {
      this._unlockScroll();
      return;
    }

    var duration = Date.now() - this._touchStart;
    var momentum = Math.abs(this._dx) / duration;
    var snap;

    if (momentum > 0.5) {
      // if the drag was fast we consider it as a swipe (move multiple cells
      // at once)
      var direction = this._dx > 0 ? -1 : 1;
      snap = this._origX + (direction * this._gridSize * this._visibleCells);
    } else {
      // we only round up if very close to the next column, this behavior is
      // better for the user than a regular round/ceil/floor
      snap = Math.round((this._destX / this._gridSize) + 0.2) *
        this._gridSize;
    }

    // we only unlock the scroll after the tween is complete to make multiple
    // consecutive swipes faster (also avoids flickering y-axis position)
    this._onTweenEnd = this._unlockScroll;
    this._updateDestination(snap);

    this.emit('release', {
      diff: Math.round((this._origX - this._destX) / this._gridSize)
    });
  },

  _set: function(x) {
    x = clamp(x, this._minX, 0);
    this.targets.forEach(el => {
      el.style.transform = 'translateX(' + x +'px)';
    });
    this._curX = x;
  },

  refresh: function() {
    var diff = Math.abs(this._curX - this._destX);
    diff *= this._curX < this._destX ? -1 : 1;

    // we update the position based on the relative distance to keep a smooth
    // transition
    if (diff) {
      this._set(this._origX + diff);
      this._updateDestination(this._origX);
    } else {
      this._set(this._origX);
    }
  }

};

// borrowed from zeh/ztween (expoOut)
function ease(t) {
  return (t >= 0.999) ? 1 : 1.001 * (-Math.pow(2, -10 * t) + 1);
}

});

/**
 * Representation of conflicts over a span of time, organized into
 * non-overlapping columns tracked by IntervalTree instances.
 */
define('conflict_span',['require','exports','module','interval_tree','timespan'],function(require, exports, module) {


var IntervalTree = require('interval_tree');
var Timespan = require('timespan');

// Smallest gap interval to use in splitting conflict spans
var MIN_SPLIT_INTERVAL = 5 * 60 * 1000;  // 5 minutes

// Auto-increment ID for instances
var _id = 0;

function ConflictSpan(parent) {
  this.id = (_id++);
  this.parent = parent;
  this.startTime = null;
  this.endTime = null;
  this.all = new IntervalTree();
  this.columnsByID = {};
  this.columns = [];
  this.addColumn();
}
module.exports = ConflictSpan;

ConflictSpan.prototype = {
  /**
   * Get a list of all the busytime IDs in this span.
   *
   * @return {Array} List of all the busytime IDs.
   */
  getIDs: function() {
    return Object.keys(this.all.byId);
  },

  /**
   * Add a new column tracked by an IntervalTree
   *
   * @return {Object} IntervalTree tracking the column.
   */
  addColumn: function() {
    var tree = new IntervalTree();
    this.columns.push(tree);
    return tree;
  },

  /**
   * Find a column where the given busytime fits without conflict, adding a
   * new column if necessary.
   *
   * @param {Object} busytime full busytime object.
   * @return {Object} IntervalTree column that can accept the busytime.
   */
  findColumn: function(busytime, skipAdd) {
    var column = null;
    var span = new Timespan(busytime._startDateMS, busytime._endDateMS);
    for (var i = 0; i < this.columns.length; i++) {
      var curr = this.columns[i];
      if (!curr.query(span).length) {
        column = curr;
        break;
      }
    }
    if (!column && !skipAdd) {
      column = this.addColumn();
    }
    return column;
  },

  /**
   * Add a busytime to the conflict span
   *
   * @param {Object} busytime full busytime object.
   */
  add: function(busytime) {
    var id = busytime._id;

    this.parent.conflicts[id] = this;
    this.all.add(busytime);

    var column = this.findColumn(busytime);
    column.add(busytime);
    this.columnsByID[id] = column;

    this._updateTimes(busytime);
    this._updateLayout();
    return this;
  },

  /**
   * Remove a busytime from the conflict span
   *
   * @param {Object} busytime full busytime object.
   * @param {Boolean} skipMaintenance skip post-removal maintenance.
   */
  remove: function(busytime, skipMaintenance) {
    var id = busytime._id;

    this.all.remove(busytime);
    var column = this.columnsByID[id];
    if (!column) { return; }

    column.remove(busytime);
    delete this.columnsByID[id];
    delete this.parent.conflicts[id];

    // Removing a single item requires maintenance after. But, this can be
    // skipped during a split, which does its own cleanup after multiple
    // removes & adds between spans.
    if (skipMaintenance) { return this; }

    this._splitIfNecessary();
    var boom = this._selfDestructIfNecessary();
    if (!boom) {
      this._resetTimes();
      this._purgeEmptyColumns();
      this._updateLayout();
    }

    return this;
  },

  /**
   * Absorb the given conflict span into this one
   *
   * @param {Object} ConflictSpan to be absorbed.
   */
  absorb: function(otherCS) {
    var self = this;
    var otherIDs = otherCS.getIDs();
    otherIDs.forEach(function(otherID) {
      var otherBusytime = self.parent.tree.byId[otherID];
      self.add(otherBusytime);
      // Cheat: skip removing from the other span, since references go away.
    });
  },

  /**
   * Update the start/end times for this span from a new busytime.
   *
   * @param {Object} busytime full busytime object.
   */
  _updateTimes: function(busytime) {
    var start = busytime._startDateMS;
    if (null === this.startTime || start < this.startTime) {
      this.startTime = start;
    }
    var end = busytime._endDateMS;
    if (null === this.endTime || end > this.endTime) {
      this.endTime = end;
    }
  },

  /**
   * Reset times with a complete re-scan of all events in the span.
   */
  _resetTimes: function() {
    this.startTime = this.endTime = null;
    var byId = this.all.byId;
    for (var k in byId) {
      this._updateTimes(byId[k]);
    }
  },

  /**
   * Scan through the events in this span. If a significant gap is found,
   * presumably after a removal, split this span in two.
   *
   * @param {Object} busytime full busytime object.
   */
  _splitIfNecessary: function() {
    var start = this.startTime;
    var end = this.endTime;

    // Scan for the end of the first gap, if any.
    var splitAt = false;
    var prevHits = null;
    for (var top = start; top < end; top += MIN_SPLIT_INTERVAL) {
      var span = new Timespan(top, top + MIN_SPLIT_INTERVAL);
      var hits = this.all.query(span).length;
      if (0 === prevHits && hits > 0) {
        // Transition from empty to non-empty is where we split.
        splitAt = top; break;
      }
      prevHits = hits;
    }

    // Bail if we never found a gap.
    if (splitAt === false) { return; }

    // Remove & collect the post-gap items for new split.
    var newItems = [];
    var splitSpan = new Timespan(splitAt, Infinity);
    var splitItems = this.all.query(splitSpan);
    var self = this;
    splitItems.forEach(function(item) {
      self.remove(item, true);
      newItems.push(item);
    });

    // Perform partial post-removal maintenance
    var boom = this._selfDestructIfNecessary();
    if (!boom) {
      this._resetTimes();
      this._purgeEmptyColumns();
      this._updateLayout();
    }

    // Bail if there's just one item for new split - no conflict.
    if (newItems.length == 1) {
      this.parent._clearLayout(newItems[0]);
      return;
    }

    // Otherwise, populate a new span with the conflicting items.
    var newCS = new ConflictSpan(this.parent);
    newItems.forEach(function(item) {
      newCS.add(item);
    });

    // Finally, recurse into the new span and split further, if necessary.
    newCS._splitIfNecessary();
  },

  /**
   * If this span has only one event left, then self-destruct because there's
   * no longer a conflict.
   */
  _selfDestructIfNecessary: function() {
    var keys = this.getIDs();
    if (keys.length > 1) {
      // There's still a conflict, so bail.
      return false;
    }
    if (keys.length == 1) {
      // Exactly one left, so clean up.
      var busytime = this.all.byId[keys[0]];
      this.remove(busytime, true);
      this.parent._clearLayout(busytime);
    }
    return true;
  },

  /**
   * Purge empty columns from the conflict span.
   */
  _purgeEmptyColumns: function() {
    var newColumns = [];
    for (var i = 0; i < this.columns.length; i++) {
      var column = this.columns[i];
      if (Object.keys(column.byId).length > 0) {
        newColumns.push(column);
      }
    }
    this.columns = newColumns;
  },

  /**
   * Update layout for all events participating in this conflict span.
   */
  _updateLayout: function() {
    var numCols = this.columns.length;
    var width = (100 / numCols);
    for (var cIdx = 0; cIdx < numCols; cIdx++) {
      var column = this.columns[cIdx];
      for (var k in column.byId) {
        var busytime = column.byId[k];
        var el = this.parent.getElement(busytime);
        el.style.width = width + '%';
        el.style.left = (width * cIdx) + '%';
        // we toggle the style based on amount of overlaps
        el.classList.toggle('many-overlaps', numCols > 4);
        el.classList.toggle('has-overlaps', numCols > 1);
      }
    }
  }
};

});

/**
 * Conflict manager
 */
define('utils/overlap',['require','exports','module','conflict_span','interval_tree','timespan'],function(require, exports, module) {


/**
 * Module dependencies
 */
var ConflictSpan = require('conflict_span');
var IntervalTree = require('interval_tree');
var Timespan = require('timespan');

function Overlap() {
  this.reset();
}
module.exports = Overlap;

Overlap.prototype = {
  reset: function() {
    this.tree = new IntervalTree();
    this.conflicts = {};
    this.elements = {};
  },

  add: function(myBusytime, element) {
    this.tree.add(myBusytime);
    this.elements[myBusytime._id] = element;

    // Check for conflicts, bail if none
    var related = this._findRelated(myBusytime);
    if (0 === related.length) {
      return;
    }

    var myID = myBusytime._id;
    var myCS = this.conflicts[myID];

    var self = this;
    related.forEach(function(otherBusytime) {
      // Get the other's ID, skip the current
      var otherID = otherBusytime._id;
      if (otherID === myID) {
        return;
      }

      var otherCS = self.conflicts[otherID];
      if (!myCS && !otherCS) {
        // This is a brand new conflict.
        myCS = new ConflictSpan(self);
        myCS.add(myBusytime).add(otherBusytime);
      } else if (myCS && !otherCS) {
        // Other time can join this one's existing span
        myCS.add(otherBusytime);
      } else if (!myCS && otherCS) {
        // This time can join the other's existing span
        myCS = otherCS.add(myBusytime);
      } else if (myCS && otherCS && myCS != otherCS) {
        // Both already in different spans, so absorb other into this
        myCS.absorb(otherCS);
      }
    });

  },

  /**
   * Remove a busytime from the collection.
   * Unlike other methods you must pass a real
   * busytime object.
   *
   * @param {Object} busytime full busytime object.
   */
  remove: function(busytime) {
    this._clearLayout(busytime);
    this.tree.remove(busytime);
    delete this.elements[busytime._id];
    var myID = busytime._id;
    var myCS = this.conflicts[myID];
    if (myCS) {
      myCS.remove(busytime);
    }
  },

  /**
   * Get the ConflictSpan associated with this busytime, if any.
   *
   * @param {Object|String} busytime id or busytime object.
   * @return {Object} associated ConflictSpan, if any.
   */
  getConflictSpan: function(busytime) {
    var id = this._busytimeId(busytime);
    return this.conflicts[id];
  },

  /**
   * @param {Object|String} busytime id or busytime object.
   * @return {HTMLElement} associated dom element.
   */
  getElement: function(busytime) {
    var id = this._busytimeId(busytime);
    return this.elements[id];
  },

  /** private */

  _busytimeId: function(busytime) {
    return (typeof(busytime) === 'string') ? busytime : busytime._id;
  },

  /**
   * Search tree for busytimes that overlap with the given.
   */
  _findRelated: function(busytime) {
    //XXX: this is bad encapsulation but
    //     we generate these when we insert
    //     the points in the tree.
    var span = new Timespan(busytime._startDateMS, busytime._endDateMS);
    return this.tree.query(span);
  },

  /**
   * Clear the layout from a busytime element, presumably because it has just
   * been removed from conflict.
   *
   * @param {Object} busytime full busytime object.
   */
  _clearLayout: function(busytime) {
    var el = this.elements[busytime._id];
    el.style.width = '';
    el.style.left = '';
    el.classList.remove('has-overlaps', 'many-overlaps');
  }
};

});

define('views/single_day',['require','exports','module','utils/overlap','date_format','day_observer','calc','calc'],function(require, exports, module) {


var Overlap = require('utils/overlap');
var dateFormat = require('date_format');
var dayObserver = require('day_observer');
var relativeDuration = require('calc').relativeDuration;
var relativeOffset = require('calc').relativeOffset;

function SingleDay(config) {
  this.date = config.date;
  this._hourHeight = config.hourHeight;
  this._daysHolder = config.daysHolder;
  this._alldaysHolder = config.alldaysHolder;
  this._render = this._render.bind(this);
  this.overlaps = new Overlap();
}
module.exports = SingleDay;

SingleDay.prototype = {
  _isActive: false,
  _borderWidth: 0.1,
  _attached: false,

  setup: function() {
    this.day = document.createElement('div');
    this.day.className = 'md__day';
    this.day.dataset.date = this.date;

    this.allday = document.createElement('div');
    this.allday.className = 'md__allday';
    this.allday.dataset.date = this.date;

    this._dayName = document.createElement('h1');
    this._dayName.className = 'md__day-name';
    this.allday.appendChild(this._dayName);

    this._alldayEvents = document.createElement('div');
    this._alldayEvents.className = 'md__allday-events';
    this.allday.appendChild(this._alldayEvents);

    this._updateDayName();

    this.onactive();
  },

  _updateDayName: function() {
    // we can't use [data-l10n-date-format] because format might change
    var format = window.navigator.mozL10n.get('week-day');
    this._dayName.textContent = dateFormat.localeFormat(
      this.date,
      format
    );
  },

  handleEvent: function(evt) {
    switch(evt.type) {
      case 'localized':
        this._updateDayName();
        break;
    }
  },

  append: function() {
    this._daysHolder.appendChild(this.day);
    this._alldaysHolder.appendChild(this.allday);
    this._attached = true;
  },

  onactive: function() {
    if (this._isActive) {
      return;
    }
    dayObserver.on(this.date, this._render);
    window.addEventListener('localized', this);
    this._isActive = true;
  },

  _render: function(records) {
    // we always remove all elements and then again since it's simpler and we
    // should not have that many busytimes on a single day.
    this._alldayEvents.innerHTML = '';
    records.allday.forEach(this._renderAlldayEvent, this);
    this.overlaps.reset();
    this.day.innerHTML = '';
    records.events.forEach(this._renderEvent, this);
  },

  _renderEvent: function(record) {
    var el = this._buildEventElement(record);

    var busytime = record.busytime;
    var {startDate, endDate} = busytime;
    var duration = relativeDuration(this.date, startDate, endDate);
    // we subtract border to keep a margin between consecutive events
    var hei = duration * this._hourHeight - this._borderWidth;
    el.style.height = hei + 'px';

    if (duration < 1) {
      el.classList.add('is-partial');
      var size = '';
      // we need to toggle layout if event lasts less than 20, 30 and 45min
      if (duration < 0.3) {
        size = 'micro';
      } else if (duration < 0.5) {
        size = 'tiny';
      } else if (duration < 0.75) {
        size = 'small';
      }
      if (size) {
        el.classList.add('is-partial-' + size);
      }
    }

    var offset = relativeOffset(this.date, startDate);
    el.style.top = (offset * this._hourHeight) + 'px';

    this.overlaps.add(busytime, el);
    this.day.appendChild(el);
  },

  _buildEventElement: function(record) {
    var {event, busytime} = record;
    var {remote} = event;

    var el = document.createElement('a');
    el.href = '/event/show/' + busytime._id;
    el.className = [
      'md__event',
      'calendar-id-' + event.calendarId,
      'calendar-border-color',
      'calendar-bg-color'
    ].join(' ');

    var title = document.createElement('span');
    title.className = 'md__event-title';
    // since we use "textContent" there is no risk of XSS
    title.textContent = remote.title;
    el.appendChild(title);

    if (remote.location) {
      var location = document.createElement('span');
      location.className = 'md__event-location';
      // since we use "textContent" there is no risk of XSS
      location.textContent = remote.location;
      el.appendChild(location);
    }

    if (remote.alarms && remote.alarms.length) {
      var icon = document.createElement('i');
      icon.className = 'gaia-icon icon-calendar-alarm calendar-text-color';
      el.appendChild(icon);
      el.classList.add('has-alarms');
    }

    return el;
  },

  _renderAlldayEvent: function(record) {
    var el = this._buildEventElement(record);
    el.classList.add('is-allday');
    this._alldayEvents.appendChild(el);
  },

  destroy: function() {
    this.oninactive();
    this._detach();
  },

  _detach: function() {
    if (this._attached) {
      this._daysHolder.removeChild(this.day);
      this._alldaysHolder.removeChild(this.allday);
      this._attached = false;
    }
  },

  oninactive: function() {
    if (!this._isActive) {
      return;
    }
    dayObserver.off(this.date, this._render);
    window.removeEventListener('localized', this);
    this._isActive = false;
  }
};

});

define('views/multi_day',['require','exports','module','calc','./current_time','templates/date_span','./hour_double_tap','./pan','./single_day','timespan','view','calc','utils/mout'],function(require, exports, module) {


var Calc = require('calc');
var CurrentTime = require('./current_time');
var DateSpan = require('templates/date_span');
var HourDoubleTap = require('./hour_double_tap');
var Pan = require('./pan');
var SingleDay = require('./single_day');
var Timespan = require('timespan');
var View = require('view');
var createDay = require('calc').createDay;
var throttle = require('utils/mout').throttle;

function MultiDay(opts) {
  this.app = opts.app;
  this.timeController = opts.app.timeController;
  this.children = [];
  this._render = throttle(this._render, 200);
}
module.exports = MultiDay;

MultiDay.prototype = {

  // override these properties on child classes to change the behavior!
  scale: 'week',
  visibleCells: 5,
  element: null,
  _hourFormat: 'hour-format',
  _addAmPmClass: false,

  childClass: SingleDay,
  children: null,
  seen: false,
  _baseDate: null,
  _hourHeight: 0,
  _prevRange: null,
  _visibleRange: null,

  set baseDate(date) {
    // it's very important that base date doesn't hold hour info otherwise we
    // could create duplicate days (because range wouldn't contain datetime)
    this._baseDate = createDay(date);
  },

  get baseDate() {
    return this._baseDate;
  },

  get daysHolder() {
    return this.element.querySelector('.md__days');
  },

  get alldaysHolder() {
    return this.element.querySelector('.md__alldays');
  },

  get main() {
    return this.element.querySelector('.md__main');
  },

  get mainContent() {
    return this.element.querySelector('.md__main-content');
  },

  get sidebar() {
    return this.element.querySelector('.md__sidebar');
  },

  onactive: function() {
    this.element.classList.add(View.ACTIVE);

    if (!this.seen) {
      this.onfirstseen();
      this.seen = true;
    }

    var controller = this.timeController;
    controller.scale = this.scale;
    controller.moveToMostRecentDay();

    var previousBaseDate = this.baseDate;
    this.baseDate = this._calcBaseDate(controller.position);
    this._render();
    // Do not scroll when come back from any other screen.
    if (!(previousBaseDate &&
          Calc.isSameDate(previousBaseDate, this.baseDate))) {
      this._resetScroll();
      this._scrollToHour();
    }

    // add listeners afterwards to avoid calling render twice
    controller.on('dayChange', this);
  },

  _calcBaseDate: function(date) {
    // this is overwritten by week view, and only called during onactivate
    return date;
  },

  onfirstseen: function() {
    this._setupPan();
    this._setupHours();
    this._setupCurrentTime();
    this._setupDoubleTap();
    // we keep the localized listener even when view is inactive to avoid
    // rebuilding the hours/dates every time we switch between views
    window.addEventListener('localized', this);
  },

  _setupPan: function() {
    var containerWidth = this.daysHolder.parentNode.offsetWidth;
    this._pan = new Pan({
      gridSize: Math.round(containerWidth / this.visibleCells),
      visibleCells: this.visibleCells,
      eventTarget: this.element,
      overflows: [
        this.main
      ],
      targets: [
        this.alldaysHolder,
        this.daysHolder
      ]
    });
    this._pan.setup();
    this._pan.on('start', () => this._hourDoubleTap.removeAddEventLink());
    this._pan.on('release', obj => this._updateBaseDateAfterScroll(obj.diff));
  },

  _setupHours: function() {
    var sidebar = this.sidebar;
    // we need to remove all children because when locale change we rebuild
    // the hours (we can't use data-l10n-id because of special format)
    sidebar.innerHTML = '';
    var hour, i = -1;
    while (++i < 24) {
      hour = this._createHour(i);
      sidebar.appendChild(hour);
    }
    this._hourHeight = hour.offsetHeight;
  },

  _createHour: function(hour) {
    var el = document.createElement('li');
    el.className = 'md__hour md__hour-' + hour;
    el.innerHTML = DateSpan.hour.render({
      hour: hour,
      format: this._hourFormat,
      addAmPmClass: this._addAmPmClass,
      className: 'md__display-hour'
    });
    return el;
  },

  _setupCurrentTime: function() {
    this._currentTime = new CurrentTime({
      container: this.element.querySelector('.md__main-content'),
      sticky: this.alldaysHolder
    });
  },

  _setupDoubleTap: function() {
    this._hourDoubleTap = new HourDoubleTap({
      app: this.app,
      main: this.main,
      daysHolder: this.daysHolder,
      alldaysHolder: this.alldaysHolder,
      hourHeight: this._hourHeight
    });
    this._hourDoubleTap.setup();
  },

  handleEvent: function(e) {
    switch (e.type) {
      case 'dayChange':
        this._onDayChange(e.data[0]);
        break;
      case 'localized':
        this._localize();
        break;
    }
  },

  _onDayChange: function(date) {
    // _render() updates the _visibleRange, so we need to check it first
    var containedToday = this._visibleRange.contains(new Date());
    this.baseDate = date;
    this._render();
    if (!containedToday) {
      this._scrollToHour({ onlyToday: true });
    }
  },

  _localize: function() {
    this._setupHours();
    this._refreshCurrentTime();
  },

  _updateBaseDateAfterScroll: function(diff) {
    var day = createDayDiff(this.baseDate, diff);
    this.timeController.move(day);
    this.timeController.selectedDay = day;
  },

  _render: function() {
    var currentRange = this._getRange();
    this._removeDatesOutsideRange(currentRange);

    // very important to re-activate child views in case we change views
    // without moving to a different date
    this.children.forEach(child => child.onactive());

    this._addDatesInsideRange(currentRange);

    this._prevRange = currentRange;
    this._visibleRange = this._getVisibleRange();
    this._sortDays();
    this._pan.refresh();
    this._refreshCurrentTime();
  },

  _refreshCurrentTime: function() {
    this._currentTime.timespan = this._visibleRange;
    this._currentTime.refresh();
  },

  _removeDatesOutsideRange: function(range) {
    if (this.children.length) {
      this.children = this.children.filter(child => {
        if (range.contains(child.date)) {
          return true;
        }
        child.destroy();
        return false;
      });
    }
  },

  _addDatesInsideRange: function(range) {
    this._getPendingDates(range)
      .forEach(date => {
        var day = new this.childClass({
          date: date,
          daysHolder: this.daysHolder,
          alldaysHolder: this.alldaysHolder,
          hourHeight: this._hourHeight
        });
        day.setup();
        this.children.push(day);
      });
  },

  _getPendingDates: function(range) {
    var dates = Calc.daysBetween(range);
    if (this._prevRange) {
      dates = dates.filter(date => {
        return !this._prevRange.contains(date);
      });
    }
    return dates;
  },

  _sortDays: function() {
    // decided to use float and reappend the elements in the right order
    // since using position:absolute or css transforms felt "slower"
    // (we have a reflow anyway since we might add new elements to the DOM)
    this.children
      .sort((a, b) => a.date - b.date)
      .forEach(day => day.append());
  },

  _getRange: function() {
    return new Timespan(
      createDayDiff(this.baseDate, -this.visibleCells),
      createDayDiff(this.baseDate, (this.visibleCells * 2) - 1)
    );
  },

  _getVisibleRange: function() {
    return new Timespan(
      this.baseDate,
      createDayDiff(this.baseDate, this.visibleCells)
    );
  },

  _resetScroll: function() {
    this.main.scrollTop = 0;
  },

  _scrollToHour: function(options) {
    var now = new Date();
    var hour;

    if (this._visibleRange.contains(now)) {
      hour = Math.max(now.getHours() - 1, 0);
    } else if (!options || !options.onlyToday) {
      hour = 8;
    }

    if (hour != null) {
      this._animatedScroll(hour * this._hourHeight);
    }
  },

  _animatedScroll: function(scrollTop) {
    var container = this.main;
    var maxScroll = container.scrollHeight - container.clientHeight;

    scrollTop = Math.min(scrollTop, maxScroll);

    var content = this.mainContent;
    var destination = container.scrollTop - scrollTop;
    var seconds = Math.abs(destination) / 500;

    container.style.overflowY = 'hidden';

    window.requestAnimationFrame(() => {
      content.style.transform = 'translateY(' + destination + 'px)';
      // easeOutQuart borrowed from http://matthewlein.com/ceaser/
      content.style.transition = 'transform ' + seconds + 's ' +
        'cubic-bezier(0.165, 0.840, 0.440, 1.000)';
    });

    content.addEventListener('transitionend', function setScrollTop() {
      content.removeEventListener('transitionend', setScrollTop);
      content.style.transform = '';
      content.style.transition = '';
      container.scrollTop = scrollTop;
      container.style.overflowY = 'scroll';
    });
  },

  oninactive: function() {
    this.element.classList.remove(View.ACTIVE);
    this.timeController.removeEventListener('dayChange', this);
    this.children.forEach(child => child.oninactive());
  }
};

function createDayDiff(date, diff) {
  return createDay(date, date.getDate() + diff);
}

});

/* exported LazyLoader */
/* globals HtmlImports, Promise */


/**
 * This contains a simple LazyLoader implementation
 * To use:
 *
 *   LazyLoader.load(
 *    ['/path/to/file.js', '/path/to/file.css', 'domNode'], callback
 *   );
 */
var LazyLoader = (function() {

  function LazyLoader() {
    this._loaded = {};
    this._isLoading = {};
  }

  LazyLoader.prototype = {

    _js: function(file, callback) {
      var script = document.createElement('script');
      script.src = file;
      // until bug 916255 lands async is the default so
      // we must disable it so scripts load in the order they where
      // required.
      script.async = false;
      script.addEventListener('load', callback);
      document.head.appendChild(script);
      this._isLoading[file] = script;
    },

    _css: function(file, callback) {
      var style = document.createElement('link');
      style.type = 'text/css';
      style.rel = 'stylesheet';
      style.href = file;
      document.head.appendChild(style);
      callback();
    },

    _html: function(domNode, callback) {

      // The next few lines are for loading html imports in DEBUG mode
      if (domNode.getAttribute('is')) {
        this.load(['/shared/js/html_imports.js'], function() {
          HtmlImports.populate(callback);
        }.bind(this));
        return;
      }

      for (var i = 0; i < domNode.childNodes.length; i++) {
        if (domNode.childNodes[i].nodeType == document.COMMENT_NODE) {
          domNode.innerHTML = domNode.childNodes[i].nodeValue;
          break;
        }
      }

      window.dispatchEvent(new CustomEvent('lazyload', {
        detail: domNode
      }));

      callback();
    },

    /**
     * Retrieves content of JSON file.
     *
     * @param {String} file Path to JSON file
     * @return {Promise} A promise that resolves to the JSON content
     * or null in case of invalid path. Rejects if an error occurs.
     */
    getJSON: function(file) {
      return new Promise(function(resolve, reject) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', file, true);
        xhr.responseType = 'json';

        xhr.onerror = function(error) {
          reject(error);
        };
        xhr.onload = function() {
          if (xhr.response !== null) {
            resolve(xhr.response);
          } else {
            reject(new Error('No valid JSON object was found (' + 
			     xhr.status + ' ' + xhr.statusText + ')'));
          }
        };

        xhr.send();
      });
    },

    load: function(files, callback) {
      var deferred = {};
      deferred.promise = new Promise(resolve => {
        deferred.resolve = resolve;
      });

      if (!Array.isArray(files)) {
        files = [files];
      }

      var loadsRemaining = files.length, self = this;
      function perFileCallback(file) {
        if (self._isLoading[file]) {
          delete self._isLoading[file];
        }
        self._loaded[file] = true;

        if (--loadsRemaining === 0) {
          deferred.resolve();
          if (callback) {
            callback();
          }
        }
      }

      for (var i = 0; i < files.length; i++) {
        var file = files[i];

        if (this._loaded[file.id || file]) {
          perFileCallback(file);
        } else if (this._isLoading[file]) {
          this._isLoading[file].addEventListener(
            'load', perFileCallback.bind(null, file));
        } else {
          var method, idx;
          if (typeof file === 'string') {
            method = file.match(/\.([^.]+)$/)[1];
            idx = file;
          } else {
            method = 'html';
            idx = file.id;
          }

          this['_' + method](file, perFileCallback.bind(null, idx));
        }
      }

      return deferred.promise;
    }
  };

  return new LazyLoader();
}());

define("shared/lazy_loader", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.LazyLoader;
    };
}(this)));

define('dom',['require','exports','module','shared/lazy_loader'],function(require, exports) {


var LazyLoader = require('shared/lazy_loader');

exports.load = function(id, require, onLoad, config) {
  if (config.isBuild) {
    return onLoad();
  }

  var node = document.getElementById(id);
  if (!node) {
    onLoad.error('can\'t find element with id #' + id);
    return;
  }

  LazyLoader.load(node, function() {
    onLoad(node);
  });
};

});

define('views/day',['require','exports','module','./multi_day','dom!day-view'],function(require, exports, module) {


var MultiDay = require('./multi_day');

require('dom!day-view');

function DayView(opts) {
  MultiDay.apply(this, arguments);
}
module.exports = DayView;

DayView.prototype = {
  __proto__: MultiDay.prototype,

  scale: 'day',
  visibleCells: 1,

  get element() {
    return document.getElementById('day-view');
  }
};

});
