
define('view',['require','exports','module'],function(require, exports, module) {


var DEFAULT_ERROR_ID = 'error-default';
const INVALID_CSS = /([^a-zA-Z\-\_0-9])/g;

/**
 * Very simple base class for views.
 * Provides functionality for active/inactive.
 *
 * The first time the view is activated
 * the onactive function/event will fire.
 *
 * The .seen property is added to each object
 * with view in its prototype. .seen can be used
 * to detect if the view has ever been activated.
 *
 * @param {String|Object} options options or a selector for element.
 */
function View(options) {
  if (typeof(options) === 'undefined') {
    options = {};
  }

  if (typeof(options) === 'string') {
    this.selectors = { element: options };
  } else {
    var key;

    if (typeof(options) === 'undefined') {
      options = {};
    }

    for (key in options) {
      if (options.hasOwnProperty(key)) {
        this[key] = options[key];
      }
    }
  }

  this.hideErrors = this.hideErrors.bind(this);
}
module.exports = View;

View.ACTIVE = 'active';

View.prototype = {
  seen: false,
  activeClass: View.ACTIVE,
  errorVisible: false,

  get element() {
    return this._findElement('element');
  },

  get status() {
    return this._findElement('status');
  },

  get errors() {
    return this._findElement('errors');
  },

  /**
   * Creates a string id for a given model.
   *
   *    view.idForModel('foo-', { _id: 1 }); // => foo-1
   *    view.idForModel('foo-', '2'); // => foo-2
   *
   * @param {String} prefix of string.
   * @param {Object|String|Numeric} objectOrString representation of model.
   */
  idForModel: function(prefix, objectOrString) {
    prefix += (typeof(objectOrString) === 'object') ?
      objectOrString._id :
      objectOrString;

    return prefix;
  },

  calendarId: function(input) {
    if (typeof(input) !== 'string') {
      input = input.calendarId;
    }

    input = this.cssClean(input);
    return 'calendar-id-' + input;
  },

  /**
   * Delegate pattern event listener.
   *
   * @param {HTMLElement} element parent element.
   * @param {String} type type of dom event.
   * @param {String} selector css selector element should match
   *                          _note_ there is no magic here this
   *                          is determined from the root of the document.
   * @param {Function|Object} handler event handler.
   *                                  first argument is the raw
   *                                  event second is the element
   *                                  matching the pattern.
   */
  delegate: function(element, type, selector, handler) {
    if (typeof(handler) === 'object') {
      var context = handler;
      handler = function() {
        context.handleEvent.apply(context, arguments);
      };
    }

    element.addEventListener(type, function(e) {
      var target = e.target;
      while (target !== element) {
        if ('mozMatchesSelector' in target &&
            target.mozMatchesSelector(selector)) {
          return handler(e, target);
        }
        target = target.parentNode;
      }
    });
  },

  /**
   * Clean a string for use with css.
   * Converts illegal chars to legal ones.
   */
  cssClean: function(string) {
    if (typeof(string) !== 'string') {
      return string;
    }

    //TODO: I am worried about the performance
    //of using this all over the place =/
    //consider sanitizing all keys to ensure
    //they don't blow up when used as a selector?
    return string.replace(INVALID_CSS, '-');
  },

  /**
   * Finds a caches a element defined
   * by selectors
   *
   * @param {String} selector name as defined in selectors.
   * @param {Boolean} all true when to find all elements. (default false).
   */
  _findElement: function(name, all, element) {
    if (typeof(all) === 'object') {
      element = all;
      all = false;
    }

    element = element || document;

    var cacheName;
    var selector;

    if (typeof(all) === 'undefined') {
      all = false;
    }

    if (name in this.selectors) {
      cacheName = '_' + name + 'Element';
      selector = this.selectors[name];

      if (!this[cacheName]) {
        if (all) {
          this[cacheName] = element.querySelectorAll(selector);
        } else {
          this[cacheName] = element.querySelector(selector);
        }
      }

      return this[cacheName];
    }

    return null;
  },

 /**
   * Displays a list of errors
   *
   * @param {Array} list error list
   *  (see Event.validaitonErrors) or Error object.
   */
  showErrors: function(list) {
    var _ = navigator.mozL10n.get;
    var errors = '';

    // We can pass Error objects or
    // Array of {name: foo} objects
    if (!Array.isArray(list)) {
        list = [list];
    }

    var i = 0;
    var len = list.length;

    for (; i < len; i++) {
      var name = list[i].l10nID || list[i].name;
      errors += _('error-' + name) || _(DEFAULT_ERROR_ID);
    }

    // populate error and display it.
    this.errors.textContent = errors;
    this.errorVisible = true;
    this.status.classList.add(this.activeClass);

    this.status.addEventListener('animationend', this.hideErrors);
  },

  hideErrors: function() {
    this.status.classList.remove(this.activeClass);
    this.status.removeEventListener('animationend', this.hideErrors);
    this.errorVisible = false;
  },

  onactive: function() {
    if (this.errorVisible) {
      this.hideErrors();
    }

    // seen can be set to anything other than false to override this behaviour
    if (this.seen === false) {
      this.onfirstseen();
    }

    // intentionally using 'in'
    if ('dispatch' in this) {
      this.dispatch.apply(this, arguments);
    }

    this.seen = true;
    if (this.element) {
      this.element.classList.add(this.activeClass);
    }
  },

  oninactive: function() {
    if (this.element) {
      this.element.classList.remove(this.activeClass);
    }
  },

  onfirstseen: function() {}
};

});

define('views/calendar_colors',['require','exports','module','models/calendar','view','app'],function(require, exports, module) {


var Calendar = require('models/calendar');
var View = require('view');
var app = require('app');

function Colors() {
  this.colorMap = Object.create(null);
  this._ruleMap = Object.create(null);

  // create style sheet for us to play with
  var sheet = document.createElement('style');
  sheet.type = 'text/css';
  sheet.id = '_dynamic-calendar-styles';

  document.head.appendChild(sheet);

  // Make no assumption about the ordering of the style sheets
  for (var i = 0, len = document.styleSheets.length; i < len; i++) {
    if (document.styleSheets[i].ownerNode.id === sheet.id) {
      this._styles = document.styleSheets[i];
    }
  }
  this._initEvents();
}
module.exports = Colors;

Colors.prototype = {
  __proto__: View.prototype,

  _initEvents: function() {
    // handle changes in calendar
    var store = app.store('Calendar');
    store.on('persist', this);
    store.on('remove', this);
  },

  handleEvent: function(e) {
    switch (e.type) {
      case 'persist':
        // 1 is the model
        this.updateRule(e.data[1]);
        break;
      case 'remove':
        // 0 is an id of a model
        this.removeRule(e.data[0]);
        break;
    }
  },

  render: function() {
    var calendarStore = app.store('Calendar');

    calendarStore.all(function(err, calendars) {
      if (err) {
        console.error('Error fetch calendars in CalendarColors');
      }

      var id;
      for (id in calendars) {
        this.updateRule(calendars[id]);
      }

      if (this.onrender) {
        this.onrender();
      }

    }.bind(this));

  },

  /**
   * Returns prefixed calendar id used
   * in css classes, etc...
   *
   * Uses _id field when given a model
   *
   * @param {Calendar.Models.Calendar|String} item model or string.
   */
  getId: function(item) {
    var id;
    if (item instanceof Calendar) {
      id = item._id;
    } else {
      id = item;
    }
    return this.calendarId(String(id));
  },

  /**
   * associates a color with a
   * calendar/calendar id with a color.
   *
   * @param {Calendar.Model.Calendar} calendar model.
   */
  updateRule: function(calendar) {
    var id = this.getId(calendar);
    var method = id in this.colorMap ? '_updateRules' : '_createRules';
    this[method](calendar, id, calendar.color);
  },

  _updateRules: function(calendar, id, color) {
    var map = this._ruleMap[id];

    var bgStyle = map.background.style;
    var borderStyle = map.border.style;
    var textStyle = map.text.style;

    bgStyle.backgroundColor = this._hexToBackgroundColor(color);
    borderStyle.borderColor = color;
    textStyle.color = color;
  },

  _createRules: function(calendar, id, color) {
    // We need to store the ids of created rules for deletion later on.
    var map = this._ruleMap[id] = {
      ruleIds: []
    };
    this.colorMap[id] = color;

    // calendar coloring
    var bg = 'background-color: ' + this._hexToBackgroundColor(color) + ';';
    map.background = this._insertRule(id, 'calendar-bg-color', bg);

    var border = 'border-color: ' + color + ';';
    map.border = this._insertRule(id, 'calendar-border-color', border);

    var textColor = 'color: ' + color + ';';
    map.text = this._insertRule(id, 'calendar-text-color', textColor);
  },

  _hexToBackgroundColor: function(hex) {
    // we need 20% opacity for background; it's simpler to use rgba than to
    // create a new layer and set opacity:20%
    var rgb = this._hexToChannels(hex);
    return 'rgba(' +
      rgb.r + ',' +
      rgb.g + ',' +
      rgb.b + ', 0.2)';
  },

  _hexToChannels: function(hex) {
    hex = hex.replace(/#/, '');
    // we slice in case calendar hex also includes opacity (#rrggbbaa)
    hex = hex.slice(0, 6);
    if (hex.length === 3) {
      // expand "abc" into "aabbcc"
      hex = hex.replace(/(\w)/g, '$1$1');
    }
    var val = parseInt(hex, 16);
    return {
      r: val >> 16,
      g: val >> 8 & 0xFF,
      b: val & 0xFF
    };
  },

  _insertRule: function(calendarId, className, body) {
    // affects root element and child elements as well
    var block = ('.%calId.%className, .%calId .%className { %body }')
      .replace(/%calId/g, calendarId)
      .replace(/%className/g, className)
      .replace(/%body/g, body);

    // increment index for rule...
    var ruleId = this._styles.cssRules.length;
    this._styles.insertRule(block, ruleId);

    // store the rule into cache
    var map = this._ruleMap[calendarId];
    map.ruleIds.push(ruleId);

    return this._styles.cssRules[ruleId];
  },

  /**
   * CSSRuleList is a zero based array like
   * object. When we remove an item from
   * the rule list we need to adjust the saved indexes
   * of all values above the one we are removing.
   */
  _adjustRuleIds: function(above) {
    var key;

    function subtract(item) {

      if (item > above) {
        return item - 1;
      }
      return item;
    }

    for (key in this._ruleMap) {
      var map = this._ruleMap[key];
      map.ruleIds = map.ruleIds.map(subtract);
    }
  },

  /**
   * Removes color rule associated
   * with given id.
   */
  removeRule: function(id) {
    var map;

    var idOffset = 0;

    id = this.getId(id);
    if (id in this.colorMap) {
      delete this.colorMap[id];
      map = this._ruleMap[id];
      delete this._ruleMap[id];

      map.ruleIds.forEach(function(idx) {
        this._adjustRuleIds(idx);
        idx = idx - idOffset;

        // XXX: this is probably a slow way to do this
        // but given this will only happen when a calendar
        // is removed it should be fine.
        this._styles.deleteRule(idx);
        idOffset += 1;
      }, this);
    }
  }
};

});
